﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace ContentsUploader.Assistants
{
    using static Constants;
    using static Models.Pms;
    using static Models.Smt;
    using static Models.Shogun;
    using static WebAccessor.Method;

    // eShop 関連操作
    public class ShopHelper
    {
        private const ContentMetaType InvalidType = (ContentMetaType)(-1);

        private Setting Setting { get; }
        private bool IsRedirection { get; }

        public ShopHelper(Setting setting, bool isRedirection = true)
        {
            Setting = setting;
            IsRedirection = isRedirection;
        }

        //  contentMetaId -> nsUids 変換
        public bool ConvertToNsUids(out List<string> nsUids, Id64 contentMetaId, string type)
        {
            nsUids = null;

            var web = new WebAccessor(Setting);
            var query = $"?country=JP&lang=ja&shop_id=4&select_first=true&title_ids={contentMetaId}&type={type}";
            var uri = new Uri($"{Setting.Api.ShogunV1}/contents/ids{query}");
            var output = string.Empty;
            if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
            {
                return false;
            }

            var info = ToolUtility.Deserialize<IdPairsInfo>(output);
            nsUids = info.GetNsUids();
            return true;
        }

        //  contentMetaId -> nsUid 変換
        public bool ConvertToNsUid(out string nsUid, Id64 contentMetaId, string type)
        {
            nsUid = string.Empty;

            List<string> nsUids = null;
            if (!ConvertToNsUids(out nsUids, contentMetaId, type))
            {
                return false;
            }
            if (nsUids != null && nsUids.Count > 0)
            {
                nsUid = nsUids[0];
            }
            return true;
        }

        // Demo 情報取得
        public bool GetDemoInfo(out DemoInfo demoInfo, string nsUid)
        {
            demoInfo = null;

            var web = new WebAccessor(Setting);
            var uri = new Uri($"{Setting.Api.PmsRestV2}/demos/{nsUid}");
            var output = string.Empty;
            if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
            {
                return false;
            }

            demoInfo = ToolUtility.Deserialize<DemoInfo>(output);
            return true;
        }

        // Aoc 情報取得
        public bool GetAocInfo(out AocInfo aocInfo, string nsUid)
        {
            aocInfo = null;

            var web = new WebAccessor(Setting);
            var uri = new Uri($"{Setting.Api.PmsRestV2}/aocs/{nsUid}");
            var output = string.Empty;
            if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
            {
                return false;
            }

            aocInfo = ToolUtility.Deserialize<AocInfo>(output);
            return true;
        }

        public bool GetBundleInfo(out BundleInfo bundleInfo, string nsUid)
        {
            bundleInfo = null;

            var web = new WebAccessor(Setting);
            var uri = new Uri($"{Setting.Api.PmsRestV2}/bundles/{nsUid}");
            var output = string.Empty;
            if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
            {
                return false;
            }

            bundleInfo = ToolUtility.Deserialize<BundleInfo>(output);
            return true;
        }

        // Aoc nsUid -> contentMetaIds 変換
        public bool ConvertToAocMetaIds(out List<Id64> contentMetaIds, string nsUid)
        {
            contentMetaIds = new List<Id64>();

            AocInfo aocInfo = null;
            if (!GetAocInfo(out aocInfo, nsUid))
            {
                return false;
            }
            if (aocInfo.roms != null && aocInfo.roms.Count > 0)
            {
                foreach (var rom in aocInfo.roms)
                {
                    var contentMetaId = new Id64(rom);
                    contentMetaIds.Add(contentMetaId);
                }
            }
            return true;
        }

        // Aoc nsUid -> contentMetaId 変換
        public bool ConvertToAocMetaId(out Id64 contentMetaId, string nsUid)
        {
            contentMetaId = Id64.Invalid;

            List<Id64> contentMetaIds = null;
            if (!ConvertToAocMetaIds(out contentMetaIds, nsUid))
            {
                return false;
            }
            if (contentMetaIds != null && contentMetaIds.Count > 0)
            {
                contentMetaId = contentMetaIds[0];
            }
            return true;
        }

        // Title 情報取得
        public bool GetTitleInfo(out TitleInfo titleInfo, string nsUid)
        {
            titleInfo = null;

            var web = new WebAccessor(Setting);
            var uri = new Uri($"{Setting.Api.PmsRestV2}/titles/{nsUid}");
            string output = string.Empty;
            if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
            {
                return false;
            }

            titleInfo = ToolUtility.Deserialize<TitleInfo>(output);
            return true;
        }

        // Title 情報にある nsUid のマップ情報を作成（Title 自身の nsUid は含まない）
        // 引数 aocNsUids は、既に判明してる Aoc の nsUid のマップ情報を渡す
        public bool CreateTitleNsUidMap(out NsUidMap allNsUids, TitleInfo titleInfo, NsUidMap aocNsUids)
        {
            allNsUids = new NsUidMap(aocNsUids);
            if (titleInfo.aocs != null && titleInfo.aocs.Count > 0)
            {
                foreach (var aoc in titleInfo.aocs)
                {
                    if (aoc.ids != null && aoc.ids.Count > 0)
                    {
                        foreach (var id in aoc.ids)
                        {
                            var nsUid = id.id.ToString();

                            var contentMetaId = allNsUids.FindKey(nsUid);
                            if (contentMetaId.IsValid)
                            {
                                // 既に nsUid が取得済みなら以降の処理をスキップ
                                continue;
                            }

                            if (ConvertToAocMetaId(out contentMetaId, nsUid))
                            {
                                if (contentMetaId.IsValid)
                                {
                                    // 重複する場合は先に取得した nsUid を優先
                                    if (!allNsUids.ContainsKey(contentMetaId))
                                    {
                                        allNsUids.Add(contentMetaId, nsUid);
                                    }
                                    else
                                    {
                                        Log.WriteLine($"Warning: Duplicated id. Aoc({contentMetaId}) has mulitiple ids of \"{allNsUids[contentMetaId]}\" and \"{nsUid}\".");
                                    }
                                }
                            }
                        }
                    }
                }
            }
            return true;
        }

        public bool RegisterRomByDirectory(out RegisterRomResult result, string dirPath, bool isUpdateNotificationEnabled = true, bool isForcedUpdateEnabled = true)
        {
            result = new RegisterRomResult();

            var nsps = NspAccessor.ReadDirectory(dirPath, false);
            foreach (var nsp in nsps)
            {
                var local = new RegisterRomResult();
                if (!RegisterRomByFile(out local, nsp, isUpdateNotificationEnabled, isForcedUpdateEnabled))
                {
                    return false;
                }
                if (!result.Merge(local))
                {
                    WriteLog($"Error: Different application id nsps are in the input directory");
                    return false;
                }
            }

            if (!result.Validate())
            {
                WriteLog($"Error: Unexpected directory. Please check the path \"{dirPath}\".");
                return false;
            }
            return true;
        }

        public bool RegisterRomByFile(out RegisterRomResult result, string filePath, bool isUpdateNotificationEnabled = true, bool isForcedUpdateEnabled = true)
        {
            var nsp = new NspAccessor(filePath, false);
            return RegisterRomByFile(out result, nsp, isUpdateNotificationEnabled, isForcedUpdateEnabled);
        }

        public bool RegisterRomByFile(out RegisterRomResult result, NspAccessor nsp, bool isUpdateNotificationEnabled = true, bool isForcedUpdateEnabled = true)
        {
            result = new RegisterRomResult(nsp.ApplicationId);

            if (nsp.IsUnregistrable())
            {
                WriteLog($"Error: Unregistrable file.");
                return false;
            }

            foreach (var content in nsp.NspInnerContent)
            {
                if (!RegisterRomByContentMeta(content.ContentMetaId, content.Type, isUpdateNotificationEnabled, isForcedUpdateEnabled))
                {
                    return false;
                }
                if (!result.Add(content))
                {
                    WriteLog($"Error: Unsupport content type.");
                    return false;
                }
            }

            if (!result.Validate())
            {
                WriteLog($"Error: Unexpected file. Please check the path \"{nsp.Path}\".");
                return false;
            }
            return true;
        }

        public bool RegisterRomByContentMeta(Id64 contentMetaId, ContentMetaType contentMetaType, bool isUpdateNotificationEnabled = true, bool isForcedUpdateEnabled = true)
        {
            if (Id64.Unregistrable.Contains(contentMetaId))
            {
                WriteLog($"Error: Unregistrable id.");
                return false;
            }

            // Rom 作成
            var web = new WebAccessor(Setting);
            {
                var uri = new Uri($"{Setting.Api.PmsRestV2}/roms");
                var entity = $"{{\"title_id\":\"{contentMetaId}\",\"type\":\"{contentMetaType}\"}}";
                var output = string.Empty;
                if (!web.HttpRequest(out output, POST, uri, entity, Account.DebugUser))
                {
                    // 登録済み時は注釈を通知して続行
                    var message = "1008";
                    if (!output.Contains(message))
                    {
                        return false;
                    }
                    WriteLog($"Note: Ignore responce, if server return \"{message}\". Process is continued.");
                }
            }

            // Rom 更新
            {
                // Rom 取得
                var uri = new Uri($"{Setting.Api.PmsRestV2}/roms/{contentMetaId}");
                var output = string.Empty;
                if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
                {
                    return false;
                }

                // 承認依頼で残っている時は承認
                var info = ToolUtility.Deserialize<StatusInfo>(output);
                if (info.status == "RequestApproval")
                {
                    // ステータス変更（→承認）
                    if (!OnceApprove($"{Setting.Api.PmsRestV2}/roms/{contentMetaId}/status"))
                    {
                        return false;
                    }

                    // Rom 再取得
                    if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
                    {
                        return false;
                    }
                    info = ToolUtility.Deserialize<StatusInfo>(output);
                }

                // ステータスのリセット、更新フラグを更新
                if (info.status == "Approved" || info.status == "Compilation")
                {
                    var entity = output;
                    entity = entity.Replace("\"status\":\"Approved\",", string.Empty);
                    entity = entity.Replace("\"status\":\"Compilation\",", string.Empty);
                    entity = ToolUtility.ReplaceJsonValue(entity, "update_notice_flag", isUpdateNotificationEnabled);
                    entity = ToolUtility.ReplaceJsonValue(entity, "force_update_flag", isForcedUpdateEnabled);

                    if (!web.HttpRequest(PUT, uri, entity, Account.DebugUser))
                    {
                        return false;
                    }
                }
            }

            // ステータス変更（→承認依頼→承認）
            return Approve($"{Setting.Api.PmsRestV2}/roms/{contentMetaId}/status");
        }

        public bool ConfirmRomByDirectory(out RomCollection roms, string dirPath)
        {
            roms = new RomCollection();

            var nsps = NspAccessor.ReadDirectory(dirPath, false);
            foreach (var nsp in nsps)
            {
                var local = new RomCollection();
                if (!ConfirmRomByFile(out local, nsp))
                {
                    return false;
                }
                roms.Add(local);
            }

            if (!roms.Distinct() || roms.IsEmpty())
            {
                WriteLog($"Error: Unexpected directory. Please check the path \"{dirPath}\".");
                return false;
            }
            return true;
        }

        public bool ConfirmRomByFile(out RomCollection roms, string filePath)
        {
            var nsp = new NspAccessor(filePath, false);
            return ConfirmRomByFile(out roms, nsp);
        }

        public bool ConfirmRomByFile(out RomCollection roms, NspAccessor nsp)
        {
            roms = new RomCollection();

            if (nsp.IsUnregistrable())
            {
                WriteLog($"Error: Unregistrable file.");
                return false;
            }

            foreach (var content in nsp.NspInnerContent)
            {
                if (!ConfirmRomByContentMeta(content.ContentMetaId, content.Type))
                {
                    return false;
                }
                if (!roms.Add(content))
                {
                    WriteLog($"Error: Unsupport content type.");
                    return false;
                }
            }

            if (!roms.Distinct() || roms.IsEmpty())
            {
                WriteLog($"Error: Unexpected file. Please check the path \"{nsp.Path}\".");
                return false;
            }
            return true;
        }

        public bool ConfirmRomByContentMeta(Id64 contentMetaId, ContentMetaType contentMetaType)
        {
            ContentMetaType result;
            if (!ConfirmRomByContentMeta(out result, contentMetaId))
            {
                return false;
            }
            if (result != contentMetaType)
            {
                WriteLog($"Error: Different content type.");
                return false;
            }
            return true;
        }

        public bool ConfirmRomByContentMeta(out ContentMetaType contentMetaType, Id64 contentMetaId)
        {
            contentMetaType = InvalidType;

            if (Id64.Unregistrable.Contains(contentMetaId))
            {
                WriteLog($"Error: Unregistrable id.");
                return false;
            }

            // Rom 取得
            var web = new WebAccessor(Setting);
            var uri = new Uri($"{Setting.Api.PmsRestV2}/roms/{contentMetaId}");
            var output = string.Empty;
            if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
            {
                return false;
            }

            // 承認済みであることを確認
            var info = ToolUtility.Deserialize<ConfirmInfo>(output);
            if (info.status != "Approved")
            {
                WriteLog($"Error: Unapproved rom {contentMetaId}.");
                return false;
            }

            // コンテンツタイプを取得
            contentMetaType = ToolUtility.ToEnumValue<ContentMetaType>(info.type, InvalidType);
            if (contentMetaType == InvalidType)
            {
                WriteLog($"Error: Unsupport content type.");
                return false;
            }
            return true;
        }

        public bool ConfrimContents(out NsUidMap nsUids, RomCollection roms)
        {
            nsUids = new NsUidMap();

            foreach (var contentMetaId in roms.Apps)
            {
                string nsUid;
                if (!ConfirmTitle(out nsUid, contentMetaId))
                {
                    return false;
                }
                nsUids.Add(contentMetaId, nsUid);
            }
            foreach (var contentMetaId in roms.Aocs)
            {
                string nsUid;
                if (!ConfrimAoc(out nsUid, contentMetaId))
                {
                    return false;
                }
                nsUids.Add(contentMetaId, nsUid);
            }
            return true;
        }

        public bool ConfrimAoc(out string nsUid, Id64 contentMetaId)
        {
            // nsUid 取得
            if (!ConvertToNsUid(out nsUid, contentMetaId, "aoc"))
            {
                return false;
            }
            if (string.IsNullOrEmpty(nsUid))
            {
                WriteLog($"Error: Not found aoc {contentMetaId}.");
                return false;
            }

            // Aoc 情報取得
            AocInfo info = null;
            if (!GetAocInfo(out info, nsUid))
            {
                return false;
            }
            if (info.status != "Approved")
            {
                WriteLog($"Error: Unapproved aoc {contentMetaId}.");
                return false;
            }
            return true;
        }

        public bool ConfirmTitle(out string nsUid, Id64 contentMetaId)
        {
            // nsUid 取得
            if (!ConvertToNsUid(out nsUid, contentMetaId, "title"))
            {
                return false;
            }
            if (string.IsNullOrEmpty(nsUid))
            {
                WriteLog($"Error: Not found title {contentMetaId}.");
                return false;
            }

            // Title 情報取得
            TitleInfo info = null;
            if (!GetTitleInfo(out info, nsUid))
            {
                return false;
            }
            if (info.status != "Approved")
            {
                WriteLog($"Error: Unapproved title {contentMetaId}.");
                return false;
            }
            return true;
        }

        public bool RegisterContent(out string nsUid, string apiType, string json)
        {
            nsUid = string.Empty;

            var web = new WebAccessor(Setting);
            var uri = new Uri($"{Setting.Api.PmsRestV2}/{apiType}");
            var output = string.Empty;
            if (!web.HttpRequest(out output, POST, uri, json, Account.DebugUser))
            {
                if (output.Contains("1002"))
                {
                    WriteLog($"Error: Your item code already be used. Please use another --initial-code.");
                }
                return false;
            }
            var info = ToolUtility.Deserialize<NsUidInfo>(output);
            nsUid = info.id.ToString();

            // ステータス変更（→承認依頼→承認）
            return Approve($"{Setting.Api.PmsRestV2}/{apiType}/{nsUid}/status");
        }

        public bool UpdateContent(string apiType, string json, string nsUid)
        {
            var web = new WebAccessor(Setting);
            var uri = new Uri($"{Setting.Api.PmsRestV2}/{apiType}/{nsUid}");
            var output = string.Empty;
            if (!web.HttpRequest(out output, PUT, uri, json, Account.DebugUser))
            {
                if (output.Contains("1002"))
                {
                    WriteLog($"Error: Your item code already be used. Please use another --initial-code.");
                }
                return false;
            }

            // ステータス変更（→承認依頼→承認）
            return Approve($"{Setting.Api.PmsRestV2}/{apiType}/{nsUid}/status");
        }

        public bool GetPrice(out PriceInfo priceInfo, string apiType, string nsUid)
        {
            priceInfo = null;

            var web = new WebAccessor(Setting);
            var smt = $"{Setting.Api.SmtRs}/{apiType}/onlineprice";
            var token = Setting.Token.Contents.accessToken;
            for (var retry = 0; retry < ApprovePriceRetryCountMax; retry++)
            {
                var output = string.Empty;
                {
                    var uri = new Uri($"{smt}/get?nsUid={nsUid}&dataSource=qa&token={token}&locale=ja");
                    if (!web.HttpRequest(out output, GET, uri))
                    {
                        return false;
                    }
                }
                priceInfo = ToolUtility.Deserialize<PriceInfo>(output);

                // 承認依頼中の価格がなければ成功扱い
                if (!priceInfo.IsAwaitingApproval())
                {
                    return true;
                }

                // 承認依頼中の価格を承認し再取得を行う
                foreach (var onlinePrice in priceInfo.onlinePrices)
                {
                    if (onlinePrice.IsAwaitingApproval())
                    {
                        var uri = new Uri($"{smt}/approve?nsUid={onlinePrice.nsUid}&token={token}&locale=ja");
                        if (!web.HttpRequest(POST, uri))
                        {
                            return false;
                        }
                    }
                }
            }
            return false;
        }

        public bool RegisterPrice(string apiType, PriceInfo priceInfo)
        {
            var web = new WebAccessor(Setting);
            var smt = $"{Setting.Api.SmtRs}/{apiType}/onlineprice";
            var token = Setting.Token.Contents.accessToken;
            {
                var uri = new Uri($"{smt}/update?token={token}&locale=en");
                var entity = ToolUtility.Serialize<PriceInfo>(priceInfo);
                var output = string.Empty;
                if (!web.HttpRequest(out output, POST, uri, entity))
                {
                    var message = "Cannot delete online price after Start Date";
                    if (output.Contains(message))
                    {
                        WriteLog($"Note: Ignore responce, if server return \"{message}\". Process is continued.");
                        return true;
                    }
                    return false;
                }
            }

            foreach (var onlinePrice in priceInfo.onlinePrices)
            {
                {
                    var uri = new Uri($"{smt}/requestApproval?nsUid={onlinePrice.nsUid}&token={token}&locale=ja");
                    if (!web.HttpRequest(POST, uri))
                    {
                        return false;
                    }
                }
                {
                    var uri = new Uri($"{smt}/approve?nsUid={onlinePrice.nsUid}&token={token}&locale=ja");
                    if (!web.HttpRequest(POST, uri))
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        public bool UpdateDisableDownloadFlag(Id64 contentMetaId, int contentMetaVersion, bool isDownloadDisabled)
        {
            var web = new WebAccessor(Setting);
            var output = string.Empty;

            // Rom 状態の取得
            var uri = new Uri($"{Setting.Api.PmsRestV2}/roms/{contentMetaId}");
            if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
            {
                return false;
            }

            var info = ToolUtility.Deserialize<RomInfo>(output);
            foreach (var version in info.versions)
            {
                if (version.version == contentMetaVersion)
                {
                    version.disableDownloadFlag = isDownloadDisabled;
                }
            }

            var entity = ToolUtility.Serialize<RomInfo>(info);
            if (!web.HttpRequestMergePatchJson(uri, entity, Account.DebugUser))
            {
                return false;
            }

            // ステータス変更（→承認依頼→承認）
            return Approve($"{Setting.Api.PmsRestV2}/roms/{contentMetaId}/status");
        }

        private bool Approve(string uri)
        {
            return Approve(new Uri(uri));
        }

        private bool Approve(Uri uri)
        {
            var web = new WebAccessor(Setting);

            // 更新：承認依頼
            {
                var entity = "{\"status\":\"RequestApproval\"}";
                if (!web.HttpRequest(PUT, uri, entity, Account.DebugUser))
                {
                    return false;
                }
            }

            // 更新：承認
            {
                var entity = "{\"status\":\"Approved\"}";
                if (!web.HttpRequest(PUT, uri, entity, Account.DebugUser))
                {
                    return false;
                }
            }
            return true;
        }

        private bool OnceApprove(string uri)
        {
            return OnceApprove(new Uri(uri));
        }

        private bool OnceApprove(Uri uri)
        {
            var web = new WebAccessor(Setting);

            // 更新：承認
            {
                var entity = "{\"status\":\"Approved\"}";
                if (!web.HttpRequest(PUT, uri, entity, Account.DebugUser))
                {
                    return false;
                }
            }
            return true;
        }

        public void WriteLog(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLine(log);
            }
        }

        public void WriteLogAsIs(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLineAsIs(log);
            }
        }
    }
}
