﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Xml;
using CommandUtility;

namespace ContentsUploader.Assistants
{
    using static WebAccessor.Method;

    public class RopsExecutor
    {
        private class Result : Tuple<bool, string, string>
        {
            public bool IsSuccess { get { return Item1; } }
            public bool IsFailure { get { return !IsSuccess; } }
            public string StandardOutput { get { return Item2; } }
            public string StandardError { get { return Item3; } }

            public Result(bool result, string stdout, string stderr) : base(result, stdout, stderr) { }
        }

        private Setting Setting { get; }
        public bool IsRedirection { get; private set; } // ExecuteRops 内の標準出力リダイレクトの有効化

        public RopsExecutor(Setting setting, bool redirection = true)
        {
            Setting = setting;
            IsRedirection = redirection;
        }

        private Tuple<StringBuilder, StringBuilder> SetupProcess(Process process, string filename, string arguments)
        {
            var stdout = new StringBuilder();
            var stderr = new StringBuilder();

            process.StartInfo.FileName = filename;
            process.StartInfo.Arguments = arguments;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.StandardOutputEncoding = System.Text.Encoding.UTF8;
            process.OutputDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                stdout.AppendLine(e.Data);
            };
            process.ErrorDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                stderr.AppendLine(e.Data);
            };

            return new Tuple<StringBuilder, StringBuilder>(stdout, stderr);
        }

        private void RunAndWaitProcess(Process process)
        {
            if (Setting.IsVerbose)
            {
                WriteLog($"Command: {process.StartInfo.FileName} {process.StartInfo.Arguments}");
            }

            process.Start();
            process.BeginOutputReadLine();
            process.BeginErrorReadLine();
            process.WaitForExit();
        }

        private Result RopsCommand(string arguments)
        {
            using (var process = new Process())
            {
                var filename = SdkPath.FindToolPath("rops", "rops/rops.exe", "rops/rops.exe").FullName;
                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                var result = new Result(process.ExitCode == 0, outputs.Item1.ToString(), outputs.Item2.ToString());
                if (result.IsFailure)
                {
                    WriteLog($"Standard Output:");
                    WriteLogAsIs(result.StandardOutput.Trim());
                    WriteLog($"Standard Error:");
                    WriteLogAsIs(result.StandardError.Trim());
                }
                else if (Setting.IsVerbose)
                {
                    WriteLog($"Standard Output:");
                    WriteLogAsIs(result.StandardOutput.Trim());
                }
                return result;
            }
        }

        private string ToArguments(string command, string arguments, params string[] options)
        {
            // 接頭辞
            var sb = new System.Text.StringBuilder();
            if (!string.IsNullOrEmpty(Setting.Proxy))
            {
                sb.Append($"--proxy {Setting.Proxy} ");
            }
            sb.Append("--insecure ");

            // コマンド
            sb.Append($"{command} ");

            // 環境
            sb.Append($"-e {Setting.Environment.Name} ");

            // 引数
            if (!string.IsNullOrEmpty(arguments))
            {
                sb.Append($"{arguments} ");
            }

            // トークン
            sb.Append($"--token \"{Setting.Token.Path}\" ");

            // オプション
            foreach (var option in options)
            {
                if (!string.IsNullOrEmpty(option))
                {
                    sb.Append($"{option} ");
                }
            }
            return sb.ToString().Trim();
        }

        public bool Version()
        {
            WriteLog($"ExecuteRopsVersion start.");
            var arguments = $"--version";
            var result = RopsCommand(arguments);
            WriteLog($"Version: {result.StandardOutput.Trim()}");

            var success = "rops";
            return result.StandardOutput.Contains(success);
        }

        public void RejectAndPending(Id64 contentMetaId)
        {
            var web = new WebAccessor(Setting, IsRedirection);
            var osiris = $"{Setting.Api.OsirisV1}/titles/{contentMetaId}";

            // 以降の処理で、HttpRequest 失敗でも処理を継続しているのは元々の実装である
            // HttpRequest 失敗時に return すると影響がでそうなので、そのままにしておく
            var uri = new Uri($"{osiris}/approval/reject");
            var output = string.Empty;
            web.HttpRequest(out output, PATCH, uri);

            var message = "000-0209";
            if (output.Contains(message))
            {
                WriteLog($"Note: Ignore responce, if server return \"{message}\". Process is continued.");
            }

            web.HttpRequest(PATCH, new Uri(osiris));
        }

        public bool Upload(out string romId, Id64 nspId, string nspPath, bool isChecking = false, int pollingTimeout = -1)
        {
            romId = string.Empty;

            WriteLog($"ExecuteRopsUpload start.");
            var checking = isChecking ? string.Empty : "--skip-checking";
            var polling = pollingTimeout <= 0 ? string.Empty : $"--polling-timeout {pollingTimeout}";
            var arguments = ToArguments("upload", $"-a {nspId} -s {nspPath}", checking, polling);
            var result = RopsCommand(arguments);

            var stdout = result.StandardOutput;
            var success = "Upload has been completed successfully. ROM ID = ";
            if (stdout.Contains(success))
            {
                int idSize = 32;
                romId = stdout.Substring(stdout.IndexOf(success) + success.Length, idSize);
                WriteLog($"ROM ID = {romId}");

                var completed = "ROM upload workflow has been completed!";
                if (string.IsNullOrEmpty(polling) || stdout.Contains(completed))
                {
                    return true;
                }
            }
            return false;
        }

        public bool Submit(out bool isRetryable, string romId, int aocArchiveNo, bool isMastering, bool isChecking, int pollingTimeout = -1)
        {
            WriteLog($"ExecuteRopsSubmit start.");
            using (var holder = new TemporaryFileHolder("ContentsUploader"))
            {
                var dir = holder.CreateTemporaryDirectory("submit");
                var path = System.IO.Path.Combine(dir.FullName, "setting.xml");
                CreateSubmitRequestEntityFile(path, aocArchiveNo, isMastering);

                var checking = isChecking ? string.Empty : "--skip-checking";
                var polling = pollingTimeout <= 0 ? string.Empty : $"--polling-timeout {pollingTimeout}";
                var arguments = ToArguments("submit", $"-i \"{path}\" -r {romId}", checking, polling);
                var result = RopsCommand(arguments);

                isRetryable = false;
                foreach (var error in new string[] {
                    "java.net.SocketException",                                     // プロトコルエラー
                    "java.net.SocketTimeoutException",                              // タイムアウト
                    "code = 409, response = {\"error\":{\"code\":\"000-0509\"" })   // サーバが busy 状態
                {
                    if (result.StandardError.Contains(error))
                    {
                        isRetryable = true;
                        break;
                    }
                }

                var success = "Submit has been completed successfully.";
                if (result.StandardOutput.Contains(success))
                {
                    var completed = "ROM submit workflow has been completed!";
                    if (string.IsNullOrEmpty(polling) || result.StandardOutput.Contains(completed))
                    {
                        return true;
                    }
                }
                return false;
            }
        }

        public bool ApproveRom(string romId, bool isAccept, int pollingTimeout = -1)
        {
            var operation = isAccept ? "accept" : "reject";
            WriteLog($"ExecuteRopsApproveRom {operation} start.");
            var polling = pollingTimeout <= 0 ? string.Empty : $"--polling-timeout {pollingTimeout}";
            var arguments = ToArguments("approve-rom", $"-r {romId} --operation {operation}", polling);
            var result = RopsCommand(arguments);

            var success = "Approve ROM has been completed successfully.";
            if (result.StandardOutput.Contains(success))
            {
                var completed = "ROM approval workflow has been completed!";
                if (string.IsNullOrEmpty(polling) || result.StandardOutput.Contains(completed))
                {
                    return true;
                }
            }

            // status が遷移先の APPROVED/REJECTED で、accept/reject 出来ない場合は成功と見なす
            var message = $"ROM status is not approvable";
            var status = isAccept ? $"Status = APPROVED" : $"Status = REJECTED";
            if (result.StandardError.Contains(message) && result.StandardError.Contains(status))
            {
                WriteLog($"Note: Ignore responce, if server return \"{message}\". Process is continued.");
                return true;
            }
            return false;
        }

        public bool RevokeRom(string romId, int pollingTimeout = -1)
        {
            WriteLog($"ExecuteRopsRevokeRom start.");
            var polling = pollingTimeout <= 0 ? string.Empty : $"--polling-timeout {pollingTimeout}";
            var arguments = ToArguments("approve-rom", $"-r {romId} --operation revoke", polling);
            var result = RopsCommand(arguments);

            var success = "Approve ROM has been completed successfully.";
            if (result.StandardOutput.Contains(success))
            {
                var completed = "ROM approval workflow has been completed!";
                if (string.IsNullOrEmpty(polling) || result.StandardOutput.Contains(completed))
                {
                    return true;
                }
            }

            // status が遷移先の CHECKING で、revoke 出来ない場合は成功と見なす
            var message = $"ROM status is not approvable";
            var status = $"Status = CHECKING";
            if (result.StandardError.Contains(message) && result.StandardError.Contains(status))
            {
                WriteLog($"Note: Ignore responce, if server return \"{message}\". Process is continued.");
                return true;
            }
            return false;
        }

        public bool Delivery(Id64 nspId, int nspVersion)
        {
            WriteLog($"ExecuteRopsDelivery start.");
            using (var holder = new TemporaryFileHolder("ContentsUploader"))
            {
                var dir = holder.CreateTemporaryDirectory("delivery");
                var path = System.IO.Path.Combine(dir.FullName, "setting.xml");
                CreateDeliveryInfoEntityFile(path);

                var arguments = ToArguments("delivery", $"-i \"{path}\" -t {nspId} -v {nspVersion}");
                var result = RopsCommand(arguments);

                var success = "Update content delivery infos has been completed successfully.";
                return result.StandardOutput.Contains(success);
            }
        }

        public bool ApproveTitle(Id64 nspId)
        {
            WriteLog($"ExecuteRopsApproveTitle start.");
            var arguments = ToArguments("approve-title", $"-t {nspId}");
            var result = RopsCommand(arguments);

            var success = "Approve title has been completed successfully.";
            return result.StandardOutput.Contains(success);
        }

        public bool GetRomStatus(out string status, string romId)
        {
            WriteLog($"ExecuteRopsIsRomStatus start.");
            var arguments = ToArguments("list-roms", $"-r {romId}");
            var result = RopsCommand(arguments);

            // 下記エラーは失敗とはせずに仮の status を返してリトライを促す
            foreach (var error in new string[] {
                "Gateway Timeout",
                "java.net.SocketException",
                "java.net.SocketTimeoutException" })
            {
                if (result.StandardError.Contains(error))
                {
                    WriteLog($"Note: \"{error}\" occurred. Retry get rom status.");
                    status = error;
                    return true;
                }
            }

            if (ToolUtility.GetTargetJsonValue(out status, result.StandardOutput, "status"))
            {
                WriteLog($"status = {status}");
            }

            var success = "\"roms\"";
            return result.StandardOutput.Contains(success);
        }

        public bool IsRomRequestDone(out bool failed, string romId)
        {
            failed = false;

            WriteLog($"ExecuteRopsIsRomRequestCompleted start.");
            var arguments = ToArguments("list-rom-requests", $"-r {romId}");
            var result = RopsCommand(arguments);

            // request status の遷移
            //  PENDING -> PROCESSING -> COMPLETED or FAILED
            var stdout = result.StandardOutput;
            if (stdout.Contains("\"status\": \"PENDING\"") || stdout.Contains("\"status\": \"PROCESSING\""))
            {
                return false; // request 処理中
            }
            failed = result.IsFailure || stdout.Contains("\"status\": \"FAILED\"");
            return true; // request 終了
        }

        public void WriteLog(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLine(log);
            }
        }

        public void WriteLogAsIs(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLineAsIs(log);
            }
        }

        private void CreateSubmitRequestEntityFile(string filePath, int aocArchiveNo, bool masteringEnabled)
        {
            XmlDocument xml = new XmlDocument();
            XmlDeclaration declaration = xml.CreateXmlDeclaration(@"1.0", @"UTF-8", @"yes");
            XmlElement RequestEntity = xml.CreateElement("submit_request_entity");

            XmlElement RequestId = xml.CreateElement("lotcheck_request_id");
            RequestId.InnerText = "00000000000000000000000000000000";
            RequestEntity.AppendChild(RequestId);

            XmlElement SubmitVersion = xml.CreateElement("submit_version");
            SubmitVersion.InnerText = "0";
            RequestEntity.AppendChild(SubmitVersion);

            XmlElement ReleaseVersion = xml.CreateElement("release_version");
            ReleaseVersion.InnerText = aocArchiveNo.ToString();
            RequestEntity.AppendChild(ReleaseVersion);

            XmlElement ProductCode = xml.CreateElement("product_code");
            ProductCode.InnerText = "XXX-Y-ZZZZZ";
            RequestEntity.AppendChild(ProductCode);

            XmlElement MasteringEnabled = xml.CreateElement("mastering_enabled");
            MasteringEnabled.InnerText = masteringEnabled.ToString();
            RequestEntity.AppendChild(MasteringEnabled);

            XmlElement Instructions = xml.CreateElement("instructions");
            XmlElement Instruction = xml.CreateElement("instruction");
            XmlElement Type = xml.CreateElement("type");
            Type.InnerText = "digital";
            Instruction.AppendChild(Type);
            Instructions.AppendChild(Instruction);
            RequestEntity.AppendChild(Instructions);

            xml.AppendChild(declaration);
            xml.AppendChild(RequestEntity);
            xml.Save(filePath);

            if (Setting.IsVerbose)
            {
                var xmlData = File.ReadAllText(filePath);
                WriteLog($"Submit Request: {filePath}");
                WriteLogAsIs(xmlData);
            }
        }

        private void CreateDeliveryInfoEntityFile(string filePath)
        {
            XmlDocument xml = new XmlDocument();
            XmlDeclaration declaration = xml.CreateXmlDeclaration(@"1.0", @"UTF-8", @"yes");
            XmlElement UpdateTitleDeliveryInfoEntity = xml.CreateElement("update_title_delivery_info_entity");
            XmlElement DeliveryEntities = xml.CreateElement("delivery_entities");
            UpdateTitleDeliveryInfoEntity.AppendChild(DeliveryEntities);

            XmlElement ControlDeliveryEntity = xml.CreateElement("delivery_entity");
            XmlElement RomDeliveryEntity = xml.CreateElement("delivery_entity");
            DeliveryEntities.AppendChild(ControlDeliveryEntity);
            DeliveryEntities.AppendChild(RomDeliveryEntity);

            XmlElement ControlContentType = xml.CreateElement("content_type");
            XmlElement ControlDeliveryDate = xml.CreateElement("delivery_date");
            XmlElement ControlIsDownloadable = xml.CreateElement("is_downloadable");
            ControlContentType.InnerText = "control";
            ControlDeliveryDate.InnerText = "2016-11-07 16:00:00";
            ControlIsDownloadable.InnerText = "true";
            ControlDeliveryEntity.AppendChild(ControlContentType);
            ControlDeliveryEntity.AppendChild(ControlDeliveryDate);
            ControlDeliveryEntity.AppendChild(ControlIsDownloadable);

            XmlElement RomContentType = xml.CreateElement("content_type");
            XmlElement RomDeliveryDate = xml.CreateElement("delivery_date");
            XmlElement RomIsDownloadable = xml.CreateElement("is_downloadable");
            RomContentType.InnerText = "rom";
            RomDeliveryDate.InnerText = "2016-11-11 14:00:00";
            RomIsDownloadable.InnerText = "true";
            RomDeliveryEntity.AppendChild(RomContentType);
            RomDeliveryEntity.AppendChild(RomDeliveryDate);
            RomDeliveryEntity.AppendChild(RomIsDownloadable);

            xml.AppendChild(declaration);
            xml.AppendChild(UpdateTitleDeliveryInfoEntity);
            xml.Save(filePath);

            if (Setting.IsVerbose)
            {
                var xmlData = File.ReadAllText(filePath);
                WriteLog($"Delivery Request: {filePath}");
                WriteLogAsIs(xmlData);
            }
        }
    }
}
