﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;

namespace ContentsUploader.Assistants
{
    using static Constants;
    using static Models.Osiris;
    using static WebAccessor.Method;

    // D4C 関連操作
    public class D4cHelper
    {
        private Setting Setting { get; }
        private bool IsRedirection { get; }

        public D4cHelper(Setting setting, bool isRedirection = true)
        {
            Setting = setting;
            IsRedirection = isRedirection;
        }

        public bool ListRomIdByApplicationId(out string responce, Id64 applicationId, int offset, int limit, string status)
        {
            var web = new WebAccessor(Setting.Proxy, false);
            var highway = $"{Setting.Api.OsirisV1}/roms";
            var request = "?application_id=" + applicationId + "&offset=" + offset + "&limit=" + limit;
            if (!string.IsNullOrEmpty(status))
            {
                request += "&status=" + status;
            }
            Uri uri = new Uri(highway + request);
            return web.HttpRequest(out responce, GET, uri);
        }

        private bool ListRomIdByApplicationId(out RomSetEntity responce, Id64 applicationId, string status)
        {
            responce = new RomSetEntity();

            int limit = 30;
            int offset = 0;
            while (true)
            {
                string output = string.Empty;
                if (!ListRomIdByApplicationId(out output, applicationId, offset, limit, status))
                {
                    WriteLog($"Error: Failed to list rom ids.");
                    return false;
                }
                if (!MergeEntity(responce, ToolUtility.Deserialize<RomSetEntity>(output)))
                {
                    WriteLog($"Error: Failed to list rom ids. List was updated during getting.");
                    return false;
                }
                if (responce.IsComplete)
                {
                    break;
                }

                // 重複部分を作って連続性の一致確認する
                offset += (limit - 1);
            }
            return true;
        }

        private bool ListRomIdByApplicationId(out List<RomTitleInfo> responce, Id64 applicationId, string status)
        {
            responce = new List<RomTitleInfo>();

            RomSetEntity result = null;
            if (!ListRomIdByApplicationId(out result, applicationId, status))
            {
                return false;
            }
            foreach (var rom in result.roms)
            {
                // Rom Id から Title Id を取得
                string output = string.Empty;
                if (!ListRomIdToTitleId(out output, rom.romId))
                {
                    WriteLog($"Error: Failed to list rom titles.");
                    return false;
                }
                RomTitlePairSetEntity title = ToolUtility.Deserialize<RomTitlePairSetEntity>(output);
                if (title.romTitles.Count > 0)
                {
                    // 同じ Rom に対して複数のタイトルやバージョンが登録されていることは考えられないが
                    // RomTitlePairSetEntity.romTitles の最も新しい更新日のものを採用する
                    title.romTitles.Sort(RomTitlePairEntity.CompareByUpdateDateDescending);
                    responce.Add(new RomTitleInfo(rom, title.romTitles[0]));
                }
                else
                {
                    // Rom に対してのタイトル情報が存在しない
                    WriteLog($"Warning: Invalid rom title.");
                    responce.Add(new RomTitleInfo(rom, RomTitlePairEntity.Invalid));
                }
            }
            return true;
        }

        private bool ListRomIdByContentMetaId(out string responce, Id64 contentMetaId, int offset, int limit, int[] version, string status)
        {
            var web = new WebAccessor(Setting.Proxy, false);
            var highway = $"{Setting.Api.OsirisV1}/rom_titles";
            var request = "?title_id=" + contentMetaId + "&offset=" + offset + "&limit=" + limit;
            if (version != null && version.Length > 0)
            {
                // ※注意
                // 下記指定で取得すると指定バージョンのみになる
                // 指定バージョン以上や以下の取り方はできない
                request += "&title_version=" + string.Join(",", version);
            }
            if (!string.IsNullOrEmpty(status))
            {
                request += "&status=" + status;
            }
            Uri uri = new Uri(highway + request);
            return web.HttpRequest(out responce, GET, uri);
        }

        private bool ListRomIdByContentMetaId(out RomTitleSetEntity responce, Id64 contentMetaId, int[] version, string status)
        {
            responce = new RomTitleSetEntity();

            int limit = 30;
            int offset = 0;
            while (true)
            {
                string output = string.Empty;
                if (!ListRomIdByContentMetaId(out output, contentMetaId, offset, limit, version, status))
                {
                    WriteLog($"Error: Failed to list rom titles.");
                    return false;
                }
                if (!MergeEntity(responce, ToolUtility.Deserialize<RomTitleSetEntity>(output)))
                {
                    WriteLog($"Error: Failed to list rom titles. List was updated during getting.");
                    return false;
                }
                if (responce.IsComplete)
                {
                    break;
                }

                // 重複部分を作って連続性の一致確認する
                offset += (limit - 1);
            }
            return true;
        }

        private bool ListRomIdByContentMetaId(out List<RomTitleInfo> responce, Id64 contentMetaId, int version, string status)
        {
            responce = new List<RomTitleInfo>();

            RomTitleSetEntity result = null;
            if (!ListRomIdByContentMetaId(out result, contentMetaId, null, status))
            {
                return false;
            }
            foreach (var rom in result.roms)
            {
                var info = new RomTitleInfo(rom);
                if (info.contentMetaVersion >= version)
                {
                    responce.Add(new RomTitleInfo(rom));
                }
            }
            return true;
        }

        public bool ListRomIdByContentMetaId(out string responce, Id64 contentMetaId, int version, string status)
        {
            // この関数では、対象 contentMetaId の全 rom を取得することはできない
            // 全ての rom を取得するには、上で定義されている別 ListRomIdByContentMetaId を利用する必要がある
            // ただし、上で定義している関数は日付順で取得できないので、直近の rom を絞るのに手間がかかる
            // 下記処理では、applicationId に紐付く romId 取得の際に、直近の rom のみに絞っている
            responce = string.Empty;
            var web = new WebAccessor(Setting.Proxy, false);

            // contentMetaId が紐付く applicationId を取得
            Id64 applicationId;
            {
                var uri = new Uri($"{Setting.Api.OsirisV1}/titles/{contentMetaId}");
                var output = string.Empty;
                if (!web.HttpRequest(out output, GET, uri))
                {
                    WriteLog($"Error: Failed to get application id of content id.");
                    return false;
                }
                var entity = ToolUtility.Deserialize<TitleDetailsEntity>(output);
                applicationId = new Id64(entity.applicationId);
            }

            // applicationId に紐付く romId を取得（status でフィルタリング）
            List<RomDetailsEntity> roms;
            {
                var output = string.Empty;
                if (!ListRomIdByApplicationId(out output, applicationId, 0, 30, status))
                {
                    WriteLog($"Error: Failed to get rom id list of application id.");
                    return false;
                }
                var entity = ToolUtility.Deserialize<RomSetEntity>(output);
                roms = entity.roms;
            }

            // 各 romId の titleId を検索
            var targets = new List<RomDetailsEntity>();
            foreach (var rom in roms)
            {
                var output = string.Empty;
                ListRomIdToTitleId(out output, rom.romId); // romId から titleId を取得

                var entity = ToolUtility.Deserialize<RomTitlePairSetEntity>(output);
                foreach (var title in entity.romTitles)
                {
                    var titleId = new Id64(title.titleId);
                    if (titleId == contentMetaId && title.titleVersion >= version)
                    {
                        targets.Add(rom);
                    }
                }
            }

            // 仮の RomDetailsResponseEntity を作成してシリアライズ
            {
                var entity = new RomSetEntity(targets.Count, targets.Count, 0, targets);
                responce = ToolUtility.Serialize<RomSetEntity>(entity);
            }
            return true;
        }

        public bool ListRomIdDetail(out string responce, string romId)
        {
            var web = new WebAccessor(Setting.Proxy, false);
            var highway = $"{Setting.Api.OsirisV1}/roms";
            var request = "/" + romId;
            Uri uri = new Uri(highway + request);
            return web.HttpRequest(out responce, GET, uri);
        }

        public bool ListRomIdToTitleId(out string responce, string romId)
        {
            var web = new WebAccessor(Setting.Proxy, false);
            var highway = $"{Setting.Api.OsirisV1}/rom_titles";
            var request = "/" + romId;
            Uri uri = new Uri(highway + request);
            return web.HttpRequest(out responce, GET, uri);
        }

        public bool RevokeApplicationId(Id64 applicationId)
        {
            List<RomTitleInfo> list = null;
            if (!ListRomIdByApplicationId(out list, applicationId, "APPROVED,CHECKING"))
            {
                return false;
            }

            list.RemoveAll(rom => (rom.romType == RomTitleInfo.RomType.Combined));
            list.Sort(RomTitleInfo.CompareForRevokeRom);
            if (Setting.IsVerbose)
            {
                if (list.Count > 0)
                {
                    WriteLog($"Revoke Rom Order:");
                    foreach (var rom in list)
                    {
                        WriteLogAsIs(rom.ToString());
                    }
                }
                else
                {
                    WriteLog($"Revoke Rom Order: none");
                }
            }

            bool result = true;
            foreach (var rom in list)
            {
                result &= RevokeRomId(rom.romId, rom.status);
            }
            return result;
        }

        public bool RevokeContentMetaId(Id64 contentMetaId, int version)
        {
            List<RomTitleInfo> list = null;
            if (!ListRomIdByContentMetaId(out list, contentMetaId, version, "APPROVED,CHECKING"))
            {
                return false;
            }

            list.Sort(RomTitleInfo.CompareForRevokeRom);
            if (Setting.IsVerbose)
            {
                if (list.Count > 0)
                {
                    WriteLog($"Revoke Rom Order:");
                    foreach (var romInfo in list)
                    {
                        WriteLogAsIs(romInfo.ToString());
                    }
                }
                else
                {
                    WriteLog($"Revoke Rom Order: none");
                }
            }

            bool result = true;
            foreach (var rom in list)
            {
                result &= RevokeRomId(rom.romId, rom.status);
            }
            return result;
        }

        private bool RevokeRomId(string romId, string status)
        {
            int timeout = RevokeRomTimeoutMinutes;
            int retry = RevokeRomRetryCountMax;
            int interval = RevokeRomIntervalMilliseconds;
            var rops = new RopsExecutor(Setting, IsRedirection);

            //! APPROVED -> CHECKING
            if (status == "APPROVED")
            {
                if (!ToolUtility.RetryUntilSuccess(() => rops.RevokeRom(romId, timeout), retry, interval))
                {
                    return false;
                }
            }

            //! CHECKING -> REJECTED
            return ToolUtility.RetryUntilSuccess(() => rops.ApproveRom(romId, false, timeout), retry, interval);
        }

        public bool RevokeRomId(string romId)
        {
            string output = string.Empty;
            if (!ListRomIdDetail(out output, romId))
            {
                WriteLog($"Error: Failed to list rom details.");
                return false;
            }

            var rom = ToolUtility.Deserialize<RomDetailsEntity>(output);
            if (Setting.IsVerbose)
            {
                WriteLogAsIs(rom.ToString());
            }

            bool result = true;
            if (rom.status == "APPROVED" || rom.status == "CHECKING")
            {
                result &= RevokeRomId(rom.romId, rom.status);
            }

            return result;
        }

        public bool ListVersionByContentMeta(Id64 contentMetaId)
        {
            var web = new WebAccessor(Setting.Proxy, true);

            // リージョンごとの通知_配信バージョンの配信設定を取得する
            {
                var uri = new Uri($"{Setting.Api.SuperflyV1}/titles/{contentMetaId}/versions");
                var output = string.Empty;
                if (!web.HttpRequest(out output, GET, uri, Account.DebugUser))
                {
                    return false;
                }
            }
            return true;
        }

        public bool RegisterVersionByContentMeta(Id64 applicationId, Id64 contentMetaId, ContentMetaType contentMetaType, int deliveryVersion, int notifyVersion)
        {
            var web = new WebAccessor(Setting);

            // 指定された Title ID のバージョン情報を全て削除する
            {
                var uri = new Uri($"{Setting.Api.SuperflyV1}/titles/{contentMetaId}/delete");
                if (!web.HttpRequest(PATCH, uri, Account.DebugUser))
                {
                    return false;
                }
            }

            // version の登録
            {
                var uri = new Uri($"{Setting.Api.SuperflyV1}/titles/{contentMetaId}/versions/qa");
                var entity = "{\"versions\": [{\"delivery_version\":" + deliveryVersion +
                    ", \"end_date\":\"\", \"notify_version\":" + notifyVersion + ", \"start_date\":\"2016-10-01T00:00:00Z\"}]}";
                if (!web.HttpRequest(PUT, uri, entity, Account.DebugUser))
                {
                    return false;
                }
            }

            // QA -> Live
            {
                var uri = new Uri($"{Setting.Api.SuperflyV1}/titles/{contentMetaId}/versions/live");
                if (!web.HttpRequest(PATCH, uri, Account.DebugUser))
                {
                    return false;
                }
            }

            // タイトル ID とアプリケーション ID の紐付け
            {
                var uri = new Uri($"{Setting.Api.SuperflyV1}/titles/{contentMetaId}/title_info/qa");
                var entity = "{\"application_id\":\"" + applicationId + "\", \"title_type\":\"" + contentMetaType + "\"}";
                var output = string.Empty;
                if (!web.HttpRequest(out output, PUT, uri, entity, Account.DebugUser))
                {
                    // 000-1206 レスポンスの時は注釈を通知して続行
                    var message = "000-1206";
                    if (!output.Contains(message))
                    {
                        return false;
                    }
                    WriteLog($"Note: Ignore responce, if server return \"{message}\". Process is continued.");
                }
            }

            // QA -> Live
            {
                var uri = new Uri($"{Setting.Api.SuperflyV1}/titles/{contentMetaId}/versions/live");
                if (!web.HttpRequest(PATCH, uri, Account.DebugUser))
                {
                    return false;
                }
            }
            return true;
        }

        public bool DeleteVersionByContentMeta(Id64 contentMetaId)
        {
            var web = new WebAccessor(Setting);

            // 指定された Title ID のバージョン情報を全て削除する
            {
                var uri = new Uri($"{Setting.Api.SuperflyV1}/titles/{contentMetaId}/delete");
                if (!web.HttpRequest(PATCH, uri, Account.DebugUser))
                {
                    return false;
                }
            }

            // QA -> Live
            {
                var uri = new Uri($"{Setting.Api.SuperflyV1}/titles/{contentMetaId}/versions/live");
                if (!web.HttpRequest(PATCH, uri, Account.DebugUser))
                {
                    return false;
                }
            }
            return true;
        }

        public void WriteLog(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLine(log);
            }
        }

        public void WriteLogAsIs(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLineAsIs(log);
            }
        }
    }
}
