﻿using System;
using System.Text.RegularExpressions;

namespace CommandUtility
{
    public class ByteUnitExpression
    {
        public long Bytes { get; set; }

        public long KibiBytes
        {
            get
            {
                return Bytes / KibiByteFactor;
            }
            set
            {
                Bytes = value * KibiByteFactor;
            }
        }

        public long MebiBytes
        {
            get
            {
                return Bytes / MebiByteFactor;
            }
            set
            {
                Bytes = value * MebiByteFactor;
            }
        }

        public long GibiBytes
        {
            get
            {
                return Bytes / GibiByteFactor;
            }
            set
            {
                Bytes = value * GibiByteFactor;
            }
        }

        public long TebiBytes
        {
            get
            {
                return Bytes / TebiByteFactor;
            }
            set
            {
                Bytes = value * TebiByteFactor;
            }
        }

        public override string ToString()
        {
            if (TebiBytes * TebiByteFactor == Bytes)
            {
                return string.Format("{0} TiB", TebiBytes);
            }
            else if (GibiBytes * GibiByteFactor == Bytes)
            {
                return string.Format("{0} GiB", GibiBytes);
            }
            else if (MebiBytes * MebiByteFactor == Bytes)
            {
                return string.Format("{0} MiB", MebiBytes);
            }
            else if (KibiBytes * KibiByteFactor == Bytes)
            {
                return string.Format("{0} KiB", KibiBytes);
            }

            return string.Format("{0} B", Bytes);
        }

        public static ByteUnitExpression Parse(string expression)
        {
            var matchObject = RegexOfByteUnitExpression.Match(expression);

            if (matchObject.Success)
            {
                var value = long.Parse(matchObject.Groups[1].Captures[0].Value);
                var unitExpression = matchObject.Groups[2].Captures[0].Value;

                switch (unitExpression)
                {
                    case "B":
                        return new ByteUnitExpression { Bytes = value };

                    case "KiB":
                        return new ByteUnitExpression { KibiBytes = value };

                    case "MiB":
                        return new ByteUnitExpression { MebiBytes = value };

                    case "GiB":
                        return new ByteUnitExpression { GibiBytes = value };

                    case "TiB":
                        return new ByteUnitExpression { TebiBytes = value };

                    default:
                        throw new FormatException(string.Format("Not supported unit type: value = '{0}'", expression));
                }
            }
            else
            {
                throw new FormatException(string.Format("Invalid expression: value = '{0}'", expression));
            }
        }

        private static Regex RegexOfByteUnitExpression = new Regex("\\s*([0-9]+)\\s?(B|KiB|MiB|GiB|TiB)\\s*");

        public const long KibiByteFactor = 1024L;
        public const long MebiByteFactor = 1024L * 1024L;
        public const long GibiByteFactor = 1024L * 1024L * 1024L;
        public const long TebiByteFactor = 1024L * 1024L * 1024L * 1024L;
    }

}
