﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using System.Runtime.InteropServices;
using System.Runtime.Remoting.Metadata.W3cXsd2001;
using System.Text.RegularExpressions;

namespace CommandUtility
{
    public class BinaryUtility
    {
        public static void WriteBinary<StructType>(Stream outputStream, StructType value)
        {
            var size = Marshal.SizeOf(typeof(StructType));
            var buffer = new byte[size];
            var handle = GCHandle.Alloc(buffer, GCHandleType.Pinned);

            try
            {
                Marshal.StructureToPtr(value, handle.AddrOfPinnedObject(), false);
            }
            finally
            {
                handle.Free();
            }

            outputStream.Write(buffer, 0, size);
        }

        public static StructType ReadBinary<StructType>(Stream inputStream)
        {
            var size = Marshal.SizeOf(typeof(StructType));
            var buffer = new byte[size];
            var handle = GCHandle.Alloc(buffer, GCHandleType.Pinned);

            try
            {
                inputStream.Read(buffer, 0, size);
                return (StructType)Marshal.PtrToStructure(handle.AddrOfPinnedObject(), typeof(StructType));
            }
            finally
            {
                handle.Free();
            }
        }

        public static byte[] ReadBytes(Stream inputStream, int size)
        {
            var buffer = new byte[size];

            var readSize = inputStream.Read(buffer, 0, size);

            if(readSize != size)
            {
                throw new Exception($"Invalid data length. expected:{size} != actual{readSize}");
            }

            return buffer;
        }

        public static byte[] ToBinary<StructType>(StructType value)
        {
            using (var stream = new MemoryStream())
            {
                WriteBinary<StructType>(stream, value);

                return stream.ToArray();
            }
        }

        public static StructType FromBinary<StructType>(byte[] binaryValue)
        {
            using (var stream = new MemoryStream(binaryValue))
            {
                return ReadBinary<StructType>(stream);
            }
        }

        public static string ToHexString(byte[] data)
        {
            return new SoapHexBinary(data).ToString();
        }

        public static string ToPrettyHexString(byte[] binaryValue)
        {
            var a = binaryValue
                .Select(g => ToHexString(new byte[] {g}))
                .ToArray();
            var ret = string.Join(":", a);
            return ret;
        }

        public static byte[] FromHexString(string hexText)
        {
            if (!HexTextPattern.IsMatch(hexText))
            {
                throw new InvalidDataException(string.Format("invalid hex text: {0}", hexText));
            }

            return SoapHexBinary.Parse(hexText).Value;
        }

        public static readonly Regex ByteUnitExpressionPattern = new Regex("^([0-9]+)(|B|KB|MB)$");

        public static long ConvertFromByteUnitExpression(string exp)
        {
            var match = ByteUnitExpressionPattern.Match(exp);

            if (!match.Success)
            {
                throw new Exception($"Invalid bytes expression. input='{exp}'");
            }

            var unit = match.Groups[2].Value;
            var value = long.Parse(match.Groups[1].Value);

            if (unit == "" || unit == "B")
            {
                return value;
            }
            else if (unit == "KB")
            {
                return value * 1024;
            }
            else if (unit == "MB")
            {
                return value * 1024 * 1024;
            }

            throw new Exception($"Invalid byte unit format. input='{exp}'");
        }

        public static long AlignUp(long a, long align)
        {
            return ((a + align - 1) / align) * align;
        }

        public static bool IsAligned(long a, long align)
        {
            return (((a + align - 1) / align) * align) == a;
        }

        public static readonly Regex HexTextPattern = new Regex("^([0-9A-Fa-f]{2})*$");
    }
}
