﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace ElfLibrary
{
    using System;
    using System.Collections.Generic;

    /// <summary>
    /// 数値変換の関数群をまとめたクラス
    /// </summary>
    public static class NumericConvert
    {
        /// <summary>
        /// 整数の切り上げを行います。
        /// </summary>
        /// <param name="val">値</param>
        /// <param name="align">境界値</param>
        /// <returns>切り上げた結果</returns>
        public static ulong RoundUp(ulong val, ulong align)
        {
            if (align <= 1)
            {
                return val;
            }

            ulong n = val % align;
            if (n > 0)
            {
                return val + (align - n);
            }
            else
            {
                return val;
            }
        }

        /// <summary>
        /// 64-bit のデータのエンディアンを交換します。
        /// </summary>
        /// <param name="x">変換前の数値</param>
        /// <returns>変換後の数値</returns>
        public static ulong SwapEndian(ulong x)
        {
            return (ulong)(
                ((x >> 56) & 0x00000000000000ffUL) |
                ((x >> 40) & 0x000000000000ff00UL) |
                ((x >> 24) & 0x0000000000ff0000UL) |
                ((x >> 8)  & 0x00000000ff000000UL) |
                ((x << 8)  & 0x000000ff00000000UL) |
                ((x << 24) & 0x0000ff0000000000UL) |
                ((x << 40) & 0x00ff000000000000UL) |
                ((x << 56) & 0xff00000000000000UL));
        }

        /// <summary>
        /// 32-bit のデータのエンディアンを交換します。
        /// </summary>
        /// <param name="x">変換前の数値</param>
        /// <returns>変換後の数値</returns>
        public static uint SwapEndian(uint x)
        {
            return (uint)(
                ((x >> 24) & 0x000000ff) |
                ((x >> 8)  & 0x0000ff00) |
                ((x << 8)  & 0x00ff0000) |
                ((x << 24) & 0xff000000));
        }

        /// <summary>
        /// 16-bit のデータのエンディアンを交換します。
        /// </summary>
        /// <param name="x">変換前の数値</param>
        /// <returns>変換後の数値</returns>
        public static ushort SwapEndian(ushort x)
        {
            return (ushort)(
                ((x >> 8) & 0x00ff) |
                ((x << 8) & 0xff00));
        }

        /// <summary>
        /// Enumの型を変換します。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="value"></param>
        /// <returns></returns>
        public static T IntToElfDefine<T>(object value) where T : struct
        {
            T result;
            if (!Enum.TryParse<T>(Enum.GetName(typeof(T), value), out result))
            {
                throw new InvalidCastException();
            }
            return result;
        }
    }
}
