﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace ElfLibrary
{
    using System;
    using System.IO;

    /// <summary>
    /// ELFファイルの種類
    /// </summary>
    public enum ElfObjectFileType
    {
        None = 0, // No file type
        Rel = 1, // Relocatable File
        Exec = 2, // ExecutableFile
        Dyn = 3, // Shared Object File
        Core = 4, // Core file
        LopRoc = 0xff00, // Processor-specific
        HipRoc = 0x00ff, // Processor-specific
    }

    /// <summary>
    /// ELF ファイルフォーマットのバージョン
    /// </summary>
    public enum ElfVersionType
    {
        None = 0,
        Current = 1,
    }

    /// <summary>
    /// ELFヘッダクラス
    /// </summary>
    public class ElfHeader : IElfFile
    {
        public const int EiNIdEnt = 16;

        // e_ident
        public const byte ELFMAG0 = 0x7f;
        public const byte ELFMAG1 = (byte)'E';
        public const byte ELFMAG2 = (byte)'L';
        public const byte ELFMAG3 = (byte)'F';

        public byte[] Identification { get; set; }
        public ushort Type { get; set; }
        public ushort Machine { get; set; }
        public uint Version { get; set; }
        public ulong Entry { get; set; }
        public ulong ProgramOffset { get; set; }
        public ulong SectionOffset { get; set; }
        public uint Flags { get; set; }
        public ushort HeaderSize { get; set; }
        public ushort ProgramEntrySize { get; set; }
        public ushort ProgramNumber { get; set; }
        public ushort SectionEntrySize { get; set; }
        public ushort SectionNumber { get; set; }
        public ushort StringTableIndex { get; set; }

        /// <summary>
        /// ArchitectureType に沿った Elf ヘッダを作成します。
        /// </summary>
        /// <param name="fs">ELF形式のファイル</param>
        /// <returns>ArchitectureTypeに沿ったElf ヘッダ</returns>
        public static ElfHeader MakeHeader(FileStream fs)
        {
            ElfHeader header = new ElfHeader();
            BinaryReader binaryReader = new BinaryReader(fs);
            ElfBinaryReader reader = new ElfBinaryReader(binaryReader, ElfByteOrderType.None);
            header.ReadElfFile(reader);

            ElfHeader newHeader = null;
            reader = new ElfBinaryReader(binaryReader, header.ByteOrder);
            fs.Seek(0, SeekOrigin.Begin);
            switch (header.Architecture)
            {
                case ElfArchitectureType.Elf32:
                    newHeader = new Elf32.Elf32Header();
                    break;
                case ElfArchitectureType.Elf64:
                    newHeader = new Elf64.Elf64Header();
                    break;
                default:
                    throw new InvalidDataException();
            }
            newHeader.ReadElfFile(reader);
            return newHeader;
        }

        /// <summary>
        /// ElfHeader のコンストラクタ
        /// </summary>
        public ElfHeader()
        {
            Identification = new byte[EiNIdEnt];
        }

        /// <summary>
        /// ELFヘッダを初期化します。
        /// </summary>
        /// <param name="machine">Machineコード</param>
        /// <param name="endian">ELFのエンディアン</param>
        public virtual void Initialize(ushort machine, byte endian)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// ELFヘッダの位置を計算。
        /// </summary>
        /// <returns>セクションヘッダ終了地点のオフセット</returns>
        public virtual ulong CalcSectionPosition()
        {
            throw new NotImplementedException();
        }

        public ElfArchitectureType Architecture
        {
            get
            {
                return NumericConvert.IntToElfDefine<ElfArchitectureType>(Identification[4]);
            }
        }

        public ElfByteOrderType ByteOrder
        {
            get
            {
                return NumericConvert.IntToElfDefine<ElfByteOrderType>(Identification[5]);
            }
        }

        public virtual void ReadElfFile(ElfBinaryReader reader)
        {
            Identification = reader.ReadBytes(EiNIdEnt);
            if (!IsElfFormat())
            {
                throw new InvalidDataException();
            }
        }

        /// <summary>
        /// ファイルに ElfHeader を書き込みます。
        /// </summary>
        /// <param name="writer">書き込みインスタンス</param>
        /// <returns>書き込みの成否</r
        public virtual void WriteElfData(ElfBinaryWriter writer)
        {
            writer.WriteBytes(Identification);
        }

        /// <summary>
        /// ELF形式のE_Identかどうかをチェックします。
        /// </summary>
        /// <returns>ELF形式ならTrue, 違っていればFalse</returns>
        public bool IsElfFormat()
        {
            return ELFMAG0 == Identification[0] && ELFMAG1 == Identification[1]
                && ELFMAG2 == Identification[2] && ELFMAG3 == Identification[3];
        }

        /// <summary>
        /// アーキテクチャやエンディアンの設定が出来ているかを確認します。
        /// </summary>
        /// <returns>設定済みかどうか</returns>
        public bool CheckSetupEnvironment()
        {
            return !(ElfArchitectureType.None == Architecture || ElfByteOrderType.None == ByteOrder);
        }
    }
}
