﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace ElfLibrary
{
    using System;
    using System.IO;

    /// <summary>
    /// プリミティブ データ型を特定のアーキテクチャタイプおよびバイトオーダタイプのバイナリ値として書き込みます。
    /// </summary>
    public sealed class ElfBinaryWriter
    {
        /// <summary>
        /// ElfBinaryWriter クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="writer">バイナリライターです。</param>
        /// <param name="byteOrder">バイトオーダタイプです。</param>
        public ElfBinaryWriter(BinaryWriter writer, ElfByteOrderType byteOrder = ElfByteOrderType.LittleEndian)
        {
            this.BinaryWriter = writer;
            this.ByteOrder = byteOrder;
        }

        /// <summary>
        /// バイトオーダタイプを取得します。
        /// </summary>
        public ElfByteOrderType ByteOrder { get; private set; }

        private BinaryWriter BinaryWriter { get; set; }

        /// <summary>
        /// ストリームに１バイト書き込み、ストリームの位置を１バイト進めます。
        /// </summary>
        /// <param name="data">１バイトのデータ</param>
        public void WriteByte(byte data)
        {
            BinaryWriter.Write(data);
        }

        /// <summary>
        /// ストリームに指定されたデータを書き込み、ストリームの位置を書き込んだデータサイズ分進めます。
        /// </summary>
        /// <param name="data">複数バイトデータ</param>
        public void WriteBytes(byte[] data)
        {
            BinaryWriter.Write(data);
        }

        /// <summary>
        /// ストリームに２バイト書き込み、ストリームの位置を２バイト進めます。
        /// </summary>
        /// <param name="data">２バイトのデータ</param>
        public void WriteUInt16(ushort data)
        {
            ushort value = (ByteOrder == ElfByteOrderType.BigEndian)
                ? NumericConvert.SwapEndian(data) : data;
            BinaryWriter.Write(value);
        }

        /// <summary>
        /// ストリームに４バイト書き込み、ストリームの位置を４バイト進めます。
        /// </summary>
        /// <param name="data">４バイトのデータ</param>
        public void WriteUInt32(uint data)
        {
            uint value = (ByteOrder == ElfByteOrderType.BigEndian)
                ? NumericConvert.SwapEndian(data) : data;
            BinaryWriter.Write(value);
        }

        /// <summary>
        /// ストリームに８バイト書き込み、ストリームの位置を８バイト進めます。
        /// </summary>
        /// <param name="data">８バイトのデータ</param>
        public void WriteUInt64(ulong data)
        {
            ulong value = (ByteOrder == ElfByteOrderType.BigEndian)
                ? NumericConvert.SwapEndian(data) : data;
            BinaryWriter.Write(value);
        }

        /// <summary>
        /// バッファのデータをファイルに書きだします。
        /// </summary>
        public void Flush()
        {
            BinaryWriter.Flush();
        }
    }
}
