﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace ElfLibrary.Elf32
{
    using System;
    using System.IO;
    using ElfLibrary;

    /// <summary>
    /// ELF 32bit ELFヘッダクラス
    /// </summary>
    public class Elf32Header : ElfHeader
    {
        public const ushort Elf32HeaderSize = 52;

        /// <summary>
        /// ELFヘッダの位置を計算。4 バイトアラインメントにする。
        /// </summary>
        /// <returns>セクションヘッダ終了地点のオフセット</returns>
        public override ulong CalcSectionPosition()
        {
            ulong offset;
            offset = HeaderSize;
            SectionOffset = NumericConvert.RoundUp(offset, 4);
            return SectionOffset + (ulong)(SectionEntrySize * SectionNumber);
        }

        /// <summary>
        /// ELFヘッダを初期化します。
        /// </summary>
        /// <param name="machine">Machineコード</param>
        /// <param name="endian">ELFのエンディアン</param>
        public override void Initialize(ushort machine, byte endian)
        {
            Identification[0] = ELFMAG0;
            Identification[1] = ELFMAG1;
            Identification[2] = ELFMAG2;
            Identification[3] = ELFMAG3;
            Identification[4] = (byte)ElfArchitectureType.Elf32;
            Identification[5] = endian;
            Identification[6] = (byte)ElfVersionType.Current;
            Type = (ushort)ElfObjectFileType.Rel;
            Machine = machine;
            Version = (uint)ElfVersionType.Current;
            Entry = 0;
            ProgramOffset = 0;
            SectionOffset = 0;
            Flags = 0;
            HeaderSize = Elf32HeaderSize;
            ProgramEntrySize = 0;
            ProgramNumber = 0;
            SectionEntrySize = Elf32SectionHeader.Elf32HeaderSize;
            SectionNumber = 0;
            StringTableIndex = 0;
        }

        /// <summary>
        /// ファイルから ElfHeader を読み込みます。
        /// </summary>
        /// <param name="reader">読み込みインスタンス</param>
        public override void ReadElfFile(ElfBinaryReader reader)
        {
            base.ReadElfFile(reader); // Read E_Ident
            Type = reader.ReadUInt16();
            Machine = reader.ReadUInt16();
            Version = reader.ReadUInt32();
            Entry = reader.ReadUInt32();
            ProgramOffset = reader.ReadUInt32();
            SectionOffset = reader.ReadUInt32();
            Flags = reader.ReadUInt32();
            HeaderSize = reader.ReadUInt16();
            ProgramEntrySize = reader.ReadUInt16();
            ProgramNumber = reader.ReadUInt16();
            SectionEntrySize = reader.ReadUInt16();
            SectionNumber = reader.ReadUInt16();
            StringTableIndex = reader.ReadUInt16();
        }

        /// <summary>
        /// ファイルに ElfHeader を書き込みます。
        /// </summary>
        /// <param name="writer">書き込みインスタンス</param>
        public override void WriteElfData(ElfBinaryWriter writer)
        {
            checked
            {
                base.WriteElfData(writer); //write E_Ident
                writer.WriteUInt16(Type);
                writer.WriteUInt16(Machine);
                writer.WriteUInt32(Version);
                writer.WriteUInt32((uint)Entry);
                writer.WriteUInt32((uint)ProgramOffset);
                writer.WriteUInt32((uint)SectionOffset);
                writer.WriteUInt32(Flags);
                writer.WriteUInt16(HeaderSize);
                writer.WriteUInt16(ProgramEntrySize);
                writer.WriteUInt16(ProgramNumber);
                writer.WriteUInt16(SectionEntrySize);
                writer.WriteUInt16(SectionNumber);
                writer.WriteUInt16(StringTableIndex);
            }
        }
    }
}
