﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Bin2Obj
{
    using System;
    using System.IO;
    using Nintendo.Foundation.IO;
    using ElfLibrary;

    /// <summary>
    /// Bin2Obj アプリケーションの引数情報を管理します。
    /// </summary>
    internal class Bin2ObjArgs
    {
        private Bin2ObjParams parameters;

        /// <summary>
        /// 引数を解析することで得られるELFヘッダを生成するために必要な情報です。
        /// </summary>
        public Bin2ObjParams Params
        {
            get { return parameters; }
        }

        /// <summary>
        /// 引数を解析し、ELFヘッダを生成するために必要な情報を生成します。
        /// </summary>
        /// <returns>解析の成否</returns>
        public bool ParseArgs(string[] args)
        {
            var settings = new CommandLineParserSettings()
            {
                ApplicationDescription = "Binary to Object converter",
                HelpWriter = text =>
                {
                    Console.WriteLine(text);
                    Console.WriteLine("  (*) special %% rules for symbols (ex. binary_file = \"filename.dat\")\n"
                                + "       %%f,%%t replaced to file name of binary   (%%f = \"filename.dat\")\n"
                                + "       %%b     replaced to base name of binary   (%%b = \"filename\")\n"
                                + "       %%e     replaced to extension of binary   (%%e = \"dat\")\n\n");
                },
            };

            if (!new CommandLineParser(settings).ParseArgs(args, out parameters))
            {
                return false;
            }

            if (Params.Compatible)
            {
                Params.SymbolBegin = Bin2ObjParams.DefaultCompatibleBegin;
                Params.SymbolEnd = Bin2ObjParams.DefaultCompatibleEnd;
            }

            if (0 == Params.Machine)
            {
                Params.Machine = Params.DefaultMachine;
            }

            if ((byte)ElfByteOrderType.None == Params.Endian)
            {
                Params.Endian = Params.DefaultEndian;
            }

            if ((byte)ElfArchitectureType.None == Params.Architecture)
            {
                switch (NumericConvert.IntToElfDefine<ElfMachineType>(Params.Machine))
                {
                    case ElfMachineType.Unknown:
                        throw new ArgumentException();
                    case ElfMachineType.AArch64:
                        Params.Architecture = (byte)ElfArchitectureType.Elf64;
                        break;
                    default:
                        Params.Architecture = (byte)ElfArchitectureType.Elf32;
                        break;
                }
            }
            return true;
        }

        /// <summary>
        /// 引数解析が失敗したときに呼ばれるメソッドです。
        /// </summary>
        /// <param name="errorInfo">エラーの情報</param>
        private void CommandLineParseError(CommandLineErrorInfo errorInfo)
        {
            Console.Error.WriteLine("CommandLine: {0}", errorInfo.ErrorMessage);
        }
    }
}
