﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>
#include <msclr/marshal.h>
#include <vector>
#include <nn/detail/nn_FirmwareVersion-spec.NX.h>

#include "../Util/DeclareAlive.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::Collections;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;
using namespace msclr::interop;

    public ref class NintendoContentMetaConstant
    {
    public:
        literal String^ ContentTypeProgram = "Program";
        literal String^ ContentTypeControl = "Control";
        literal String^ ContentTypeData = "Data";
        literal String^ ContentTypeMeta = "Meta";
        literal String^ ContentTypeHtmlDocument = "HtmlDocument";
        literal String^ ContentTypeLegalInformation = "LegalInformation";
        literal String^ ContentTypeDeltaFragment = "DeltaFragment";
        literal String^ ContentTypePublicData = "PublicData"; // ncm レイヤには存在しない

        literal String^ ContentMetaTypeApplication = "Application";
        literal String^ ContentMetaTypePatch = "Patch";
        literal String^ ContentMetaTypeAddOnContent = "AddOnContent";
        literal String^ ContentMetaTypeSystemProgram = "SystemProgram";
        literal String^ ContentMetaTypeSystemData = "SystemData";
        literal String^ ContentMetaTypeSystemUpdate = "SystemUpdate";
        literal String^ ContentMetaTypeBootImagePackage = "BootImagePackage";
        literal String^ ContentMetaTypeBootImagePackageSafe = "BootImagePackageSafe";
        literal String^ ContentMetaTypeDelta = "Delta";
    };

    public ref class NintendoContentInfo
    {
    public:
        NintendoContentInfo(String^ type, Int64 size, byte keyGeneration, byte idOffset)
        {
            Type = VerifyContentType(type);
            Size = size;
            KeyGeneration = keyGeneration;
            IdOffset = idOffset;
            GC::KeepAlive(this);
        }

        NintendoContentInfo(String^ type, array<byte>^ id, Int64 size, array<byte>^ hash, byte idOffset) : NintendoContentInfo(type, id, size, hash, Nullable<byte>(), idOffset)
        {
        }

        NintendoContentInfo(String^ type, array<byte>^ id, Int64 size, array<byte>^ hash, Nullable<byte> keyGeneration, byte idOffset)
        {
            System::Diagnostics::Debug::Assert(id->Length == sizeof(nn::ncm::ContentId));
            System::Diagnostics::Debug::Assert(hash->Length == sizeof(nn::ncm::Hash));
            Type = VerifyContentType(type);
            Id = id;
            Size = size;
            Hash = hash;
            if (keyGeneration.HasValue)
            {
                KeyGeneration = keyGeneration.Value;
            }
            IdOffset = idOffset;

            GC::KeepAlive(this);
        }

        property String^ Type;
        property array<byte>^ Id;
        property Int64 Size;
        property array<byte>^ Hash;
        property byte KeyGeneration
        {
        public:
            byte get ()
            {
                return _KeyGeneration.Value;
            }
        private:
            void set(byte value)
            {
                _KeyGeneration = value;
            }
        }
        property byte IdOffset;

    private:
        Nullable<byte> _KeyGeneration;

        String^ VerifyContentType(String^ type)
        {
            if (type == "PublicData")
            {
                // コンテンツメタ、ncm のレイヤでは PublicData も Data コンテンツとみなす
                return "Data";
            }
            return type;
        }
    };
    public ref class NintendoContentFragmentInfo
    {
    public:
        NintendoContentFragmentInfo()
        {
            FragmentTargetContentType = nullptr;
            SourceContentId = nullptr;
            DestinationContentId = nullptr;
            UpdateType = nullptr;
            FragmentIndex = 0;
            GC::KeepAlive(this);
        }
        property String^ FragmentTargetContentType;
        property String^ UpdateType;
        property String^ SourceContentId;
        property String^ DestinationContentId;
        property Int64   SourceSize;
        property Int64   DestinationSize;
        property uint16_t DeltaContentIndex;
        property uint16_t FragmentIndex;
    };
    public ref class NintendoContentDescriptor
    {
    public:
        NintendoContentDescriptor()
        {}
        property NintendoContentInfo^ ContentInfo;
        property NintendoContentFragmentInfo^ ContentFragmentInfo;
    };
    public ref class NintendoContentMetaInfo
    {
    public:
        NintendoContentMetaInfo(String^ type, UInt64 id, UInt32 version, Byte attributes)
        {
            Id = id;
            Version = version;
            Type = type;
            Attributes = attributes;
            GC::KeepAlive(this);
        }

        property String^ Type;
        property UInt64 Id;
        property UInt32 Version;
        property Byte Attributes;
    };


    public ref class NintendoMetaExtendedHeader
    {
    };

    public ref class NintendoApplicationMetaExtendedHeader : NintendoMetaExtendedHeader
    {
    public:
        NintendoApplicationMetaExtendedHeader(UInt64 patchId, UInt32 requiredSystemVersion)
        {
            PatchId = patchId;
            RequiredSystemVersion = requiredSystemVersion;
            GC::KeepAlive(this);
        }
        property UInt64 PatchId;
        property UInt32 RequiredSystemVersion;
    };

    public ref class NintendoAddOnContentMetaExtendedHeader : NintendoMetaExtendedHeader
    {
    public:
        NintendoAddOnContentMetaExtendedHeader(UInt64 applicationId, UInt32 requiredApplicationVersion, String^ tag)
        {
            ApplicationId = applicationId;
            RequiredApplicationVersion = requiredApplicationVersion;
            Tag = tag;
            GC::KeepAlive(this);
        }
        property UInt64     ApplicationId;
        property UInt32     RequiredApplicationVersion;
        property String^    Tag;
    };

    public ref class NintendoPatchMetaExtendedHeader : NintendoMetaExtendedHeader
    {
    public:
        NintendoPatchMetaExtendedHeader(UInt64 applicationId, UInt32 requiredSystemVersion)
        {
            ApplicationId = applicationId;
            RequiredSystemVersion = requiredSystemVersion;
            ExtendedDataSize = 0;
            GC::KeepAlive(this);
        }
        property UInt64 ApplicationId;
        property UInt32 RequiredSystemVersion;
        property UInt32 ExtendedDataSize;
    };
    public ref class NintendoDeltaMetaExtendedHeader : NintendoMetaExtendedHeader
    {
    public:
        NintendoDeltaMetaExtendedHeader(UInt64 applicationId,  UInt32 extendedDataSize)
        {
            ApplicationId = applicationId;
            ExtendedDataSize = extendedDataSize;
            GC::KeepAlive(this);
        }
        property UInt64 ApplicationId;
        property UInt32 ExtendedDataSize;
    };

    public ref class NintendoMetaExtendedData
    {
    };

    public ref class NintendoDeltaMetaExtendedData : NintendoMetaExtendedData
    {
    public:
        ref class NintendoDeltaMetaPatchIndicator
        {
        public:
            property UInt64 PatchId;
            property UInt32 Version;
        };
        ref class NintendoDeltaMetaFragmentIndicator
        {
        public:
            property UInt16 ContentInfoIndex;
            property UInt16 FragmentIndex;
        };
        ref class NintendoDeltaMetaFragmentSet
        {
        public:
            property array<byte>^ SourceContentId;
            property array<byte>^ DestinationContentId;
            property Int64        SourceSize;
            property Int64        DestinationSize;
            property String^      FragmentTargetContentType;
            property String^      UpdateType;
            property System::Collections::Generic::List<NintendoDeltaMetaFragmentIndicator^>^ Fragments;
        };

        property NintendoDeltaMetaPatchIndicator^ Source;
        property NintendoDeltaMetaPatchIndicator^ Destination;
        property System::Collections::Generic::List<NintendoDeltaMetaFragmentSet^>^ FragmentSets;

        NintendoDeltaMetaExtendedData()
        {
            Source = gcnew NintendoDeltaMetaPatchIndicator();
            Destination = gcnew NintendoDeltaMetaPatchIndicator();
        }
        UInt64 GetSize()
        {
            int fragmentSetCount = FragmentSets->Count;
            int fragmentSetContentCount = 0;
            for (int i = 0; i < fragmentSetCount; ++i)
            {
                fragmentSetContentCount += FragmentSets[i]->Fragments->Count;
            }
            return nn::ncm::DeltaMetaExtendedDataWriter::CalculateSize(fragmentSetCount, fragmentSetContentCount);
        }
    };

    public ref class NintendoPatchMetaExtendedData : NintendoMetaExtendedData
    {
    public:
        ref class NintendoPatchMetaHisotry
        {
        public:
            property String^ Type;
            property UInt64 ProgramId; // Application or Patch
            property UInt32 Version;
            property array<byte>^ Digest;
            property System::Collections::Generic::List<NintendoContentInfo^>^ ContentInfos;
        };
        ref class NintendoPatchMetaDeltaHisotry
        {
        public:
            property NintendoDeltaMetaExtendedData^ Delta;
            property UInt64 DownloadSize;
        };
        ref class NintendoPatchMetaDelta
        {
        public:
            property System::Collections::Generic::List<NintendoContentInfo^>^ ContentInfos;
            property NintendoDeltaMetaExtendedData^ Delta;
        };

        property System::Collections::Generic::List<NintendoPatchMetaHisotry^>^ Histories;
        property System::Collections::Generic::List<NintendoPatchMetaDeltaHisotry^>^ DeltaHistories;
        property System::Collections::Generic::List<NintendoPatchMetaDelta^>^ Deltas;

        int GetHistoryContentCount()
        {
            int historyContentCount = 0;
            for (int i = 0; i < Histories->Count; ++i)
            {
                historyContentCount += Histories[i]->ContentInfos->Count;
            }
            return historyContentCount;
        }
        int GetDeltaContentCount()
        {
            int deltaContentCount = 0;
            for (int i = 0; i < Deltas->Count; ++i)
            {
                deltaContentCount += Deltas[i]->ContentInfos->Count;
            }
            return deltaContentCount;
        }
        int GetFragmentSetCount()
        {
            int fragmentSetCount = 0;
            for (int i = 0; i < Deltas->Count; ++i)
            {
                fragmentSetCount += Deltas[i]->Delta->FragmentSets->Count;
            }
            return fragmentSetCount;
        }
        int GetFragmentIndicatorCount()
        {
            int fragmentIndicatorCount = 0;

            for (int i = 0; i < Deltas->Count; ++i)
            {
                for (int j = 0; j < Deltas[i]->Delta->FragmentSets->Count; ++j)
                {
                    fragmentIndicatorCount += Deltas[i]->Delta->FragmentSets[j]->Fragments->Count;
                }
            }
            return fragmentIndicatorCount;
        }

        UInt64 GetSize()
        {
            int historyCount = Histories->Count;
            int deltaHistoryCount = DeltaHistories->Count;
            int deltaCount = Deltas->Count;
            int historyContentCount = GetHistoryContentCount();
            int deltaContentCount = GetDeltaContentCount();
            int fragmentSetCount = GetFragmentSetCount();
            int fragmentIndicatorCount = GetFragmentIndicatorCount();

            return nn::ncm::PatchMetaExtendedDataWriter::CalculateSize(historyCount, historyContentCount, deltaHistoryCount, deltaCount, deltaContentCount, fragmentSetCount, fragmentIndicatorCount);
        }
    };

    public ref class NintendoContentMeta
    {
    public:
        static NintendoContentMeta^ CreateApplicationMeta(UInt64 id, UInt32 version, Byte attributes, UInt32 requiredDownloadSystemVersion, NintendoApplicationMetaExtendedHeader^ appExHeader, List<NintendoContentDescriptor^>^ contentDescriptorList, List<NintendoContentMetaInfo^>^ contentMetaInfoList, array<byte>^ digest)
        {
            auto meta = gcnew NintendoContentMeta(nn::ncm::ContentMetaType::Application, id, version, attributes, requiredDownloadSystemVersion, contentDescriptorList, contentMetaInfoList, 0, digest);
            nn::ncm::ApplicationMetaExtendedHeader extendedHeader = {};
            extendedHeader.patchId.value = appExHeader->PatchId;
            extendedHeader.requiredSystemVersion = appExHeader->RequiredSystemVersion;
            meta->WriteExtendedHeader(extendedHeader);
            return meta;
        }

        static NintendoContentMeta^ CreateAddOnContentMeta(UInt64 id, UInt32 version, Byte attributes, UInt32 requiredDownloadSystemVersion, NintendoAddOnContentMetaExtendedHeader^ aocExHeader, List<NintendoContentDescriptor^>^ contentDescriptorList, List<NintendoContentMetaInfo^>^ contentMetaInfoList, array<byte>^ digest)
        {
            auto meta = gcnew NintendoContentMeta(nn::ncm::ContentMetaType::AddOnContent, id, version, attributes, requiredDownloadSystemVersion, contentDescriptorList, contentMetaInfoList, 0,  digest);
            nn::ncm::AddOnContentMetaExtendedHeader extendedHeader = {};
            extendedHeader.applicationId.value = aocExHeader->ApplicationId;
            extendedHeader.requiredApplicationVersion = aocExHeader->RequiredApplicationVersion;
            meta->WriteExtendedHeader(extendedHeader);
            return meta;
        }

        static NintendoContentMeta^ CreatePatchMeta(UInt64 id, UInt32 version, Byte attributes, UInt32 requiredDownloadSystemVersion, NintendoPatchMetaExtendedHeader^ patchExHeader, List<NintendoContentDescriptor^>^ contentDescriptorList, List<NintendoContentMetaInfo^>^ contentMetaInfoList, NintendoPatchMetaExtendedData^ patchExData, array<byte>^ digest)
        {
            auto size = patchExHeader->ExtendedDataSize;
            nn::ncm::PatchMetaExtendedHeader extendedHeader = {};
            extendedHeader.applicationId.value = patchExHeader->ApplicationId;
            extendedHeader.requiredSystemVersion = patchExHeader->RequiredSystemVersion;
            extendedHeader.extendedDataSize = size;

            auto meta = gcnew NintendoContentMeta(nn::ncm::ContentMetaType::Patch, id, version, attributes, requiredDownloadSystemVersion, contentDescriptorList, contentMetaInfoList, size, digest);
            meta->WriteExtendedHeader(extendedHeader);

            if (patchExData != nullptr)
            {
                std::unique_ptr<byte[]> extendedDataBuffer(new byte[size]);
                std::unique_ptr<nn::ncm::PatchMetaExtendedDataWriter> extendedData(new nn::ncm::PatchMetaExtendedDataWriter(extendedDataBuffer.get(), size));
                std::memset(extendedDataBuffer.get(), 0, size);

                // header の設定
                {
                    nn::ncm::PatchMetaExtendedDataHeader header = {};
                    header.historyCount = patchExData->Histories->Count;
                    header.historyContentTotalCount = patchExData->GetHistoryContentCount();
                    header.deltaHistoryCount = patchExData->DeltaHistories->Count;
                    header.deltaCount = patchExData->Deltas->Count;
                    header.deltaContentTotalCount = patchExData->GetDeltaContentCount();
                    header.fragmentSetCount = patchExData->GetFragmentSetCount();
                    extendedData->WritePatchMetaExtendedDataHeader(header);
                }
                // history の設定
                {
                    for (int i = 0; i < patchExData->Histories->Count; ++i)
                    {
                        auto history = patchExData->Histories[i];

                        nn::ncm::PatchHistoryHeader header = {};
                        header.key = nn::ncm::ContentMetaKey::Make(history->ProgramId, history->Version, ConvertMetaType(history->Type));
                        Marshal::Copy(history->Digest, 0, IntPtr(&(header.digest)), sizeof(nn::ncm::Hash));
                        header.contentCount = history->ContentInfos->Count;
                        extendedData->WritePatchHistoryHeader(i, header);

                        for (int j = 0; j < history->ContentInfos->Count; ++j)
                        {
                            nn::ncm::ContentInfo contentInfo = {};
                            WriteContentInfo(&contentInfo, history->ContentInfos[j]);
                            extendedData->WritePatchHistoryContentInfo(i, j, contentInfo);
                        }
                    }
                }
                // delta history の設定
                {
                    for (int i = 0; i < patchExData->DeltaHistories->Count; ++i)
                    {
                        auto history = patchExData->DeltaHistories[i];

                        nn::ncm::PatchDeltaHistory historyRaw = {};
                        historyRaw.sourceId.value = history->Delta->Source->PatchId;
                        historyRaw.sourceVersion = history->Delta->Source->Version;
                        historyRaw.destinationId.value = history->Delta->Destination->PatchId;
                        historyRaw.destinationVersion = history->Delta->Destination->Version;

                        historyRaw.downloadSize = history->DownloadSize;

                        extendedData->WritePatchDeltaHistory(i, historyRaw);
                    }
                }
                // delta の設定
                {
                    for (int i = 0; i < patchExData->Deltas->Count; ++i)
                    {
                        auto delta = patchExData->Deltas[i];

                        // delta 自体の情報
                        nn::ncm::PatchDeltaHeader header = {};
                        header.delta.sourceId.value = delta->Delta->Source->PatchId;
                        header.delta.sourceVersion = delta->Delta->Source->Version;
                        header.delta.destinationId.value = delta->Delta->Destination->PatchId;
                        header.delta.destinationVersion = delta->Delta->Destination->Version;
                        header.contentCount = delta->ContentInfos->Count;
                        header.delta.fragmentSetCount = delta->Delta->FragmentSets->Count;
                        extendedData->WritePatchDeltaHeader(i, header);

                        for (int j = 0; j < delta->ContentInfos->Count; ++j)
                        {
                            nn::ncm::PackagedContentInfo contentInfo = {};
                            Marshal::Copy(delta->ContentInfos[j]->Hash, 0, IntPtr(&(contentInfo.hash)), sizeof(nn::ncm::Hash));
                            WriteContentInfo(&(contentInfo.info), delta->ContentInfos[j]);
                            extendedData->WritePatchDeltaPackagedContentInfo(i, j, contentInfo);
                        }
                        // delta に付随する fragmentSet の情報
                        for (int j = 0; j < delta->Delta->FragmentSets->Count; ++j)
                        {
                            nn::ncm::FragmentSet set = {};
                            set.fragmentCount = delta->Delta->FragmentSets[j]->Fragments->Count;
                            Marshal::Copy(delta->Delta->FragmentSets[j]->SourceContentId, 0, IntPtr(&(set.sourceContentId)), sizeof(nn::ncm::ContentId));
                            Marshal::Copy(delta->Delta->FragmentSets[j]->DestinationContentId, 0, IntPtr(&(set.destinationContentId)), sizeof(nn::ncm::ContentId));
                            set.SetSourceSize(delta->Delta->FragmentSets[j]->SourceSize);
                            set.SetDestinationSize(delta->Delta->FragmentSets[j]->DestinationSize);
                            set.targetContentType = ConvertType(delta->Delta->FragmentSets[j]->FragmentTargetContentType);
                            set.updateType = ConvertUpdateType(delta->Delta->FragmentSets[j]->UpdateType);
                            extendedData->WriteFragmentSet(i, j, set);

                            for (int k = 0; k < delta->Delta->FragmentSets[j]->Fragments->Count; ++k)
                            {
                                nn::ncm::FragmentIndicator indicator = {};
                                indicator.fragmentIndex = delta->Delta->FragmentSets[j]->Fragments[k]->FragmentIndex;
                                indicator.contentInfoIndex = delta->Delta->FragmentSets[j]->Fragments[k]->ContentInfoIndex;
                                extendedData->WriteFragmentIndicator(i, j, k, indicator);
                            }
                        }
                    }
                }
                meta->WriteExtendedData(extendedDataBuffer.get(), size);
            }
            return meta;
        }

        static NintendoContentMeta^ CreateDeltaMeta(UInt64 id, UInt32 version, Byte attributes, UInt32 requiredDownloadSystemVersion, NintendoDeltaMetaExtendedHeader^ deltaExHeader, List<NintendoContentDescriptor^>^ contentDescriptorList, List<NintendoContentMetaInfo^>^ contentMetaInfoList, NintendoDeltaMetaExtendedData^ deltaExData, array<byte>^ digest)
        {
            // ex-header
            auto size = deltaExHeader->ExtendedDataSize;
            nn::ncm::DeltaMetaExtendedHeader extendedHeader = {};
            extendedHeader.extendedDataSize = size;
            extendedHeader.applicationId.value = deltaExHeader->ApplicationId;

            // meta の生成
            auto meta = gcnew NintendoContentMeta(nn::ncm::ContentMetaType::Delta, id, version, attributes, requiredDownloadSystemVersion, contentDescriptorList, contentMetaInfoList, size, digest);
            meta->WriteExtendedHeader(extendedHeader);

            // ex-data
            if (deltaExData != nullptr)
            {
                std::unique_ptr<byte[]> extendedDataBuffer(new byte[size]);
                std::unique_ptr<nn::ncm::DeltaMetaExtendedDataWriter> extendedData(new nn::ncm::DeltaMetaExtendedDataWriter(extendedDataBuffer.get(), size));
                std::memset(extendedDataBuffer.get(), 0, size);

                // header の設定
                {
                    nn::ncm::DeltaMetaExtendedDataHeader header = {};
                    header.sourceId.value = deltaExData->Source->PatchId;
                    header.sourceVersion = deltaExData->Source->Version;

                    header.destinationId.value = deltaExData->Destination->PatchId;
                    header.destinationVersion = deltaExData->Destination->Version;
                    header.fragmentSetCount = deltaExData->FragmentSets->Count;
                    extendedData->WriteHeader(header);
                }
                // fragmentSet の設定
                {
                    for (int i = 0; i < deltaExData->FragmentSets->Count; ++i)
                    {
                        System::Diagnostics::Debug::Assert(deltaExData->FragmentSets[i]->Fragments->Count > 0);

                        nn::ncm::FragmentSet set = {};
                        set.fragmentCount = deltaExData->FragmentSets[i]->Fragments->Count;
                        Marshal::Copy(deltaExData->FragmentSets[i]->SourceContentId, 0, IntPtr(&(set.sourceContentId)), sizeof(nn::ncm::ContentId));
                        Marshal::Copy(deltaExData->FragmentSets[i]->DestinationContentId, 0, IntPtr(&(set.destinationContentId)), sizeof(nn::ncm::ContentId));
                        set.SetSourceSize(deltaExData->FragmentSets[i]->SourceSize);
                        set.SetDestinationSize(deltaExData->FragmentSets[i]->DestinationSize);
                        set.targetContentType = ConvertType(deltaExData->FragmentSets[i]->FragmentTargetContentType);
                        set.updateType = ConvertUpdateType(deltaExData->FragmentSets[i]->UpdateType);

                        extendedData->WriteFragmentSet(set, i);
                        for (int j = 0; j < deltaExData->FragmentSets[i]->Fragments->Count; ++j)
                        {
                            nn::ncm::FragmentIndicator indicator = {};
                            indicator.fragmentIndex = deltaExData->FragmentSets[i]->Fragments[j]->FragmentIndex;
                            indicator.contentInfoIndex = deltaExData->FragmentSets[i]->Fragments[j]->ContentInfoIndex;
                            extendedData->WriteFragmentIndicator(indicator, i, j);
                        }
                    }
                }
                meta->WriteExtendedData(extendedDataBuffer.get(), size);
            }

            return meta;
        }

        static NintendoContentMeta^ CreateSystemUpdateMeta(UInt64 id, UInt32 version, String^ type, Byte attributes, UInt32 requiredDownloadSystemVersion, List<NintendoContentDescriptor^>^ contentDescriptorList, List<NintendoContentMetaInfo^>^ contentMetaInfoList, array<byte>^ digest)
        {
            // SystemUpdate ならば常に再起動不要フラグを立てる
            attributes |= nn::ncm::ContentMetaAttribute::ContentMetaAttribute_Rebootless;
            return gcnew NintendoContentMeta(ConvertMetaType(type), id, version, attributes, requiredDownloadSystemVersion, contentDescriptorList, contentMetaInfoList, 0, digest);
        }

        static NintendoContentMeta^ CreateDefault(UInt64 id, UInt32 version, String^ type, Byte attributes, UInt32 requiredDownloadSystemVersion, List<NintendoContentDescriptor^>^ contentDescriptorList, List<NintendoContentMetaInfo^>^ contentMetaInfoList, array<byte>^ digest)
        {
            return gcnew NintendoContentMeta(ConvertMetaType(type), id, version, attributes, requiredDownloadSystemVersion, contentDescriptorList, contentMetaInfoList, 0, digest);
        }

        static UInt32 GetRequiredSystemVersion()
        {
            // 製品化時にこの値を下回っているUPPをつけるとエラーになります。
            // マイナーバージョン、マイクロバージョンが更新されるNUPがリリースされた場合でも、それに対応するUPPがすぐに許可される訳ではないため
            // メジャーバージョンのみを採用しています。
            return (NN_FIRMWARE_VERSION_MAJOR << 26 | 0);
        }

        static UInt32 GetDefaultVersion()
        {
            // システムタイトルのデフォルトのバージョンです。
            // 主にOceanでビルドされるタイトルにバージョンを設定するために使用されます。
            // https://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=222894944
            // (Major Version : 6bit | Minor Version : 6bit | Micro Version : 4bit | Relstep Major : 8bit | Relstep Minor 8bit)
            return (NN_FIRMWARE_VERSION_MAJOR << 26 |
                    NN_FIRMWARE_VERSION_MINOR << 20 |
                    NN_FIRMWARE_VERSION_MICRO << 16 |
                    NN_FIRMWARE_VERSION_MAJORRELSTEP << 8 |
                    NN_FIRMWARE_VERSION_MINORRELSTEP);
        }

        ~NintendoContentMeta()
        {
            this->!NintendoContentMeta();
        }

        !NintendoContentMeta()
        {
            delete m_Writer;
            delete[] m_Buffer;
        }

        array<Byte>^ GetBytes()
        {
            array<unsigned char>^ buf = gcnew array<unsigned char>(m_Writer->GetSize());
            {
                pin_ptr<unsigned char> ptr = &buf[0];
                memcpy(ptr, m_Writer->GetData(), m_Writer->GetSize());
                ptr = nullptr;
            }

            return Util::ReturnAndDeclareAlive(this, buf);
        }

        Int64 GetContentInfoIdDataOffset(Int32 index)
        {
            uintptr_t dataPointer = reinterpret_cast<uintptr_t>(m_Writer->GetData());
            uintptr_t infoDataPointer = reinterpret_cast<uintptr_t>(m_Writer->GetWritableContentInfo(index));
            if (! infoDataPointer)
            {
                throw gcnew InvalidOperationException();
            }
            uintptr_t offset = offsetof(struct nn::ncm::PackagedContentInfo, info.id);

            return Util::ReturnAndDeclareAlive(this, infoDataPointer - dataPointer + offset);
        }

        static Int32 GetContentInfoIdDataSize()
        {
            return sizeof(((nn::ncm::PackagedContentInfo*)0)->info.id);
        }

        Int64 GetContentInfoHashOffset(Int32 index)
        {
            uintptr_t dataPointer = reinterpret_cast<uintptr_t>(m_Writer->GetData());
            uintptr_t infoDataPointer = reinterpret_cast<uintptr_t>(m_Writer->GetWritableContentInfo(index));
            if (! infoDataPointer)
            {
                throw gcnew InvalidOperationException();
            }
            uintptr_t offset = offsetof(struct nn::ncm::PackagedContentInfo, hash);

            return Util::ReturnAndDeclareAlive(this, infoDataPointer - dataPointer + offset);
        }

        static Int32 GetContentInfoHashSize()
        {
            return sizeof(((nn::ncm::PackagedContentInfo*)0)->hash);
        }

        Int64 GetDigestOffset()
        {
            uintptr_t dataPointer = reinterpret_cast<uintptr_t>(m_Writer->GetData());
            uintptr_t digestPointer = m_Writer->GetDigestAddress();
            if (! digestPointer)
            {
                throw gcnew InvalidOperationException();
            }
            return Util::ReturnAndDeclareAlive(this, digestPointer - dataPointer);
        }

        static Int32 GetDigestSize()
        {
            return sizeof(nn::ncm::Hash);
        }

    public:
        static Int32 ConvertContentType(String^ type)
        {
            return static_cast<Int32>(ConvertType(type));
        }

    private:
        char* m_Buffer;
        nn::ncm::PackagedContentMetaWriter* m_Writer;

        NintendoContentMeta(nn::ncm::ContentMetaType metaType, UInt64 id, UInt32 version, Byte attributes, UInt32 requiredDownloadSystemVersion, List<NintendoContentDescriptor^>^ contentDescriptorList, List<NintendoContentMetaInfo^>^ contentMetaInfoList, size_t extendedDataSize, array<byte>^ digest)
        {
            auto metaSize = nn::ncm::PackagedContentMetaWriter::CalculateSize(metaType, contentDescriptorList->Count, contentMetaInfoList->Count, extendedDataSize);
            m_Buffer = new char[metaSize];
            m_Writer = new nn::ncm::PackagedContentMetaWriter(m_Buffer, metaSize);
            m_Writer->WriteHeader(id, version, metaType, attributes, requiredDownloadSystemVersion, contentDescriptorList->Count, contentMetaInfoList->Count);

            // ex-header を書かないと、サイズ情報がないためにその後のデータアクセスが問題になるので、特定の場合は先に書く
            switch (metaType)
            {
            case nn::ncm::ContentMetaType::Patch:
            {
                // とりあえずサイズだけ
                nn::ncm::PatchMetaExtendedHeader exHeader = { 0, };
                exHeader.extendedDataSize = static_cast<uint32_t>(extendedDataSize);
                m_Writer->WriteExtendedHeader(exHeader);
                break;
            }
            case nn::ncm::ContentMetaType::Delta:
            {
                // とりあえずサイズだけ
                nn::ncm::DeltaMetaExtendedHeader exHeader = { 0, };
                exHeader.extendedDataSize = static_cast<uint32_t>(extendedDataSize);
                m_Writer->WriteExtendedHeader(exHeader);
                break;
            }
            default:
                break;
            }

            for (int i = 0; i < contentDescriptorList->Count; i++)
            {
                nn::ncm::PackagedContentInfo data = {};
                nn::ncm::ContentId id = {};
                if (contentDescriptorList[i]->ContentInfo->Id != nullptr)
                {
                    Marshal::Copy(contentDescriptorList[i]->ContentInfo->Id, 0, IntPtr(&id), sizeof(nn::ncm::ContentId));
                }

                data.info = nn::ncm::ContentInfo::Make(id, contentDescriptorList[i]->ContentInfo->Size, ConvertType(contentDescriptorList[i]->ContentInfo->Type), contentDescriptorList[i]->ContentInfo->IdOffset);

                if (contentDescriptorList[i]->ContentInfo->Hash != nullptr)
                {
                    Marshal::Copy(contentDescriptorList[i]->ContentInfo->Hash, 0, IntPtr(&data.hash), sizeof(nn::ncm::Hash));
                }
                m_Writer->WriteContentInfo(data, i);
            }

            for (int i = 0; i < contentMetaInfoList->Count; i++)
            {
                auto metaInfo = contentMetaInfoList[i];
                auto info = nn::ncm::ContentMetaInfo::Make(metaInfo->Id, metaInfo->Version, ConvertMetaType(metaInfo->Type), metaInfo->Attributes);
                m_Writer->WriteContentMetaInfo(info, i);
            }

            {
                nn::ncm::Hash data;
                {
                    pin_ptr<unsigned char> ptr = &digest[0];
                    memcpy(data.data, ptr, sizeof(data.data));
                    ptr = nullptr;
                }
                m_Writer->WriteDigest(data);
            }

            GC::KeepAlive(this);
        }
        static void WriteContentInfo(nn::ncm::ContentInfo* outContentInfo, NintendoContentInfo^ contentInfo)
        {
            // copy id
            Marshal::Copy(contentInfo->Id, 0, IntPtr(&(outContentInfo->id)), sizeof(nn::ncm::ContentId));

            // copy size
            outContentInfo->sizeLow = static_cast<uint32_t>(contentInfo->Size);
            outContentInfo->sizeHigh = static_cast<uint16_t>((contentInfo->Size) >> 32);

            // copy content type
            outContentInfo->type = ConvertType(contentInfo->Type);

            // copy index
            outContentInfo->idOffset = contentInfo->IdOffset;
        }

        static nn::ncm::ContentMetaType ConvertMetaType(String^ type)
        {
            if(type->Equals("Application")) return nn::ncm::ContentMetaType::Application;
            if(type->Equals("Patch")) return nn::ncm::ContentMetaType::Patch;
            if(type->Equals("AddOnContent")) return nn::ncm::ContentMetaType::AddOnContent;
            if(type->Equals("SystemProgram")) return nn::ncm::ContentMetaType::SystemProgram;
            if(type->Equals("SystemData")) return nn::ncm::ContentMetaType::SystemData;
            if(type->Equals("SystemUpdate")) return nn::ncm::ContentMetaType::SystemUpdate;
            if (type->Equals("BootImagePackage")) return nn::ncm::ContentMetaType::BootImagePackage;
            if (type->Equals("BootImagePackageSafe")) return nn::ncm::ContentMetaType::BootImagePackageSafe;
            if (type->Equals("Delta")) return nn::ncm::ContentMetaType::Delta;

            throw gcnew ArgumentException(String::Format("Unknown content meta type {0}.", type));
        }

        static nn::ncm::ContentType ConvertType(String^ type)
        {
            if(type->Equals("Program")) return nn::ncm::ContentType::Program;
            if(type->Equals("Control")) return nn::ncm::ContentType::Control;
            if(type->Equals("Data")) return nn::ncm::ContentType::Data;
            if(type->Equals("Meta")) return nn::ncm::ContentType::Meta;
            if(type->Equals("HtmlDocument")) return nn::ncm::ContentType::HtmlDocument;
            if(type->Equals("LegalInformation")) return nn::ncm::ContentType::LegalInformation;
            if(type->Equals("DeltaFragment")) return nn::ncm::ContentType::DeltaFragment;

            throw gcnew ArgumentException(String::Format("Unknown content type {0}.", type));
        }

        static nn::ncm::UpdateType ConvertUpdateType(String^ type)
        {
            if (type->Equals("ApplyAsDelta")) return nn::ncm::UpdateType::ApplyAsDelta;
            if (type->Equals("Overwrite")) return nn::ncm::UpdateType::Overwrite;
            if (type->Equals("Create")) return nn::ncm::UpdateType::Create;

            throw gcnew ArgumentException(String::Format("Unknown update type {0}.", type));
        }

        template<typename ExtendedHeaderT>
        void WriteExtendedHeader(const ExtendedHeaderT& data)
        {
            m_Writer->WriteExtendedHeader(data);
            GC::KeepAlive(this);
        }
        void WriteExtendedData(const void* data, size_t size)
        {
            m_Writer->WriteExtendedData(data, size);
            GC::KeepAlive(this);
        }
    };

    public ref class NintendoContentMetaReader
    {
    public:
        NintendoContentMetaReader(array<byte>^ buffer) : m_Size(buffer->Length)
        {
            m_Buffer = new char[m_Size];
            pin_ptr<unsigned char> pin = &buffer[0];
            memcpy(m_Buffer, pin, m_Size);
            pin = nullptr;
            m_Reader = new nn::ncm::PackagedContentMetaReader(m_Buffer, m_Size);
            GC::KeepAlive(this);
        }

        ~NintendoContentMetaReader()
        {
            this->!NintendoContentMetaReader();
        }

        !NintendoContentMetaReader()
        {
            delete m_Reader;
            delete[] m_Buffer;
        }

        String^ GetType()
        {
            auto header = m_Reader->GetHeader();
            return Util::ReturnAndDeclareAlive(this, ConvertMetaType(header->type));
        }

        UInt64 GetId()
        {
            auto header = m_Reader->GetHeader();
            return Util::ReturnAndDeclareAlive(this, header->id);
        }

        UInt32 GetVersion()
        {
            auto header = m_Reader->GetHeader();
            return Util::ReturnAndDeclareAlive(this, header->version);
        }

        Byte GetAttributes()
        {
            auto header = m_Reader->GetHeader();
            return Util::ReturnAndDeclareAlive(this, static_cast<nn::Bit8>(header->attributes));
        }

        UInt64 GetApplicationId()
        {
            return Util::ReturnAndDeclareAlive(this, 0);
        }

        UInt32 GetRequiredDownloadSystemVersion()
        {
            auto header = m_Reader->GetHeader();
            return Util::ReturnAndDeclareAlive(this, header->requiredDownloadSystemVersion);
        }

        bool CanUpdateContentMetaKeyGeneration()
        {
            auto contentMetaType = GetType();
            return contentMetaType != NintendoContentMetaConstant::ContentMetaTypeSystemProgram &&
                   contentMetaType != NintendoContentMetaConstant::ContentMetaTypeSystemData &&
                   contentMetaType != NintendoContentMetaConstant::ContentMetaTypeSystemUpdate &&
                   contentMetaType != NintendoContentMetaConstant::ContentMetaTypeBootImagePackage &&
                   contentMetaType != NintendoContentMetaConstant::ContentMetaTypeBootImagePackageSafe;
        }

        List<NintendoContentDescriptor^>^ GetContentDescriptorList()
        {
            auto list = gcnew List<NintendoContentDescriptor^>();
            for (int i = 0; i < m_Reader->CountContent(); i++)
            {
                auto tmpInfo = m_Reader->GetContentInfo(i);
                auto id = gcnew array<byte>(sizeof(nn::ncm::ContentId));
                {
                    pin_ptr<unsigned char> ptr = &id[0];
                    memcpy(ptr, tmpInfo->GetId().data, sizeof(nn::ncm::ContentId));
                    ptr = nullptr;
                }
                auto hash = gcnew array<byte>(sizeof(nn::ncm::Hash));
                {
                    pin_ptr<unsigned char> ptr = &hash[0];
                    memcpy(ptr, tmpInfo->hash.data, sizeof(nn::ncm::Hash));
                    ptr = nullptr;
                }
                auto info = gcnew NintendoContentInfo(ConvertType(tmpInfo->GetType()), id, tmpInfo->info.GetSize(), hash, tmpInfo->info.GetIdOffset());
                auto descriptor = gcnew NintendoContentDescriptor();
                descriptor->ContentInfo = info;
                list->Add(descriptor);
            }
            return Util::ReturnAndDeclareAlive(this, list);
        }

        List<NintendoContentMetaInfo^>^ GetContentMetaInfoList()
        {
            auto list = gcnew List<NintendoContentMetaInfo^>();
            for (int i = 0; i < m_Reader->CountContentMeta(); i++)
            {
                auto tmpInfo = m_Reader->GetContentMetaInfo(i);
                auto info = gcnew NintendoContentMetaInfo(ConvertMetaType(tmpInfo->type), tmpInfo->id, tmpInfo->version, tmpInfo->attributes);
                list->Add(info);
            }
            return Util::ReturnAndDeclareAlive(this, list);
        }

        array<byte>^ GetDigest()
        {
            auto buf = gcnew array<byte>(sizeof(nn::ncm::Hash));
            {
                pin_ptr<unsigned char> ptr = &buf[0];
                memcpy(ptr, m_Reader->GetDigest()->data, sizeof(nn::ncm::Hash));
                ptr = nullptr;
            }
            return Util::ReturnAndDeclareAlive(this, buf);
        }

        NintendoMetaExtendedHeader^ GetExtendedHeader()
        {
            auto header = m_Reader->GetHeader();
            nn::ncm::ContentMetaType type = header->type;
            switch (type)
            {
                case nn::ncm::ContentMetaType::Application:
                {
                    auto exHeader = m_Reader->GetExtendedHeader<nn::ncm::ApplicationMetaExtendedHeader>();
                    return Util::ReturnAndDeclareAlive(this, gcnew NintendoApplicationMetaExtendedHeader(
                        exHeader->patchId.value,
                        exHeader->requiredSystemVersion));
                }
                break;

                case nn::ncm::ContentMetaType::Patch:
                {
                    auto exHeader = m_Reader->GetExtendedHeader<nn::ncm::PatchMetaExtendedHeader>();
                    return Util::ReturnAndDeclareAlive(this, gcnew NintendoPatchMetaExtendedHeader(
                        exHeader->applicationId.value,
                        exHeader->requiredSystemVersion));
                }
                break;

                case nn::ncm::ContentMetaType::AddOnContent:
                {
                    auto exHeader = m_Reader->GetExtendedHeader<nn::ncm::AddOnContentMetaExtendedHeader>();
                    return Util::ReturnAndDeclareAlive(this, gcnew NintendoAddOnContentMetaExtendedHeader(exHeader->applicationId.value, exHeader->requiredApplicationVersion, nullptr));
                }
                break;

                case nn::ncm::ContentMetaType::Delta:
                {
                    auto exHeader = m_Reader->GetExtendedHeader<nn::ncm::DeltaMetaExtendedHeader>();
                    return Util::ReturnAndDeclareAlive(this, gcnew NintendoDeltaMetaExtendedHeader(exHeader->applicationId.value, exHeader->extendedDataSize));
                }
                break;

                case nn::ncm::ContentMetaType::SystemProgram:
                case nn::ncm::ContentMetaType::SystemData:
                case nn::ncm::ContentMetaType::SystemUpdate:
                case nn::ncm::ContentMetaType::BootImagePackage:
                case nn::ncm::ContentMetaType::BootImagePackageSafe:
                {
                    GC::KeepAlive(this);
                    return nullptr;
                }

            default:
                throw gcnew ArgumentException(String::Format("Unknown content meta type."));
            };
        }

        static NintendoContentInfo^ GetContentInfoFromPackaged(const nn::ncm::PackagedContentInfo* contentInfo)
        {
            auto id = gcnew array<byte>(sizeof(nn::ncm::ContentId));
            Marshal::Copy(IntPtr(const_cast<nn::ncm::ContentId*>(&(contentInfo->info.id))), id, 0, id->Length);

            auto hash = gcnew array<byte>(sizeof(nn::ncm::Hash));
            Marshal::Copy(IntPtr(const_cast<nn::ncm::Hash*>(&(contentInfo->hash))), hash, 0, hash->Length);

            return gcnew NintendoContentInfo(ConvertType(contentInfo->GetType()), id, contentInfo->info.GetSize(), hash, contentInfo->info.GetIdOffset());
        }

        static NintendoDeltaMetaExtendedData^ GetDeltaMetaExtendedData(nn::ncm::IDeltaReaderWrapper* reader)
        {
            auto extendedData = gcnew NintendoDeltaMetaExtendedData();
            auto header = reader->GetHeader();

            extendedData->Source->PatchId = header->sourceId.value;
            extendedData->Source->Version = header->sourceVersion;
            extendedData->Destination->PatchId = header->destinationId.value;
            extendedData->Destination->Version = header->destinationVersion;

            auto fragmentSets = gcnew System::Collections::Generic::List<NintendoDeltaMetaExtendedData::NintendoDeltaMetaFragmentSet^>();
            for (int i = 0; i < header->fragmentSetCount; ++i)
            {
                auto fragmentSetRaw = reader->GetFragmentSet(i);

                auto fragmentSet = gcnew NintendoDeltaMetaExtendedData::NintendoDeltaMetaFragmentSet();
                fragmentSet->FragmentTargetContentType = ConvertType(fragmentSetRaw->targetContentType);
                fragmentSet->UpdateType = ConvertUpdateType(fragmentSetRaw->updateType);

                fragmentSet->SourceContentId = gcnew array<byte>(sizeof(nn::ncm::ContentId));
                fragmentSet->DestinationContentId = gcnew array<byte>(sizeof(nn::ncm::ContentId));
                Marshal::Copy(IntPtr(const_cast<nn::ncm::ContentId*>(&(fragmentSetRaw->sourceContentId))), fragmentSet->SourceContentId, 0, fragmentSet->SourceContentId->Length);
                Marshal::Copy(IntPtr(const_cast<nn::ncm::ContentId*>(&(fragmentSetRaw->destinationContentId))), fragmentSet->DestinationContentId, 0, fragmentSet->SourceContentId->Length);

                fragmentSet->SourceSize = fragmentSetRaw->GetSourceSize();
                fragmentSet->DestinationSize = fragmentSetRaw->GetDestinationSize();

                auto fragmentIndicators = gcnew System::Collections::Generic::List<NintendoDeltaMetaExtendedData::NintendoDeltaMetaFragmentIndicator^>();
                for (int j = 0; j < fragmentSetRaw->fragmentCount; ++j)
                {
                    auto fragmentRaw = reader->GetFragmentIndicator(i, j);

                    auto fragmentIndicator = gcnew NintendoDeltaMetaExtendedData::NintendoDeltaMetaFragmentIndicator();
                    fragmentIndicator->ContentInfoIndex = fragmentRaw->contentInfoIndex;
                    fragmentIndicator->FragmentIndex = fragmentRaw->fragmentIndex;

                    fragmentIndicators->Add(fragmentIndicator);
                }
                fragmentSet->Fragments = fragmentIndicators;
                fragmentSets->Add(fragmentSet);
            }
            extendedData->FragmentSets = fragmentSets;

            return extendedData;
        }

        NintendoMetaExtendedData^ GetExtendedData()
        {
            auto header = m_Reader->GetHeader();
            nn::ncm::ContentMetaType type = header->type;
            switch (type)
            {
            case nn::ncm::ContentMetaType::Patch:
            {
                auto exDataSize = m_Reader->GetExtendedDataSize();
                if (exDataSize == 0)
                {
                    return nullptr;
                }
                auto exDataPointer = m_Reader->GetExtendedData();
                std::unique_ptr<nn::ncm::PatchMetaExtendedDataReader> patchExtendedDataReader(new nn::ncm::PatchMetaExtendedDataReader(exDataPointer, exDataSize));
                auto extendedData = gcnew NintendoPatchMetaExtendedData();
                extendedData->Histories = gcnew System::Collections::Generic::List<NintendoPatchMetaExtendedData::NintendoPatchMetaHisotry^>();
                extendedData->Deltas = gcnew System::Collections::Generic::List<NintendoPatchMetaExtendedData::NintendoPatchMetaDelta^>();
                auto patchHeader = patchExtendedDataReader->GetHeader();

                // history の生成
                for (int i = 0; i < static_cast<int>(patchHeader->historyCount); ++i)
                {
                    auto patchHistory = gcnew NintendoPatchMetaExtendedData::NintendoPatchMetaHisotry();
                    patchHistory->ContentInfos = gcnew System::Collections::Generic::List<NintendoContentInfo^>();

                    auto historyRaw = patchExtendedDataReader->GetPatchHistoryHeader(i);
                    patchHistory->Type = ConvertMetaType(historyRaw->key.type);
                    patchHistory->ProgramId = historyRaw->key.id;
                    patchHistory->Version = historyRaw->key.version;
                    patchHistory->Digest = gcnew array<byte>(sizeof(nn::ncm::Hash));
                    Marshal::Copy(IntPtr(const_cast<nn::ncm::Hash*>(&(historyRaw->digest))), patchHistory->Digest, 0, patchHistory->Digest->Length);

                    for (int j = 0; j < historyRaw->contentCount; ++j)
                    {
                        patchHistory->ContentInfos->Add(GetContentInfo(patchExtendedDataReader->GetPatchHistoryContentInfo(i, j)));
                    }
                    extendedData->Histories->Add(patchHistory);
                }
                // delta history の生成
                for (int i = 0; i < static_cast<int>(patchHeader->deltaHistoryCount); ++i)
                {
                    auto deltaHistoryRaw = patchExtendedDataReader->GetPatchDeltaHistory(i);

                    auto patchDeltaHistory = gcnew NintendoPatchMetaExtendedData::NintendoPatchMetaDeltaHisotry();
                    patchDeltaHistory->Delta = gcnew NintendoDeltaMetaExtendedData();
                    patchDeltaHistory->Delta->Source->PatchId = deltaHistoryRaw->sourceId.value;
                    patchDeltaHistory->Delta->Source->Version = deltaHistoryRaw->sourceVersion;
                    patchDeltaHistory->Delta->Destination->PatchId = deltaHistoryRaw->destinationId.value;
                    patchDeltaHistory->Delta->Destination->Version = deltaHistoryRaw->destinationVersion;
                    patchDeltaHistory->DownloadSize = deltaHistoryRaw->downloadSize;
                }

                // delta の生成
                nn::ncm::PatchDeltaReaderWrapper binder(exDataPointer, exDataSize, 0);
                for (int i = 0; i < static_cast<int>(patchHeader->deltaCount); ++i)
                {
                    auto patchDelta = gcnew NintendoPatchMetaExtendedData::NintendoPatchMetaDelta();
                    patchDelta->ContentInfos = gcnew System::Collections::Generic::List<NintendoContentInfo^>();

                    auto deltaRaw = patchExtendedDataReader->GetPatchDeltaHeader(i);

                    for (int j = 0; j < deltaRaw->contentCount; ++j)
                    {
                        patchDelta->ContentInfos->Add(GetContentInfoFromPackaged(patchExtendedDataReader->GetPatchDeltaPackagedContentInfo(i, j)));
                    }

                    binder.SetIndex(i);
                    patchDelta->Delta = GetDeltaMetaExtendedData(&binder);

                    extendedData->Deltas->Add(patchDelta);
                }
                return extendedData;
            }

            case nn::ncm::ContentMetaType::Delta:
            {
                auto exDataSize = m_Reader->GetExtendedDataSize();
                if (exDataSize == 0)
                {
                    return nullptr;
                }
                auto exDataPointer = m_Reader->GetExtendedData();
                nn::ncm::DeltaReaderWrapper reader(m_Buffer, m_Size, exDataPointer, exDataSize);

                return GetDeltaMetaExtendedData(&reader);
            }
            default:
                break;
            }
            return nullptr;
        }
    private:
        char* m_Buffer;
        nn::ncm::PackagedContentMetaReader* m_Reader;
        size_t m_Size;

        static String^ ConvertMetaType(nn::ncm::ContentMetaType type)
        {
            switch (type)
            {
            case nn::ncm::ContentMetaType::Application: return "Application";
            case nn::ncm::ContentMetaType::Patch: return "Patch";
            case nn::ncm::ContentMetaType::AddOnContent: return "AddOnContent";
            case nn::ncm::ContentMetaType::SystemProgram: return "SystemProgram";
            case nn::ncm::ContentMetaType::SystemData: return "SystemData";
            case nn::ncm::ContentMetaType::SystemUpdate: return "SystemUpdate";
            case nn::ncm::ContentMetaType::BootImagePackage: return "BootImagePackage";
            case nn::ncm::ContentMetaType::BootImagePackageSafe: return "BootImagePackageSafe";
            case nn::ncm::ContentMetaType::Delta: return "Delta";
            default:
                throw gcnew ArgumentException(String::Format("Unknown content meta type."));
            }
        }

        static String^ ConvertType(nn::ncm::ContentType type)
        {
            switch (type)
            {
            case nn::ncm::ContentType::Program: return "Program";
            case nn::ncm::ContentType::Control: return "Control";
            case nn::ncm::ContentType::Data: return "Data";
            case nn::ncm::ContentType::Meta: return "Meta";
            case nn::ncm::ContentType::HtmlDocument: return "HtmlDocument";
            case nn::ncm::ContentType::LegalInformation: return "LegalInformation";
            case nn::ncm::ContentType::DeltaFragment: return "DeltaFragment";
            default:
                throw gcnew ArgumentException(String::Format("Unknown content type."));
            }
        }
        static String^ ConvertUpdateType(nn::ncm::UpdateType type)
        {
            switch (type)
            {
            case nn::ncm::UpdateType::ApplyAsDelta: return "ApplyAsDelta";
            case nn::ncm::UpdateType::Overwrite: return "Overwrite";
            case nn::ncm::UpdateType::Create: return "Create";
            default:
                throw gcnew ArgumentException(String::Format("Unknown update type."));
            }
        }
        static NintendoContentInfo^ GetContentInfo(const nn::ncm::ContentInfo* contentInfo)
        {
            auto id = gcnew array<byte>(sizeof(nn::ncm::ContentId));
            Marshal::Copy(IntPtr(const_cast<nn::ncm::ContentId*>(&(contentInfo->id))), id, 0, sizeof(nn::ncm::ContentId));

            auto hash = gcnew array<byte>(sizeof(nn::ncm::Hash));
            UInt64 size = (static_cast<uint64_t>(contentInfo->sizeHigh) << 32) + contentInfo->sizeLow;

            return gcnew NintendoContentInfo(ConvertType(contentInfo->type), id, size, hash, contentInfo->idOffset);
        }
    };

}}}
