﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_NcaHeader.h>
#include <FileSystemInfo.h>
#include <SourceInterface.h>
#include "AesCtrExGenerationTable.h"

#include "../Util/DeclareAlive.h"
#include "NintendoContentArchiveReader.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::Collections;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;
using namespace msclr::interop;
using namespace nn::fssystem;

    public ref class NintendoContentFileSystemMetaConstant
    {
    public:
        literal Int32 ContentTypeProgram = (Int32)NcaHeader::ContentType::Program;
        literal Int32 ContentTypeMeta = (Int32)NcaHeader::ContentType::Meta;
        literal Int32 ContentTypeControl = (Int32)NcaHeader::ContentType::Control;
        literal Int32 ContentTypeManual = (Int32)NcaHeader::ContentType::Manual;
        literal Int32 ContentTypeData = (Int32)NcaHeader::ContentType::Data;
        literal Int32 ContentTypePublicData = (Int32)NcaHeader::ContentType::PublicData;

        literal Int32 HashTypeAuto = (Int32)NcaFsHeader::HashType::Auto;
        literal Int32 HashTypeHierarchicalSha256 = (Int32)NcaFsHeader::HashType::HierarchicalSha256Hash;
        literal Int32 HashTypeHierarchicalIntegrity = (Int32)NcaFsHeader::HashType::HierarchicalIntegrityHash;

        literal Int32 HeaderEncryptionTypeAuto = (Int32)NcaHeader::EncryptionType::Auto;
        literal Int32 HeaderEncryptionTypeNone = (Int32)NcaHeader::EncryptionType::None;

        literal Int32 EncryptionTypeAuto = (Int32)NcaFsHeader::EncryptionType::Auto;
        literal Int32 EncryptionTypeNone = (Int32)NcaFsHeader::EncryptionType::None;
        literal Int32 EncryptionTypeAesCtr = (Int32)NcaFsHeader::EncryptionType::AesCtr;
        literal Int32 EncryptionTypeAesCtrEx = (Int32)NcaFsHeader::EncryptionType::AesCtrEx;

        literal Int32 DecryptionKeyIndexAesCtr   = (Int32)NcaHeader::DecryptionKey_AesCtr;
        literal Int32 DecryptionKeyIndexAesCtrHw = (Int32)NcaHeader::DecryptionKey_AesCtrHw;
        literal Int32 DecryptionKeyIndexMax      = (Int32)NcaHeader::DecryptionKey_Count;

        literal Byte DistributionTypeDownload = (Byte)NcaHeader::DistributionType::Download;
        literal Byte DistributionTypeGameCard = (Byte)NcaHeader::DistributionType::GameCard;

        literal Int32 PartitionAlignmentTypeHashBlockSize = 0;
        literal Int32 PartitionAlignmentTypeSectorOrEncryptionBlockSize = 1;

        literal String^ FormatTypeRomFs                = "RomFs";
        literal String^ FormatTypePatchableRomFs       = "PatchableRomFs";
        literal String^ FormatTypePartitionFs          = "PartitionFs";
        literal String^ FormatTypePatchablePartitionFs = "PatchablePartitionFs";

        literal Byte CurrentKeyGeneration = (Byte)5;
        literal Byte SupportedKeyGenerationMax = (Byte)5;
        literal Byte InvalidKeyGeneration = 0xFF;
    };

    public ref class NintendoContentFileSystemInfo : public FileSystemInfo
    {
    public:
        interface class IHashInfo
        {
        };

        ref class HierarchicalSha256HashInfo : public IHashInfo
        {
        public:
            initonly UInt64 hashDataSize;
            initonly UInt64 hashTargetOffset;
            initonly UInt64 hashTargetSize;

            HierarchicalSha256HashInfo(UInt64 hashDataSize, UInt64 hashTargetOffset, UInt64 hashTargetSize)
            {
                this->hashDataSize = hashDataSize;
                this->hashTargetOffset = hashTargetOffset;
                this->hashTargetSize = hashTargetSize;
            }
        };

        ref class HierarchicalIntegrityHashInfo : public IHashInfo
        {
        public:
            initonly array<Byte>^ metaInfo;
            HierarchicalIntegrityHashInfo(array<Byte>^ metaInfo)
            {
                this->metaInfo = metaInfo;
            }
        };

        value class EntryInfo
        {
        public: // FS n ヘッダ生成用コンテキスト
            String^ formatType;
            int     partitionIndex;
            UInt16 version;
            Byte hashType;
            Byte encryptionType;
            UInt32 secureValue; // [31:24]: programIdOffset, [23:8]: reserved, [7:0]: contentType * partitionIndex
            UInt32 generation;
            UInt64 startOffset;
            UInt64 endOffset;
            IHashInfo^ hashInfo;

        public: // 内部処理用コンテキスト
            String^ type;
            String^ partitionType;
            FileSystemInfo^ fileSystemInfo;
            SourceInterface^ sourceInterface;
            AesCtrExGenerationTable^ generationTable;
            NintendoContentArchivePatchInfo^ patchInfo;
            UInt64 hashTargetOffset; // 消せたら消す
            Boolean existsSparseLayer;

        public:
            void SetHashInfo(UInt64 hashDataSize, UInt64 hashTargetOffset, UInt64 hashTargetSize)
            {
                if (hashType != NintendoContentFileSystemMetaConstant::HashTypeHierarchicalSha256)
                {
                    throw gcnew ArgumentException();
                }
                hashInfo = gcnew HierarchicalSha256HashInfo(hashDataSize, hashTargetOffset, hashTargetSize);
            }
            void SetHashInfo(array<Byte>^ metaInfo)
            {
                if (hashType != NintendoContentFileSystemMetaConstant::HashTypeHierarchicalIntegrity)
                {
                    throw gcnew ArgumentException();
                }
                hashInfo = gcnew HierarchicalIntegrityHashInfo(metaInfo);
            }
            static Byte ConvertSecureValueToIdOffset(UInt32 secureValue)
            {
                return (byte)((secureValue & NcaFsHeader::ProgramIdOffsetMaskOnSecureValue) >> 24);
            }
        };

    public: // NCA ヘッダ生成用コンテキスト
        Byte contentType;
        Byte distributionType;
        Byte keyGeneration;
        Byte keyAreaEncryptionKeyIndex;

        UInt64 contentSize;
        UInt64 programId;
        array<Byte>^ rightsId;
        UInt32 contentIndex;
        UInt32 sdkAddonVersion;

        List<EntryInfo>^ fsEntries;
        array<Byte>^ encryptedKey;

    public: // 内部処理用コンテキスト
        bool isHardwareEncryptionKeyEmbedded;
        Int32 partitionAlignmentType;

        String^ descFileName;
        array<Byte>^ header2SignKeyModulus;
        array<Byte>^ header2SignKeyPublicExponent;
        array<Byte>^ header2SignKeyPrivateExponent;

        Boolean isProdEncryption;
        Nullable<UInt64> contentMetaId;
        Byte headerEncryptionType;

        List<Int32>^ existentFsIndices;

    public:
        NintendoContentFileSystemInfo();
        void GenerateExistentFsIndicesFromFsEntries();
        Byte GetRepresentProgramIdOffset();

    private:
        static Int32 SelectFsIndex(EntryInfo entryInfo);
    };

    public ref class NintendoContentFileSystemMeta
    {
    public:
        literal Int32 HeaderSize = NcaHeader::Size;
        literal Int32 FsHeaderSize = NcaFsHeader::Size;
        literal Int32 PatchInfoSize = NcaPatchInfo::Size;
        literal Int32 PatchInfoOffset = NcaPatchInfo::Offset;
        literal Int32 PatchInfoHeaderSize = NcaBucketInfo::HeaderSize;

        static NcaFsHeader::FsType ConvertFormatTypeToFsType(String^ formatType);
        static property Int32 IntegrityMasterHashOffset
        {
            Int32 get() { return NcaFsHeader::HashData::IntegrityMetaInfo::MasterHashOffset; }
        }
        static property Int32 HierarchicalSha256MasterHashOffset
        {
            Int32 get() { return NcaFsHeader::HashData::HierarchicalSha256Data::MasterHashOffset; }
        }
        static Int32 GetHashBlockSize(Int64 targetSize);
        static array<Byte>^ CreateNcaHeader(NintendoContentFileSystemInfo^ fileSystemInfo);
        static array<Byte>^ CreateNcaFsHeader(NintendoContentFileSystemInfo::EntryInfo^ fsEntry);
        static array<Byte>^ UpdatePatchInfo(array<Byte>^ header, NintendoContentArchivePatchInfo^ patchInfo, UInt32 generation, UInt32 secureValue);
    };

}}}
