﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NOMINMAX

#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <memory>
#include <string>
#include <msclr/marshal.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssrv/fssrv_NcaCryptoConfiguration.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/fssystem/utilTool/fs_BlockSignatureContainer.h>
#include <nn/fssystem/utilTool/fs_StorageChecker.h>
#include <nn/fssystem/utilTool/fs_NcaSparseStorage.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_NcaFileSystemDriverUtility.h>
#include <nn/fssystem/fs_SpeedEmulationConfiguration.h>

#include "../Util/DeclareAlive.h"
#include "DefaultMemoryResource.h"
#include "IndirectStorageArchiveReader.h"
#include "StorageArchiveReader.h"
#include "ManagedStreamStorage.h"
#include "NintendoContentArchiveReader.h"
#include "NintendoContentFileSystemMeta.h"
#include "PartitionFileSystemArchiveReader.h"
#include "RomFsFileSystemArchiveReader.h"
#include "BufferManager.h"
#include "BufferPool.h"

namespace nn { namespace fs { namespace detail {
    void GenerateKeyWin(void* pOutKey, size_t outKeySize, const void* pEncryptedKey, size_t encryptedKeySize, int keyTypeValue, const fssystem::NcaCryptoConfiguration& cryptoConfiguration) NN_NOEXCEPT; // TORIAEZU: 共有
}}}

namespace {
    void DecryptAesCtrFunc(void* pDst, size_t dstSize, int keyTypeValue, const void* pEncryptedKey, size_t encryptedKeySize, const void* pIv, size_t ivSize, const void* pSrc, size_t srcSize)
    {
        // pEncryptedKey をそのまま使って復号化する
        nn::crypto::DecryptAes128Ctr(pDst, dstSize, pEncryptedKey, encryptedKeySize, pIv, ivSize, pSrc, srcSize);
    }

    struct VerifyStorageInfo
    {
        nn::fssystem::utilTool::AesCtrCounterExtendedStorageChecker::StorageInfo info;
        nn::fssystem::IndirectStorage* pIndirectStorage;
    };

}

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Collections;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;

using namespace nn;
using namespace nn::fssystem;

    private class NintendoContentArchiveStorageReaderImpl : public StorageArchiveReaderImplBase
    {
    public:
        typedef NcaFsHeader::HashData::IntegrityMetaInfo::InfoLevelHash LayerInfo;

    public:
        NintendoContentArchiveStorageReaderImpl(std::shared_ptr<fs::IStorage> storage, const NcaFileSystemDriver::StorageOption& option)
            : StorageArchiveReaderImplBase(&m_DataStorage)
            , m_StorageHolder(std::move(storage))
            , m_DataStorage(option.GetDataStorage())
            , m_pSparseStorage(option.GetSparseStorage())
        {
            const auto& headerReader = option.GetHeaderReader();
            if( headerReader.GetHashType() == NcaFsHeader::HashType::HierarchicalIntegrityHash )
            {
                m_LayerInfo.reset(new LayerInfo);
                *m_LayerInfo = headerReader.GetHashData().integrityMetaInfo.infoLevelHash;
            }
        }

        const LayerInfo* GetLayerInfo() const
        {
            return m_LayerInfo.get();
        }

        List<IndirectStorageStream::ExcludeRange>^ CollectExcludeRange()
        {
            List<IndirectStorageStream::ExcludeRange>^ output = gcnew List<IndirectStorageStream::ExcludeRange>();

            if( m_pSparseStorage != nullptr )
            {
                utilTool::SparseStorageBuilder builder;

                auto result = builder.Import(m_pSparseStorage);
                if( result.IsFailure() )
                {
                    throw gcnew Exception(String::Format(
                        "Failed to Import SparseStorageBuilder 0x{0:X8}.", result.GetInnerValueForDebug()));
                }

                int64_t virtualOffset = 0;

                const auto& rangeArray = builder.GetRangeArray();
                for( int i = 0; i < rangeArray.size(); ++i )
                {
                    const auto& range = rangeArray[i];

                    if( virtualOffset < range.virtualOffset )
                    {
                        IndirectStorageStream::ExcludeRange value;
                        value.offset = virtualOffset;
                        value.size = range.virtualOffset - virtualOffset;

                        output->Add(value);
                    }

                    virtualOffset = range.virtualOffset + range.size;
                }

                const auto endOffset = builder.GetOriginalStorageSize();
                if( virtualOffset < endOffset )
                {
                    IndirectStorageStream::ExcludeRange value;
                    value.offset = virtualOffset;
                    value.size = endOffset - virtualOffset;

                    output->Add(value);
                }
            }

            return output;
        }

    private:
        std::shared_ptr<fs::IStorage> m_StorageHolder;
        fs::SubStorage m_DataStorage;
        std::unique_ptr<LayerInfo> m_LayerInfo;
        SparseStorage* m_pSparseStorage;
    };

    NintendoContentArchiveStorageReader::NintendoContentArchiveStorageReader(std::shared_ptr<fs::IStorage> storage, const NcaFileSystemDriver::StorageOption& option)
    {
        m_Impl = new NintendoContentArchiveStorageReaderImpl(std::move(storage), option);
        GC::KeepAlive(this);
    }

    NintendoContentArchiveStorageReader::~NintendoContentArchiveStorageReader()
    {
        this->!NintendoContentArchiveStorageReader();
    }

    NintendoContentArchiveStorageReader::!NintendoContentArchiveStorageReader()
    {
        delete m_Impl;
    }

    array<Byte>^ NintendoContentArchiveStorageReader::Read(Int64 offset, Int32 size)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->Read(offset, size));
    }

    Int64 NintendoContentArchiveStorageReader::GetSize()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetSize());
    }

    List<IndirectStorageStream::ExcludeRange>^
        NintendoContentArchiveStorageReader::CollectExcludeRange(List<IndirectStorageStream::ExcludeRange>^ ranges)
    {
        const NintendoContentArchiveStorageReaderImpl::LayerInfo* pLayerInfo = m_Impl->GetLayerInfo();
        if( pLayerInfo == nullptr )
        {
            return nullptr;
        }

        const auto& dataRange = pLayerInfo->info[pLayerInfo->maxLayers - 2];

        utilTool::BlockSignatureContainer container;
        container.Initialize(*pLayerInfo);

        for each (auto range in ranges)
        {
            if( dataRange.size <= range.offset ||
                dataRange.size < range.offset + range.size )
            {
                throw gcnew System::Exception("Out of data range.");
            }

            container.Collect(range.offset, range.size);
        }
        container.Optimize();

        List<IndirectStorageStream::ExcludeRange>^ output = m_Impl->CollectExcludeRange();

        for( const auto& block : container.GetBlock() )
        {
            IndirectStorageStream::ExcludeRange range;
            range.offset = block.position;
            range.size = block.size;

            output->Add(range);
        }

        return Util::ReturnAndDeclareAlive(this, output);
    }

    private class NintendoContentArchiveReaderImpl
    {
    public:
        NintendoContentArchiveReaderImpl(std::shared_ptr<fs::IStorage> storage, KeyGenerationFunction keyGenFunc, bool isProd)
            : m_NcaReader(std::make_shared<NcaReader>())
            , m_Allocator(new DefaultMemoryResource())
            , m_pBufferManager(GetBufferManager())
            , m_AesCtrGeneration(0)
        {
            EnsureBufferPool();

            auto m_NcaCryptoConfiguration = *fssrv::GetNcaDefaultCryptoConfiguration(isProd);
            m_NcaCryptoConfiguration.pGenerateKey = keyGenFunc;
            m_NcaCryptoConfiguration.pDecryptAesCtr = DecryptAesCtrFunc;
            m_NcaCryptoConfiguration.pDecryptAesCtrForExternalKey = DecryptAesCtrFunc;
            m_NcaCryptoConfiguration.isAvailableRawHeader = !isProd;

            nn::fssystem::SpeedEmulationConfiguration::SetSpeedEmulationMode(nn::fs::SpeedEmulationMode::Faster);

            auto result = m_NcaReader->Initialize(std::move(storage), m_NcaCryptoConfiguration);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to Open Nca 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            CheckAesCtrGeneration();
        }

        List<Tuple<Int32, Int64>^>^ ListFsInfo()
        {
            return ListFsInfo(nullptr, nullptr);
        }

        List<Tuple<Int32, Int64>^>^ ListFsInfo(std::shared_ptr<NcaReader> pOriginalReader, std::shared_ptr<utilTool::NcaSparseStorage> pSparseStorage)
        {
            auto fsInfoList = gcnew List<Tuple<Int32, Int64>^>();
            for each (auto i in GetExistentFsIndices())
            {
                NcaFileSystemDriver ncaFsDriver(pOriginalReader, m_NcaReader, m_Allocator, m_pBufferManager);
                NcaFileSystemDriver::StorageOptionWithHeaderReader option(i);
                option.SetOriginalSparseStorage(pSparseStorage);

                std::shared_ptr<fs::IStorage> storage;
                auto result = ncaFsDriver.OpenStorage(&storage, &option);
                if (result.IsFailure())
                {
                    throw gcnew ArgumentException(String::Format("Failed to OpenStorage {0} 0x{1:X8}.", i, result.GetInnerValueForDebug()));
                }

                int64_t storageSize;
                result = storage->GetSize(&storageSize);
                if (result.IsFailure())
                {
                    throw gcnew ArgumentException(String::Format("Failed to GetStorageSize {0} 0x{1:X8}.", i, result.GetInnerValueForDebug()));
                }

                // TORIAEZU: Tuple(index, storageSize)
                fsInfoList->Add(Tuple::Create(i, storageSize));
            }
            // TODO: 何度も呼び出されるのでキャッシュする？
            return fsInfoList;
        }

        byte GetDistributionType()
        {
            return static_cast<byte>(m_NcaReader->GetDistributionType());
        }

        byte GetContentType()
        {
            return (byte)m_NcaReader->GetContentType();
        }

        byte GetKeyGeneration()
        {
            return m_NcaReader->GetKeyGeneration();
        }

        byte GetKeyIndex()
        {
            return m_NcaReader->GetKeyIndex();
        }

        UInt64 GetProgramId()
        {
            return m_NcaReader->GetProgramId();
        }

        array<byte>^ GetRightsId()
        {
            auto rightsIdBytes = gcnew array<byte>(fssystem::NcaHeader::RightIdSize);
            pin_ptr<unsigned char> ptr = &rightsIdBytes[0];
            m_NcaReader->GetRightsId(ptr, fssystem::NcaHeader::RightIdSize);
            ptr = nullptr;
            return rightsIdBytes;
        }

        UInt32 GetContentIndex()
        {
            return m_NcaReader->GetContentIndex();
        }

        UInt32 GetSdkAddonVersion()
        {
            return m_NcaReader->GetSdkAddonVersion();
        }

        bool HasFsInfo(int index)
        {
            return m_NcaReader->HasFsInfo(index);
        }

        List<Int32>^ GetExistentFsIndices()
        {
            List<Int32>^ listExists = gcnew List<Int32>();
            for (int i = 0; i < NcaHeader::FsCountMax; ++i)
            {
                if (m_NcaReader->HasFsInfo(i))
                {
                    listExists->Add(i);
                }
            }
            return listExists;
        }

        Int64 GetFsStartOffset(int index)
        {
            return m_NcaReader->GetFsOffset(index);
        }

        Int64 GetFsEndOffset(int index)
        {
            return m_NcaReader->GetFsEndOffset(index);
        }

        UInt32 GetAesCtrGeneration()
        {
            return m_AesCtrGeneration;
        }

        bool IsHardwareEncryptionKeyEmbedded()
        {
            return m_NcaReader->HasInternalDecryptionKeyForAesHw();
        }

        void SetExternalKey(array<byte>^ externalKey)
        {
            pin_ptr<unsigned char> ptr = &externalKey[0];
            m_NcaReader->SetExternalDecryptionKey(ptr, externalKey->Length);
            ptr = nullptr;
        }

        bool HasValidInternalKey()
        {
            return m_NcaReader->HasValidInternalKey();
        }

        array<byte>^ GetInternalKey(int index)
        {
            static const auto KeyLength = 16;

            array<byte>^ key = nullptr;

            auto* pKey = m_NcaReader->GetDecryptionKey(index);
            if( pKey != nullptr )
            {
                key = gcnew array<byte>(KeyLength);
                Marshal::Copy(IntPtr(reinterpret_cast<intptr_t>(pKey)), key, 0, KeyLength);
            }

            return key;
        }

        void SetExternalSparseStorage(std::shared_ptr<utilTool::NcaSparseStorage> pStorage)
        {
            if( !pStorage->Verify(*m_NcaReader) )
            {
                throw gcnew System::Exception("Failed to Verify NcaSparseStorage.");
            }

            m_pExternalSparseStorage = std::move(pStorage);
        }

        std::shared_ptr<utilTool::NcaSparseStorage> GetExternalSparseStorage()
        {
            return m_pExternalSparseStorage;
        }

        bool HasExternalSparseStorage()
        {
            return m_pExternalSparseStorage != nullptr;
        }

        void OpenRawStorage(std::shared_ptr<fs::IStorage>* outValue, int index)
        {
            NcaFileSystemDriver ncaFsDriver(m_NcaReader, m_Allocator, m_pBufferManager);
            auto result = ncaFsDriver.OpenRawStorage(outValue, index);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenRawStorage {0} 0x{1:X8}.", index, result.GetInnerValueForDebug()));
            }
        }

        void OpenStorage(std::shared_ptr<fs::IStorage>* outValue, NcaFsHeaderReader* outFsHeaderReader, int index)
        {
            NcaFileSystemDriver ncaFsDriver(m_NcaReader, m_Allocator, m_pBufferManager);
            NcaFileSystemDriver::StorageOption option(outFsHeaderReader, index);

            auto result = ncaFsDriver.OpenStorage(outValue, &option);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenStorage {0} 0x{1:X8}.", index, result.GetInnerValueForDebug()));
            }
        }

        void OpenStorage(std::shared_ptr<fs::IStorage>* outValue, NcaFsHeaderReader* outFsHeaderReader, int index, NintendoContentArchiveReader^ reader)
        {
            NcaFileSystemDriver ncaFsDriver(reader->GetImpl(), m_NcaReader, m_Allocator, m_pBufferManager);
            NcaFileSystemDriver::StorageOption option(outFsHeaderReader, index);
            option.SetOriginalSparseStorage(reader->GetExternalSparseStorage());

            auto result = ncaFsDriver.OpenStorage(outValue, &option);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenStorage {0} 0x{1:X8}.", index, result.GetInnerValueForDebug()));
            }
        }

        void OpenFsDataStorage(std::shared_ptr<fs::IStorage>* outValue, NcaFileSystemDriver::StorageOption* pOption)
        {
            pOption->SetExternalSparseStorage(m_pExternalSparseStorage);

            const auto index = pOption->GetFsIndex();

            // FS 領域がない場合はサイズ 0 のストレージを返す
            if( !m_NcaReader->HasFsInfo(index) )
            {
                *outValue = AllocateShared<fs::MemoryStorage>(__nullptr, 0);
            }
            else
            {
                NcaFileSystemDriver ncaDriver(m_NcaReader, m_Allocator, m_pBufferManager);

                auto result = ncaDriver.OpenDecryptableStorage(outValue, pOption, false);
                if( result.IsFailure() )
                {
                    System::Diagnostics::Debug::Assert(false);
                    throw gcnew ArgumentException(String::Format("Failed to OpenDecryptableStorage {0} 0x{1:X8}.", index, result.GetInnerValueForDebug()));
                }
            }
        }

        void OpenFsHeaderReader(NcaFsHeaderReader* outFsHeaderReader, int index)
        {
            auto result = NcaFileSystemDriver::SetupFsHeaderReader(outFsHeaderReader, *m_NcaReader, index);
            if( result.IsFailure() )
            {
                throw gcnew ArgumentException(String::Format("Failed to Setup FsHeaderReader {0} 0x{1:X8}.", index, result.GetInnerValueForDebug()));
            }
        }

        bool TryOpenFsHeaderReader(NcaFsHeaderReader* outFsHeaderReader, int index)
        {
            return TryOpenFsHeaderReader(outFsHeaderReader, *m_NcaReader, index);
        }

        void Verify()
        {
            // NcaHeader, NcaFsHeader の検証
            auto result = m_NcaReader->Verify();
            if( result.IsFailure() )
            {
                throw gcnew ArgumentException(
                    String::Format("Failed to Verify NcaReader 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            // NcaFileSystemDriver::OpenStorage() が動作するか確認
            for( int i = 0; i < NcaHeader::FsCountMax; ++i )
            {
                NcaFsHeaderReader headerReader;
                if( !TryOpenFsHeaderReader(&headerReader, i) )
                {
                    continue;
                }

                NcaFileSystemDriver driver(m_NcaReader, m_Allocator, GetBufferManager());

                // SparseStorage のチェックは別対応
                if( headerReader.ExistsSparseLayer() )
                {
                    std::shared_ptr<fs::IStorage> storage;
                    NcaFileSystemDriver::StorageOption option(&headerReader, i);

                    result = driver.OpenDecryptableStorage(&storage, &option, false);
                    if( result.IsFailure() )
                    {
                        System::Diagnostics::Debug::Assert(false);
                        throw gcnew ArgumentException(
                            String::Format("Failed to OpenDecryptableStorage NcaFileSystemDriver {0} 0x{1:X8}.", i, result.GetInnerValueForDebug()));
                    }

                    result = utilTool::IndirectStorageChecker::Verify(option.GetSparseStorage());
                    if( result.IsFailure() )
                    {
                        throw gcnew ArgumentException(
                            String::Format("Failed to Verify SparseStorageChecker {0} 0x{1:X8}.", i, result.GetInnerValueForDebug()));
                    }
                }
                else
                {
                    std::shared_ptr<fs::IStorage> storage;

                    result = driver.OpenStorage(&storage, i);
                    if( result.IsFailure() )
                    {
                        throw gcnew ArgumentException(
                            String::Format("Failed to OpenStorage NcaFileSystemDriver {0} 0x{1:X8}.", i, result.GetInnerValueForDebug()));
                    }
                }
            }

            // パッチ単体の検証
            for( int i = 0; i < NcaHeader::FsCountMax; ++i )
            {
                NcaFsHeaderReader headerReader;
                if( !TryOpenFsHeaderReader(&headerReader, i) )
                {
                    continue;
                }
                if( !headerReader.GetPatchInfo().HasAesCtrExTable() )
                {
                    continue;
                }

                VerifyStorageInfo storageInfo = {};
                auto pStorage = GetVerifyStorage(&storageInfo, m_NcaReader, &headerReader, true);
                {
                    auto result = utilTool::IndirectStorageChecker::Verify(storageInfo.pIndirectStorage);
                    if( result.IsFailure() )
                    {
                        throw gcnew ArgumentException(
                            String::Format("Failed to Verify IndirectStorageChecker {0} 0x{1:X8}.", i, result.GetInnerValueForDebug()));
                    }
                }
                {
                    auto result = utilTool::AesCtrCounterExtendedStorageChecker::Verify(storageInfo.info.pCheckStorage, storageInfo.info.generation);
                    if( result.IsFailure() )
                    {
                        throw gcnew ArgumentException(
                            String::Format("Failed to Verify AesCtrCounterExtendedStorageChecker {0} 0x{1:X8}.", i, result.GetInnerValueForDebug()));
                    }
                }
            }
        }

        void Verify(std::shared_ptr<NcaReader> pPrevious)
        {
            for( int i = 0; i < NcaHeader::FsCountMax; ++i )
            {
                NcaFsHeaderReader currentHeaderReader;
                if( !TryOpenFsHeaderReader(&currentHeaderReader, i) )
                {
                    continue;
                }
                if( !currentHeaderReader.GetPatchInfo().HasAesCtrExTable() )
                {
                    continue;
                }

                NcaFsHeaderReader previousHeaderReader;
                if( !TryOpenFsHeaderReader(&previousHeaderReader, *pPrevious, i) )
                {
                    continue;
                }

                VerifyStorageInfo currentStorageInfo = {};
                auto pCurrentStorage = GetVerifyStorage(&currentStorageInfo, m_NcaReader, &currentHeaderReader, false);

                VerifyStorageInfo previousStorageInfo = {};
                auto pPreviousStorage = GetVerifyStorage(&previousStorageInfo, pPrevious, &previousHeaderReader, false);

                utilTool::AesCtrCounterExtendedStorageChecker checker;
                auto result = checker.Verify(previousStorageInfo.info, currentStorageInfo.info);
                if( result.IsFailure() )
                {
                    throw gcnew ArgumentException(
                        String::Format("Failed to Verify AesCtrCounterExtendedStorageChecker {0} 0x{1:X8}.", i, result.GetInnerValueForDebug()));
                }
            }
        }

        byte GetHeaderEncryptionType()
        {
            return (byte)m_NcaReader->GetEncryptionType();
        }

        std::shared_ptr<NcaReader> GetImpl()
        {
            return m_NcaReader;
        }

        nn::MemoryResource* GetAllocator()
        {
            return m_Allocator;
        }

    private:
        static bool TryOpenFsHeaderReader(NcaFsHeaderReader* outFsHeaderReader, const NcaReader& ncaReader, int index)
        {
            auto result = NcaFileSystemDriver::SetupFsHeaderReader(outFsHeaderReader, ncaReader, index);

            if( nn::fs::ResultPartitionNotFound::Includes(result) )
            {
                return false;
            }
            if( result.IsFailure() )
            {
                throw gcnew ArgumentException(String::Format("Failed to Setup FsHeaderReader {0} 0x{1:X8}.", index, result.GetInnerValueForDebug()));
            }

            return true;
        }

        void CheckAesCtrGeneration()
        {
            for( int i = 0; i < NcaHeader::FsCountMax; ++i )
            {
                NcaFsHeaderReader reader;
                if( !TryOpenFsHeaderReader(&reader, i) )
                {
                    continue;
                }
                if( m_AesCtrGeneration < reader.GetAesCtrUpperIv().part.generation )
                {
                    m_AesCtrGeneration = reader.GetAesCtrUpperIv().part.generation;
                }
            }
        }

        std::shared_ptr<fs::IStorage> GetVerifyStorage(VerifyStorageInfo* pOutInfo, std::shared_ptr<NcaReader> pReader, NcaFsHeaderReader* pHeaderReader, bool needsIndirectStorage)
        {
            NcaFileSystemDriver::StorageOption option(pHeaderReader);
            std::shared_ptr<fs::IStorage> pStorage;
            {
                NcaFileSystemDriver driver(pReader, m_Allocator, m_pBufferManager);
                auto result = driver.OpenDecryptableStorage(&pStorage, &option, needsIndirectStorage);
                if( result.IsFailure() )
                {
                    System::Diagnostics::Debug::Assert(false);
                    throw gcnew ArgumentException(
                        String::Format("Failed to OpenDecryptableStorage NcaFileSystemDriver {0} 0x{1:X8}.", pHeaderReader->GetFsIndex(), result.GetInnerValueForDebug()));
                }
            }

            pOutInfo->info.dataOffset = pReader->GetFsOffset(pHeaderReader->GetFsIndex());
            pOutInfo->info.generation = pHeaderReader->GetAesCtrUpperIv().part.generation;
            pOutInfo->info.pDataStorage = option.GetAesCtrExStorage();
            pOutInfo->info.pCheckStorage = option.GetAesCtrExStorageRaw();
            pOutInfo->pIndirectStorage = option.GetIndirectStorage();

            return std::move(pStorage);
        }

    private:
        std::shared_ptr<NcaReader> m_NcaReader;
        nn::MemoryResource* m_Allocator;
        fssystem::NcaCryptoConfiguration m_NcaCryptoConfiguration;
        nn::fssystem::FileSystemBufferManager* m_pBufferManager;
        uint32_t m_AesCtrGeneration;
        std::shared_ptr<utilTool::NcaSparseStorage> m_pExternalSparseStorage;
    };

    NintendoContentArchiveFsHeaderInfo::NintendoContentArchiveFsHeaderInfo()
    {
    }

    NintendoContentArchiveFsHeaderInfo::NintendoContentArchiveFsHeaderInfo(std::unique_ptr<NcaFsHeaderReader> fsHeaderReader, UInt32 generation)
        : m_Version(fsHeaderReader->GetVersion()),
          m_FsType((byte)fsHeaderReader->GetFsType()),
          m_HashType((byte)fsHeaderReader->GetHashType()),
          m_EncryptionType((byte)fsHeaderReader->GetEncryptionType()),
          m_SecureValue(fsHeaderReader->GetAesCtrUpperIv().part.secureValue),
          m_Generation(generation),
          m_pFsHeaderReader(fsHeaderReader.release())
    {
        GC::KeepAlive(this);
    }

    NintendoContentArchiveFsHeaderInfo::~NintendoContentArchiveFsHeaderInfo()
    {
        this->!NintendoContentArchiveFsHeaderInfo();
    }

    NintendoContentArchiveFsHeaderInfo::!NintendoContentArchiveFsHeaderInfo()
    {
        if (m_pFsHeaderReader != nullptr)
        {
            delete m_pFsHeaderReader;
            m_pFsHeaderReader = nullptr;
        }
    }

    Int32 NintendoContentArchiveFsHeaderInfo::GetFsIndex()
    {
        return m_pFsHeaderReader->GetFsIndex();
    }

    UInt16 NintendoContentArchiveFsHeaderInfo::GetVersion()
    {
        return m_Version;
    }

    String^ NintendoContentArchiveFsHeaderInfo::GetFormatType()
    {
        return ConvertFsTypeToFormatType(m_FsType, false);
    }

    String^ NintendoContentArchiveFsHeaderInfo::GetFormatTypeWithPatch() // TODO: ↑と統合
    {
        auto patchInfo = GetPatchInfo();
        return ConvertFsTypeToFormatType(m_FsType, patchInfo != nullptr);
    }

    byte NintendoContentArchiveFsHeaderInfo::GetHashType()
    {
        return m_HashType;
    }

    byte NintendoContentArchiveFsHeaderInfo::GetEncryptionType()
    {
        return m_EncryptionType;
    }

    NintendoContentArchivePatchInfo^ NintendoContentArchiveFsHeaderInfo::GetPatchInfo()
    {
        auto patchInfo = m_pFsHeaderReader->GetPatchInfo();

        const char zero[fssystem::NcaPatchInfo::Size] = {0};
        if (memcmp(zero, &patchInfo, sizeof(zero)) == 0)
        {
            return nullptr;
        }

        auto info = gcnew NintendoContentArchivePatchInfo();
        {
            info->indirectOffset = patchInfo.indirectOffset;
            info->indirectSize = patchInfo.indirectSize;
            info->indirectHeader = gcnew array<Byte>(fssystem::NcaBucketInfo::HeaderSize);
            {
                pin_ptr<unsigned char> ptr = &info->indirectHeader[0];
                memcpy(ptr, patchInfo.indirectHeader, info->indirectHeader->Length);
                ptr = nullptr;
            }
            info->aesCtrExOffset = patchInfo.aesCtrExOffset;
            info->aesCtrExSize = patchInfo.aesCtrExSize;
            info->aesCtrExHeader = gcnew array<Byte>(fssystem::NcaBucketInfo::HeaderSize);
            {
                pin_ptr<unsigned char> ptr = &info->aesCtrExHeader[0];
                memcpy(ptr, patchInfo.aesCtrExHeader, info->aesCtrExHeader->Length);
                ptr = nullptr;
            }
            info->fsHeaderRawData = GetRawData();
            info->masterHash = GetMasterHash();
        }

        return info;
    }

    UInt32 NintendoContentArchiveFsHeaderInfo::GetSecureValue()
    {
        return m_SecureValue;
    }

    UInt32 NintendoContentArchiveFsHeaderInfo::GetGeneration()
    {
        return m_Generation;
    }

    Boolean NintendoContentArchiveFsHeaderInfo::ExistsSparseLayer()
    {
        return m_pFsHeaderReader->ExistsSparseLayer();
    }

    UInt32 NintendoContentArchiveFsHeaderInfo::GetSparseGeneration()
    {
        return m_pFsHeaderReader->GetSparseInfo().GetGeneration();
    }

    array<byte>^ NintendoContentArchiveFsHeaderInfo::GetRawData()
    {
        char data[NcaFsHeader::Size];
        m_pFsHeaderReader->GetRawData(&data, NcaFsHeader::Size);

        array<byte>^ rawData = gcnew array<byte>(NcaFsHeader::Size);
        Marshal::Copy(IntPtr(data), rawData, 0, NcaFsHeader::Size);

        return rawData;
    }

    array<Byte>^ NintendoContentArchiveFsHeaderInfo::GetMasterHash()
    {
        if (m_pFsHeaderReader->GetHashType() == NcaFsHeader::HashType::HierarchicalIntegrityHash)
        {
            auto& info = m_pFsHeaderReader->GetHashData().integrityMetaInfo;
            auto buf = gcnew array<Byte>(info.sizeMasterHash);
            Marshal::Copy(IntPtr(&info.masterHash), buf, 0, buf->Length);
            return buf;
        }
        else if (m_pFsHeaderReader->GetHashType() == NcaFsHeader::HashType::HierarchicalSha256Hash)
        {
            auto& info = m_pFsHeaderReader->GetHashData().hierarchicalSha256Data;
            auto buf = gcnew array<Byte>(sizeof(info.fsDataMasterHash));
            Marshal::Copy(IntPtr(&info.fsDataMasterHash), buf, 0, buf->Length);
            return buf;
        }
        else
        {
            throw gcnew NotImplementedException();
        }
    }

    UInt64 NintendoContentArchiveFsHeaderInfo::GetHashTargetOffset()
    {
        if (m_pFsHeaderReader->GetHashType() == NcaFsHeader::HashType::HierarchicalIntegrityHash)
        {
            auto& layerInfo = m_pFsHeaderReader->GetHashData().integrityMetaInfo.infoLevelHash;
            return layerInfo.info[layerInfo.maxLayers - 2].offset;
        }
        else if (m_pFsHeaderReader->GetHashType() == NcaFsHeader::HashType::HierarchicalSha256Hash)
        {
            auto& layerInfo = m_pFsHeaderReader->GetHashData().hierarchicalSha256Data;
            return layerInfo.hashLayerRegion[layerInfo.hashLayerCount - 1].offset;
        }
        else
        {
            throw gcnew NotImplementedException();
        }
    }

    String^ NintendoContentArchiveFsHeaderInfo::ConvertFsTypeToFormatType(Byte fsType, bool isPatchable)
    {
        if (static_cast<NcaFsHeader::FsType>(fsType) == NcaFsHeader::FsType::RomFs)
        {
            return isPatchable ? NintendoContentFileSystemMetaConstant::FormatTypePatchableRomFs : NintendoContentFileSystemMetaConstant::FormatTypeRomFs;
        }
        else if (static_cast<NcaFsHeader::FsType>(fsType) == NcaFsHeader::FsType::PartitionFs)
        {
            return isPatchable ? NintendoContentFileSystemMetaConstant::FormatTypePatchablePartitionFs : NintendoContentFileSystemMetaConstant::FormatTypePartitionFs;
        }
        else
        {
            throw gcnew ArgumentException("Invalid fs type.");
        }
    }

    NcaFsHeaderReader* NintendoContentArchiveFsHeaderInfo::GetFsHeaderReader()
    {
        return m_pFsHeaderReader;
    }

    NintendoContentArchiveReader::NintendoContentArchiveReader(Stream^ stream, IKeyGenerator^ keyGenerator)
    {
        m_keyGenerator = keyGenerator;
        m_GenerateKeyDelegate = gcnew GenerateKeyDelegate(this, &NintendoContentArchiveReader::GenerateKey);
        m_GenerateKeyFunc = Marshal::GetFunctionPointerForDelegate(m_GenerateKeyDelegate);

        auto storage = std::shared_ptr<fs::IStorage>(new ManagedStreamStorage(stream));
        m_Impl = new NintendoContentArchiveReaderImpl(storage, static_cast<KeyGenerationFunction>(m_GenerateKeyFunc.ToPointer()), !m_keyGenerator->GetUseDevHsm());
        GC::KeepAlive(this);
    }

    NintendoContentArchiveReader::~NintendoContentArchiveReader()
    {
        this->!NintendoContentArchiveReader();
    }

    NintendoContentArchiveReader::!NintendoContentArchiveReader()
    {
        delete m_Impl;
    }

    List<Tuple<Int32, Int64>^>^ NintendoContentArchiveReader::ListFsInfo()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ListFsInfo());
    }

    List<Tuple<Int32, Int64>^>^ NintendoContentArchiveReader::ListFsInfo(NintendoContentArchiveReader^ originalReader)
    {
        if( originalReader == nullptr )
        {
            return Util::ReturnAndDeclareAlive(this, m_Impl->ListFsInfo(nullptr, nullptr));
        }
        else
        {
            return Util::ReturnAndDeclareAlive(this, m_Impl->ListFsInfo(originalReader->GetImpl(), originalReader->GetExternalSparseStorage()));
        }
    }

    byte NintendoContentArchiveReader::GetDistributionType()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetDistributionType());
    }

    byte NintendoContentArchiveReader::GetContentType()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetContentType());
    }

    byte NintendoContentArchiveReader::GetKeyGeneration()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetKeyGeneration());
    }

    byte NintendoContentArchiveReader::GetKeyIndex()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetKeyIndex());
    }

    UInt64 NintendoContentArchiveReader::GetProgramId()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetProgramId());
    }

    array<byte>^ NintendoContentArchiveReader::GetRightsId()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetRightsId());
    }

    UInt32 NintendoContentArchiveReader::GetContentIndex()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetContentIndex());
    }

    UInt32 NintendoContentArchiveReader::GetSdkAddonVersion()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetSdkAddonVersion());
    }

    bool NintendoContentArchiveReader::HasFsInfo(int index)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->HasFsInfo(index));
    }

    List<Int32>^ NintendoContentArchiveReader::GetExistentFsIndices()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetExistentFsIndices());
    }

    Int64 NintendoContentArchiveReader::GetFsStartOffset(int index)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFsStartOffset(index));
    }

    Int64 NintendoContentArchiveReader::GetFsEndOffset(int index)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFsEndOffset(index));
    }

    UInt32 NintendoContentArchiveReader::GetAesCtrGeneration()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetAesCtrGeneration());
    }

    bool NintendoContentArchiveReader::IsHardwareEncryptionKeyEmbedded()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->IsHardwareEncryptionKeyEmbedded());
    }

    void NintendoContentArchiveReader::SetExternalKey(array<byte>^ externalKey)
    {
        m_Impl->SetExternalKey(externalKey);
        GC::KeepAlive(this);
    }

    bool NintendoContentArchiveReader::HasValidInternalKey()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->HasValidInternalKey());
    }

    array<Byte>^ NintendoContentArchiveReader::GetInternalKey(int index)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetInternalKey(index));
    }

    bool NintendoContentArchiveReader::HasSparseStorageArchive()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->HasExternalSparseStorage());
    }

    NintendoContentArchiveFsHeaderInfo^ NintendoContentArchiveReader::GetFsHeaderInfo(int index)
    {
        std::unique_ptr<NcaFsHeaderReader> fsHeaderReader(new NcaFsHeaderReader);
        m_Impl->OpenFsHeaderReader(fsHeaderReader.get(), index);
        return Util::ReturnAndDeclareAlive(this, gcnew NintendoContentArchiveFsHeaderInfo(std::move(fsHeaderReader), m_Impl->GetAesCtrGeneration()));
    }

    byte NintendoContentArchiveReader::GetRepresentProgramIdOffset()
    {
        Nullable<byte> programIdOffset;
        for( int i = 0; i < NcaHeader::FsCountMax; ++i )
        {
            NcaFsHeaderReader fsHeaderReader;
            if( !m_Impl->TryOpenFsHeaderReader(&fsHeaderReader, i) )
            {
                continue;
            }
            if(fsHeaderReader.GetEncryptionType() != NcaFsHeader::EncryptionType::AesCtr &&
               fsHeaderReader.GetEncryptionType() != NcaFsHeader::EncryptionType::AesCtrEx)
            {
                continue;
            }

            auto tmpProgramIdOffset = NintendoContentFileSystemInfo::EntryInfo::ConvertSecureValueToIdOffset(fsHeaderReader.GetAesCtrUpperIv().part.secureValue);
            System::Diagnostics::Trace::Assert(!programIdOffset.HasValue || programIdOffset.Value == tmpProgramIdOffset);
            programIdOffset = tmpProgramIdOffset;
        }
        if (!programIdOffset.HasValue)
        {
            return Util::ReturnAndDeclareAlive(this, 0);
        }
        return Util::ReturnAndDeclareAlive(this, programIdOffset.Value);
    }

    byte NintendoContentArchiveReader::GetHeaderEncryptionType()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetHeaderEncryptionType());
    }

    void NintendoContentArchiveReader::SetExternalSparseStorage(std::shared_ptr<utilTool::NcaSparseStorage> pStorage)
    {
        m_Impl->SetExternalSparseStorage(pStorage);
    }

    std::shared_ptr<utilTool::NcaSparseStorage> NintendoContentArchiveReader::GetExternalSparseStorage()
    {
        return m_Impl->GetExternalSparseStorage();
    }

    NintendoContentArchiveReader::NintendoContentArchiveReader(std::shared_ptr<fs::IStorage> storage, IKeyGenerator^ keyGenerator)
    {
        m_keyGenerator = keyGenerator;
        m_GenerateKeyDelegate = gcnew GenerateKeyDelegate(this, &NintendoContentArchiveReader::GenerateKey);
        m_GenerateKeyFunc = Marshal::GetFunctionPointerForDelegate(m_GenerateKeyDelegate);

        m_Impl = new NintendoContentArchiveReaderImpl(storage, static_cast<KeyGenerationFunction>(m_GenerateKeyFunc.ToPointer()), !m_keyGenerator->GetUseDevHsm());
        GC::KeepAlive(this);
    }

    void NintendoContentArchiveReader::GenerateKey(void* pOutKey, size_t outKeySize, const void* pEncryptedKey, size_t encryptedKeySize, int keyTypeValue, const NcaCryptoConfiguration& cryptoConfiguration)
    {
        auto encryptedKey = gcnew array<Byte>(encryptedKeySize);
        {
            pin_ptr<unsigned char> ptr = &encryptedKey[0];
            memcpy(ptr, pEncryptedKey, encryptedKeySize);
            ptr = nullptr;
        }
        auto key = m_keyGenerator->Generate(encryptedKey, keyTypeValue);
        if (static_cast<size_t>(key->Length) > outKeySize)
        {
            throw gcnew ArgumentException();
        }
        {
            pin_ptr<unsigned char> ptr = &key[0];
            memcpy(pOutKey, ptr, outKeySize);
            ptr = nullptr;
        }
    }

    IFileSystemArchiveReader^ NintendoContentArchiveReader::OpenFileSystemArchiveReader(int index)
    {
        NintendoContentArchiveFsHeaderInfo^ fsInfo;
        return Util::ReturnAndDeclareAlive(this, OpenFileSystemArchiveReader(index, fsInfo, nullptr));
    }

    IFileSystemArchiveReader^ NintendoContentArchiveReader::OpenFileSystemArchiveReader(int index, NintendoContentArchiveReader^ originalReader)
    {
        NintendoContentArchiveFsHeaderInfo^ fsInfo;
        return Util::ReturnAndDeclareAlive(this, OpenFileSystemArchiveReader(index, fsInfo, originalReader));
    }

    IFileSystemArchiveReader^ NintendoContentArchiveReader::OpenFileSystemArchiveReader(int index, NintendoContentArchiveFsHeaderInfo^% fsInfo)
    {
        return Util::ReturnAndDeclareAlive(this, OpenFileSystemArchiveReader(index, fsInfo, nullptr));
    }

    IFileSystemArchiveReader^ NintendoContentArchiveReader::OpenFileSystemArchiveReader(int index, NintendoContentArchiveFsHeaderInfo^% fsInfo, NintendoContentArchiveReader^ originalReader)
    {
        std::shared_ptr<fs::IStorage> storage;

        std::unique_ptr<NcaFsHeaderReader> fsHeaderReader(new NcaFsHeaderReader);
        if( originalReader == nullptr )
        {
            m_Impl->OpenStorage(&storage, fsHeaderReader.get(), index);
        }
        else
        {
            m_Impl->OpenStorage(&storage, fsHeaderReader.get(), index, originalReader);
        }

        auto fsType = fsHeaderReader->GetFsType();
        fsInfo = gcnew NintendoContentArchiveFsHeaderInfo(std::move(fsHeaderReader), m_Impl->GetAesCtrGeneration());

        switch(fsType)
        {
        case fssystem::NcaFsHeader::FsType::RomFs:
        {
            return Util::ReturnAndDeclareAlive(this, gcnew RomFsFileSystemArchiveReader(std::move(storage)));
        }
        case fssystem::NcaFsHeader::FsType::PartitionFs:
        {
            return Util::ReturnAndDeclareAlive(this, gcnew PartitionFileSystemArchiveReader(std::move(storage)));
        }
        default:
            throw Util::ReturnAndDeclareAlive(this, gcnew ArgumentException(String::Format("Invalid FsType.")));
        }
    }

    IStorageArchiveReader^ NintendoContentArchiveReader::OpenFsRawStorageArchiveReader(int index)
    {
        std::shared_ptr<fs::IStorage> pStorage;
        m_Impl->OpenRawStorage(&pStorage, index);

        return Util::ReturnAndDeclareAlive(this, gcnew StorageArchiveReader(std::move(pStorage)));
    }

    IStorageArchiveReader^ NintendoContentArchiveReader::OpenFsDataStorageArchiveReader(int index)
    {
        std::shared_ptr<fs::IStorage> pStorage;
        NcaFileSystemDriver::StorageOptionWithHeaderReader option(index);

        m_Impl->OpenFsDataStorage(&pStorage, &option);

        return Util::ReturnAndDeclareAlive(this, gcnew NintendoContentArchiveStorageReader(pStorage, option));
    }

    IStorageArchiveReader^ NintendoContentArchiveReader::OpenFsDataStorageArchiveReader(NintendoContentArchiveFsHeaderInfo^ fsInfo)
    {
        std::shared_ptr<fs::IStorage> pStorage;
        NcaFileSystemDriver::StorageOption option(fsInfo->GetFsHeaderReader());

        m_Impl->OpenFsDataStorage(&pStorage, &option);

        return Util::ReturnAndDeclareAlive(this, gcnew NintendoContentArchiveStorageReader(pStorage, option));
    }

    NintendoContentArchiveReader::StoragePair NintendoContentArchiveReader::OpenFsDataAndTableStorageArchiveReader(NintendoContentArchiveFsHeaderInfo^ fsInfo)
    {
        std::shared_ptr<fs::IStorage> pStorage;
        NcaFileSystemDriver::StorageOption option(fsInfo->GetFsHeaderReader());

        m_Impl->OpenFsDataStorage(&pStorage, &option);

        auto pTableStorage = option.GetAesCtrExTableStorage();
        if( pTableStorage == nullptr )
        {
            throw gcnew System::Exception("Not found AesCtrCounterExtendedStorage table.");
        }

        StoragePair pair;
        pair.data = gcnew NintendoContentArchiveStorageReader(pStorage, option);
        pair.table = gcnew StorageArchiveReader(pTableStorage, pStorage);

        return Util::ReturnAndDeclareAlive(this, pair);
    }

    void NintendoContentArchiveReader::OpenFsDataStorage(std::shared_ptr<fs::IStorage>* outStorage, NcaFileSystemDriver::StorageOption* pOption)
    {
        m_Impl->OpenFsDataStorage(outStorage, pOption);
    }

    void NintendoContentArchiveReader::Verify()
    {
        m_Impl->Verify();
        GC::KeepAlive(this);
    }

    void NintendoContentArchiveReader::Verify(NintendoContentArchiveReader^ previous)
    {
        m_Impl->Verify(previous->GetImpl());
        GC::KeepAlive(this);
    }

    std::shared_ptr<NcaReader> NintendoContentArchiveReader::GetImpl()
    {
        return m_Impl->GetImpl();
    }

    nn::MemoryResource* NintendoContentArchiveReader::GetAllocator()
    {
        return m_Impl->GetAllocator();
    }

}}}
