﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <map>
#include <nn/nn_Allocator.h>
#include <nn/nn_Macro.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fssystem/fs_BucketTree.h>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>
#include <nn/fssystem/utilTool/fs_BinaryRegionFile.h>
#include <msclr/marshal.h>

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;

public interface class IReadOnlySource
{
    Int32 Read(IntPtr address, Int64 offset, Int32 size);
    Int64 GetSize();
};

class ManagedSourceStorage : public nn::fs::IStorage
{
    NN_DISALLOW_COPY(ManagedSourceStorage);

public:
    NN_IMPLICIT ManagedSourceStorage(IReadOnlySource^ source) NN_NOEXCEPT
        : m_Source(source)
    {
    }

    virtual ~ManagedSourceStorage() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( 0x7FFFFFFF < size )
        {
            return nn::fs::ResultInvalidSize();
        }

        size_t readSize = 0;
        try
        {
            readSize = m_Source->Read(IntPtr(buffer), offset, static_cast<int>(size));
        }
        catch( ... )
        {
            return nn::fs::ResultUnexpected();
        }

        if( readSize < size )
        {
            std::memset(reinterpret_cast<char*>(buffer) + readSize, 0, size - readSize);
        }

        NN_RESULT_SUCCESS;
    }

    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result GetSize(int64_t* pOutSize) NN_NOEXCEPT NN_OVERRIDE
    {
        *pOutSize = m_Source->GetSize();
        NN_RESULT_SUCCESS;
    }

private:
    gcroot<IReadOnlySource^> m_Source;
};

class BinaryRegionInfo
{
public:
    typedef nn::fssystem::utilTool::BinaryRegionFile RegionFile;
    typedef nn::fssystem::utilTool::BinaryRegionArray RegionArray;

public:
    BinaryRegionInfo(String^ originalPath, String^ outputPath, array<Byte>^ rawHeader) NN_NOEXCEPT;

    nn::Result Load(nn::fs::IStorage* pStorage, int64_t storageOffset, size_t blockSize, size_t regionSize) NN_NOEXCEPT;

    void Save(const nn::fssystem::utilTool::BinaryRegionArray& regions) NN_NOEXCEPT;

    bool IsValid() const NN_NOEXCEPT
    {
        return m_pRegionData != nullptr;
    }

    const RegionArray GetRegionArray() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsValid());
        return m_RegionFile.GetRegion();
    }

private:
    std::wstring m_Timestamp;
    std::string m_OutputPath;
    RegionFile m_RegionFile;
    std::unique_ptr<char[]> m_RegionBuffer;
    char* m_pRegionData;
    size_t m_RegionDataSize;
};

class BucketTreeUtility
{
public:
    static void CopyHeader(nn::fssystem::BucketTree::Header* outValue, array<Byte>^ headerData);
};

public value class IndirectStorageProgress
{
public:
    enum class Phase
    {
        MakeRegionHash = int(nn::fssystem::utilTool::BinaryMatchPhase::MakeRegionHash),
        FileBaseMatch = int(nn::fssystem::utilTool::BinaryMatchPhase::FileBaseMatch),
        BinaryMatch1stPass = int(nn::fssystem::utilTool::BinaryMatchPhase::BinaryMatch1stPass),
        BinaryMatch2ndPass = int(nn::fssystem::utilTool::BinaryMatchPhase::BinaryMatch2ndPass),
        Optimize = int(nn::fssystem::utilTool::BinaryMatchPhase::Optimize),
    };

public:
    Phase phase;
    Int64 total;
    Int64 value;

public:
    static bool operator==(IndirectStorageProgress x, IndirectStorageProgress y)
    {
        return (x.phase == y.phase) && (x.total == y.total) && (x.value == y.value);
    }

    static bool operator!=(IndirectStorageProgress x, IndirectStorageProgress y)
    {
        return (x.phase != y.phase) || (x.total != y.total) || (x.value != y.value);
    }

    property Double rate
    {
        Double get()
        {
            if( this->total <= 0 )
            {
                return 0.0;
            }
            if( this->total <= this->value )
            {
                return 1.0;
            }
            return this->value / Double(this->total);
        }
    }
};

}}}
