﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_IndirectStorage.h>
#include <nn/fssystem/fs_NcaFileSystemDriverUtility.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <memory>
#include <msclr/marshal.h>

#include "../Util/DeclareAlive.h"
#include "DefaultMemoryResource.h"
#include "IndirectStorageArchiveReader.h"
#include "ManagedByteArrayStorage.h"
#include "ManagedStreamStorage.h"
#include "NintendoContentArchiveReader.h"
#include "StorageArchiveReaderImplBase.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;

    public class IndirectStorageArchiveReaderImpl : public StorageArchiveReaderImplBase
    {
    private:
        static std::shared_ptr<fs::IStorage> CreateIndirectStorage(MemoryResource* allocator, const BucketTree::Header& header, fs::IStorage* tableStorage, fs::IStorage* oldStorage, fs::IStorage* differenceStorage)
        {
            std::shared_ptr<fssystem::IndirectStorage> indirectStorage(new fssystem::IndirectStorage());

            const auto nodeOffset = 0;
            const auto nodeSize = fssystem::IndirectStorage::QueryNodeStorageSize(header.entryCount);
            const auto entryOffset = nodeOffset + nodeSize;
            const auto entrySize = fssystem::IndirectStorage::QueryEntryStorageSize(header.entryCount);

            auto result = indirectStorage->Initialize(allocator, fs::SubStorage(tableStorage, nodeOffset, nodeSize), fs::SubStorage(tableStorage, entryOffset, entrySize), header.entryCount);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to initialize indirect storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            int64_t oldStorageSize = 0;
            result = oldStorage->GetSize(&oldStorageSize);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to get size of old storage on indirect storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
            int64_t differenceStorageSize = 0;
            result = differenceStorage->GetSize(&differenceStorageSize);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to get size of difference storage on indirect storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            indirectStorage->SetStorage(0, oldStorage, 0, oldStorageSize);
            indirectStorage->SetStorage(1, differenceStorage, 0, differenceStorageSize);

            return std::move(indirectStorage);
        }

    public:
        IndirectStorageArchiveReaderImpl(const BucketTree::Header& header, std::shared_ptr<fs::IStorage> tableStorage, std::shared_ptr<fs::IStorage> oldStorage, std::shared_ptr<fs::IStorage> differenceStorage)
            : StorageArchiveReaderImplBase()
        {
            m_IndirectStorage = CreateIndirectStorage(&m_Allocator, header, tableStorage.get(), oldStorage.get(), differenceStorage.get());
            m_pStorage = m_IndirectStorage.get();
            m_TableStorage = std::move(tableStorage);
            m_OldStorage = std::move(oldStorage);
            m_DifferenceStorage = std::move(differenceStorage);
        }

        List<IndirectStorageEntry^>^ GetFragmentListOnDifference(Int64 offset, Int64 size)
        {
            int entryNum = 0;
            auto indirectStorage = reinterpret_cast<fssystem::IndirectStorage*>(m_IndirectStorage.get());
            auto result = indirectStorage->GetEntryList(nullptr, &entryNum, 0, offset, size);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to get entry list on indirect storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            auto entryList = gcnew List<IndirectStorageEntry^>();
            if (entryNum == 0)
            {
                return entryList;
            }

            std::unique_ptr<fssystem::IndirectStorage::Entry[]> entries(new fssystem::IndirectStorage::Entry[entryNum]);
            if (entries == nullptr)
            {
                throw gcnew ArgumentException("Failed to allocate memory for entry list on indirect storage.");
            }

            result = indirectStorage->GetEntryList(entries.get(), &entryNum, entryNum, offset, size);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to get entry list on indirect storage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            for (int i = 0; i < entryNum; i++)
            {
                auto entryOffset = entries[i].GetVirtualOffset();
                if (entryList->Count == 0)
                {
                    entryOffset += offset - entryOffset;
                }
                int64_t endOffset = i == entryNum - 1 ? offset + size : entries[i + 1].GetVirtualOffset();
                auto entrySize = endOffset - entryOffset;
                entryList->Add(gcnew IndirectStorageEntry(entryOffset - offset, entrySize, entries[i].storageIndex));
            }

            return entryList;
        }

    private:
        DefaultAlignedMemoryResource m_Allocator;
        std::shared_ptr<fs::IStorage> m_TableStorage;
        std::shared_ptr<fs::IStorage> m_OldStorage;
        std::shared_ptr<fs::IStorage> m_DifferenceStorage;
        std::shared_ptr<fs::IStorage> m_IndirectStorage;
    };

    IndirectStorageArchiveReader::IndirectStorageArchiveReader(array<Byte>^ headerData, array<Byte>^ tableData, Stream^ oldStream, Stream^ differenceStream)
    {
        BucketTree::Header header;
        BucketTreeUtility::CopyHeader(&header, headerData);
        auto tableStorage = std::shared_ptr<fs::IStorage>(new ManagedByteArrayStorage(tableData));
        auto oldStorage = std::shared_ptr<fs::IStorage>(new ManagedStreamStorage(oldStream));
        auto differenceStorage = std::shared_ptr<fs::IStorage>(new ManagedStreamStorage(differenceStream));
        m_Impl = new IndirectStorageArchiveReaderImpl(header, std::move(tableStorage), std::move(oldStorage), std::move(differenceStorage));
        GC::KeepAlive(this);
    }

    IndirectStorageArchiveReader::IndirectStorageArchiveReader(NintendoContentArchiveReader^ originalNcaReader, NintendoContentArchiveReader^ patchNcaReader, int fsIndex)
    {
        NcaFileSystemDriver::StorageOptionWithHeaderReader patchOption(fsIndex);
        std::shared_ptr<fs::IStorage> patchDataStorage;

        patchNcaReader->OpenFsDataStorage(&patchDataStorage, &patchOption);

        const auto& patchInfo = patchOption.GetHeaderReader().GetPatchInfo();
        if( !patchInfo.HasIndirectTable() )
        {
            throw gcnew ArgumentException("Failed to open indirect storage. No patch.");
        }

        std::shared_ptr<fs::IStorage> oldStorage;
        if( originalNcaReader != nullptr )
        {
            NcaFileSystemDriver::StorageOptionWithHeaderReader originalOption(fsIndex);
            originalNcaReader->OpenFsDataStorage(&oldStorage, &originalOption);
        }
        else
        {
            oldStorage = fssystem::AllocateShared<fs::MemoryStorage>(__nullptr, 0);
        }

        auto differenceStorage = AllocateShared<fs::SubStorage>(patchDataStorage, 0, patchInfo.indirectOffset);
        auto tableStorage = AllocateShared<fs::SubStorage>(patchDataStorage, patchInfo.indirectOffset, patchInfo.indirectSize);

        BucketTree::Header header;
        std::memcpy(&header, &patchInfo.indirectHeader, sizeof(header));

        m_Impl = new IndirectStorageArchiveReaderImpl(header, std::move(tableStorage), std::move(oldStorage), std::move(differenceStorage));
        GC::KeepAlive(this);
    }

    IndirectStorageArchiveReader::~IndirectStorageArchiveReader()
    {
        this->!IndirectStorageArchiveReader();
    }

    IndirectStorageArchiveReader::!IndirectStorageArchiveReader()
    {
        delete m_Impl;
    }

    array<byte>^ IndirectStorageArchiveReader::Read(Int64 offset, Int32 size)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->Read(offset, size));
    }

    int64_t IndirectStorageArchiveReader::GetSize()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetSize());
    }

    List<IndirectStorageEntry^>^ IndirectStorageArchiveReader::GetFragmentListOnDifference(Int64 offset, Int64 size)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFragmentListOnDifference(offset, size));
    }

}}}
