﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include "IEncryptor.h"

namespace Nintendo { namespace Authoring { namespace CryptoLibrary {

using namespace System;

    public ref class Rsa2048OaepSha256CryptoDriver : IEncryptor
    {
    public:
        static const int _KeySize = 256;
        property int KeySize
        {
            virtual int get() { return _KeySize; }
        };

        Rsa2048OaepSha256CryptoDriver(Rsa2048OaepSha256KeyIndex keyIndex)
        {
            switch (keyIndex)
            {
            case Rsa2048OaepSha256KeyIndex::XcieKeyArea:
                m_KeyModulus = XcieKeyAreaKeyModulus;
                m_KeyPublicExponent = XcieKeyAreaKeyPublicExponent;
                m_LabelDigest = XcieKeyAreaLabelDigest;
                break;
            case Rsa2048OaepSha256KeyIndex::XcirKeyArea:
                m_KeyModulus = XcirKeyAreaKeyModulus;
                m_KeyPublicExponent = XcirKeyAreaKeyPublicExponent;
                m_LabelDigest = XcirKeyAreaLabelDigest;
                break;
            default:
                throw gcnew ArgumentException("Invalid KeyIndex.");
            }
            GC::KeepAlive(this);
        }

        ~Rsa2048OaepSha256CryptoDriver()
        {
            this->!Rsa2048OaepSha256CryptoDriver();
        }

        !Rsa2048OaepSha256CryptoDriver()
        {
        }

        virtual void EncryptBlock(array<Byte>^ src, int srcOffset, int srcSize, array<Byte>^ dst, int dstOffset)
        {
            System::Diagnostics::Debug::Assert(dst->Length - dstOffset >= _KeySize);
            array<unsigned char>^ seed = gcnew array<unsigned char>(nn::crypto::Sha256Generator::HashSize);
            m_Rng.GetBytes(seed);
            pin_ptr<unsigned char> pinSrc = &src[srcOffset];
            pin_ptr<unsigned char> pinDst = &dst[dstOffset];
            pin_ptr<unsigned char> pinKeyModulus = &m_KeyModulus[0];
            pin_ptr<unsigned char> pinKeyPublicExponent = &m_KeyPublicExponent[0];
            pin_ptr<unsigned char> pinLabelDigest = &m_LabelDigest[0];
            pin_ptr<unsigned char> pinSeed = &seed[0];

            if (m_LabelDigest->Length != 0)
            {
                nn::crypto::RsaOaepEncryptor<_KeySize, nn::crypto::Sha256Generator> rsaEncryptor;
                rsaEncryptor.Initialize(pinKeyModulus, m_KeyModulus->Length, pinKeyPublicExponent, m_KeyPublicExponent->Length);
                rsaEncryptor.SetLabelDigest(pinLabelDigest, m_LabelDigest->Length);
                rsaEncryptor.Encrypt(pinDst, _KeySize, pinSrc, srcSize, pinSeed, seed->Length);
            }
            else
            {
                nn::crypto::RsaOaepEncryptor<_KeySize, nn::crypto::Sha256Generator>::Encrypt(
                    pinDst, _KeySize,
                    pinKeyModulus, m_KeyModulus->Length,
                    pinKeyPublicExponent, m_KeyPublicExponent->Length,
                    pinSrc, srcSize,
                    pinSeed, seed->Length,
                    "", 0);
            }

            pinKeyModulus = nullptr;
            pinKeyPublicExponent = nullptr;
            pinLabelDigest = nullptr;
            pinSeed = nullptr;
            pinSrc = nullptr;
            pinDst = nullptr;

            GC::KeepAlive(this);
        }

        virtual void DecryptBlock(array<Byte>^ src, int srcOffset, int srcSize, array<Byte>^ dst, int dstOffset)
        {
            throw gcnew NotImplementedException();
        }

    private:
        System::Security::Cryptography::RNGCryptoServiceProvider m_Rng;

        array<Byte>^ m_KeyModulus;
        array<Byte>^ m_KeyPublicExponent;
        array<Byte>^ m_LabelDigest;

        static array<Byte>^ XcieKeyAreaLabelDigest =
        {
            0xB1, 0x7B, 0x5F, 0x65, 0x0E, 0x75, 0x90, 0xED, 0xFE, 0x3A, 0xF0, 0x75, 0x0B, 0x92, 0x05, 0xB3,
            0x0E, 0x7D, 0x49, 0x89, 0x24, 0xD8, 0xE3, 0x85, 0xCB, 0x40, 0xE0, 0xE7, 0xF8, 0xDE, 0xF0, 0x77
        };

        static array<Byte>^ XcieKeyAreaKeyPublicExponent =
        {
            0x01, 0x00, 0x01
        };

        static array<Byte>^ XcieKeyAreaKeyModulus =
        {
            0xa6, 0x1a, 0x3f, 0x5e, 0x16, 0x94, 0x4d, 0x5f, 0x88, 0x07, 0x45, 0x92, 0x42, 0x02, 0x5c, 0xf8,
            0xf8, 0x01, 0xa6, 0x8f, 0xf0, 0x77, 0x76, 0x06, 0x06, 0x4b, 0x67, 0x02, 0x35, 0x2f, 0xc2, 0xf6,
            0xcd, 0x74, 0xbc, 0x57, 0x36, 0xa0, 0x61, 0x1d, 0xd6, 0x66, 0x35, 0x79, 0x6e, 0xb7, 0xff, 0xc1,
            0x05, 0xc6, 0x59, 0xd9, 0x1e, 0xc9, 0xc3, 0x1f, 0x5b, 0x17, 0xa9, 0x24, 0x46, 0xb3, 0xab, 0x1d,
            0x7b, 0x79, 0xa8, 0xf2, 0x80, 0xc8, 0x6f, 0xa7, 0x9b, 0x2d, 0xe8, 0xbf, 0x91, 0xd7, 0xd4, 0xa2,
            0x07, 0x53, 0x98, 0x82, 0x7d, 0x91, 0x8a, 0x68, 0x79, 0x9b, 0xb6, 0x8f, 0xa7, 0x05, 0x7b, 0x2b,
            0x6c, 0x09, 0x2a, 0x58, 0x8a, 0x21, 0xe5, 0xb6, 0xcd, 0xac, 0xcb, 0x84, 0x5f, 0x22, 0x16, 0x82,
            0x54, 0x14, 0x03, 0xeb, 0xbb, 0x86, 0xec, 0x59, 0x36, 0x38, 0xf8, 0x41, 0x41, 0xca, 0x23, 0x8a,
            0xf2, 0x98, 0x01, 0xad, 0x14, 0xb6, 0x65, 0x15, 0x3b, 0x63, 0x42, 0xc1, 0x3e, 0x55, 0x8c, 0xc2,
            0x38, 0x4e, 0xc0, 0x57, 0x5e, 0xfa, 0x2a, 0x2c, 0xc7, 0x9c, 0xbe, 0x5c, 0xd7, 0x56, 0x4d, 0xcf,
            0x36, 0x31, 0x93, 0xd2, 0xb1, 0x0a, 0x8c, 0x7c, 0x7f, 0xfe, 0x46, 0x20, 0xe3, 0x91, 0x7c, 0x63,
            0x3f, 0x24, 0x81, 0xff, 0x4c, 0x68, 0x4f, 0xbe, 0x16, 0x68, 0x66, 0x1c, 0x90, 0xcd, 0xfe, 0x8d,
            0xc5, 0x63, 0x3b, 0x2d, 0x5c, 0x25, 0x95, 0xa8, 0x4b, 0x3c, 0xaf, 0x57, 0x07, 0x84, 0x31, 0x37,
            0xfa, 0x63, 0x98, 0x1b, 0xac, 0xb2, 0x81, 0xb1, 0x29, 0x67, 0x0a, 0x31, 0x3d, 0x7e, 0x2d, 0x7f,
            0xfe, 0xa5, 0x29, 0x3d, 0xbf, 0xd3, 0xcc, 0x9d, 0x73, 0x75, 0x40, 0x8b, 0x79, 0x98, 0x39, 0x8a,
            0xad, 0x90, 0x65, 0x82, 0x90, 0x09, 0xa4, 0x74, 0xa2, 0x3f, 0xee, 0xc7, 0x75, 0x06, 0x29, 0x85
        };

        static array<Byte>^ XcirKeyAreaLabelDigest =
        {
            0x97, 0x6C, 0x3F, 0xE3, 0x28, 0xF3, 0xAC, 0xEC, 0x2B, 0x71, 0x64, 0xB2, 0xB5, 0x98, 0x55, 0x8A,
            0xAB, 0x03, 0x60, 0x8C, 0x2A, 0xF6, 0x97, 0x00, 0xBB, 0xD5, 0x6D, 0x20, 0xE5, 0xC6, 0xD4, 0x1F
        };

        static array<Byte>^ XcirKeyAreaKeyPublicExponent =
        {
            0x01, 0x00, 0x01
        };

        static array<Byte>^ XcirKeyAreaKeyModulus =
        {
            0xde, 0xda, 0xfa, 0xac, 0x44, 0xcf, 0x23, 0xcf, 0xdb, 0xeb, 0x60, 0x8e, 0x7c, 0xb0, 0x31, 0x9c,
            0xbe, 0x59, 0xd6, 0x8c, 0x69, 0x7b, 0x84, 0x06, 0xd5, 0xdf, 0x67, 0x97, 0xc3, 0x5a, 0x72, 0xa0,
            0xec, 0xf5, 0x49, 0x76, 0xaa, 0x7e, 0xef, 0x7b, 0xe6, 0xbd, 0x41, 0x6b, 0x2f, 0x3f, 0xe6, 0x08,
            0x56, 0x0d, 0x1a, 0x93, 0x17, 0xd0, 0x65, 0xd4, 0x00, 0x12, 0xda, 0xeb, 0x5c, 0xf9, 0xac, 0xf5,
            0x3d, 0xb8, 0x74, 0x7c, 0x6c, 0x1d, 0x24, 0x23, 0x68, 0x40, 0x4b, 0x2b, 0xc1, 0xb4, 0x7f, 0x8a,
            0xd9, 0x05, 0x19, 0xbd, 0x80, 0x3a, 0xd3, 0xd9, 0x94, 0xd1, 0xa3, 0xb4, 0x2f, 0x8e, 0xcd, 0x77,
            0x71, 0xbd, 0x6d, 0x68, 0x21, 0x6a, 0xb6, 0x25, 0x56, 0x25, 0x19, 0x35, 0xd0, 0x8f, 0xc1, 0x02,
            0x88, 0xbb, 0xbc, 0xcd, 0x1c, 0x4d, 0xd4, 0x3c, 0x87, 0x4c, 0x6d, 0x46, 0x9c, 0x5f, 0x4f, 0xc8,
            0xac, 0x42, 0x76, 0xf6, 0xe5, 0xb0, 0x65, 0xbd, 0xa0, 0x86, 0x25, 0x7b, 0x94, 0x20, 0x60, 0xa2,
            0x02, 0x5f, 0x14, 0x6c, 0x5e, 0xac, 0xcb, 0x69, 0x80, 0x99, 0xe0, 0x51, 0xc8, 0xfa, 0x80, 0x93,
            0x61, 0xcf, 0x94, 0xfc, 0x47, 0xc6, 0x14, 0x20, 0xc3, 0x9f, 0x15, 0x33, 0x17, 0x50, 0xd5, 0x3e,
            0xe2, 0x63, 0xe9, 0xfb, 0xb3, 0xd1, 0x4c, 0xbb, 0x45, 0x8f, 0xdb, 0x43, 0x4d, 0xaa, 0x6f, 0x09,
            0xec, 0x8c, 0xaf, 0xf7, 0xc8, 0xdd, 0xf2, 0x0c, 0x7d, 0x20, 0x0d, 0xae, 0x09, 0x07, 0x77, 0xa5,
            0x75, 0x60, 0x8a, 0x36, 0xc4, 0x0e, 0x0c, 0x3f, 0x2a, 0xde, 0xac, 0x72, 0x2b, 0x99, 0x31, 0xce,
            0xa3, 0x49, 0x97, 0xf9, 0x49, 0x9d, 0x0e, 0x98, 0xa4, 0xf0, 0x8d, 0xce, 0xcd, 0xab, 0xe2, 0x3d,
            0x34, 0x21, 0x8d, 0xe3, 0x12, 0x15, 0x2e, 0x06, 0x9f, 0x2b, 0x4a, 0x95, 0x31, 0x6f, 0xad, 0x4d
        };
    };




}}}
