﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include "cryptoki.h"
#include <msclr/marshal.h>
#include <windows.h>

#include "IEncryptor.h"
#include "IHashCalculator.h"
#include "ISigner.h"

namespace Nintendo { namespace Authoring { namespace CryptoLibrary {

using namespace System;
using namespace System::Runtime::InteropServices;

    class HsmInterfaceManager;
    public class HsmInterfaceImpl
    {
    public:
        HsmInterfaceImpl(::std::shared_ptr<HsmInterfaceManager> pManager);
        ~HsmInterfaceImpl() NN_NOEXCEPT;

        void SignRsa2048PssSha256(::CK_OBJECT_HANDLE objectHandle, void* pOutSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize);
        bool VerifyRsa2048PssSha256(::CK_OBJECT_HANDLE objectHandle, void* pSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize);

        void SignRsa2048Pkcs1Sha256(::CK_OBJECT_HANDLE objectHandle, void* pOutSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize);
        bool VerifyRsa2048Pkcs1Sha256(::CK_OBJECT_HANDLE objectHandle, void* pSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize);

        void GetRsa2048PssSha256PublicKey(::CK_OBJECT_HANDLE objectHandle, void* pOutModulus, size_t modulusSize, void* pOutPublicExponent, size_t publicExponentSize);
        void GetRsa2048Pkcs1Sha256PublicKey(::CK_OBJECT_HANDLE objectHandle, void* pOutModulus, size_t modulusSize, void* pOutPublicExponent, size_t publicExponentSize);

        void EncryptAes128(::CK_OBJECT_HANDLE objectHandle, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);
        void DecryptAes128(::CK_OBJECT_HANDLE objectHandle, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);

        void EncryptAes128Cbc(::CK_OBJECT_HANDLE objectHandle, void* pDst, size_t dstSize, void* pSrc, size_t srcSize, void* pIv, size_t ivSize);
        void DecryptAes128Cbc(::CK_OBJECT_HANDLE objectHandle, void* pDst, size_t dstSize, void* pSrc, size_t srcSize, void* pIv, size_t ivSize);

        void GenerateHmacSha256(::CK_OBJECT_HANDLE objectHandle, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);

        ::CK_OBJECT_HANDLE FindObject(const char* labelName);

    private:
        void LoadObject();

        void EncryptAes(::CK_OBJECT_HANDLE objectHandle, const ::CK_MECHANISM& mech, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);
        void DecryptAes(::CK_OBJECT_HANDLE objectHandle, const ::CK_MECHANISM& mech, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);

        ::CK_FUNCTION_LIST* m_P11Functions;
        ::CK_SESSION_HANDLE m_SessionHandle;

        ::CK_UTF8CHAR_PTR m_Pin;
        ::CK_ULONG m_PinLength;

        ::std::shared_ptr<HsmInterfaceManager> m_Manager;

        static const int KeyLabelLengthMax = 128;

        class ScopedLogin
        {
        public:
            ScopedLogin(::CK_SESSION_HANDLE sessionHandle, ::CK_FUNCTION_LIST* pP11Functions, ::CK_UTF8CHAR_PTR pPin, ::CK_ULONG pinLength);
            ~ScopedLogin() NN_NOEXCEPT;
        private:
            ::CK_FUNCTION_LIST* m_P11Functions;
            ::CK_SESSION_HANDLE m_SessionHandle;
        };
    };

    public class HsmInterfaceManager
    {
    public:
        ~HsmInterfaceManager()
        {
            FinalizeCryptoki();
            UnloadModule();
        }

        static ::std::shared_ptr<HsmInterfaceManager> GetInstance();

        ::CK_FUNCTION_LIST* GetFunctionList()
        {
            return m_P11Functions;
        }

        ::CK_SLOT_ID GetSlotId()
        {
            return m_UseDev ? m_SlotIdForDev : m_SlotIdForProd;
        }

        ::CK_UTF8CHAR_PTR GetPin()
        {
            return m_Pin.get();
        }

        ::CK_ULONG GetPinLength()
        {
            return m_PinLength;
        }

        bool GetUseDev()
        {
            return m_UseDev;
        }

    private:
        HsmInterfaceManager();

        void Initialize();

        void LoadModule();
        void UnloadModule();

        void InitializeCryptoki();
        void FinalizeCryptoki();

        void SetSlotId();
        void SetPin();

        array<Byte>^ GetPinFromEnv(String^ env);

        ::HINSTANCE m_LibHandle;
        ::CK_FUNCTION_LIST* m_P11Functions;
        ::CK_SLOT_ID m_SlotIdForProd;
        ::CK_SLOT_ID m_SlotIdForDev;
        ::std::unique_ptr<::CK_UTF8CHAR> m_Pin;
        ::CK_ULONG m_PinLength;
        bool m_UseDev;
    };

    public ref class HsmInterface
    {
    public:
        HsmInterface()
        {
            auto pManager = HsmInterfaceManager::GetInstance();
            m_Impl = new HsmInterfaceImpl(pManager);
            m_UseDev = pManager->GetUseDev();
            GC::KeepAlive(this);
        }

        static HsmInterface()
        {
            m_LockObject = gcnew Object();
        }

        ~HsmInterface()
        {
            this->!HsmInterface();
        };

        !HsmInterface()
        {
            delete m_Impl;
        }

        bool GetUseDev()
        {
            return m_UseDev;
        }

        void SignRsa2048PssSha256(Rsa2048PssSha256KeyIndex index, void* pOutSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize);
        bool VerifyRsa2048PssSha256(Rsa2048PssSha256KeyIndex index, void* pSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize);
        void GetRsa2048PssSha256PublicKey(Rsa2048PssSha256KeyIndex index, void* pOutModulus, size_t modulusSize, void* pOutPublicExponent, size_t publicExponentSize);

        void SignRsa2048Pkcs1Sha256(Rsa2048Pkcs1Sha256KeyIndex index, void* pOutSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize);
        bool VerifyRsa2048Pkcs1Sha256(Rsa2048Pkcs1Sha256KeyIndex index, void* pSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize);
        void GetRsa2048Pkcs1Sha256PublicKey(Rsa2048Pkcs1Sha256KeyIndex index, void* pOutModulus, size_t modulusSize, void* pOutPublicExponent, size_t publicExponentSize);

        void EncryptAes128(Aes128KeyIndex index, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);
        void DecryptAes128(Aes128KeyIndex index, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);
        void EncryptAes128(Aes128KeyIndex index, byte keyGeneration, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);
        void DecryptAes128(Aes128KeyIndex index, byte keyGeneration, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);

        void EncryptAes128Cbc(Aes128KeyIndex index, void* pDst, size_t dstSize, void* pSrc, size_t srcSize, void* pIv, size_t ivSize);
        void DecryptAes128Cbc(Aes128KeyIndex index, void* pDst, size_t dstSize, void* pSrc, size_t srcSize, void* pIv, size_t ivSize);
        void EncryptAes128Cbc(Aes128KeyIndex index, byte keyGeneration, void* pDst, size_t dstSize, void* pSrc, size_t srcSize, void* pIv, size_t ivSize);
        void DecryptAes128Cbc(Aes128KeyIndex index, byte keyGeneration, void* pDst, size_t dstSize, void* pSrc, size_t srcSize, void* pIv, size_t ivSize);

        void GenerateHmacSha256(HmacSha256KeyIndex index, void* pDst, size_t dstSize, void* pSrc, size_t srcSize);

    private:
        ::CK_OBJECT_HANDLE GetRsa2048PssSha256KeyObjectHandle(Rsa2048PssSha256KeyIndex index, bool isPrivate);
        ::CK_OBJECT_HANDLE GetRsa2048Pkcs1Sha256KeyObjectHandle(Rsa2048Pkcs1Sha256KeyIndex index, bool isPrivate);
        ::CK_OBJECT_HANDLE GetAes128KeyObjectHandle(Aes128KeyIndex index, byte keyGeneration);
        ::CK_OBJECT_HANDLE GetHmacSha256ObjectHandle(HmacSha256KeyIndex index);

        static Object^ m_LockObject;

        HsmInterfaceImpl* m_Impl;
        bool m_UseDev;
    };

}}}
