﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Collections.Generic;
using System.Xml.Serialization;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class XmlModelWriter
    {
        static readonly private int IndentSize = 2;

        static private bool IsExistWritableChildElement(Type root)
        {
            // 書き出し可能な子要素がないかをチェック

            var elements = root.GetProperties();
            if (elements.Count() == 0)
            {
                // 子プロパティが存在しない
                return false;
            }

            if (root.GetCustomAttributes(typeof(XmlRootAttribute), false).Count() > 0)
            {
                // XML属性がXmlRoot
                return true;
            }

            if (elements.Where(entry => entry.GetCustomAttributes(typeof(XmlElementAttribute), false).Count() > 0).Count() > 0)
            {
                // 属性がXmlElementAttributeの子プロパティが存在する
                return true;
            }

            return false;
        }

        static private void XmlModelRecursiveWrite(StreamWriter sw, Type root, List<string> parentModelList, string tagName, int indent)
        {
            string indentString = new string(' ', indent * IndentSize);
            indent += IndentSize;

            if (!IsExistWritableChildElement(root) || parentModelList.Contains(root.Name))
            {
                // 書き出し可能な子要素がない or 上位階層に同じXmlModelが存在する場合、子要素は無視する
                sw.WriteLine(indentString + "<" + tagName + ">" + "</" + tagName + ">");
                return;
            }
            var elements = root.GetProperties();

            sw.WriteLine(indentString + "<" + tagName + ">");

            foreach (var element in elements)
            {
                parentModelList.Add(root.Name); // 親Modelのリストに自分を追加

                string elementName = null;
                var elementAttributes = element.GetCustomAttributes(typeof(XmlElementAttribute), false);
                if (elementAttributes.Count() > 0)
                {
                    elementName = (elementAttributes.First() as XmlElementAttribute).ElementName;
                }
                else
                {
                    // XmlElementAttributeが設定されていない場合は、プロパティ名をタグ名にする
                    elementName = element.Name;
                }

                if (element.PropertyType.IsGenericType && element.PropertyType.GetGenericTypeDefinition() == typeof(List<>))
                {
                    var type = element.PropertyType.GetGenericArguments().First(); // List<>の中身を取得
                    XmlModelRecursiveWrite(sw, type, parentModelList, elementName, indent);
                }
                else
                {
                    XmlModelRecursiveWrite(sw, element.PropertyType, parentModelList, elementName, indent);
                }
                parentModelList.Remove(root.Name);
            }

            sw.WriteLine(indentString + "</" + tagName + ">");
        }
        static private void XmlModelWrite(StreamWriter sw, Type type)
        {
            var element = type.GetCustomAttributes(typeof(XmlRootAttribute), false).Single();
            XmlModelRecursiveWrite(sw, type, new List<string>(), (element as XmlRootAttribute).ElementName, 0);
        }
        static public void XmlModelWrite(StreamWriter sw, List<Type> types)
        {
            foreach (var type in types)
            {
                XmlModelWrite(sw, type);
            }

            sw.Flush();
        }
    }
}
