﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using Nintendo.Authoring.ETicketLibrary;
using Nintendo.Authoring.CryptoLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class TicketSource : ISource
    {
        private ISource TicketBufferSource;

        public TicketSource(Byte[] ticket, int ticketLength)
        {
            TicketBufferSource = new MemorySource(ticket, 0, ticketLength);
        }

        public long Size
        {
            get
            {
                return TicketBufferSource.Size;
            }
        }

        public ByteData PullData(long offset, int size)
        {
            return TicketBufferSource.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return TicketBufferSource.QueryStatus();
        }
    }

    public class Ticket
    {
        private const int MaxTicketLength = 1024;
        private int m_TicketDataLength;
        private Byte[] m_TicketData;

        public Ticket()
        {
            m_TicketData = new Byte[MaxTicketLength];
        }

        public int Length
        {
            get
            {
                return m_TicketDataLength;
            }
        }

        public Byte[] Data
        {
            get
            {
                return m_TicketData;
            }
        }

        public Byte[] PublishTicket(UInt64 titleId, bool isProdEncryption, byte keyGeneration, KeyConfiguration keyConfiguration)
        {
            // 生のタイトルキーを作成する
            IHashCalculator hashCalculator;
            if (isProdEncryption)
            {
                if (keyConfiguration.GetProdTitleKeyGenarateKey() != null)
                {
                    hashCalculator = new HmacSha256HashCryptoDriver(keyConfiguration.GetProdTitleKeyGenarateKey().Key);
                }
                else
                {
                    hashCalculator = new HsmHmacSha256HashCryptoDriver(HmacSha256KeyIndex.TitleKeyGenarateKey);
                }
            }
            else
            {
                hashCalculator = new HmacSha256HashCryptoDriver(HmacSha256KeyIndex.TitleKeyGenarateKey);
            }

            AesKey titleKey = ExternalContentKeyGenerator.GetNcaExternalContentKey(hashCalculator, titleId, keyGeneration);

            // タイトルキーを暗号化する
            ICbcModeEncryptor encryptor;
            if (isProdEncryption)
            {
                if (keyConfiguration.GetProdETicketCommonKey(keyGeneration) != null)
                {
                    encryptor = new Aes128CbcCryptoDriver(keyConfiguration.GetProdETicketCommonKey(keyGeneration).Key);
                }
                else
                {
                    encryptor = new HsmAes128CbcCryptoDriver(Aes128KeyIndex.ETicketCommonKey, keyGeneration);
                }
            }
            else
            {
                encryptor = new Aes128CbcCryptoDriver(Aes128KeyIndex.ETicketCommonKey, keyGeneration);
            }

            AesKey encryptedTitleKey = EncryptTitleKey(encryptor, titleKey);

            // チケットを発行する
            UInt64 deviceId = 0;
            UInt64 ticketId = 0;
            Byte[] rightsId = TicketUtility.CreateRightsId(titleId, keyGeneration);

            if (isProdEncryption)
            {
                if (keyConfiguration.GetProdETicketSignKey() != null)
                {
                    m_TicketDataLength = (int)TicketPublication.PublishTicket(m_TicketData, (uint)m_TicketData.Length, encryptedTitleKey.Key, deviceId, ticketId, rightsId, keyGeneration, System.Text.Encoding.ASCII.GetBytes("Root-CA00000004-XS00000020"));
                }
                else
                {
                    // 開発版 HSM サーバを使うかどうか
                    HsmInterface hsmInterfase = new HsmInterface();

                    if (hsmInterfase.GetUseDev())
                    {
                        // devel-XS00000021 で署名をする(CA:CA00000004)
                        m_TicketDataLength = (int)TicketPublication.PublishTicket(m_TicketData, (uint)m_TicketData.Length, encryptedTitleKey.Key, deviceId, ticketId, rightsId, keyGeneration, System.Text.Encoding.ASCII.GetBytes("Root-CA00000004-XS00000021"));
                    }
                    else
                    {
                        // prod-XS00000020 で署名をする(CA:CA00000003)
                        m_TicketDataLength = (int)TicketPublication.PublishTicket(m_TicketData, (uint)m_TicketData.Length, encryptedTitleKey.Key, deviceId, ticketId, rightsId, keyGeneration, System.Text.Encoding.ASCII.GetBytes("Root-CA00000003-XS00000020"));
                    }
                }
            }
            else
            {
                m_TicketDataLength = (int)TicketPublication.PublishTicket(m_TicketData, (uint)m_TicketData.Length, encryptedTitleKey.Key, deviceId, ticketId, rightsId, keyGeneration, System.Text.Encoding.ASCII.GetBytes("Root-CA00000004-XS00000020"));
            }

            // チケットに署名を付ける
            if (isProdEncryption)
            {
                if (keyConfiguration.GetProdETicketSignKey() != null)
                {
                    RsaKey key = keyConfiguration.GetProdETicketSignKey();
                    ISigner signer = new Rsa2048Pkcs1Sha256SignCryptoDriver(key.KeyModulus, key.KeyPublicExponent, key.KeyPrivateExponent);

                    this.SignTicket(signer);
                }
                else
                {
                    ISigner signer = new HsmRsa2048Pkcs1Sha256SignCryptoDriver(Rsa2048Pkcs1Sha256KeyIndex.ETicket);

                    this.SignTicket(signer);
                }
            }
            else
            {
                ISigner signer = new Rsa2048Pkcs1Sha256SignCryptoDriver(Rsa2048Pkcs1Sha256KeyIndex.ETicket);

                this.SignTicket(signer);
            }

            return m_TicketData;
        }

        private void SignTicket(ISigner signer)
        {
            Byte[] ticketDataForSign = new byte[MaxTicketLength];
            int ticketDataAreaLength = TicketPublication.GetTicketDataForSign(ticketDataForSign, MaxTicketLength, m_TicketData, (uint)m_TicketDataLength);

            Byte[] signBuffer = signer.SignBlock(ticketDataForSign, 0, ticketDataAreaLength);
            TicketPublication.SignTicket(m_TicketData, (uint)m_TicketDataLength, signBuffer, signBuffer.Length);
        }

        public static AesKey EncryptTitleKey(ICbcModeEncryptor encryptor, AesKey key)
        {
            var encryptedKey = new Byte[key.Key.Length];
            Byte[] iv = new byte[16];

            // タイトルキーを暗号化する(iv は 0 とする)
            encryptor.EncryptBlock(iv, key.Key, 0, key.Key.Length, encryptedKey, 0);

            return new AesKey(encryptedKey);
        }
    }

    public class TicketReader
    {
        byte[] m_EncryptedKey;
        byte[] m_Key;
        ICbcModeEncryptor m_TitleKeyEncryptor;

        public TicketReader(Stream ticketStream, byte keyGeneration) : this(GetBytesFromStream(ticketStream),  new Aes128CbcCryptoDriver(Aes128KeyIndex.ETicketCommonKey, keyGeneration))
        {
        }

        public TicketReader(Stream ticketStream, ICbcModeEncryptor titleKeyEncryptor) : this(GetBytesFromStream(ticketStream), titleKeyEncryptor)
        {
        }

        public TicketReader(byte[] ticketBytes, byte keyGeneration) : this(ticketBytes, new Aes128CbcCryptoDriver(Aes128KeyIndex.ETicketCommonKey, keyGeneration))
        {
        }

        public TicketReader(byte[] ticketBytes, ICbcModeEncryptor titleKeyEncryptor)
        {
            const int KeySize = 16;
            const int TitleKeyOffset = 0x180;
            m_TitleKeyEncryptor = titleKeyEncryptor;
            m_EncryptedKey = ticketBytes.Skip(TitleKeyOffset).Take(KeySize).ToArray();
        }

        private static byte[] GetBytesFromStream(Stream ticketStream)
        {
            var data = new byte[ticketStream.Length];
            ticketStream.Seek(0, SeekOrigin.Begin);
            ticketStream.Read(data, 0, data.Length);
            return data;
        }

        public byte[] GetTitleKey()
        {
            if (m_Key == null)
            {
                byte[] iv = new byte[16];
                m_Key = new byte[16];
                m_TitleKeyEncryptor.DecryptBlock(iv, m_EncryptedKey, 0, m_EncryptedKey.Length, m_Key, 0);
            }

            return m_Key;
        }
    }

    public class TicketUtility
    {
        public const int RightsIdLength = 16;
        public const byte ChangeRightsIdAndTitleKeyKeyGenerationMin = 3; // 鍵世代 3 から RightsId と TitleKey に鍵世代を埋め込むようにした

        // 上位 64 bit を contentMetaId、下位 64 bit のうち KeyId として 128 bit の RightId を作成する
        // KeyId のうち、下位 8 bit は keyGeneration とする（keyGeneration が ChangeRightsIdAndTitleKeyKeyGenerationMin 以上の場合のみ）
        public static Byte[] CreateRightsId(UInt64 contentMetaId, byte keyGeneration)
        {
            UInt64 keyId = 0;
            if (keyGeneration >= ChangeRightsIdAndTitleKeyKeyGenerationMin)
            {
                // 鍵世代を keyId に埋め込む
                keyId += (ulong) keyGeneration;
            }

            // リトルエンディアンなのでビックエンディアンに変換する
            Byte[] contentMetaIdBytes = BitConverter.GetBytes(contentMetaId).Reverse().ToArray();
            Byte[] keyIdBytes = BitConverter.GetBytes(keyId).Reverse().ToArray();

            Byte[] rightsId = new Byte[RightsIdLength];
            Array.Copy(contentMetaIdBytes, rightsId, contentMetaIdBytes.Length);
            Array.Copy(keyIdBytes, 0, rightsId, contentMetaIdBytes.Length, keyIdBytes.Length);

            return rightsId;
        }

        public static UInt64 GetContentMetaIdFromRightsId(byte[] rightsId)
        {
            Debug.Assert(rightsId.Length == RightsIdLength);
            var contentMetaIdBytesReverse = rightsId.Take(sizeof(UInt64));
            return BitConverter.ToUInt64(contentMetaIdBytesReverse.Reverse().ToArray(), 0);
        }

        public static bool IsValidRightsId(byte[] rightsId)
        {
            if (rightsId == null)
            {
                return false;
            }
            var zeroBytes = new byte[rightsId.Length];
            return !rightsId.SequenceEqual(zeroBytes);
        }

        public static string CreateRightsIdText(byte[] rightsId)
        {
            string rightsIdText = "";
            foreach (byte byteValue in rightsId)
            {
                rightsIdText += byteValue.ToString("x2");
            }
            return rightsIdText;
        }

        public static string CreateRightsIdText(UInt64 contentMetaId, byte keyGeneration)
        {
            return CreateRightsIdText(CreateRightsId(contentMetaId, keyGeneration));
        }

        public static bool NeedCreateTicket(string metaType)
        {
            return metaType == "Application" || metaType == "Patch" || metaType == "AddOnContent";
        }

        public static void SetExternalKey(ref NintendoContentArchiveReader ncaReader, IFileSystemArchiveReader parentReader)
        {
            var rightsId = ncaReader.GetRightsId();
            if (!TicketUtility.IsValidRightsId(rightsId))
            {
                return;
            }

            var ticketName = TicketUtility.CreateRightsIdText(rightsId) + ".tik";
            var ticketReader = new TicketReader(parentReader.ReadFile(ticketName, 0, parentReader.GetFileSize(ticketName)), ncaReader.GetKeyGeneration());
            ncaReader.SetExternalKey(ticketReader.GetTitleKey());
        }

        public static void SetExternalKey(ref NintendoContentArchiveReader ncaReader, string ticketPath)
        {
            if (!TicketUtility.IsValidRightsId(ncaReader.GetRightsId()) || string.IsNullOrEmpty(ticketPath))
            {
                return;
            }

            using (var ticketStream = new FileStream(ticketPath, FileMode.Open, FileAccess.Read, FileShare.Read, 4096, FileOptions.SequentialScan))
            {
                var ticketReader = new TicketReader(ticketStream, ncaReader.GetKeyGeneration());
                ncaReader.SetExternalKey(ticketReader.GetTitleKey());
            }
        }
    }
}
