﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Xml.Serialization;
using System.Text.RegularExpressions;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    [XmlRoot("DescFlags", IsNullable = false)]
    public class DescFlags
    {
        [XmlElement("Production")]
        public bool Production { get; set; }

        [XmlElement("UnqualifiedApproval")]
        public bool UnqualifiedApproval { get; set; }
    }

    [XmlRoot("ProgramInfo", IsNullable = false)]
    public class ProgramInfoModel
    {
        [XmlElement("SdkVersion", IsNullable = false)]
        public string SdkVersion { get; set; }

        [XmlElement("ToolVersion", IsNullable = false)]
        public string ToolVersion { get; set; }

        [XmlElement("PatchToolVersion", IsNullable = false)]
        public string PatchToolVersion { get; set; }

        [XmlElement("BuildTarget", IsNullable = false)]
        public string BuildTarget { get; set; }

        [XmlElement("BuildType", IsNullable = false)]
        public string BuildType { get; set; }

        [XmlElement("Desc", IsNullable = false)]
        public string Desc { get; set; }

        [XmlElement("DescFileName", IsNullable = false)]
        public string DescFileName { get; set; }

        [XmlElement("DescFlags", IsNullable = false)]
        public DescFlags DescFlags { get; set; }

        [XmlElement("MiddlewareList")]
        public MiddlewareList MiddlewareListData { get; set; }

        [XmlElement("DebugApiList")]
        public DebugApiList DebugApiListData { get; set; }

        [XmlElement("PrivateApiList")]
        public PrivateApiList PrivateApiListData { get; set; }

        [XmlElement("UnresolvedApiList")]
        public UnresolvedApiList UnresolvedApiListData { get; set; }

        [XmlElement("FsAccessControlData", IsNullable = false)]
        public FsAccessControlData FsAccessControlData { get; set; }

        public struct SdkVersionInfo
        {
            public string SdkVersion { get; set; }
            public string BuildType { get; set; }
        }

        static private SdkVersionInfo GetSdkVersionInfo(ModuleInfo[] moduleInfos)
        {
            SdkVersionInfo sdkVersionInfo = new SdkVersionInfo();
            foreach (var moduleInfo in moduleInfos)
            {
                // 複数見つかることは通常ないが、最後の値を利用
                string moduleName = moduleInfo.ModuleName;
                Regex sdkVersionRule = new Regex(@"NintendoSdk_nnSdk-([^-]*)\-([^-]*)");
                if (sdkVersionRule.IsMatch(moduleName))
                {
                    sdkVersionInfo.SdkVersion = sdkVersionRule.Match(moduleName).Groups[1].Value;
                    sdkVersionInfo.BuildType = sdkVersionRule.Match(moduleName).Groups[2].Value;
                }
                else
                {
                    throw new Exception("This SdkVersion string in nnSdk.nso is Invalid.\n" + moduleName);
                }
            }
            return sdkVersionInfo;
        }

        static private string GetToolVersion()
        {
            // AuthoringTool または ContentArchiveLibrary のバージョンを取得
            var asm = System.AppDomain.CurrentDomain.GetAssemblies().Where(x => x.GetName().Name == "AuthoringTool");
            if (!asm.Any())
            {
                asm = System.AppDomain.CurrentDomain.GetAssemblies().Where(x => x.GetName().Name == "ContentArchiveLibrary");
            }
            return NxAddonVersion.GetCurrentVersion() + ":" + asm.Single().GetName().Version.ToString().Replace(".", "_");
        }

        static public ProgramInfoModel Create(SymbolExtract symbolExtract, bool elf32Mode)
        {
            var model = new ProgramInfoModel();
            model.MiddlewareListData = MiddlewareList.Create(symbolExtract.GetMiddlewareInfos());
            model.DebugApiListData = DebugApiList.Create(symbolExtract.GetDebugApiInfos());
            model.PrivateApiListData = PrivateApiList.Create(symbolExtract.GetPrivateApiInfos());
            model.UnresolvedApiListData = UnresolvedApiList.Create(symbolExtract.GetUndefSymbolInfos());

            SdkVersionInfo sdkVersionInfo = GetSdkVersionInfo(symbolExtract.GetSdkVersionInfos());
            model.SdkVersion = sdkVersionInfo.SdkVersion;
            model.ToolVersion = GetToolVersion();
            model.BuildType = sdkVersionInfo.BuildType;
            if (elf32Mode == true)
            {
                model.BuildTarget = "32";
            }
            else
            {
                model.BuildTarget = "64";
            }

            return model;
        }

        static public ProgramInfoModel Create(List<ProgramInfoModel> models)
        {
            return Create(models, null, null, null);
        }

        static public ProgramInfoModel Create(List<ProgramInfoModel> models, byte[] acidBinary, List<FsAccessControlData.SaveDataOwnerId> saveDataOwnerIdList, string descFilePath)
        {
            var model = new ProgramInfoModel();
            model.MiddlewareListData = MiddlewareList.Create();
            model.DebugApiListData = DebugApiList.Create();
            model.PrivateApiListData = PrivateApiList.Create();
            model.UnresolvedApiListData = UnresolvedApiList.Create();
            model.FsAccessControlData = FsAccessControlData.Create();
            model.SdkVersion = null;
            model.ToolVersion = GetToolVersion();
            model.BuildTarget = null;
            model.BuildType = null;
            if (acidBinary != null)
            {
                model.Desc = Convert.ToBase64String(acidBinary);
                model.DescFlags = new DescFlags();
                model.DescFlags.UnqualifiedApproval = NpdmParser.GetUnqualifiedApproval(acidBinary);
                model.DescFlags.Production = NpdmParser.GetProductionFlag(acidBinary);
                model.DescFileName = Path.GetFileName(descFilePath);
            }
            if (saveDataOwnerIdList != null)
            {
                model.FsAccessControlData = FsAccessControlData.Create(saveDataOwnerIdList.ToArray());
            }

            foreach (var i in models)
            {
                model.MiddlewareListData.Entries.AddRange(i.MiddlewareListData.Entries);
                model.DebugApiListData.Entries.AddRange(i.DebugApiListData.Entries);
                model.PrivateApiListData.Entries.AddRange(i.PrivateApiListData.Entries);
                model.UnresolvedApiListData.Entries.AddRange(i.UnresolvedApiListData.Entries);

                if (string.IsNullOrEmpty(i.SdkVersion) == false)
                {
                    model.SdkVersion = i.SdkVersion;
                }
                if (string.IsNullOrEmpty(i.BuildType) == false)
                {
                    model.BuildType = i.BuildType;
                }
                if (string.IsNullOrEmpty(i.BuildTarget) == false)
                {
                    model.BuildTarget = i.BuildTarget;
                }
            }

            return model;
        }

        public void CheckAcidIntegrity(byte[] acidBinary)
        {
            if (Convert.ToBase64String(acidBinary) != Desc)
            {
                throw new ArgumentException("Invalid programinfo. Acids between programinfo.xml and main.npdm don't match.");
            }
        }
        public void SetPatchToolVersion()
        {
            PatchToolVersion = GetToolVersion();
        }

        public byte[] GetBytes()
        {
            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(string.Empty, string.Empty);

            using (var memoryStream = new MemoryStream())
            {
                var sw = new StreamWriter(memoryStream, Encoding.UTF8);
                var serializer = new XmlSerializer(typeof(ProgramInfoModel));
                serializer.Serialize(sw, this, nameSpace);
                return memoryStream.ToArray();
            }
        }
    }

    public class ProgramInfoXmlSource : ISource
    {
        public long Size { get; set; }
        private ISource m_source;

        public ProgramInfoXmlSource(ProgramInfoModel model)
        {
            // model に Desc が含まれていることをチェック
            Trace.Assert(model.Desc != null);

            var bytes = model.GetBytes();
            m_source = new MemorySource(bytes, 0, bytes.Length);
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }
}
