﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Text;
using System.Xml.Linq;
using System.Xml.Serialization;
using YamlDotNet.RepresentationModel;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.FileSystemMetaLibrary;
using Nintendo.Authoring.CryptoLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class XciUtils
    {
        static internal long GetAvailableAreaSize(int size)
        {
            long gigaByte = 1024 * 1024 * 1024;
            return (long)size * gigaByte - XciInfo.GetUnusedAreaSize(size);
        }

        static internal int GetRomSizeMinForAutoSet(int size)
        {
            // 自動決定時は RomSize2GB を最小値とする
            const int RomSizeMin = 2;
            return (size > 0 && size < RomSizeMin) ? RomSizeMin : size;
        }

        static internal int GetRomSize(long xciSize)
        {
            var possible = XciInfo.RomSizeTable.Where(x => GetAvailableAreaSize(x) >= xciSize);
            if (!possible.Any())
            {
                return XciInfo.InvalidRomSize;
            }
            return possible.First();
        }

        static internal int GetClockRate(int size)
        {
            CheckRomSize(size);
            return XciInfo.GetClockRateFromRomSize(size);
        }

        static internal void CheckRomSize(int size)
        {
            if (!XciInfo.RomSizeTable.Where(x => x == size).Any())
            {
                throw new ArgumentException("Invalid card size. (valid: " + string.Join(", ", XciInfo.RomSizeTable) + ")");
            }
        }

        static internal void CheckClockRate(int clockRate)
        {
            if (!XciInfo.ClockRateTable.Where(x => x == clockRate).Any())
            {
                throw new ArgumentException("Invalid card clock rate. (valid: " + string.Join(", ", XciInfo.ClockRateTable) + ")");
            }
        }

        static internal void CheckRomSizeAndClockRate(int size, int clockRate)
        {
            CheckRomSize(size);
            CheckClockRate(clockRate);

            // 50 MHz 動作する場合はサイズ制限がある
            if (clockRate == 50 && XciInfo.GetClockRateFromRomSize(size) != 50)
            {
                throw new ArgumentException("Invalid card clock rate. 50 MHz clock rate setting can be used for only the card whose size more than 8 GB.");
            }
        }

        static public byte[] GetCardHeaderHash(byte[] xciHeaderData) // xciHeaderData は署名領域含む
        {
            const int SignatureTargetSize = 256;

            var hashCalculator = new SHA256CryptoServiceProvider();
            return hashCalculator.ComputeHash(xciHeaderData, SignatureTargetSize, xciHeaderData.Length - SignatureTargetSize);
        }

        static public byte[] DecryptHeader(byte[] xciHeaderData) // xciHeaderData は署名領域含む
        {
            const int SignatureTargetSize = 256;

            var xciHeaderBytes = xciHeaderData.Skip(SignatureTargetSize).ToArray();

            // 一度読み出してみて、開発版専用パラメータだったら開発鍵で復号する
            bool isDev;
            {
                var reader = new XciHeaderReader(xciHeaderBytes);
                isDev = reader.GetKekIndex() == XciInfo.KekIndexVersionForDev;
            }

            // 暗号化部分を復号
            {
                Aes128CbcCryptoDriver headerEncryptor;
                headerEncryptor = new Aes128CbcCryptoDriver(Aes128KeyIndex.XciHeader, isDev);
                var tmpIv = xciHeaderBytes.Skip(XciMeta.GetIvOffset()).Take(XciInfo.IvSize).ToArray();
                var iv = new byte[tmpIv.Length];
                for (int i = 0; i < tmpIv.Length; i++) // IV はビッグエンディアン
                {
                    iv[i] = tmpIv[tmpIv.Length - 1 - i];
                }
                headerEncryptor.DecryptBlock(iv, xciHeaderBytes, XciMeta.GetEncryptionTargetOffset(), XciMeta.GetEncryptionTargetSize(), xciHeaderBytes, XciMeta.GetEncryptionTargetOffset());
            }

            // TODO: 署名の検証

            return xciHeaderBytes;
        }

        static public string GetBytesString<T>(T value)
        {
            var format = "{0:x" + 2 * Marshal.SizeOf(typeof(T)) + "}";
            return "0x" + string.Format(format, value);
        }

        static public string GetBytesString(byte[] bytes)
        {
            return "0x" + BitConverter.ToString(bytes, 0, bytes.Length).Replace("-", string.Empty).ToLower();
        }

        static public void DumpHeader(byte[] decryptedXciHeaderData) // decryptedXciHeaderData は署名領域含まない
        {
            var reader = new XciHeaderReader(decryptedXciHeaderData);
            Console.WriteLine("Rom Area Start Page Address: {0}", GetBytesString(reader.GetRomAreaStartPageAddress()));
            Console.WriteLine("Backup Area Start Page Address: {0}", GetBytesString(reader.GetBackupAreaStartPageAddress()));
            Console.WriteLine("KekIndex: {0}", GetBytesString(reader.GetKekIndex()));
            Console.WriteLine("TitleKeyDecIndex: {0}", GetBytesString(reader.GetTitleKeyDecIndex()));
            Console.WriteLine("Rom Size: {0}", GetBytesString(reader.GetRomSize()));
            Console.WriteLine("Card Header Version: {0}", GetBytesString(reader.GetCardHeaderVersion()));
            Console.WriteLine("Flags: {0}", GetBytesString(reader.GetFlags()));
            Console.WriteLine("PackageId: {0}", GetBytesString(reader.GetPackageId()));
            Console.WriteLine("Valid Data End Addess: {0}", GetBytesString(reader.GetValidDataEndAddress()));
            Console.WriteLine("Iv: {0}", GetBytesString(reader.GetIv()));
            Console.WriteLine("PartitionFsHeader Address: {0}", GetBytesString(reader.GetPartitionFsHeaderAddress()));
            Console.WriteLine("PartitionFsHeader Size: {0}", GetBytesString(reader.GetPartitionFsHeaderSize()));
            Console.WriteLine("PartitionFsHeader Hash: {0}", GetBytesString(reader.GetPartitionFsHeaderHash()));
            Console.WriteLine("Initial Data Hash: {0}", GetBytesString(reader.GetInitialDataHash()));
            Console.WriteLine("SelSec: {0}", GetBytesString(reader.GetSelSec()));
            Console.WriteLine("SelT1Key: {0}", GetBytesString(reader.GetSelT1Key()));
            Console.WriteLine("SelKey: {0}", GetBytesString(reader.GetSelKey()));
            Console.WriteLine("LimArea: {0}", GetBytesString(reader.GetLimArea()));
            Console.WriteLine("FwVersion: {0}", GetBytesString(reader.GetFwVersion()));
            Console.WriteLine("AccCtrl1: {0}", GetBytesString(reader.GetAccCtrl1()));
            Console.WriteLine("Wait1TimeRead: {0}", GetBytesString(reader.GetWait1TimeRead()));
            Console.WriteLine("Wait2TimeRead: {0}", GetBytesString(reader.GetWait2TimeRead()));
            Console.WriteLine("Wait1TimeWrite: {0}", GetBytesString(reader.GetWait1TimeWrite()));
            Console.WriteLine("Wait2TimeWrite: {0}", GetBytesString(reader.GetWait2TimeWrite()));
            Console.WriteLine("FwMode: {0}", GetBytesString(reader.GetFwMode()));
            Console.WriteLine("UPP version: {0}", GetBytesString(reader.GetUppVersion()));
            Console.WriteLine("UPP Id: {0}", GetBytesString(reader.GetUppId()));
            Console.WriteLine("UPP Hash: {0}", GetBytesString(reader.GetUppHash()));
        }

        static public byte[] GetRootPartitionHash(byte[] xciHeaderData) // xciHeaderData は署名領域含む
        {
            const int SignatureTargetSize = 256;
            var xciHeaderBytes = xciHeaderData.Skip(SignatureTargetSize).ToArray();
            return XciMeta.GetRootPartitionHash(xciHeaderBytes);
        }
    }

    internal class XciHeaderSource : ISource
    {
        public long Size { get; private set; }

        private ISource m_rootPartitionFsHeaderHashSource;

        private XciInfo m_xciInfo;
        private XciMeta m_xciMeta;

        private ICbcModeEncryptor m_headerEncryptor;
        private ISigner m_headerSigner;

        public XciHeaderSource(XciInfo xciInfo, XciMeta xciMeta, ISource rootPartitionFsHeaderHashSource, ICbcModeEncryptor headerEncryptor, ISigner headerSigner)
        {
            Debug.Assert(xciInfo.partitionFsHeaderHash == null);
            m_xciInfo = xciInfo;
            m_xciMeta = xciMeta;

            Debug.Assert(rootPartitionFsHeaderHashSource.Size == 32);
            m_rootPartitionFsHeaderHashSource = rootPartitionFsHeaderHashSource;

            m_headerEncryptor = headerEncryptor;
            m_headerSigner = headerSigner;

            Size = XciInfo.PageSize * XciInfo.CardHeaderPageCount;
        }

        public ByteData PullData(long offset, int size)
        {
            var data = m_rootPartitionFsHeaderHashSource.PullData(0, 32);
            Debug.Assert(data.Buffer.Count == 32);
            m_xciInfo.partitionFsHeaderHash = new byte[32];
            Buffer.BlockCopy(data.Buffer.Array, data.Buffer.Offset, m_xciInfo.partitionFsHeaderHash, 0, data.Buffer.Count);

            var xciHeaderBytes = m_xciMeta.CreateHeader(m_xciInfo);

            // 暗号化
            if (m_headerEncryptor != null)
            {
                m_headerEncryptor.EncryptBlock(m_xciInfo.iv, xciHeaderBytes, XciMeta.GetEncryptionTargetOffset(), XciMeta.GetEncryptionTargetSize(), xciHeaderBytes, XciMeta.GetEncryptionTargetOffset());
            }

            // 署名
            var sign = m_headerSigner.SignBlock(xciHeaderBytes, 0, xciHeaderBytes.Length);

            var xciHeaderBytesWithSign = sign.Concat(xciHeaderBytes).ToArray();
            Debug.Assert(xciHeaderBytesWithSign.Length == Size);
            return new ByteData(new ArraySegment<byte>(xciHeaderBytesWithSign, 0, xciHeaderBytesWithSign.Length));
        }

        public SourceStatus QueryStatus()
        {
            // rootPartitionFsHeaderHashSource が利用可能になったら Pull 可
            var rangeList = m_rootPartitionFsHeaderHashSource.QueryStatus().AvailableRangeList;
            if(rangeList.Count == 0 || rangeList[0].Size != m_rootPartitionFsHeaderHashSource.Size)
            {
                return new SourceStatus();
            }
            var status = new SourceStatus();
            status.AvailableRangeList.MergingAdd(new Range(0, Size));
            return status;
        }
    }

    internal class XciSizeInfo
    {
        public long RootPartitionOffset;
        public long RootPartitionHeaderSize;

        public long UpdatePartitionOffset;
        public long UpdatePartitionHeaderSize;
        public long UpdatePartitionSize;

        public long LogoPartitionOffset;
        public long LogoPartitionHeaderSize;
        public long LogoPartitionSize;

        public long NormalPartitionOffset;
        public long NormalPartitionHeaderSize;
        public long NormalPartitionSize;

        public long SecurePartitionOffset;
        public long SecurePartitionHeaderSize;
        public long SecurePartitionSize;

        public long TotalSize;
    }

    [XmlRoot("CardHeader", IsNullable = false)]
    public class CardHeaderModel
    {
        [XmlElement("RomAreaStartPageAddress")]
        public string RomAreaStartPageAddress { get; set; }

        [XmlElement("BackupAreaStartPageAddress")]
        public string BackupAreaStartPageAddress { get; set; }

        [XmlElement("KekIndex")]
        public string KekIndex { get; set; }

        [XmlElement("TitleKeyDecIndex")]
        public string TitleKeyDecIndex { get; set; }

        [XmlElement("RomSize")]
        public string RomSize { get; set; }

        [XmlElement("CardHeaderVersion")]
        public string CardHeaderVersion { get; set; }

        [XmlElement("Flags")]
        public string Flags { get; set; }

        [XmlElement("PackageId")]
        public string PackageId { get; set; }

        [XmlElement("ValidDataEndAddess")]
        public string ValidDataEndAddess { get; set; }

        [XmlElement("Iv")]
        public string Iv { get; set; }

        [XmlElement("PartitionFsHeaderAddress")]
        public string PartitionFsHeaderAddress { get; set; }

        [XmlElement("PartitionFsHeaderSize")]
        public string PartitionFsHeaderSize { get; set; }

        [XmlElement("PartitionFsHeaderHash")]
        public string PartitionFsHeaderHash { get; set; }

        [XmlElement("InitialDataHash")]
        public string InitialDataHash { get; set; }

        [XmlElement("SelSec")]
        public string SelSec { get; set; }

        [XmlElement("SelT1Key")]
        public string SelT1Key { get; set; }

        [XmlElement("SelKey")]
        public string SelKey { get; set; }

        [XmlElement("LimArea")]
        public string LimArea { get; set; }

        [XmlElement("FwVersion")]
        public string FwVersion { get; set; }

        [XmlElement("AccCtrl1")]
        public string AccCtrl1 { get; set; }

        [XmlElement("Wait1TimeRead")]
        public string Wait1TimeRead { get; set; }

        [XmlElement("Wait2TimeRead")]
        public string Wait2TimeRead { get; set; }

        [XmlElement("Wait1TimeWrite")]
        public string Wait1TimeWrite { get; set; }

        [XmlElement("Wait2TimeWrite")]
        public string Wait2TimeWrite { get; set; }

        [XmlElement("FwMode")]
        public string FwMode { get; set; }

        [XmlElement("UppVersion")]
        public string UppVersion { get; set; }

        [XmlElement("UppId")]
        public string UppId { get; set; }

        [XmlElement("UppHash")]
        public string UppHash { get; set; }

        static public CardHeaderModel Create(byte[] bytes)
        {
            var model = new CardHeaderModel();
            var reader = new XciHeaderReader(bytes);
            model.RomAreaStartPageAddress = XciUtils.GetBytesString(reader.GetRomAreaStartPageAddress());
            model.BackupAreaStartPageAddress = XciUtils.GetBytesString(reader.GetBackupAreaStartPageAddress());
            model.KekIndex = XciUtils.GetBytesString(reader.GetKekIndex());
            model.TitleKeyDecIndex = XciUtils.GetBytesString(reader.GetTitleKeyDecIndex());
            model.RomSize = XciUtils.GetBytesString(reader.GetRomSize());
            model.CardHeaderVersion = XciUtils.GetBytesString(reader.GetCardHeaderVersion());
            model.Flags = XciUtils.GetBytesString(reader.GetFlags());
            model.PackageId = XciUtils.GetBytesString(reader.GetPackageId());
            model.ValidDataEndAddess = XciUtils.GetBytesString(reader.GetValidDataEndAddress());
            model.Iv = XciUtils.GetBytesString(reader.GetIv());
            model.PartitionFsHeaderAddress = XciUtils.GetBytesString(reader.GetPartitionFsHeaderAddress());
            model.PartitionFsHeaderSize = XciUtils.GetBytesString(reader.GetPartitionFsHeaderSize());
            model.PartitionFsHeaderHash = XciUtils.GetBytesString(reader.GetPartitionFsHeaderHash());
            model.InitialDataHash = XciUtils.GetBytesString(reader.GetInitialDataHash());
            model.SelSec = XciUtils.GetBytesString(reader.GetSelSec());
            model.SelT1Key = XciUtils.GetBytesString(reader.GetSelT1Key());
            model.SelKey = XciUtils.GetBytesString(reader.GetSelKey());
            model.LimArea = XciUtils.GetBytesString(reader.GetLimArea());
            model.FwVersion = XciUtils.GetBytesString(reader.GetFwVersion());
            model.AccCtrl1 = XciUtils.GetBytesString(reader.GetAccCtrl1());
            model.Wait1TimeRead = XciUtils.GetBytesString(reader.GetWait1TimeRead());
            model.Wait2TimeRead = XciUtils.GetBytesString(reader.GetWait2TimeRead());
            model.Wait1TimeWrite = XciUtils.GetBytesString(reader.GetWait1TimeWrite());
            model.Wait2TimeWrite = XciUtils.GetBytesString(reader.GetWait2TimeWrite());
            model.FwMode = XciUtils.GetBytesString(reader.GetFwMode());
            model.UppVersion = XciUtils.GetBytesString(reader.GetUppVersion());
            model.UppId = XciUtils.GetBytesString(reader.GetUppId());
            model.UppHash = XciUtils.GetBytesString(reader.GetUppHash());
            return model;
        }
    }

    internal class ProdEncryptedXciArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        private KeyConfiguration m_keyConfig;
        private IEncryptor m_keyAreaEncryptionKeyIvEncryptor;
        private IEncryptor m_initialDataEncryptor;
        private ICbcModeEncryptor m_headerEncryptor;
        private ISigner m_headerSigner;

        private bool m_UseDevHsm;

        private RNGCryptoServiceProvider m_rng;

        private void SetCryptor(KeyConfiguration keyConfig)
        {
            if (keyConfig.GetProdXciHeaderSignKey() != null || keyConfig.GetProdXciInitialDataEncryptionKey() != null)
            {
                m_UseDevHsm = false;
            }
            else
            {
                var hsm = new HsmInterface();
                m_UseDevHsm = hsm.GetUseDev();
            }

            m_keyAreaEncryptionKeyIvEncryptor = new Rsa2048OaepSha256CryptoDriver((Rsa2048OaepSha256KeyIndex)keyConfig.KeyIndexForEncryptedXci);

            if (keyConfig.GetProdXciHeaderSignKey() != null)
            {
                var rsaKey = keyConfig.GetProdXciHeaderSignKey();
                m_headerSigner = new Rsa2048Pkcs1Sha256SignCryptoDriver(rsaKey.KeyModulus, rsaKey.KeyPublicExponent, rsaKey.KeyPrivateExponent);
            }
            else
            {
                m_headerSigner = new HsmRsa2048Pkcs1Sha256SignCryptoDriver(Rsa2048Pkcs1Sha256KeyIndex.XciHeader);
            }

            if (keyConfig.GetProdXciInitialDataEncryptionKey() != null)
            {
                m_initialDataEncryptor = new Aes128CryptoDriver(keyConfig.GetProdXciInitialDataEncryptionKey().Key);
            }
            else
            {
                m_initialDataEncryptor = new HsmAes128CryptoDriver(Aes128KeyIndex.XciInitialData);
            }
            m_headerEncryptor = new Aes128CbcCryptoDriver(Aes128KeyIndex.XciHeader, m_UseDevHsm);
        }

        static private long RoundupPageSize(long value)
        {
            long alignment = XciInfo.PageSize;
            return (value + (alignment - 1)) & ~(alignment - 1);
        }

        static private long RoundupRomAreaStartPageAddress(long value)
        {
            long alignment = XciInfo.RomAreaStartPageCountAlignment * XciInfo.PageSize;
            return (value + (alignment - 1)) & ~(alignment - 1);
        }

        static private void CheckAlignmentPageSize(long value)
        {
            if ((value & (XciInfo.PageSize - 1)) != 0)
            {
                throw new ArgumentException();
            }
        }

        static private string GetTicketNameFromProdNsp(NintendoSubmissionPackageReader nspReader, ulong contentMetaId, string postfix)
        {
            var rightsIdTextHalf = TicketUtility.CreateRightsIdText(contentMetaId, (byte)0).Substring(0, TicketUtility.RightsIdLength);
            return nspReader.ListFileInfo().Where(x => x.Item1.StartsWith(rightsIdTextHalf) && x.Item1.EndsWith(postfix)).Select(x => x.Item1).Single();
        }

        static private void CalculateRootPartitionHeaderSize(out long rootPartitionHeaderSize)
        {
            var entryNameList = new List<string>() { "update", "logo", "normal", "secure" };
            rootPartitionHeaderSize = HashNameEntryPartitionFsHeaderSource<Sha256PartitionFileSystemMeta>.GetDummySize(entryNameList);
            CheckAlignmentPageSize(rootPartitionHeaderSize);
        }

        static private void CalculateUpdatePartitionSize(out long updatePartitionHeaderSize, out long updatePartitionSize, bool isEmpty, NintendoSubmissionPackageReader uppReader)
        {
            if (isEmpty)
            {
                Trace.Assert(uppReader == null);
            }

            var entryNameList = new List<string>();
            if (uppReader != null)
            {
                foreach (var ncaName in uppReader.ListFileInfo().Where(x => x.Item1.EndsWith(".nca")).Select(x => x.Item1))
                {
                    entryNameList.Add(ncaName);
                }
            }
            else
            {
                entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.nca");
                entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.nca");
            }

            updatePartitionHeaderSize = HashNameEntryPartitionFsHeaderSource<Sha256PartitionFileSystemMeta>.GetDummySize(entryNameList);
            CheckAlignmentPageSize(updatePartitionHeaderSize);
            updatePartitionSize = isEmpty ? XciInfo.EmptyUpdatePartitionSize : XciInfo.UpdatePartitionLimitSize - updatePartitionHeaderSize;
        }

        static private void CalculateLogoPartitionSize(out long logoPartitionHeaderSize, out long logoPartitionSize, NintendoSubmissionPackageFileSystemInfo nspInfo, KeyConfiguration keyConfig)
        {
            var entryNameList = new List<string>();

            long contentSize = 0;
            foreach (var logoData in LogoManager.GetDefaultLogoData())
            {
                entryNameList.Add(logoData.FileName);
                contentSize += RoundupPageSize(logoData.Data.Length);
            }

            logoPartitionHeaderSize = HashNameEntryPartitionFsHeaderSource<Sha256PartitionFileSystemMeta>.GetDummySize(entryNameList);
            CheckAlignmentPageSize(logoPartitionHeaderSize);
            logoPartitionSize = contentSize;
        }

        static private void CalculatePartitionSizeOne(out long partitionSize, List<string> entryNameList, NintendoSubmissionPackageFileSystemInfo nspInfo, bool isNormal, KeyConfiguration keyConfig)
        {
            long contentSize = 0;
            for (int i = 0; i < nspInfo.Entries.Count; i++)
            {
                var contentSourceList = new List<Tuple<ISource, NintendoContentDescriptor>>();

                var entry = nspInfo.Entries[i];
                if (isNormal)
                {
                    // normal 領域には現状では何も置かない
                    continue;
                }
                for (int j = 0; j < entry.Contents.Count; j++)
                {
                    var content = entry.Contents[j];
                    if (content.ContentType == NintendoContentMetaConstant.ContentTypeMeta)
                    {
                        continue;
                    }

                    // 開発版 nsp のサイズチェック用。この contentSource は出力には利用しない
                    ISource contentSource;
                    if (content.Source != null)
                    {
                        contentSource = content.Source;
                    }
                    else if (content.SourceSizeCache.HasValue)
                    {
                        contentSource = new UnreadableSource(new PaddingSource(content.SourceSizeCache.Value));
                    }
                    else
                    {
                        throw new NotImplementedException();
                    }

                    contentSourceList.Add(Tuple.Create(contentSource, new NintendoContentDescriptor { ContentInfo = new NintendoContentInfo(content.ContentType, contentSource.Size, content.KeyGeneration, content.IdOffset) }));

                    entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.nca"); // 32 文字+ 拡張子
                    contentSize += RoundupPageSize(contentSource.Size);
                }

                if (entry.MetaType == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    // オンカードパッチの場合は、チケットと証明書が必要
                    {
                        var rightsIdText = TicketUtility.CreateRightsIdText(0, 0); // サイズさえわかればよいので仮の ID
                        var ticket = new Ticket();
                        ticket.PublishTicket(0, nspInfo.IsProdEncryption, 0, keyConfig);
                        var ticketSource = new TicketSource(ticket.Data, ticket.Length);

                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.tik");
                        contentSize += RoundupPageSize(ticketSource.Size);
                    }
                    {
                        var ticketCertificateSource = new TicketCertificateSource(nspInfo.IsProdEncryption, keyConfig);
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cert");
                        contentSize += RoundupPageSize(ticketCertificateSource.Size);
                    }
                }

                entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.nca"); // 32 文字+ 拡張子
                // 開発版 nsp のサイズチェック用
                NintendoContentMetaBase contentMetaBase;
                if (entry.ContentMetaInfo?.Data != null &&
                    entry.ContentMetaInfo?.Model != null)
                {
                    contentMetaBase = new NintendoContentMetaBase(contentSourceList, entry.ContentMetaInfo.Data, entry.ContentMetaInfo.Model, true);
                }
                else
                {
                    contentMetaBase = new NintendoContentMetaBase(contentSourceList, entry.MetaType, entry.MetaFilePath, entry.MetaExtendedData);
                }
                var contentMetaSource = new NintendoContentMetaArchiveSource(contentMetaBase, entry.KeyIndex, keyConfig, nspInfo.IsProdEncryption, true, false);
                contentSize += RoundupPageSize(contentMetaSource.Size);
            }
            partitionSize = contentSize;
        }

        // コンテンツメタリストに含まれるコンテンツのカード上でのサイズを計算する
        static private void CalculatePartitionSizeOne(out long partitionSize, List<string> entryNameList, List<ContentMetaLiteModel> additionalContentMetaList, bool isNormal, KeyConfiguration keyConfig)
        {
            long contentSize = 0;
            foreach (var model in additionalContentMetaList)
            {
                if (model.Type != NintendoContentMetaConstant.ContentMetaTypeApplication &&
                    model.Type != NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    throw new ArgumentException(string.Format("Invalid type {0}", model.Type));
                }

                var normalTypes = new List<string>();
                if (model.Type == NintendoContentMetaConstant.ContentMetaTypeApplication)
                {
                    normalTypes.Add(NintendoContentMetaConstant.ContentTypeMeta);
                    normalTypes.Add(NintendoContentMetaConstant.ContentTypeControl);
                }

                foreach(var content in model.ContentList.Where(p => isNormal ? normalTypes.Contains(p.Type) : true))
                {
                    entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" + (content.Type == NintendoContentMetaConstant.ContentTypeMeta ? ".cnmt.nca" : ".nca"));
                    contentSize += RoundupPageSize(content.Size);
                }

                if (model.Type == NintendoContentMetaConstant.ContentMetaTypePatch && !isNormal)
                {
                    // オンカードパッチの場合は、チケットと証明書が必要
                    {
                        var rightsIdText = TicketUtility.CreateRightsIdText(0, 0); // サイズさえわかればよいので仮の ID
                        var ticket = new Ticket();
                        ticket.PublishTicket(0, false, 0, keyConfig);
                        var ticketSource = new TicketSource(ticket.Data, ticket.Length);

                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.tik");
                        contentSize += RoundupPageSize(ticketSource.Size);
                    }
                    {
                        var ticketCertificateSource = new TicketCertificateSource(false, keyConfig);
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cert");
                        contentSize += RoundupPageSize(ticketCertificateSource.Size);
                    }
                }
            }
            partitionSize = contentSize;
        }

        // 製品化済みの nsp のコンテンツサイズを計算する
        static private void CalculatePartitionSizeOne(out long partitionSize, List<string> entryNameList, NintendoSubmissionPackageReader nspReader)
        {
            long contentSize = 0;
            foreach (var model in ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader))
            {
                foreach (var content in model.ContentList)
                {
                    if (model.Type == NintendoContentMetaConstant.ContentMetaTypePatch && content.Type == NintendoContentMetaConstant.ContentTypeDeltaFragment)
                    {
                        // パッチ間差分コンテンツはカードに含めない
                        continue;
                    }
                    entryNameList.Add(content.Id + (content.Type == NintendoContentMetaConstant.ContentTypeMeta ? ".cnmt.nca" : ".nca"));
                    contentSize += RoundupPageSize(content.Size);
                }

                // オンカードパッチの場合は、チケットと証明書が必要
                if (model.Type == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    var ticketName = GetTicketNameFromProdNsp(nspReader, model.GetUInt64Id(), ".tik");
                    var ticketCertName = GetTicketNameFromProdNsp(nspReader, model.GetUInt64Id(), ".cert");
                    // 既に含まれている場合には入れない
                    if (!entryNameList.Contains(ticketName))
                    {
                        entryNameList.Add(ticketName);
                        contentSize += RoundupPageSize(nspReader.GetFileSize(ticketName));
                    }
                    if (!entryNameList.Contains(ticketCertName))
                    {
                        entryNameList.Add(ticketCertName);
                        contentSize += RoundupPageSize(nspReader.GetFileSize(ticketCertName));
                    }
                }
            }
            partitionSize = contentSize;
        }

        static private void CalculatePartitionSize(out long partitionHeaderSize, out long partitionSize, NintendoSubmissionPackageFileSystemInfo nspInfo, List<ContentMetaLiteModel> additionalContentMetaList, List<NintendoSubmissionPackageReader> patchReaders, bool isNormal, KeyConfiguration keyConfig)
        {
            if (patchReaders.Count != 0)
            {
                // OnCardPatch では additionalContentMetaList の指定は不要
                Trace.Assert(additionalContentMetaList == null);
            }

            // 各コンテンツ、コンテンツメタのサイズ計算
            long contentSize = 0;
            var entryNameList = new List<string>();

            long nspSize;
            {
                CalculatePartitionSizeOne(out nspSize, entryNameList, nspInfo, isNormal, keyConfig);
                contentSize += nspSize;
            }

            if (additionalContentMetaList != null)
            {
                CalculatePartitionSizeOne(out nspSize, entryNameList, additionalContentMetaList, isNormal, keyConfig);
                contentSize += nspSize;
            }

            if (!isNormal && patchReaders.Count != 0)
            {
                foreach (var patchReader in patchReaders)
                {
                    CalculatePartitionSizeOne(out nspSize, entryNameList, patchReader);
                    contentSize += nspSize;
                }
            }

            partitionHeaderSize = HashNameEntryPartitionFsHeaderSource<Sha256PartitionFileSystemMeta>.GetDummySize(entryNameList);
            CheckAlignmentPageSize(partitionHeaderSize);
            partitionSize = contentSize;
        }

        // カード製品化処理用
        static private XciSizeInfo CalculateXciSize(NintendoSubmissionPackageFileSystemInfo nspInfo, List<NintendoSubmissionPackageReader> patchReaders, NintendoSubmissionPackageReader uppReader, KeyConfiguration keyConfig)
        {
            return CalculateXciSize(nspInfo, null, uppReader == null, keyConfig, patchReaders, uppReader);
        }

        static internal XciSizeInfo CalculateXciSize(NintendoSubmissionPackageFileSystemInfo nspInfo, List<ContentMetaLiteModel> additionalContentMetaList, bool useEmptyUpp, KeyConfiguration keyConfig)
        {
            return CalculateXciSize(nspInfo, additionalContentMetaList, useEmptyUpp, keyConfig, new List<NintendoSubmissionPackageReader>(), null);
        }

        // nsp のカードサイズ計算用
        static internal XciSizeInfo CalculateXciSize(NintendoSubmissionPackageFileSystemInfo nspInfo, List<ContentMetaLiteModel> additionalContentMetaList, bool useEmptyUpp, KeyConfiguration keyConfig, List<NintendoSubmissionPackageReader> patchReaders, NintendoSubmissionPackageReader uppReader)
        {
            XciSizeInfo xciSizeInfo = new XciSizeInfo();

            // ルートパーティションのサイズを計算
            xciSizeInfo.RootPartitionOffset = (long)XciInfo.NormalAreaStartPageAddress * XciInfo.PageSize;
            CalculateRootPartitionHeaderSize(out xciSizeInfo.RootPartitionHeaderSize);

            // それぞれのパーティションのオフセットとサイズを計算する
            xciSizeInfo.UpdatePartitionOffset = xciSizeInfo.RootPartitionOffset + xciSizeInfo.RootPartitionHeaderSize;
            CalculateUpdatePartitionSize(out xciSizeInfo.UpdatePartitionHeaderSize, out xciSizeInfo.UpdatePartitionSize, useEmptyUpp, uppReader);

            xciSizeInfo.LogoPartitionOffset = xciSizeInfo.UpdatePartitionOffset + xciSizeInfo.UpdatePartitionHeaderSize + xciSizeInfo.UpdatePartitionSize;
            CalculateLogoPartitionSize(out xciSizeInfo.LogoPartitionHeaderSize, out xciSizeInfo.LogoPartitionSize, nspInfo, keyConfig);

            xciSizeInfo.NormalPartitionOffset = xciSizeInfo.LogoPartitionOffset + xciSizeInfo.LogoPartitionHeaderSize + xciSizeInfo.LogoPartitionSize;
            CalculatePartitionSize(out xciSizeInfo.NormalPartitionHeaderSize, out xciSizeInfo.NormalPartitionSize, nspInfo, additionalContentMetaList, patchReaders, true, keyConfig);

            // Normal 領域の終端は読み込み時（KeyArea がない）に 32 KB アラインされている必要がある
            xciSizeInfo.SecurePartitionOffset = RoundupRomAreaStartPageAddress(xciSizeInfo.NormalPartitionOffset + xciSizeInfo.NormalPartitionHeaderSize + xciSizeInfo.NormalPartitionSize) + XciInfo.CardKeyAreaPageCount * XciInfo.PageSize;
            CalculatePartitionSize(out xciSizeInfo.SecurePartitionHeaderSize, out xciSizeInfo.SecurePartitionSize, nspInfo, additionalContentMetaList, patchReaders, false, keyConfig);

            // 有効終端出力サイズの決定
            xciSizeInfo.TotalSize = xciSizeInfo.SecurePartitionOffset + xciSizeInfo.SecurePartitionHeaderSize + xciSizeInfo.SecurePartitionSize;

            return xciSizeInfo;
        }

        // 各コンテンツの ISource の生成処理を行い、プログラムコンテンツ生成時の SDK version を取得
        private void UpdateNintendoContentInfo(ref uint programContentSdkAddonVersion, NintendoSubmissionPackageFileSystemInfo nspInfo, byte? uppKeyGeneration)
        {
            for (int i = 0; i < nspInfo.Entries.Count; i++)
            {
                var entry = nspInfo.Entries[i];
                for (int j = 0; j < entry.Contents.Count; j++)
                {
                    var content = entry.Contents[j];
                    Trace.Assert(content.FsInfo != null);
                    var ncaFsInfo = (NintendoContentFileSystemInfo)content.FsInfo;

                    if (uppKeyGeneration.HasValue && !(entry.ContentMetaInfo.Model.KeepGeneration ?? false))
                    {
                        if (ncaFsInfo.keyGeneration > uppKeyGeneration.Value)
                        {
                            throw new ArgumentException("Target nsp includes content has newer key generation than upp has.");
                        }
                        else if (ncaFsInfo.keyGeneration < uppKeyGeneration.Value)
                        {
                            ncaFsInfo.keyGeneration = uppKeyGeneration.Value;
                        }
                    }

                    if (content.ContentType == NintendoContentMetaConstant.ContentTypeMeta)
                    {
                        continue;
                    }

                    if (content.ContentType == NintendoContentMetaConstant.ContentTypeProgram)
                    {
                        programContentSdkAddonVersion = ncaFsInfo.sdkAddonVersion;
                    }

                    ncaFsInfo.distributionType = NintendoContentFileSystemMetaConstant.DistributionTypeGameCard;

                    Trace.Assert(!TicketUtility.IsValidRightsId(ncaFsInfo.rightsId));

                    var contentSource = new NintendoContentArchiveSource(ncaFsInfo, m_keyConfig);
                    content.SetSource(content.ContentType, ncaFsInfo.GetRepresentProgramIdOffset(), ncaFsInfo.keyGeneration, null, contentSource);
                    entry.Contents[j] = content;
                }
            }
        }

        internal Tuple<ulong, uint, byte[]> GetUpdatePartitionInfo(NintendoSubmissionPackageReader uppReader)
        {
            var xmlData = uppReader.ReadFile(UpdatePartitionUtils.UpdatePartitionXmlName, 0, uppReader.GetFileSize(UpdatePartitionUtils.UpdatePartitionXmlName));
            var xmlReader = new UpdatePartitionUtils.UpdatePartitionXmlReader(xmlData);
            return Tuple.Create(xmlReader.GetId(), xmlReader.GetVersion(), xmlReader.GetHash());
        }

        internal byte[] CreateInitialData(ulong packageId, Pair<byte[], byte[]> titleKeys)
        {
            var titleKeyConcat = titleKeys.first.Concat(titleKeys.second).ToArray();
            Debug.Assert(titleKeyConcat.Length == 16);

            var initialDataA = new byte[XciMeta.InitialDataASize];
            {
                // A は PackageId + 0 padding
                var packageIdBytes = BitConverter.GetBytes(packageId);
                Buffer.BlockCopy(packageIdBytes, 0, initialDataA, 0, packageIdBytes.Length);
            }

            var key = new byte[m_initialDataEncryptor.KeySize];
            if (!m_UseDevHsm)
            {
                Debug.Assert(initialDataA.Length == key.Length);
                m_initialDataEncryptor.DecryptBlock(initialDataA, 0, initialDataA.Length, key, 0);
            }

            byte[] mac = new byte[XciMeta.InitialDataMacSize];
            byte[] aad = new byte[0];
            byte[] nonce = new byte[XciMeta.InitialDataNonceSize];
            m_rng.GetBytes(nonce);

            var initialDataB = new byte[titleKeyConcat.Length];
            {
                var titleKeyEncryptor = new Aes128CcmCryptoDriver(key);
                titleKeyEncryptor.EncryptBlock(nonce, aad, titleKeyConcat, 0, titleKeyConcat.Length, mac, 0, mac.Length, initialDataB, 0);
            }

            var initialData = new byte[XciMeta.InitialDataSize];
            var initialDataBytes = initialDataA.Concat(initialDataB).Concat(mac).Concat(nonce).ToArray();
            Buffer.BlockCopy(initialDataBytes, 0, initialData, 0, initialDataBytes.Length);

            return initialData;
        }

        internal byte[] CreateKeyArea(byte[] initialData, Pair<byte[], byte[]> titleKeys)
        {
            var keyArea = new byte[XciInfo.PageSize * XciInfo.CardKeyAreaPageCount];
            var keyAreaBytes = initialData.Concat(titleKeys.first).Concat(titleKeys.second).ToArray();
            Buffer.BlockCopy(keyAreaBytes, 0, keyArea, 0, keyAreaBytes.Length);

            return keyArea;
        }

        internal byte[] EncryptKeyArea(byte[] keyArea)
        {
            var encryptedKeyArea = new byte[keyArea.Length];
            Buffer.BlockCopy(keyArea, 0, encryptedKeyArea, 0, keyArea.Length);

            var key = new byte[16];
            var iv = new byte[16];
            m_rng.GetBytes(key);
            m_rng.GetBytes(iv);

            // KeyArea を暗号化
            var keyAreaEncryptor = new Aes128CtrCryptoDriver(key);
            Debug.Assert(keyAreaEncryptor.KeySize == key.Length);
            keyAreaEncryptor.EncryptBlock(iv, encryptedKeyArea, XciMeta.KeyAreaEncryptionTargetOffset, XciMeta.KeyAreaEncryptionTargetSize, encryptedKeyArea, XciMeta.KeyAreaEncryptionTargetOffset);

            // 暗号鍵と IV を RSA 暗号化
            var keyIv = key.Concat(iv).ToArray();
            var encryptedKeyIv = new byte[m_keyAreaEncryptionKeyIvEncryptor.KeySize];
            m_keyAreaEncryptionKeyIvEncryptor.EncryptBlock(keyIv, 0, keyIv.Length, encryptedKeyIv, 0);

            Buffer.BlockCopy(encryptedKeyIv, 0, encryptedKeyArea, XciMeta.KeyAreaEncryptionTargetOffset + XciMeta.KeyAreaEncryptionTargetSize, encryptedKeyIv.Length);
            return encryptedKeyArea;
        }

        private void AddPaddingConnection(ISink outSink, long offset, long size)
        {
            if (size == 0)
            {
                return;
            }

            var source = new PaddingSource(size);
            var targetSink = new SubSink(outSink, offset, size);
            ConnectionList.Add(new Connection(source, targetSink));
        }

        private void SetUpPartitionFromProdNsp(ref long currentOffset, ref PartitionFileSystemInfo partFsInfo, ref List<ContentHashSource> contentHashSources, ref List<Sha256PartitionFsHashSource> partitionFsHashSources, IReadableSink outSink, NintendoSubmissionPackageReader nspReader, string fileName, long partitionHeaderSize, long partitionOffset, long partitionSize)
        {
            {
                ulong fileSize = (ulong)nspReader.GetFileSize(fileName);
                var entryOffset = (ulong)(currentOffset - partitionHeaderSize);
                var partFsEntry = PartitionFileSystemInfo.EntryInfo.Make(fileName, fileSize, entryOffset, 0, XciInfo.PageSize);
                partFsInfo.entries.Add(partFsEntry);
            }

            var contentSource = new FileSystemArchvieFileSource(nspReader, fileName);
            IReadableSink contentSink = new ReadableSubSink(outSink, partitionOffset + currentOffset, contentSource.Size);
            ConnectionList.Add(new Connection(contentSource, contentSink));

            {
                contentHashSources.Add(new ContentHashSource(null, "no_change")); // エントリ名の変更不要
                var partitionFsHashSource = new SinkLinkedSource(contentSink, new Sha256StreamHashSource(new SubSource(contentSink.ToSource(), 0, XciInfo.PageSize)));
                partitionFsHashSources.Add(new Sha256PartitionFsHashSource(partitionFsHashSource, partitionFsHashSources.Count));
            }

            currentOffset += RoundupPageSize(contentSource.Size);
        }

        private bool IsKeepGenerationSpecified(NintendoSubmissionPackageReader nspReader)
        {
            foreach (var cnmt in ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader))
            {
                if (cnmt.KeepGeneration.HasValue && cnmt.KeepGeneration.Value)
                {
                    return true;
                }
            }
            return false;
        }

        public ProdEncryptedXciArchive(IReadableSink outSink, out List<IReadableSink> outContentMetaXmlSinks, ref ResultXmlInfo resultXmlInfo, NintendoSubmissionPackageReader nspReader, NintendoSubmissionPackageReader uppReader, NintendoSubmissionPackageReader patchReader, NintendoSubmissionPackageReader aocReader, int cardSize, int cardClockRate, byte launchFlags, bool noPadding, bool keepGeneration, KeyConfiguration keyConfig)
            : this(outSink, out outContentMetaXmlSinks, ref resultXmlInfo, new List<NintendoSubmissionPackageReader>() { nspReader }, uppReader, patchReader != null ? new List<NintendoSubmissionPackageReader>() { patchReader } : new List<NintendoSubmissionPackageReader>(), aocReader, cardSize, cardClockRate, launchFlags, noPadding, keepGeneration, keyConfig)
        {
        }

        public ProdEncryptedXciArchive(IReadableSink outSink, out List<IReadableSink> outContentMetaXmlSinks, ref ResultXmlInfo resultXmlInfo, List<NintendoSubmissionPackageReader> nspReaders, NintendoSubmissionPackageReader uppReader, List<NintendoSubmissionPackageReader> patchReaders, NintendoSubmissionPackageReader aocReader, int cardSize, int cardClockRate, byte launchFlags, bool noPadding, bool keepGeneration, KeyConfiguration keyConfig)
        {
            ConnectionList = new List<Connection>();
            m_rng = new RNGCryptoServiceProvider();

            m_keyConfig = keyConfig;
            SetCryptor(m_keyConfig);

            outContentMetaXmlSinks = new List<IReadableSink>();
            resultXmlInfo.ProgramInfoList = new List<ProgramInfoModel>();
            resultXmlInfo.ApplicationControlPropertyList = new List<ApplicationControlPropertyModel>();
            resultXmlInfo.HtmlDocumentList = new List<HtmlDocumentXmlModel>();
            resultXmlInfo.LegalInformationList = new List<LegalInformationModel>();
            resultXmlInfo.AuthoringToolInfoList = new List<AuthoringToolInfoModel>();

            uint programContentSdkAddonVersion = 0;
            uint cupVersion = 0;
            ulong cupId = 0;
            byte[] uppHash = null;
            byte? uppKeyGeneration = null;

            var prodNspInfo = ArchiveReconstructionUtils.GetProdNspInfo(nspReaders, m_keyConfig);

            int currentCardSize = cardSize;
            int currentCardClockRate = cardClockRate;
            byte currentLaunchFlags = launchFlags;
            bool currentKeepGeneration = keepGeneration;
            var currentBaseDigestList = new List<string>();

            if (uppReader != null)
            {
                // upp のハッシュを取得
                var info = GetUpdatePartitionInfo(uppReader);
                // CUP Id は SystemUpdateMeta の ContentMeta Id とする
                cupId = info.Item1;
                // CUP Version は SystemUpdateMeta の ContentMeta Version とする
                cupVersion = info.Item2;
                uppHash = info.Item3;
                // UPP の中で使われている最も新しい鍵世代を基準とする
                uppKeyGeneration = UpdatePartitionUtils.GetLatestKeyGeneration(uppReader);
            }

            foreach (var patchReader in patchReaders)
            {
                // OnCardPatch ではパッチに設定されたのカードサイズ・動作周波数を適用する
                var patchInfo = ArchiveReconstructionUtils.GetProdNspInfo(patchReader);
                currentCardSize = Math.Max(currentCardSize, patchInfo.CardSize);
                currentCardClockRate = Math.Max(currentCardClockRate, patchInfo.CardClockRate);
                currentLaunchFlags |= patchInfo.CardLaunchFlags ?? 0x0;
                currentKeepGeneration |= IsKeepGenerationSpecified(patchReader);
                {
                    var patchContentMeta = ArchiveReconstructionUtils.ReadContentMetaInNsp(patchReader);
                    currentBaseDigestList.AddRange(patchContentMeta.Select(x => x.Digest));
                    currentBaseDigestList.AddRange(patchContentMeta.SelectMany(x => (x as PatchContentMetaModel).HistoryList.Where(y => y.Type == NintendoContentMetaConstant.ContentMetaTypeApplication).Select(z => z.Digest)));
                }

                // CUP Version がパッチの RequiredSystemVersion より高いことをチェック
                UpdatePartitionUtils.CheckUpdatePartitionVersionHighEnough(patchReader, cupVersion);

                // パッチに UPP より新しい鍵世代が使われていないかチェック
                if (uppKeyGeneration.HasValue && UpdatePartitionUtils.GetLatestKeyGeneration(patchReader) > uppKeyGeneration.Value)
                {
                    throw new ArgumentException("The patch includes content has newer key generation than upp has.");
                }
            }

            currentCardSize = Math.Max(currentCardSize, prodNspInfo.CardSize);
            currentCardClockRate = Math.Max(currentCardClockRate, prodNspInfo.CardClockRate);
            currentLaunchFlags |= prodNspInfo.CardLaunchFlags ?? 0x0;
            if (patchReaders.Count == 0)
            {
                currentBaseDigestList.AddRange(prodNspInfo.Entries.Select(x => x.ContentMetaInfo.Model.Digest));
            }

            foreach (var nspReader in nspReaders)
            {
                currentKeepGeneration |= IsKeepGenerationSpecified(nspReader);
                // CUP Version がアプリの RequiredSystemVersion より高いことをチェック
                UpdatePartitionUtils.CheckUpdatePartitionVersionHighEnough(nspReader, cupVersion);
            }

            if (aocReader != null)
            {
                var aocInfo = ArchiveReconstructionUtils.GetProdNspInfo(aocReader, m_keyConfig);
                if (aocInfo.OnCardAddOnContentInfo == null)
                {
                    throw new ArgumentException("This AddOnContent is not converted for card.");
                }

                // OnCardAocInfo に記載されたアプリ・パッチが指定されているかチェック
                {
                    var contentMetaList = aocInfo.OnCardAddOnContentInfo.Application.ContentMetaList;
                    if (aocInfo.OnCardAddOnContentInfo.Patch != null)
                    {
                        contentMetaList.AddRange(aocInfo.OnCardAddOnContentInfo.Patch.ContentMetaList);
                    }
                    foreach (var contentMeta in contentMetaList)
                    {
                        if (!currentBaseDigestList.Contains(contentMeta.Digest))
                        {
                            throw new ArgumentException(string.Format("Application or Patch (Id: {0}) is not input, which was used to convert AddOnContent.", contentMeta.Id));
                        }
                    }
                }

                // カード設定
                currentCardSize = aocInfo.CardSize;
                currentCardClockRate = aocInfo.CardClockRate;
                currentLaunchFlags |= aocInfo.CardLaunchFlags ?? 0x0;

                // aoc はアプリといっしょに再暗号化する
                prodNspInfo.Entries.AddRange(aocInfo.Entries);
            }

            UpdateNintendoContentInfo(ref programContentSdkAddonVersion, prodNspInfo, currentKeepGeneration ? null : uppKeyGeneration);

            XciUtils.CheckRomSizeAndClockRate(currentCardSize, currentCardClockRate);

            var xciSizeInfo = CalculateXciSize(prodNspInfo, patchReaders, uppReader, m_keyConfig);

            {
                // nsp 生成時に決定されたカードサイズに収まらない場合はエラー
                var refCardSize = XciUtils.GetRomSize(xciSizeInfo.TotalSize);
                if (refCardSize == XciInfo.InvalidRomSize)
                {
                    throw new ArgumentException(string.Format("Xci size exceed the maximum size of the game card."));
                }
                if (currentCardSize < refCardSize)
                {
                    throw new ArgumentException(string.Format("Xci requires CardSpec/Size = {0}, though it is set as {1}.", refCardSize, currentCardSize));
                }

                if (!noPadding)
                {
                    // nsp 生成時に決定されたカードサイズの容量いっぱいまで確保（余白は 0xFF Padding）
                    const long GiBytes = 1024 * 1024 * 1024;
                    outSink.SetSize((long)currentCardSize * GiBytes);
                }
                else
                {
                    // noPadding = true の場合は、収まるちょうどのサイズまで確保
                    outSink.SetSize(RoundupPageSize(xciSizeInfo.TotalSize));
                    // 最後まで書き込まれない場合があるので、先に末尾に 16 バイト書き込んでおく
                    {
                        var zero = new byte[16];
                        outSink.PushData(new ByteData(new ArraySegment<byte>(zero, 0, zero.Length)), outSink.Size - zero.Length);
                    }
                }
            }

            var rootPartFsHashSources = new List<Sha256PartitionFsHashSource>();

            // TODO: 以降辺りのコードを nsp 作成と共通化

            // Update パーティションの構築
            {
                var updatePartFsInfo = new PartitionFileSystemInfo();
                var hashSourcesForUpdate = new List<ContentHashSource>();
                var partitionFsHashSourcesForUpdate = new List<Sha256PartitionFsHashSource>();

                if (uppReader != null)
                {
                    long currentOffsetForUpdate = xciSizeInfo.UpdatePartitionHeaderSize;
                    foreach (var ncaFileName in uppReader.ListFileInfo().Where(x => x.Item1.EndsWith(".nca")).Select(x => x.Item1))
                    {
                        SetUpPartitionFromProdNsp(ref currentOffsetForUpdate, ref updatePartFsInfo, ref hashSourcesForUpdate, ref partitionFsHashSourcesForUpdate, outSink, uppReader, ncaFileName, xciSizeInfo.UpdatePartitionHeaderSize, xciSizeInfo.UpdatePartitionOffset, xciSizeInfo.UpdatePartitionSize);
                    }
                    Trace.Assert(currentOffsetForUpdate <= xciSizeInfo.UpdatePartitionHeaderSize + xciSizeInfo.UpdatePartitionSize);
                }

                var headerBaseSourceForUpdate = new HashNameEntryPartitionFsHeaderSource<Sha256PartitionFileSystemMeta>(hashSourcesForUpdate, updatePartFsInfo, xciSizeInfo.UpdatePartitionHeaderSize);
                var headerSourceForUpdate = new HashAdaptedSha256PartitionFsHeaderSource(headerBaseSourceForUpdate, partitionFsHashSourcesForUpdate);
                IReadableSink updatePartitionSink = new ReadableSubSink(outSink, xciSizeInfo.UpdatePartitionOffset, headerSourceForUpdate.Size);
                ConnectionList.Add(new Connection(headerSourceForUpdate, updatePartitionSink));

                var updatePartitionHeaderHashSource = new SinkLinkedSource(updatePartitionSink, new Sha256StreamHashSource(new SubSource(updatePartitionSink.ToSource(), 0, xciSizeInfo.UpdatePartitionHeaderSize)));
                rootPartFsHashSources.Add(new Sha256PartitionFsHashSource(updatePartitionHeaderHashSource, rootPartFsHashSources.Count));
            }

            // Logo 領域の構築
            {
                var logoPartFsInfo = new PartitionFileSystemInfo();
                var hashSourcesForLogo = new List<ContentHashSource>();
                var partitionFsHashSourcesForLogo = new List<Sha256PartitionFsHashSource>();
                var logoPartitionSources = new List<ConcatenatedSource.Element>();

                long currentOffsetForLogo = xciSizeInfo.LogoPartitionHeaderSize;

                foreach (var logoData in LogoManager.GetDefaultLogoData())
                {
                    var logoDataBytes = logoData.Data;

                    {
                        var entryOffset = (ulong)(currentOffsetForLogo - xciSizeInfo.LogoPartitionHeaderSize);
                        var logoPartFsEntry = PartitionFileSystemInfo.EntryInfo.Make(logoData.FileName, (ulong)logoDataBytes.Length, entryOffset, 0, XciInfo.PageSize);
                        logoPartFsInfo.entries.Add(logoPartFsEntry);
                    }

                    IReadableSink logoDataSink = new ReadableSubSink(outSink, xciSizeInfo.LogoPartitionOffset + currentOffsetForLogo, logoDataBytes.Length);
                    ConnectionList.Add(new Connection(new MemorySource(logoDataBytes, 0, logoDataBytes.Length), logoDataSink));
                    logoPartitionSources.Add(new ConcatenatedSource.Element(new SinkLinkedSource(logoDataSink, logoDataSink.ToSource()), "logoData", currentOffsetForLogo));

                    var partitionFsHashSource = new SinkLinkedSource(logoDataSink, new Sha256StreamHashSource(new SubSource(logoDataSink.ToSource(), 0, XciInfo.PageSize)));

                    hashSourcesForLogo.Add(new ContentHashSource(null, "no_change")); // エントリ名の変更不要
                    partitionFsHashSourcesForLogo.Add(new Sha256PartitionFsHashSource(partitionFsHashSource, partitionFsHashSourcesForLogo.Count));

                    currentOffsetForLogo += RoundupPageSize(logoDataBytes.Length);
                }

                // padding
                {
                    var lastOffset = logoPartitionSources.Last().Offset + logoPartitionSources.Last().Source.Size;
                    logoPartitionSources.Add(new ConcatenatedSource.Element(new PaddingSource(currentOffsetForLogo - lastOffset), "logoDataPadding", lastOffset));
                }

                Trace.Assert(currentOffsetForLogo == xciSizeInfo.LogoPartitionHeaderSize + xciSizeInfo.LogoPartitionSize);

                var headerBaseSourceForLogo = new HashNameEntryPartitionFsHeaderSource<Sha256PartitionFileSystemMeta>(hashSourcesForLogo, logoPartFsInfo, xciSizeInfo.LogoPartitionHeaderSize);
                var headerSourceForLogo = new HashAdaptedSha256PartitionFsHeaderSource(headerBaseSourceForLogo, partitionFsHashSourcesForLogo);
                IReadableSink logoPartitionSink = new ReadableSubSink(outSink, xciSizeInfo.LogoPartitionOffset, headerSourceForLogo.Size);
                ConnectionList.Add(new Connection(headerSourceForLogo, logoPartitionSink));
                logoPartitionSources.Insert(0, new ConcatenatedSource.Element(new SinkLinkedSource(logoPartitionSink, logoPartitionSink.ToSource()), "logoPartitionHeader", 0));

                // Logo 領域はパーティション全体のハッシュを検証対象とする
                var logoPartitionHashSource = new Sha256StreamHashSource(new ConcatenatedSource(logoPartitionSources));
                rootPartFsHashSources.Add(new Sha256PartitionFsHashSource(logoPartitionHashSource, rootPartFsHashSources.Count));
            }

            // Normal, Secure 領域の構築
            {
                var securePartFsInfo = new PartitionFileSystemInfo();
                var normalPartFsInfo = new PartitionFileSystemInfo();

                var hashSourcesForSecure = new List<ContentHashSource>();
                var hashSourcesForNormal = new List<ContentHashSource>();

                var partitionFsHashSourcesForSecure = new List<Sha256PartitionFsHashSource>();
                var partitionFsHashSourcesForNormal = new List<Sha256PartitionFsHashSource>();

                long currentOffsetForSecure = xciSizeInfo.SecurePartitionHeaderSize;
                long currentOffsetForNormal = xciSizeInfo.NormalPartitionHeaderSize;

                // Application and AddOnContent
                for (int i = 0; i < prodNspInfo.Entries.Count; i++)
                {
                    var entry = prodNspInfo.Entries[i];
                    var contentSourceListForSecure = new List<Tuple<ISource, NintendoContentDescriptor>>();

                    // 各コンテンツ
                    for (int j = 0; j < entry.Contents.Count; j++)
                    {
                        var content = entry.Contents[j];
                        Trace.Assert(content.Source != null);
                        var contentSource = content.Source;

                        {
                            var entryOffset = (ulong)(currentOffsetForSecure - xciSizeInfo.SecurePartitionHeaderSize);
                            var securePartFsEntry = PartitionFileSystemInfo.EntryInfo.Make("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.nca", (ulong)contentSource.Size, entryOffset, 0, XciInfo.PageSize);
                            securePartFsInfo.entries.Add(securePartFsEntry);
                        }

                        IReadableSink contentSink = new ReadableSubSink(outSink, xciSizeInfo.SecurePartitionOffset + currentOffsetForSecure, contentSource.Size);
                        ConnectionList.Add(new Connection(contentSource, contentSink));

                        var contentHashSource = new SinkLinkedSource(contentSink, new Sha256StreamHashSource(contentSink.ToSource()));
                        var partitionFsHashSource = new SinkLinkedSource(contentSink, new Sha256StreamHashSource(new SubSource(contentSink.ToSource(), 0, XciInfo.PageSize)));

                        hashSourcesForSecure.Add(new ContentHashSource(contentHashSource, ".nca"));
                        partitionFsHashSourcesForSecure.Add(new Sha256PartitionFsHashSource(partitionFsHashSource, partitionFsHashSourcesForSecure.Count));

                        contentSourceListForSecure.Add(Tuple.Create(hashSourcesForSecure[hashSourcesForSecure.Count - 1].Source, new NintendoContentDescriptor { ContentInfo = new NintendoContentInfo(content.ContentType, contentSource.Size, content.KeyGeneration, content.IdOffset) }));

                        currentOffsetForSecure += RoundupPageSize(contentSource.Size);
                    }

                    // コンテンツメタ
                    {
                        var contentMetaBase = new NintendoContentMetaBase(contentSourceListForSecure, entry.ContentMetaInfo.Data, entry.ContentMetaInfo.Model, true);
                        var contentMetaSource = new NintendoContentMetaArchiveSource(contentMetaBase, entry.KeyIndex, m_keyConfig, prodNspInfo.IsProdEncryption, true, false);
                        {
                            var entryOffset = (ulong)(currentOffsetForSecure - xciSizeInfo.SecurePartitionHeaderSize);
                            var securePartFsEntry = PartitionFileSystemInfo.EntryInfo.Make("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.nca", (ulong)contentMetaSource.Size, entryOffset, 0, XciInfo.PageSize);
                            securePartFsInfo.entries.Add(securePartFsEntry);
                        }

                        IReadableSink contentMetaSink = new ReadableSubSink(outSink, xciSizeInfo.SecurePartitionOffset + currentOffsetForSecure, contentMetaSource.Size);
                        ConnectionList.Add(new Connection(contentMetaSource, contentMetaSink));

                        var contentMetaHashSource = new SinkLinkedSource(contentMetaSink, new Sha256StreamHashSource(contentMetaSink.ToSource()));
                        var partitionFsHashSource = new SinkLinkedSource(contentMetaSink, new Sha256StreamHashSource(new SubSource(contentMetaSink.ToSource(), 0, XciInfo.PageSize)));

                        hashSourcesForSecure.Add(new ContentHashSource(contentMetaHashSource, ".cnmt.nca"));
                        partitionFsHashSourcesForSecure.Add(new Sha256PartitionFsHashSource(partitionFsHashSource, partitionFsHashSourcesForSecure.Count));

                        currentOffsetForSecure += RoundupPageSize(contentMetaSource.Size);

                        // コンテンツメタ XML を出力
                        {
                            var xmlSource = new NintendoContentMetaXmlSource(contentMetaBase, contentMetaHashSource, contentMetaSource.Size);
                            var xmlSink = new MemorySink((int)xmlSource.Size);
                            ConnectionList.Add(new Connection(xmlSource, xmlSink));
                            outContentMetaXmlSinks.Add(xmlSink);
                        }
                    }

                    // Result XML に ProgramInfo を保持
                    if (entry.ProgramInfo != null)
                    {
                        resultXmlInfo.ProgramInfoList.Add(entry.ProgramInfo.Model);
                    }

                    // Result XML に ApplicationControlProperty を保持
                    if (entry.ApplicationControlPropertyInfo != null)
                    {
                        resultXmlInfo.ApplicationControlPropertyList.Add(entry.ApplicationControlPropertyInfo.Model);
                    }

                    // Result XML に HtmlDocument を保持
                    if (entry.HtmlDocumentInfo != null)
                    {
                        resultXmlInfo.HtmlDocumentList.Add(entry.HtmlDocumentInfo.Model);
                    }

                    // Result XML に LegalInfomation を保持
                    if (entry.LegalInformationInfo != null)
                    {
                        resultXmlInfo.LegalInformationList.Add(entry.LegalInformationInfo.Model);
                    }
                }

                // Result XML に AuthoringToolInfo を保持
                if (prodNspInfo.AuthoringToolInfo != null)
                {
                    resultXmlInfo.AuthoringToolInfoList.Add(prodNspInfo.AuthoringToolInfo);
                }

                // Patch
                foreach (var patchReader in patchReaders)
                {
                    foreach (var model in ArchiveReconstructionUtils.ReadContentMetaInNsp(patchReader).Where(x => x.Type == NintendoContentMetaConstant.ContentMetaTypePatch).Select(x => x as PatchContentMetaModel))
                    {
                        // コンテンツメタは作り直すので一度除外
                        model.ContentList.RemoveAll(p => p.Type == "Meta");
                        foreach (var content in model.ContentList)
                        {
                            if (content.Type == NintendoContentMetaConstant.ContentTypeDeltaFragment)
                            {
                                // パッチ間差分コンテンツはカードに含めない
                                continue;
                            }
                            var ncaFileName = content.Id + (content.Type == NintendoContentMetaConstant.ContentTypeMeta ? ".cnmt.nca" : ".nca");
                            SetUpPartitionFromProdNsp(ref currentOffsetForSecure, ref securePartFsInfo, ref hashSourcesForSecure, ref partitionFsHashSourcesForSecure, outSink, patchReader, ncaFileName, xciSizeInfo.SecurePartitionHeaderSize, xciSizeInfo.SecurePartitionOffset, xciSizeInfo.SecurePartitionSize);
                        }

                        {
                            // コンテンツメタは RequiredSystemVersion を再設定して作り直す。
                            if (cupVersion != 0)
                            {
                                // CheckUpdatePartitionVersionHighEnough 大小関係は確認済み
                                model.RequiredSystemVersion = Math.Min(model.RequiredSystemVersion, cupVersion);
                            }
                            else if (model.OriginalRequiredSystemVersion.HasValue)
                            {
                                model.RequiredSystemVersion = model.OriginalRequiredSystemVersion.Value;
                            }

                            var contentMetaBase = new NintendoContentMetaBase(model as PatchContentMetaModel, prodNspInfo.IsProdEncryption);
                            var contentMetaSource = new NintendoContentMetaArchiveSource(contentMetaBase, 0, m_keyConfig, prodNspInfo.IsProdEncryption, false, false); // NAND にコピーする可能性がある
                            {
                                var entryOffset = (ulong)(currentOffsetForSecure - xciSizeInfo.SecurePartitionHeaderSize);
                                var securePartFsEntry = PartitionFileSystemInfo.EntryInfo.Make("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.nca", (ulong)contentMetaSource.Size, entryOffset, 0, XciInfo.PageSize);
                                securePartFsInfo.entries.Add(securePartFsEntry);
                            }

                            IReadableSink contentMetaSink = new ReadableSubSink(outSink, xciSizeInfo.SecurePartitionOffset + currentOffsetForSecure, contentMetaSource.Size);
                            ConnectionList.Add(new Connection(contentMetaSource, contentMetaSink));

                            var contentMetaHashSource = new SinkLinkedSource(contentMetaSink, new Sha256StreamHashSource(contentMetaSink.ToSource()));
                            var partitionFsHashSource = new SinkLinkedSource(contentMetaSink, new Sha256StreamHashSource(new SubSource(contentMetaSink.ToSource(), 0, XciInfo.PageSize)));

                            hashSourcesForSecure.Add(new ContentHashSource(contentMetaHashSource, ".cnmt.nca"));
                            partitionFsHashSourcesForSecure.Add(new Sha256PartitionFsHashSource(partitionFsHashSource, partitionFsHashSourcesForSecure.Count));
                            currentOffsetForSecure += RoundupPageSize(contentMetaSource.Size);

                            // パッチのコンテンツメタ XML を出力
                            {
                                var xmlSource = new NintendoContentMetaXmlSource(contentMetaBase, contentMetaHashSource, contentMetaSource.Size);
                                var xmlSink = new MemorySink((int)xmlSource.Size);
                                ConnectionList.Add(new Connection(xmlSource, xmlSink));
                                outContentMetaXmlSinks.Add(xmlSink);
                            }
                        }

                        // オンカードパッチの場合は、チケットと証明書が必要
                        {
                            var ticketName = GetTicketNameFromProdNsp(patchReader, model.GetUInt64Id(), ".tik");
                            var ticketCertName = GetTicketNameFromProdNsp(patchReader, model.GetUInt64Id(), ".cert");
                            if (!securePartFsInfo.entries.Any(x => x.name == ticketName))
                            {
                                SetUpPartitionFromProdNsp(ref currentOffsetForSecure, ref securePartFsInfo, ref hashSourcesForSecure, ref partitionFsHashSourcesForSecure, outSink, patchReader, ticketName, xciSizeInfo.SecurePartitionHeaderSize, xciSizeInfo.SecurePartitionOffset, xciSizeInfo.SecurePartitionSize);
                            }
                            if (!securePartFsInfo.entries.Any(x => x.name == ticketCertName))
                            {
                                SetUpPartitionFromProdNsp(ref currentOffsetForSecure, ref securePartFsInfo, ref hashSourcesForSecure, ref partitionFsHashSourcesForSecure, outSink, patchReader, ticketCertName, xciSizeInfo.SecurePartitionHeaderSize, xciSizeInfo.SecurePartitionOffset, xciSizeInfo.SecurePartitionSize);
                            }
                        }
                    }

                    // Result XML にパッチの ProgramInfo を保持
                    var programInfoXmlFiles = patchReader.ListFileInfo().Where(x => x.Item1.EndsWith(".programinfo.xml"));
                    foreach (var programInfoXml in programInfoXmlFiles)
                    {
                        var model = ArchiveReconstructionUtils.ReadXml<ProgramInfoModel>(patchReader, programInfoXml.Item1, programInfoXml.Item2);
                        resultXmlInfo.ProgramInfoList.Add(model);
                    }

                    // Result XML にパッチの ApplicationControlProperty を保持
                    var nacpXmlFiles = patchReader.ListFileInfo().Where(x => x.Item1.EndsWith(".nacp.xml"));
                    foreach (var nacpXml in nacpXmlFiles)
                    {
                        var model = ArchiveReconstructionUtils.ReadXml<ApplicationControlPropertyModel>(patchReader, nacpXml.Item1, nacpXml.Item2);
                        resultXmlInfo.ApplicationControlPropertyList.Add(model);
                    }

                    // Result XML にパッチの HtmlDocument を保持
                    var htmlDocumentXmlFiles = patchReader.ListFileInfo().Where(x => x.Item1.EndsWith(".htmldocument.xml"));
                    foreach (var htmlDocumentXml in htmlDocumentXmlFiles)
                    {
                        var model = ArchiveReconstructionUtils.ReadXml<HtmlDocumentXmlModel>(patchReader, htmlDocumentXml.Item1, htmlDocumentXml.Item2);
                        resultXmlInfo.HtmlDocumentList.Add(model);
                    }

                    // Result XML にパッチの LegalInformation を保持
                    var legalInfoXmlFiles = patchReader.ListFileInfo().Where(x => x.Item1.EndsWith(".legalinfo.xml"));
                    foreach (var legalInfoXml in legalInfoXmlFiles)
                    {
                        var model = ArchiveReconstructionUtils.ReadXml<LegalInformationModel>(patchReader, legalInfoXml.Item1, legalInfoXml.Item2);
                        resultXmlInfo.LegalInformationList.Add(model);
                    }

                    // Result XML にパッチの AuthoringToolInfo を保持
                    var toolInfoXmlFiles = patchReader.ListFileInfo().Where(x => x.Item1.EndsWith("authoringtoolinfo.xml"));
                    foreach (var toolInfoXml in toolInfoXmlFiles)
                    {
                        var model = ArchiveReconstructionUtils.ReadXml<AuthoringToolInfoModel>(patchReader, toolInfoXml.Item1, toolInfoXml.Item2);
                        resultXmlInfo.AuthoringToolInfoList.Add(model);
                    }
                }

                Trace.Assert(currentOffsetForSecure == xciSizeInfo.SecurePartitionHeaderSize + xciSizeInfo.SecurePartitionSize);
                Trace.Assert(currentOffsetForNormal == xciSizeInfo.NormalPartitionHeaderSize + xciSizeInfo.NormalPartitionSize);

                // Normal Header
                {
                    var headerBaseSourceForNormal = new HashNameEntryPartitionFsHeaderSource<Sha256PartitionFileSystemMeta>(hashSourcesForNormal, normalPartFsInfo, xciSizeInfo.NormalPartitionHeaderSize);
                    var headerSourceForNormal = new HashAdaptedSha256PartitionFsHeaderSource(headerBaseSourceForNormal, partitionFsHashSourcesForNormal);
                    IReadableSink normalPartitionSink = new ReadableSubSink(outSink, xciSizeInfo.NormalPartitionOffset, headerSourceForNormal.Size);
                    ConnectionList.Add(new Connection(headerSourceForNormal, normalPartitionSink));

                    var normalPartitionHeaderHashSource = new SinkLinkedSource(normalPartitionSink, new Sha256StreamHashSource(new SubSource(normalPartitionSink.ToSource(), 0, xciSizeInfo.NormalPartitionHeaderSize)));
                    rootPartFsHashSources.Add(new Sha256PartitionFsHashSource(normalPartitionHeaderHashSource, rootPartFsHashSources.Count));
                }

                // Secure Header
                {
                    var headerBaseSourceForSecure = new HashNameEntryPartitionFsHeaderSource<Sha256PartitionFileSystemMeta>(hashSourcesForSecure, securePartFsInfo, xciSizeInfo.SecurePartitionHeaderSize);
                    var headerSourceForSecure = new HashAdaptedSha256PartitionFsHeaderSource(headerBaseSourceForSecure, partitionFsHashSourcesForSecure);
                    IReadableSink securePartitionSink = new ReadableSubSink(outSink, xciSizeInfo.SecurePartitionOffset, headerSourceForSecure.Size);
                    ConnectionList.Add(new Connection(headerSourceForSecure, securePartitionSink));

                    var securePartitionHeaderHashSource = new SinkLinkedSource(securePartitionSink, new Sha256StreamHashSource(new SubSource(securePartitionSink.ToSource(), 0, xciSizeInfo.SecurePartitionHeaderSize)));
                    rootPartFsHashSources.Add(new Sha256PartitionFsHashSource(securePartitionHeaderHashSource, rootPartFsHashSources.Count));
                }
            }

            // ルートパーティションの作成
            ISource rootPartitionHeaderHashSource;
            var rootPartFsInfo = new PartitionFileSystemInfo();
            {
                var hashSources = new List<ContentHashSource>();
                {
                    var entryOffset = (ulong)(xciSizeInfo.UpdatePartitionOffset - xciSizeInfo.UpdatePartitionOffset); // Update パーティション開始位置が基準の Offset
                    var entrySize = (ulong)(xciSizeInfo.UpdatePartitionHeaderSize + xciSizeInfo.UpdatePartitionSize);
                    var rootPartFsEntry = PartitionFileSystemInfo.EntryInfo.Make("update", entrySize, entryOffset, 0, (uint)xciSizeInfo.UpdatePartitionHeaderSize);
                    rootPartFsInfo.entries.Add(rootPartFsEntry);
                    hashSources.Add(new ContentHashSource(null, "update"));
                }

                {
                    var entryOffset = (ulong)(xciSizeInfo.LogoPartitionOffset - xciSizeInfo.UpdatePartitionOffset);
                    var entrySize = (ulong)(xciSizeInfo.LogoPartitionHeaderSize + xciSizeInfo.LogoPartitionSize);
                    // Logo 領域はパーティション全体のハッシュを検証対象とする
                    var rootPartFsEntry = PartitionFileSystemInfo.EntryInfo.Make("logo", entrySize, entryOffset, 0, (uint)(xciSizeInfo.LogoPartitionHeaderSize + xciSizeInfo.LogoPartitionSize));
                    rootPartFsInfo.entries.Add(rootPartFsEntry);
                    hashSources.Add(new ContentHashSource(null, "logo"));
                }

                {
                    var entryOffset = (ulong)(xciSizeInfo.NormalPartitionOffset - xciSizeInfo.UpdatePartitionOffset);
                    var entrySize = (ulong)(xciSizeInfo.NormalPartitionHeaderSize + xciSizeInfo.NormalPartitionSize);
                    var rootPartFsEntry = PartitionFileSystemInfo.EntryInfo.Make("normal", entrySize, entryOffset, 0, (uint)xciSizeInfo.NormalPartitionHeaderSize);
                    rootPartFsInfo.entries.Add(rootPartFsEntry);
                    hashSources.Add(new ContentHashSource(null, "normal"));
                }

                {
                    var entryOffset = (ulong)(xciSizeInfo.SecurePartitionOffset - xciSizeInfo.UpdatePartitionOffset);
                    var entrySize = (ulong)(xciSizeInfo.SecurePartitionHeaderSize + xciSizeInfo.SecurePartitionSize);
                    var rootPartFsEntry = PartitionFileSystemInfo.EntryInfo.Make("secure", entrySize, entryOffset, 0, (uint)xciSizeInfo.SecurePartitionHeaderSize);
                    rootPartFsInfo.entries.Add(rootPartFsEntry);
                    hashSources.Add(new ContentHashSource(null, "secure"));
                }

                var headerBaseSource = new HashNameEntryPartitionFsHeaderSource<Sha256PartitionFileSystemMeta>(hashSources, rootPartFsInfo, xciSizeInfo.RootPartitionHeaderSize);
                var headerSource = new HashAdaptedSha256PartitionFsHeaderSource(headerBaseSource, rootPartFsHashSources);
                IReadableSink rootPartitionSink = new ReadableSubSink(outSink, xciSizeInfo.RootPartitionOffset, headerSource.Size);
                ConnectionList.Add(new Connection(headerSource, rootPartitionSink));

                rootPartitionHeaderHashSource = new SinkLinkedSource(rootPartitionSink, new Sha256StreamHashSource(new SubSource(rootPartitionSink.ToSource(), 0, xciSizeInfo.RootPartitionHeaderSize)));
            }

            // packageId を乱数生成
            ulong packageId;
            {
                var bytes = new byte[8];
                m_rng.GetBytes(bytes);
                packageId = BitConverter.ToUInt64(bytes, 0);
            }

            // InitialData の生成
            var titleKey1 = new byte[XciMeta.TitleKey1Size];
            var titleKey2 = new byte[XciMeta.TitleKey2Size];
            {
                m_rng.GetBytes(titleKey1);
                m_rng.GetBytes(titleKey2);
            }
            var initialData = CreateInitialData(packageId, new Pair<byte[], byte[]>(titleKey1, titleKey2));

            // KeyArea の生成
            var keyArea = CreateKeyArea(initialData, new Pair<byte[], byte[]>(titleKey1, titleKey2));
            var keyAreaSource = new MemorySource(keyArea, 0, keyArea.Length);

            var iv = new byte[XciInfo.IvSize];
            {
                m_rng.GetBytes(iv);
            }

            // カードパラメータの設定
            var xciInfo = new XciInfo();
            {
                xciInfo.romAreaStartPageAddress = (uint)(xciSizeInfo.SecurePartitionOffset / XciInfo.PageSize) - XciInfo.CardKeyAreaPageCount;
                xciInfo.kekIndex = m_UseDevHsm ? XciInfo.KekIndexVersionForDev : XciInfo.KekIndexVersion0;
                xciInfo.romSize = XciInfo.ConvertRomSizeToRomSizeByte(currentCardSize);
                xciInfo.flags = currentLaunchFlags;
                xciInfo.packageId = packageId;
                // KeyArea を除く有効なデータが書き込まれている最大のブロックアドレスを最終有効アドレスとする
                xciInfo.validDataEndAddress = (uint)(RoundupPageSize(xciSizeInfo.TotalSize) / XciInfo.PageSize) - XciInfo.CardKeyAreaPageCount - 1;
                xciInfo.iv = iv;
                xciInfo.partitionFsHeaderAddress = (ulong)(XciInfo.NormalAreaStartPageAddress - XciInfo.CardKeyAreaPageCount) * XciInfo.PageSize; // rootPartition の書き込みアドレス - KeyArea サイズ
                xciInfo.partitionFsHeaderSize = (ulong)xciSizeInfo.RootPartitionHeaderSize;
                xciInfo.partitionFsHeaderHash = null;
                xciInfo.selSec = XciInfo.SelSecForT1; // TODO: T2 対応
                // FwVersion は搭載される UPP バージョンによって切り替える
                var fwVersionForProd = currentKeepGeneration ? XciInfo.FwVersion : XciInfo.GetFwVersion(cupVersion);
                xciInfo.fwVersion = m_UseDevHsm ? XciInfo.FwVersionForDev : fwVersionForProd;
                xciInfo.accCtrl1 = XciInfo.ConvertClockRateToAccCtrl1(currentCardClockRate);
                Debug.Assert(programContentSdkAddonVersion != 0);
                xciInfo.fwMode = programContentSdkAddonVersion;
                xciInfo.cupVersion = cupVersion;
                xciInfo.cupId  = cupId;
                xciInfo.uppHash = uppHash != null ? uppHash : new byte[XciInfo.UppHashSize]; // 先頭 8 byte
            }

            // カードヘッダ
            var xciMeta = new XciMeta(initialData);
            var xciHeaderSource = new XciHeaderSource(xciInfo, xciMeta, rootPartitionHeaderHashSource, m_headerEncryptor, m_headerSigner);

            // KeyArea を出力用 Sink と接続
            {
                var keyAreaSink = new ReadableSubSink(outSink, 0, keyAreaSource.Size);
                ConnectionList.Add(new Connection(keyAreaSource, keyAreaSink));
            }

            // カードヘッダを出力用 Sink と接続
            {
                var xciHeaderSink = new ReadableSubSink(outSink, keyAreaSource.Size, xciHeaderSource.Size);
                ConnectionList.Add(new Connection(xciHeaderSource, xciHeaderSink));
            }

            // 証明書領域を 0xFF 埋め
            {
                var paddingSize = XciInfo.CertAreaPageCount * XciInfo.PageSize;
                var paddingOffset = (long)XciInfo.CertAreaStartPageAddress * XciInfo.PageSize;
                var paddingSink = new SubSink(outSink, paddingOffset, paddingSize);
                ConnectionList.Add(new Connection(new PaddingSource(paddingSize, 0xFF), paddingSink));
            }

            // イメージ終端まで 0xFF 埋め
            if (!noPadding)
            {
                var paddingSize = outSink.Size - xciSizeInfo.TotalSize;
                var paddingSink = new SubSink(outSink, xciSizeInfo.TotalSize, paddingSize);
                ConnectionList.Add(new Connection(new PaddingSource(paddingSize, 0xFF), paddingSink));
            }
        }

        // Xcie
        public ProdEncryptedXciArchive(ISink outSink, ISource xciSource, KeyConfiguration keyConfig)
        {
            ConnectionList = new List<Connection>();
            m_rng = new RNGCryptoServiceProvider();

            m_keyConfig = keyConfig;
            SetCryptor(m_keyConfig);

            var keyArea = new byte[XciInfo.PageSize * XciInfo.CardKeyAreaPageCount];
            {
                var data = xciSource.PullData(0, (int)(XciInfo.PageSize * XciInfo.CardKeyAreaPageCount));
                Debug.Assert(data.Buffer.Count == keyArea.Length);
                Buffer.BlockCopy(data.Buffer.Array, data.Buffer.Offset, keyArea, 0, data.Buffer.Count);
            }

            var encryptedKeyArea = EncryptKeyArea(keyArea);
            var encryptedKeyAreaSource = new MemorySource(encryptedKeyArea, 0, encryptedKeyArea.Length);

            var xcieSource = new AdaptedSource(xciSource, encryptedKeyAreaSource, 0, encryptedKeyAreaSource.Size);
            outSink.SetSize(xcieSource.Size);
            ConnectionList.Add(new Connection(xcieSource, outSink));
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }
}
