﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class NpdmParser
    {
        public static Pair<uint, uint> GetAciOffsetSize(byte[] npdmSrc)
        {
            const int offsetAciOffset = 0x70;
            var offset = BitConverter.ToUInt32(npdmSrc, offsetAciOffset);

            const int offsetAciSize = 0x74;
            var size = BitConverter.ToUInt32(npdmSrc, offsetAciSize);

            return new Pair<uint, uint>(offset, size);
        }

        public static Pair<uint, uint> GetAcidOffsetSize(byte[] npdmSrc)
        {
            const int offsetAcidOffset = 0x78;
            var offset = BitConverter.ToUInt32(npdmSrc, offsetAcidOffset);

            const int offsetAcidSize = 0x7C;
            var size = BitConverter.ToUInt32(npdmSrc, offsetAcidSize);

            return new Pair<uint, uint>(offset, size);
        }

        public static byte[] GetAcid(byte[] npdmSrc)
        {
            var acidOffsetSize = NpdmParser.GetAcidOffsetSize(npdmSrc);
            return npdmSrc.Skip((int)acidOffsetSize.first).Take((int)acidOffsetSize.second).ToArray();
        }

        public static bool GetProductionFlag(byte[] acid)
        {
            const int offsetFlagsOffset = 0x20C;
            const uint productionFlag = 0x00000001;
            var flags = BitConverter.ToUInt32(acid, offsetFlagsOffset);

            return (flags & productionFlag) != 0;
        }

        public static bool GetUnqualifiedApproval(byte[] acid)
        {
            const int offsetFlagsOffset = 0x20C;
            const uint unqualifiedApprovalFlag = 0x00000002;
            var flags = BitConverter.ToUInt32(acid, offsetFlagsOffset);

            return (flags & unqualifiedApprovalFlag) != 0;
        }

        public static List<FsAccessControlData.SaveDataOwnerId> ParseSaveDataOwnerId(byte[] npdmSrc)
        {
            var saveDataOwnerIdList = new List<FsAccessControlData.SaveDataOwnerId>();

            const uint offsetFacOffset = 0x40;
            var aciOffsetSize = NpdmParser.GetAciOffsetSize(npdmSrc);
            int readPosition = (int)(aciOffsetSize.first + offsetFacOffset);

            GCHandle gch = GCHandle.Alloc(npdmSrc.Skip(readPosition).ToArray(), GCHandleType.Pinned);
            var fac = (FsAccessControlStruct)Marshal.PtrToStructure(gch.AddrOfPinnedObject(), typeof(FsAccessControlStruct));
            gch.Free();
            readPosition += (int)fac.saveDataOwnerInfoOffset;

            const int saveDataOwnerIdCountSize = 4;
            uint saveDataOwnerInfoCount = BitConverter.ToUInt32(npdmSrc.Skip(readPosition).Take(saveDataOwnerIdCountSize).ToArray(), 0);
            readPosition += saveDataOwnerIdCountSize;

            const int accessibilitySize = 1;
            for (int i = 0; i < saveDataOwnerInfoCount; i += accessibilitySize)
            {
                var accessibility = npdmSrc.Skip(readPosition).Take(accessibilitySize).ToArray().First();
                readPosition += accessibilitySize;

                var saveDataOwnerIdElem = new FsAccessControlData.SaveDataOwnerId();
                saveDataOwnerIdElem.Accessibility = FsAccessControlData.SaveDataOwnerId.GetAccessibilityString(accessibility);
                saveDataOwnerIdList.Add(saveDataOwnerIdElem);
            }
            readPosition = RoundUp(readPosition, 4);

            const int saveDataOwnerIdSize = 8;
            for (int i = 0; i < saveDataOwnerInfoCount; i++)
            {
                byte[] saveDataOwnerId = new byte[saveDataOwnerIdSize];
                Array.Copy(npdmSrc, readPosition, saveDataOwnerId, 0, saveDataOwnerIdSize);
                readPosition += saveDataOwnerIdSize;

                Array.Reverse(saveDataOwnerId);
                saveDataOwnerIdList[i].Id = "0x" + BitConverter.ToString(saveDataOwnerId).Replace("-", "").ToLower();
            }

            return saveDataOwnerIdList;
        }

        private static int RoundUp(int target, uint align)
        {
            return (int)((target + (align - 1)) & ~(align - 1));
        }
    }
}
