﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.CryptoLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;
using Nintendo.ApplicationControlProperty;

namespace Nintendo.Authoring.AuthoringLibrary
{
    using EntryFilterRule = Pair<FilterType, string>;
    using DirectoryConnector = Pair<string, string>;

    public class NintendoSubmissionPackageContentResource
    {
        public NintendoSubmissionPackageContentResource()
        {
            PathList = new List<DirectoryConnector>();
        }

        public string ContentType { get; set; }
        public List<DirectoryConnector> PathList { get; set; }
    }

    public class NintendoSubmissionPackageContentInfo
    {
        public const UInt32 DefaultNxIconMaxSize = 100 * 1024;
        public string MetaType { get; set; }
        public string MetaFilePath { get; set; }
        public string DescFilePath { get; set; }
        public string DeltaSourceFilePath { get; set; }
        public string DeltaDestinationFilePath { get; set; }
        public string NroDirectoryPath { get; set; }
        public string FilterDescriptionFilePath { get; set; }
        public int KeyAreaEncryptionKeyIndex { get; set; }
        public bool IsHardwareEncryptionKeyEmbedded { get; set; }
        public List<NintendoSubmissionPackageContentResource> ResourceList { get; set; }
        public List<Tuple<string, string>> IconList { get; set; }
        public List<Tuple<string, string>> NxIconList { get; set; }
        public UInt32 NxIconMaxSize { get; set; }
        public bool HasTicket { get; set; }
        public byte? KeyGeneration { get; set; }
        public bool NoEncryption { get; set; }
        public NintendoSubmissionPackageContentInfo()
        {
            ResourceList = new List<NintendoSubmissionPackageContentResource>();
            IconList = new List<Tuple<string, string>>();
            NxIconList = new List<Tuple<string, string>>();
            NxIconMaxSize = DefaultNxIconMaxSize;
            KeyAreaEncryptionKeyIndex = -1;
            KeyGeneration = null;
        }

        public NintendoSubmissionPackageContentResource GetResource(string contentType)
        {
            foreach (var resource in ResourceList)
            {
                if (resource.ContentType.Equals(contentType))
                {
                    return resource;
                }
            }

            return null;
        }

        public bool HasResource(string contentType)
        {
            return GetResource(contentType) != null;
        }
    }

    public class NintendoSubmissionPackageAdfWriter
    {
        private string m_adfPath;

        public NintendoSubmissionPackageAdfWriter(string adfPath)
        {
            m_adfPath = adfPath;
        }

        private void WriteContentInfo(StreamWriter writer, int index, List<DirectoryConnector> contentPaths, string contentType, string metaFilePath, string descFilePath, int keyAreaEncryptionKeyIndex, byte keyGeneration, List<EntryFilterRule> filterRules, ulong programId, byte programIdOffset, byte[] rightsId, bool needHardwareEncryption, bool noEncryption, List<Tuple<string, Int64>> originalRomFsListFileInfo)
        {
            if (contentPaths.Count > 0)
            {
                if ((File.GetAttributes(contentPaths[0].first) & FileAttributes.Directory) != FileAttributes.Directory)
                {
                    writer.WriteLine("      - type : file");
                    writer.WriteLine("        contentType : {0}", contentType);
                    writer.WriteLine("        path : {0}", contentPaths[0].first);
                }
                else
                {
                    string contentAdfPath = Path.GetFullPath(Path.GetDirectoryName(m_adfPath)) + "\\" +
                        Path.GetFileNameWithoutExtension(m_adfPath) + ".c" + index + "." + contentType + ".nca.adf";

                    NintendoContentAdfWriter contentAdfWriter = new NintendoContentAdfWriter(contentAdfPath, contentType, metaFilePath, descFilePath, keyAreaEncryptionKeyIndex, keyGeneration, programId, programIdOffset, rightsId, needHardwareEncryption, noEncryption);

                    contentAdfWriter.Write(contentPaths, filterRules, originalRomFsListFileInfo);

                    writer.WriteLine("      - type : format");
                    writer.WriteLine("        contentType : {0}", contentType);
                    writer.WriteLine("        path : {0}", contentAdfPath);
                }
            }
        }

        private ulong GetProgramId(MetaFileReader reader, string metaType, int index)
        {
            if(reader == null)
            {
                return 0xffffffffffffffff;
            }

            if(metaType == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
            {
                return reader.GetContentMetaList().ElementAt(index).Id;
            }
            if(metaType == NintendoContentMetaConstant.ContentMetaTypeApplication)
            {
                return ((ApplicationMeta)reader.GetContentMetaList().First()).ProgramId;
            }
            if(metaType == NintendoContentMetaConstant.ContentMetaTypePatch)
            {
                return ((PatchMeta)reader.GetContentMetaList().First()).ProgramId;
            }
            return reader.GetContentMetaList().First().Id;
        }

        private byte GetProgramIndex(MetaFileReader reader, string metaType)
        {
            if(reader == null)
            {
                return 0;
            }

            if(metaType == NintendoContentMetaConstant.ContentMetaTypeApplication)
            {
                return ((ApplicationMeta)reader.GetContentMetaList().First()).ProgramIndex ?? 0;
            }
            else if(metaType == NintendoContentMetaConstant.ContentMetaTypePatch)
            {
                return ((PatchMeta)reader.GetContentMetaList().First()).ProgramIndex ?? 0;
            }
            return 0;
        }

        private byte[] GetRightsId(MetaFileReader reader, string metaType, int index, bool isUseExternalKey, byte keyGeneration)
        {
            // 内部鍵の場合、RightsId は全て 0 埋めとする
            if(!isUseExternalKey)
            {
                return new byte[16];
            }

            if (reader == null)
            {
                return TicketUtility.CreateRightsId(0xffffffffffffffff, keyGeneration);
            }

            if (metaType == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
            {
                return TicketUtility.CreateRightsId(reader.GetContentMetaList().ElementAt(index).Id, keyGeneration);
            }
            return TicketUtility.CreateRightsId(reader.GetContentMetaList().First().Id, keyGeneration);
        }

        public void Write(List<NintendoSubmissionPackageContentInfo> contentInfos)
        {
            Write(contentInfos, null, null);
        }

        public void Write(List<NintendoSubmissionPackageContentInfo> contentInfos, List<EntryFilterRule> filterRules, List<Tuple<string, Int64>> originalRomFsListFileInfo)
        {
            using (var adf = new StreamWriter(m_adfPath, false, Encoding.UTF8))
            {
                adf.WriteLine("formatType : NintendoSubmissionPackage");
                adf.WriteLine("version : 0");
                adf.WriteLine("entries :");
                for (int i = 0; i < contentInfos.Count; i++)
                {
                    adf.WriteLine("  - contents :");

                    int keyIndex;
                    if (contentInfos[i].KeyAreaEncryptionKeyIndex != -1)
                    {
                        // 引数による key index の明示指定
                        keyIndex = contentInfos[i].KeyAreaEncryptionKeyIndex;
                    }
                    else
                    {
                        // type からデフォルト値を判定
                        if (contentInfos[i].MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication ||
                            contentInfos[i].MetaType == NintendoContentMetaConstant.ContentMetaTypePatch       ||
                            contentInfos[i].MetaType == NintendoContentMetaConstant.ContentMetaTypeDelta       ||
                            contentInfos[i].MetaType == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                        {
                            keyIndex = 0; // application
                        }
                        else
                        {
                            keyIndex = 1; // ocean
                        }
                    }

                    for (var j = 0; j < contentInfos[i].ResourceList.Count; ++j)
                    {
                        MetaFileReader metaReader = null;
                        if (contentInfos[i].MetaFilePath != null)
                        {
                            metaReader = new MetaFileReader(contentInfos[i].MetaFilePath, contentInfos[i].MetaType);
                        }

                        var programId = GetProgramId(metaReader, contentInfos[i].MetaType, i);
                        var programIndex = GetProgramIndex(metaReader, contentInfos[i].MetaType);
                        // アプリ系の Program/HtmlDocument は HW 版内部鍵または外部鍵を入れる。
                        bool needHardwareEncryption = (contentInfos[i].MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication ||
                                                       contentInfos[i].MetaType == NintendoContentMetaConstant.ContentMetaTypePatch ||
                                                       contentInfos[i].MetaType == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                                                       &&
                                                      (contentInfos[i].ResourceList[j].ContentType == NintendoContentMetaConstant.ContentTypeProgram ||
                                                       contentInfos[i].ResourceList[j].ContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument ||
                                                       contentInfos[i].ResourceList[j].ContentType == NintendoContentMetaConstant.ContentTypePublicData);
                        var resource = contentInfos[i].ResourceList.ElementAt(j);
                        var keyGeneration = contentInfos[i].KeyGeneration.HasValue ? contentInfos[i].KeyGeneration.Value : NintendoContentFileSystemMetaConstant.CurrentKeyGeneration;
                        var rightsId = GetRightsId(metaReader, contentInfos[i].MetaType, i, contentInfos[i].HasTicket && needHardwareEncryption, keyGeneration);
                        // TORIAEZU: フィルタは nmeta で指定された方を優先
                        var filterRulesToBeUsed = contentInfos[i].FilterDescriptionFilePath != null ? FilterDescription.ParseFdf(contentInfos[i].FilterDescriptionFilePath) : filterRules;
                        WriteContentInfo(adf, i, resource.PathList, resource.ContentType, contentInfos[i].MetaFilePath, contentInfos[i].DescFilePath, keyIndex, keyGeneration, filterRulesToBeUsed, programId, programIndex, rightsId, needHardwareEncryption, contentInfos[i].NoEncryption, originalRomFsListFileInfo);
                    }

                    adf.WriteLine("    metaType : {0}", contentInfos[i].MetaType);
                    adf.WriteLine("    metaFilePath : {0}", contentInfos[i].MetaFilePath);
                    adf.WriteLine("    nxIconMaxSize: {0}", contentInfos[i].NxIconMaxSize);
                    adf.WriteLine("    nroDirectoryPath : {0}", contentInfos[i].NroDirectoryPath);
                    adf.WriteLine("    keyIndex : {0}", keyIndex);
                    adf.WriteLine("    hasTicket: {0}", !contentInfos[i].NoEncryption && contentInfos[i].HasTicket);
                    if(contentInfos[i].DeltaSourceFilePath != null && contentInfos[i].DeltaDestinationFilePath != null)
                    {
                        adf.WriteLine("    delta :");
                        adf.WriteLine("      source : {0}", contentInfos[i].DeltaSourceFilePath);
                        adf.WriteLine("      destination : {0}", contentInfos[i].DeltaDestinationFilePath);
                    }
                    if (contentInfos[i].IconList.Count > 0)
                    {
                        adf.WriteLine("    icon:");
                        contentInfos[i].IconList.ForEach(info =>
                        {
                            adf.WriteLine("        - language: {0}", info.Item1);
                            adf.WriteLine("          path: {0}", info.Item2);
                        });
                    }
                    if (contentInfos[i].NxIconList.Count > 0)
                    {
                        adf.WriteLine("    nxIcon:");
                        contentInfos[i].NxIconList.ForEach(info =>
                        {
                            adf.WriteLine("        - language: {0}", info.Item1);
                            adf.WriteLine("          path: {0}", info.Item2);
                        });
                    }
                }
            }
        }
        public void WriteDelta(string sourceNsp, string destinationNsp, string metaFilePath, int keyAreaKeyIndex, long commandSizeMax, bool saveDelta)
        {
            int keyIndex = (keyAreaKeyIndex == -1) ? 0 : keyAreaKeyIndex; // delta はデフォルト 0
            using (var adf = new StreamWriter(m_adfPath, false, Encoding.UTF8))
            {
                adf.WriteLine("formatType : NintendoSubmissionPackageForDelta");
                adf.WriteLine("version : 0");
                adf.WriteLine("entries :");
                adf.WriteLine("  - metaType : {0}", NintendoContentMetaConstant.ContentMetaTypeDelta);
                adf.WriteLine("    metaFilePath : {0}", metaFilePath);
                adf.WriteLine("    keyIndex : {0}", keyIndex);
                adf.WriteLine("    delta :");
                adf.WriteLine("      source : {0}", sourceNsp);
                adf.WriteLine("      destination : {0}", destinationNsp);
                adf.WriteLine("      commandSizeMax : {0}", commandSizeMax);
                adf.WriteLine("      save : {0}", saveDelta);
            }
        }
    }

    public class NintendoSubmissionPackageFileSystemInfo
    {
        public byte Version { get; set; }
        public class ContentInfo
        {
            public string ContentType { get; private set; }
            public string ContentId { get; private set; }
            public string DescFileName { get; private set; }
            private byte _KeyGeneration;
            public byte KeyGeneration
            {
                get
                {
                    if (_KeyGeneration == NintendoContentFileSystemMetaConstant.InvalidKeyGeneration)
                    {
                        throw new ArgumentException("Invalid key generation is referred.");
                    }
                    return _KeyGeneration;
                }
                private set
                {
                    _KeyGeneration = value;
                }
            }
            public byte IdOffset { get; private set; }

            public ISource Source { get; private set; }
            public void SetSource(string contentType, byte idOffset, byte keyGeneration, string contentId, ISource source)
            {
                ContentType = contentType;
                IdOffset = idOffset;
                KeyGeneration = keyGeneration;
                ContentId = contentId;
                Source = source;
                FsInfo = null;
            }
            public ISource OriginalSource { get; set; }

            public FileSystemMetaLibrary.FileSystemInfo FsInfo { get; private set; }
            public void SetFsInfo(string contentType, string contentId, FileSystemMetaLibrary.FileSystemInfo fsInfo)
            {
                ContentType = contentType;
                ContentId = contentId;
                FsInfo = fsInfo;
                IdOffset = (fsInfo as NintendoContentFileSystemInfo).GetRepresentProgramIdOffset();
                KeyGeneration = (fsInfo as NintendoContentFileSystemInfo).keyGeneration;
                DescFileName =(fsInfo as NintendoContentFileSystemInfo).descFileName;
                Source = null;
            }

            public long? SourceSizeCache { get; private set; }
            public void SetSourceSizeCache(ISource source)
            {
                if (FsInfo == null)
                {
                    throw new ArgumentException();
                }
                SourceSizeCache = source.Size;
            }

            public FragmentInfo Fragment { get; private set; }
            public void SetFragment(FragmentInfo fragment)
            {
                if (ContentType != NintendoContentMetaConstant.ContentTypeDeltaFragment)
                {
                    throw new ArgumentException();
                }
                Fragment = fragment;
            }
        }
        public class FragmentInfo
        {
            public string TargetContentType { get; set; }
            public string UpdateType { get; set; }
            public string SourceContentId { get; set; }
            public string DestinationContentId { get; set; }
            public Int64 SourceSize { get; set; }
            public Int64 DestinationSize { get; set; }
            public UInt16 Index { get; set; }
        }
        // コンテンツメタの引き継ぎ情報
        public class ContentMetaInfo
        {
            public byte[] Data { get; set; }
            public ContentMetaModel Model { get; set; }
            public ContentMetaInfo(byte[] data, ContentMetaModel model) { Data = data; Model = model; }
        }
        // プログラム情報の引き継ぎ情報
        public class ProgramInfo
        {
            public ProgramInfoModel Model { get { return Models.First(); } }
            public List<ProgramInfoModel> Models { get; set; }
            public ProgramInfo(ProgramInfoModel model) { Models = new List<ProgramInfoModel>() { model }; }
            public ProgramInfo(List<ProgramInfoModel> models) { Models = new List<ProgramInfoModel>(); Models.AddRange(models); }
        }
        // アプリケーション管理データの引き継ぎ情報
        public class ApplicationControlPropertyInfo
        {
            public ApplicationControlPropertyModel Model { get { return Models.First(); } }
            public List<ApplicationControlPropertyModel> Models { get; set; }
            public ApplicationControlPropertyInfo(ApplicationControlPropertyModel model) { Models = new List<ApplicationControlPropertyModel>() { model }; }
            public ApplicationControlPropertyInfo(List<ApplicationControlPropertyModel> models) { Models = new List<ApplicationControlPropertyModel>(); Models.AddRange(models); }
        }
        // htmldocument.xml の引き継ぎ情報
        public class HtmlDocumentInfo
        {
            public HtmlDocumentXmlModel Model { get { return Models.First(); } }
            public List<HtmlDocumentXmlModel> Models { get; set; }
            public HtmlDocumentInfo(HtmlDocumentXmlModel model) { Models = new List<HtmlDocumentXmlModel>() { model }; }
            public HtmlDocumentInfo(List<HtmlDocumentXmlModel> models) { Models = new List<HtmlDocumentXmlModel>(); Models.AddRange(models); }
        }
        // legalinformation.xml の引き継ぎ情報
        public class LegalInformationInfo
        {
            public LegalInformationModel Model { get { return Models.First(); } }
            public List<LegalInformationModel> Models { get; set; }
            public LegalInformationInfo(LegalInformationModel model) { Models = new List<LegalInformationModel>() { model }; }
            public LegalInformationInfo(List<LegalInformationModel> models) { Models = new List<LegalInformationModel>(); Models.AddRange(models); }
        }
        public class EntryInfo
        {
            public List<ContentInfo> Contents { get; set; }
            public string MetaType { get; set; }
            public string MetaFilePath { get; set; }
            public string DescFilePath { get; set; }
            public string DeltaSource { get; set; }
            public string DeltaDestination { get; set; }
            public string NroDirectoryPath { get; set; }
            public int KeyIndex { get; set; }
            public bool HasTicket { get; set; }
            public List<IContentMetaExtendedData> MetaExtendedData { get; set; }
            public ContentMetaInfo ContentMetaInfo;
            public ProgramInfo ProgramInfo { get; set; }
            public HtmlDocumentInfo HtmlDocumentInfo { get; set; }
            public LegalInformationInfo LegalInformationInfo { get; set; }
            public ApplicationControlPropertyInfo ApplicationControlPropertyInfo { get; set; }
            public List<NintendoSubmissionPackageExtraData> ExtraData { get; set; }

            public EntryInfo()
            {
                ExtraData = new List<NintendoSubmissionPackageExtraData>();
            }

            public byte GetProgramIdOffset()
            {
                var programIdOffset = Contents.First().IdOffset;
                for (int i = 1; i < Contents.Count(); i++)
                {
                    var tmpProgramIdOffset = Contents[i].IdOffset;
                    Trace.Assert(programIdOffset == tmpProgramIdOffset);
                }
                return programIdOffset;
            }
        }
        public List<EntryInfo> Entries { get; set; }

        public int CardSize { get; set; }
        public int CardClockRate { get; set; }
        public bool AutoSetSize { get; set; }
        public bool AutoSetClockRate { get; set; }
        public bool SkipSizeCheck { get; set; }
        public byte? CardLaunchFlags { get; set; }

        public AuthoringToolInfoModel AuthoringToolInfo { get; set; }
        public OnCardAddOnContentInfoModel OnCardAddOnContentInfo { get; set; }
        public MultiApplicationCardInfoModel MultiApplicationCardInfo { get; set; }

        public bool IsProdEncryption { get; set; }
        public bool EnableContentMetaBinaryExport { get; set; }
        public bool? KeepGeneration { get; set; }

        public NintendoSubmissionPackageFileSystemInfo()
        {
            Entries = new List<EntryInfo>();
        }
    }

    public class NintendoSubmissionPackageAdfReader
    {
        private string m_adfPath;

        public NintendoSubmissionPackageAdfReader(string adfPath)
        {
            m_adfPath = adfPath;
        }

        public NintendoSubmissionPackageFileSystemInfo GetFileSystemInfo(KeyConfiguration keyConfig)
        {
            NintendoSubmissionPackageFileSystemInfo fileSystemInfo = new NintendoSubmissionPackageFileSystemInfo();

            using (var adf = new StreamReader(m_adfPath, Encoding.UTF8))
            {
                var yamlStream = new YamlStream();
                yamlStream.Load(adf);

                YamlSequenceNode entries;
                try
                {
                    YamlMappingNode rootNode = (YamlMappingNode)yamlStream.Documents[0].RootNode;
                    YamlScalarNode formatType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("formatType")];
                    if (formatType.Value != "NintendoSubmissionPackage")
                    {
                        throw new ArgumentException();
                    }
                    YamlScalarNode version = (YamlScalarNode)rootNode.Children[new YamlScalarNode("version")];
                    entries = (YamlSequenceNode)rootNode.Children[new YamlScalarNode("entries")];

                    fileSystemInfo.Version = Convert.ToByte(version.Value);
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file.");
                }

                foreach (YamlMappingNode entry in entries)
                {
                    NintendoSubmissionPackageFileSystemInfo.EntryInfo entryInfo = new NintendoSubmissionPackageFileSystemInfo.EntryInfo();
                    entryInfo.Contents = new List<NintendoSubmissionPackageFileSystemInfo.ContentInfo>();
                    entryInfo.MetaExtendedData = new List<IContentMetaExtendedData>();

                    // コンテンツ情報の生成に必要なため、先にメタタイプを取得
                    {
                        var metaType = entry.Children[new YamlScalarNode("metaType")];
                        entryInfo.MetaType = ((YamlScalarNode)metaType).Value;
                    }

                    var contents = entry.Children[new YamlScalarNode("contents")];
                    // コンテンツを持っていたら
                    if (contents is YamlSequenceNode)
                    {
                        foreach (YamlMappingNode content in (YamlSequenceNode)contents)
                        {
                            NintendoSubmissionPackageFileSystemInfo.ContentInfo contentInfo = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                            string type = string.Empty;
                            string path = string.Empty;
                            string contentType = string.Empty;
                            foreach (var child in content)
                            {
                                switch (((YamlScalarNode)child.Key).Value)
                                {
                                    case "type":
                                        type = ((YamlScalarNode)child.Value).Value;
                                        break;
                                    case "contentType":
                                        contentType = ((YamlScalarNode)child.Value).Value;
                                        break;
                                    case "path":
                                        path = ((YamlScalarNode)child.Value).Value;
                                        break;
                                    default:
                                        throw new ArgumentException("invalid format .adf file. invalid key is specified\n" + content.ToString());
                                }
                            }

                            if (path == null)
                            {
                                throw new ArgumentException("invalid format .adf file. \"path\" is not specified\n" + content.ToString());
                            }

                            if (type == "format")
                            {
                                var adfReader = new NintendoContentAdfReader(path);
                                contentInfo.SetFsInfo(contentType, null, adfReader.GetFileSystemInfo(entryInfo.MetaType));
                            }
                            else if (type == "file")
                            {
                                FileInfo fi = new FileInfo(path);
                                var keyGenerator = new NcaKeyGenerator(keyConfig);
                                using (var stream = new FileStream(path, FileMode.Open))
                                using (var ncaReader = new NintendoContentArchiveReader(stream, keyGenerator))
                                {
                                    if (NintendoContentAdfReader.ConvertToContentTypeByte(contentType) != ncaReader.GetContentType())
                                    {
                                        throw new ArgumentException(string.Format("Invalid nca file is specified, its content type is not {0}.", contentType));
                                    }
                                    contentInfo.SetSource(contentType, ncaReader.GetRepresentProgramIdOffset(), ncaReader.GetKeyGeneration(), null, new FileSource(path, 0, fi.Length));
                                }
                            }
                            else
                            {
                                throw new ArgumentException("invalid format .adf file. unknown \"type\" is specified\n" + content.ToString());
                            }

                            Debug.Assert(contentInfo.Source != null ^ contentInfo.FsInfo != null);

                            entryInfo.Contents.Add(contentInfo);
                        }
                    }
                    List<Tuple<string, string>> iconList = new List<Tuple<string, string>>();
                    List<Tuple<string, string>> nxIconList = new List<Tuple<string, string>>();
                    UInt32 nxIconMaxSize = NintendoSubmissionPackageContentInfo.DefaultNxIconMaxSize;

                    foreach (var child in entry)
                    {
                        switch (((YamlScalarNode)child.Key).Value)
                        {
                            case "contents":
                                break;
                            case "metaType":
                                break;
                            case "metaFilePath":
                                entryInfo.MetaFilePath = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "nroDirectoryPath":
                                entryInfo.NroDirectoryPath = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "keyIndex":
                                entryInfo.KeyIndex = int.Parse(((YamlScalarNode)child.Value).Value);
                                break;
                            case "hasTicket":
                                entryInfo.HasTicket = ((YamlScalarNode)child.Value).Value == "True" ? true : false;
                                break;
                            case "icon":
                                var iconInfos = ((YamlSequenceNode)child.Value);
                                foreach(YamlMappingNode iconInfo in iconInfos)
                                {
                                    var language = ((YamlScalarNode)iconInfo.Children[new YamlScalarNode("language")]).Value;
                                    var path = ((YamlScalarNode)iconInfo.Children[new YamlScalarNode("path")]).Value;
                                    iconList.Add(new Tuple<string, string>(language, path));
                                }
                                break;
                            case "nxIcon":
                                var nxIconInfos = ((YamlSequenceNode)child.Value);
                                foreach (YamlMappingNode iconInfo in nxIconInfos)
                                {
                                    var language = ((YamlScalarNode)iconInfo.Children[new YamlScalarNode("language")]).Value;
                                    var path = ((YamlScalarNode)iconInfo.Children[new YamlScalarNode("path")]).Value;
                                    nxIconList.Add(new Tuple<string, string>(language, path));
                                }
                                break;
                            case "nxIconMaxSize":
                                nxIconMaxSize = uint.Parse(((YamlScalarNode)child.Value).Value);
                                break;
                            default:
                                throw new ArgumentException("invalid format .adf file. invalid key is specified\n" + entry.ToString());
                        }
                    }

                    if (entryInfo.MetaFilePath == null)
                    {
                        throw new ArgumentException();
                    }

                    var metaReader = new MetaFileReader(entryInfo.MetaFilePath, entryInfo.MetaType);

                    // カード情報の読み込み
                    {
                        var cardSpec = metaReader.GetCardSpec();
                        // TODO: 複数 entry がある場合の CardSpec の決定ロジック
                        fileSystemInfo.CardSize = cardSpec.Item1;
                        fileSystemInfo.CardClockRate = cardSpec.Item2;
                        fileSystemInfo.CardLaunchFlags = cardSpec.Item3;
                        if (entryInfo.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication && (fileSystemInfo.CardSize == 0 || fileSystemInfo.CardClockRate == 0))
                        {
                            Log.Info(string.Format("{0} will be calculated automatically.",
                                                   (fileSystemInfo.CardSize == 0 && fileSystemInfo.CardClockRate == 0) ? "CardSize and ClockRate" :
                                                   fileSystemInfo.CardSize == 0 ? "CardSize" : "ClockRate"));
                        }
                    }

                    // 世代更新抑制フラグ
                    fileSystemInfo.KeepGeneration = metaReader.GetKeepGeneration();

                    entryInfo.ExtraData = CreateExtraSource(iconList, nxIconList, nxIconMaxSize, (byte)0);

                    fileSystemInfo.Entries.Add(entryInfo);
                }
            }
            return fileSystemInfo;
        }
        public NintendoSubmissionPackageFileSystemInfo GetFileSystemInfoForDelta(DeltaFragmentContentGenerator generator)
        {
            var fileSystemInfo = new NintendoSubmissionPackageFileSystemInfo();

            using (var adf = new StreamReader(m_adfPath, Encoding.UTF8))
            {
                var yamlStream = new YamlStream();
                yamlStream.Load(adf);

                YamlSequenceNode entries;
                try
                {
                    YamlMappingNode rootNode = (YamlMappingNode)yamlStream.Documents[0].RootNode;
                    YamlScalarNode formatType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("formatType")];
                    if (formatType.Value != "NintendoSubmissionPackageForDelta")
                    {
                        throw new ArgumentException();
                    }
                    YamlScalarNode version = (YamlScalarNode)rootNode.Children[new YamlScalarNode("version")];
                    entries = (YamlSequenceNode)rootNode.Children[new YamlScalarNode("entries")];

                    fileSystemInfo.Version = Convert.ToByte(version.Value);
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file.");
                }
                foreach (YamlMappingNode entry in entries)
                {
                    NintendoSubmissionPackageFileSystemInfo.EntryInfo entryInfo = new NintendoSubmissionPackageFileSystemInfo.EntryInfo();
                    entryInfo.Contents = new List<NintendoSubmissionPackageFileSystemInfo.ContentInfo>();
                    entryInfo.MetaExtendedData = new List<IContentMetaExtendedData>();
                    foreach (var child in entry)
                    {
                        switch (((YamlScalarNode)child.Key).Value)
                        {
                            case "metaType":
                                entryInfo.MetaType = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "metaFilePath":
                                entryInfo.MetaFilePath = ((YamlScalarNode)child.Value).Value;
                                break;
                            case "keyIndex":
                                entryInfo.KeyIndex = int.Parse(((YamlScalarNode)child.Value).Value);
                                break;
                            case "delta":
                                var delta = ((YamlMappingNode)child.Value);
                                var deltaSource = ((YamlScalarNode)delta.Children[new YamlScalarNode("source")]).Value;
                                var deltaDestination = ((YamlScalarNode)delta.Children[new YamlScalarNode("destination")]).Value;
                                var deltaCommandSizeMax = long.Parse(((YamlScalarNode)delta.Children[new YamlScalarNode("commandSizeMax")]).Value);
                                var saveDelta = bool.Parse(((YamlScalarNode)delta.Children[new YamlScalarNode("save")]).Value);
                                entryInfo.MetaExtendedData.Add(new DeltaMetaExtendedData
                                {
                                    Source = deltaSource,
                                    Destination = deltaDestination,
                                    DeltaCommandSizeMax = deltaCommandSizeMax,
                                    WillSave = saveDelta
                                });
                                break;
                            default:
                                throw new ArgumentException("invalid format .adf file. invalid key is specified\n" + entry.ToString());
                        }
                    }
                    var deltaMetaExtendedData = entryInfo.MetaExtendedData.Find(p => (p as DeltaMetaExtendedData) != null) as DeltaMetaExtendedData;
                    // 下記 API の中で、ProgramId の設定も行う
                    entryInfo.Contents = generator.GetContentInfo(
                        Path.GetDirectoryName(m_adfPath),
                        deltaMetaExtendedData.Source,
                        deltaMetaExtendedData.Destination,
                        entryInfo.KeyIndex,
                        deltaMetaExtendedData.DeltaCommandSizeMax,
                        deltaMetaExtendedData.WillSave);

                    fileSystemInfo.Entries.Add(entryInfo);
                }
            }
            return fileSystemInfo;
        }

        private List<NintendoSubmissionPackageExtraData> CreateExtraSource(List<Tuple<string, string>> iconList, List<Tuple<string, string>> nxIconList, UInt32 maxNxIconSize, byte programIdOffset)
        {
            var mergedIconPathMap = IconConverter.GetMergedIconPathMap(iconList, nxIconList);
            var extraSources = new List<NintendoSubmissionPackageExtraData>();
            foreach (var kv in mergedIconPathMap)
            {
                var language = kv.Key;
                var pathSet = kv.Value;

                var iconSet = IconConverter.Convert(pathSet.Item1, pathSet.Item2, maxNxIconSize);
                extraSources.Add(new NintendoSubmissionPackageExtraData("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX." + ApplicationControl.GetRootIconName(language, false), new MemorySource(iconSet.Item1, 0, iconSet.Item1.Length), programIdOffset));
                extraSources.Add(new NintendoSubmissionPackageExtraData("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX." + ApplicationControl.GetRootIconName(language, true),  new MemorySource(iconSet.Item2, 0, iconSet.Item2.Length), programIdOffset));
            }
            return extraSources;
        }
    }

    public class NintendoSubmissionPackageExtraData : ISource
    {
        public string EntryName { get; private set; }
        public byte IdOffset { get; private set; }

        public long Size
        {
            get
            {
                return m_Source.Size;
            }
        }

        public NintendoSubmissionPackageExtraData(string name, ISource source, byte programIdOffset)
        {
            EntryName = name;
            m_Source = source;
            IdOffset = programIdOffset;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_Source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_Source.QueryStatus();
        }
        ISource m_Source;
    }

    public class NintendoSubmissionPackageArchiveUtils
    {
        public static bool HasExternalSparseStorage(NintendoSubmissionPackageFileSystemInfo.ContentMetaInfo metaInfo, string contentType)
        {
            return (metaInfo?.Model as PatchContentMetaModel)?.SparseStorages?.Find(value => value.Type == contentType) != null;
        }

        public static bool HasSparseStorage(NintendoSubmissionPackageFileSystemInfo.EntryInfo entryInfo)
        {
            var sparseContents = entryInfo.Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeProgram || x.ContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument);
            foreach (var content in sparseContents)
            {
                // スパースパッチかチェック
                if (HasExternalSparseStorage(entryInfo.ContentMetaInfo, content.ContentType))
                {
                    return true;
                }
                // スパースアプリかチェック
                var fsInfo = content.FsInfo as NintendoContentFileSystemInfo;
                if (fsInfo != null)
                {
                    foreach (var entry in fsInfo.fsEntries)
                    {
                        if (entry.existsSparseLayer)
                        {
                            return true;
                        }
                    }
                }
            }
            return false;
        }

        private static IFileSystemArchiveReader OpenFsReader(NintendoContentFileSystemInfo.EntryInfo entry, Stream stream)
        {
            switch (entry.formatType)
            {
                case NintendoContentFileSystemMetaConstant.FormatTypePartitionFs:
                    return new PartitionFileSystemArchiveReader(stream);
                case NintendoContentFileSystemMetaConstant.FormatTypeRomFs:
                case NintendoContentFileSystemMetaConstant.FormatTypePatchableRomFs:
                    return new RomFsFileSystemArchiveReader(stream);
                default:
                    throw new NotImplementedException();
            }
        }

        private delegate bool FileNameMatchingDelegate(string fileName);

        private static void UpdateExternalKey(NintendoContentArchiveReader ncaReader)
        {
            var rightsId = ncaReader.GetRightsId();
            if (TicketUtility.IsValidRightsId(rightsId))
            {
                // TORIAEZU: RightsId から鍵を生成
                var hashCalculator = new HmacSha256HashCryptoDriver(HmacSha256KeyIndex.TitleKeyGenarateKey);
                var contentMetaId = TicketUtility.GetContentMetaIdFromRightsId(rightsId);
                ncaReader.SetExternalKey(ExternalContentKeyGenerator.GetNcaExternalContentKey(hashCalculator, contentMetaId, ncaReader.GetKeyGeneration()).Key);
            }
        }

        private static List<Tuple<string, byte[]>> GetFileDataFromContent(NintendoSubmissionPackageFileSystemInfo.ContentInfo content, KeyConfiguration config, int fsIndex, FileNameMatchingDelegate fileNameMatching)
        {
            var bytesList = new List<Tuple<string, byte[]>>();

            if (content.FsInfo != null)
            {
                var fsEntries = (content.FsInfo as NintendoContentFileSystemInfo).fsEntries;
                if (fsEntries.Count <= fsIndex)
                {
                    return bytesList;
                }
                var entry = fsEntries[fsIndex];
                if (entry.formatType == "SparseFs")
                {
                    var sparseInfo = (entry.sourceInterface as FileSystemArchiveSparseSource).ExtraInfo;
                    var fsReader = sparseInfo.patchReader.OpenFileSystemArchiveReader(fsIndex, sparseInfo.originalReader);
                    foreach (var file in fsReader.ListFileInfo().Where(x => fileNameMatching(x.Item1)))
                    {
                        bytesList.Add(Tuple.Create(file.Item1, fsReader.ReadFile(file.Item1, 0, file.Item2)));
                    }
                }
                else
                {
                    ISource fsSource;
                    if (entry.formatType == NintendoContentFileSystemMetaConstant.FormatTypePatchableRomFs)
                    {
                        var patchFsInfo = entry.fileSystemInfo as NintendoContentPatchFileSystemInfo;
                        var integrityRomFsSource = patchFsInfo.NewSource;
                        fsSource = new SubSource(integrityRomFsSource, patchFsInfo.HashTargetOffset, integrityRomFsSource.Size - patchFsInfo.HashTargetOffset);
                    }
                    else
                    {
                        fsSource = NintendoContentArchiveSource.GetDataSource(entry);
                    }

                    using (var stream = new SourceBasedStream(fsSource))
                    {
                        var fsReader = OpenFsReader(entry, stream);
                        foreach (var file in fsReader.ListFileInfo().Where(x => fileNameMatching(x.Item1)))
                        {
                            bytesList.Add(Tuple.Create(file.Item1, fsReader.ReadFile(file.Item1, 0, file.Item2)));
                        }
                    }
                }
            }
            else if (content.Source != null)
            {
                var keyGenerator = new NcaKeyGenerator(config);
                using (var stream = new SourceBasedStream(content.Source))
                using (var originalStream = content.OriginalSource != null ? new SourceBasedStream(content.OriginalSource) : null)
                {
                    var ncaReader = new NintendoContentArchiveReader(stream, keyGenerator);
                    UpdateExternalKey(ncaReader);
                    if (!ncaReader.HasFsInfo(fsIndex))
                    {
                        return bytesList;
                    }

                    NintendoContentArchiveReader originalNcaReader = null;
                    if (originalStream != null)
                    {
                        originalNcaReader = new NintendoContentArchiveReader(originalStream, keyGenerator);
                        UpdateExternalKey(originalNcaReader);
                    }

                    var fsReader = ncaReader.OpenFileSystemArchiveReader(fsIndex, originalNcaReader);
                    foreach (var file in fsReader.ListFileInfo().Where(x => fileNameMatching(x.Item1)))
                    {
                        bytesList.Add(Tuple.Create(file.Item1, fsReader.ReadFile(file.Item1, 0, file.Item2)));
                    }
                }
            }
            else
            {
                throw new NotImplementedException();
            }

            return bytesList;
        }

        public static bool IsNoEncryptionContentIncluded(NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, KeyConfiguration config)
        {
            foreach (var content in entry.Contents)
            {
                if (content.FsInfo != null)
                {
                    if ((content.FsInfo as NintendoContentFileSystemInfo).headerEncryptionType == (byte)NintendoContentArchiveHeaderEncryptionType.None)
                    {
                        return true;
                    }
                }
                else if (content.Source != null)
                {
                    if (content.Source is NintendoDeltaFragmentArchiveSource)
                    {
                        continue;
                    }

                    var keyGenerator = new NcaKeyGenerator(config);
                    using (var stream = new SourceBasedStream(content.Source))
                    {
                        var ncaReader = new NintendoContentArchiveReader(stream, keyGenerator);
                        if (ncaReader.GetHeaderEncryptionType() == (byte)NintendoContentArchiveHeaderEncryptionType.None)
                        {
                            return true;
                        }
                    }
                }
                else
                {
                    throw new NotImplementedException();
                }
            }
            return false;
        }

        public static List<byte[]> GetUsefulNroHashListFromNrr(NintendoSubmissionPackageFileSystemInfo.ContentInfo content, KeyConfiguration config)
        {
            FileNameMatchingDelegate matchingImpl = delegate (string fileName)
            {
                return Regex.IsMatch(fileName, "^(.nrr/).+(.nrr)$");
            };

            List<byte[]> usefulNroHashList = new List<byte[]>();

            Trace.Assert(content.ContentType == NintendoContentMetaConstant.ContentTypeProgram);

            var bytesList = GetFileDataFromContent(content, config, 1, matchingImpl);
            foreach (var nrrBytes in bytesList.Select(x => x.Item2))
            {
                usefulNroHashList.AddRange(NrrFile.GetUsefulNroHashList(nrrBytes));
            }

            return usefulNroHashList;
        }

        public static string GetDescFileNameFromContent(NintendoSubmissionPackageFileSystemInfo.ContentInfo content)
        {
            return content.DescFileName;
        }

        public static ProgramInfoModel GetProgramInfoModelFromContent(NintendoSubmissionPackageFileSystemInfo.ContentInfo content, KeyConfiguration config, out bool elf32Mode)
        {
            elf32Mode = false;

            Trace.Assert(content.ContentType == NintendoContentMetaConstant.ContentTypeProgram);

            // main, main.npdm の取得
            FileNameMatchingDelegate matchingImpl = delegate (string fileName)
            {
                return fileName == "main" || fileName == "main.npdm";
            };

            var bytesList = GetFileDataFromContent(content, config, 0, matchingImpl);
            elf32Mode = SymbolExtract.IsElf32(bytesList.Where(x => x.Item1 == "main.npdm").Select(x => x.Item2).Single());

            byte[] bytes;
            if (!bytesList.Where(x => x.Item1 == "main").Any())
            {
                // mainが存在しない場合は空リストを返す(テスト時のみ発生する)
                Log.Warning("\"main\" was not found in code region. Api list in programinfo.xml will be empty.");
                bytes = new byte[0];
            }
            else
            {
                bytes = bytesList.Where(x => x.Item1 == "main").Select(x => x.Item2).Single();
            }

            return ProgramInfoModel.Create(new SymbolExtract(bytes, "main", ObjectType.NsoObject, elf32Mode), elf32Mode);
        }
        public static ProgramInfoModel GetProgramInfoModelFromLibrary(NintendoSubmissionPackageFileSystemInfo.ContentInfo content, KeyConfiguration config, bool elf32Mode)
        {
            List<ProgramInfoModel> programinfoModelList = new List<ProgramInfoModel>();

            Trace.Assert(content.ContentType == NintendoContentMetaConstant.ContentTypeProgram);

            // sdk, subsdk* の取得
            FileNameMatchingDelegate matchingImpl = delegate (string fileName)
            {
                return fileName == "sdk" || Regex.IsMatch(fileName, "^(subsdk).*$");
            };

            var bytesList = GetFileDataFromContent(content, config, 0, matchingImpl);
            foreach (var bytes in bytesList)
            {
                programinfoModelList.Add(ProgramInfoModel.Create(new SymbolExtract(bytes.Item2, bytes.Item1, ObjectType.NsoLibraryObject, elf32Mode), elf32Mode));
            }

            var programinfoModel = ProgramInfoModel.Create(programinfoModelList);

            return programinfoModel;
        }
        public static ProgramInfoModel GetProgramInfoModelFromNro(string nroDirectoryPath, bool elf32Mode, List<byte[]> usefulHashList)
        {
            List<ProgramInfoModel> programinfoModelList = new List<ProgramInfoModel>();

            //nroディレクトリ以下にある".nro"ファイルを取得
            string[] nroFiles = System.IO.Directory.GetFiles(nroDirectoryPath, "*.nro", System.IO.SearchOption.AllDirectories);

            foreach (string filePath in nroFiles)
            {
                var fs = File.OpenRead(filePath);

                byte[] data = new byte[fs.Length];
                var readSize = fs.Read(data, 0, (int)fs.Length);
                Debug.Assert((int)fs.Length == readSize);
                if (NrrFile.SetCheckUsefulHashValue(usefulHashList, NrrFile.GetHashValue(data)) == true)
                {
                    programinfoModelList.Add(ProgramInfoModel.Create(new SymbolExtract(data, filePath, ObjectType.NroObject, elf32Mode), elf32Mode));
                }
            }

            var programinfoModel = ProgramInfoModel.Create(programinfoModelList);

            return programinfoModel;
        }

        public static byte[] GetNpdmBinary(NintendoSubmissionPackageFileSystemInfo.ContentInfo content, KeyConfiguration config)
        {
            Trace.Assert(content.ContentType == NintendoContentMetaConstant.ContentTypeProgram);

            // main.npdm の取得
            FileNameMatchingDelegate matchingImpl = delegate (string fileName)
            {
                return fileName == "main.npdm";
            };

            var bytesList = GetFileDataFromContent(content, config, 0, matchingImpl);
            Debug.Assert(bytesList.Count == 1);
            var npdmBytes = bytesList.Select(x => x.Item2).Single();

            return npdmBytes;
        }

        public static ApplicationControlPropertyModel GetApplicationControlProperty(NintendoSubmissionPackageFileSystemInfo.ContentInfo content, KeyConfiguration config)
        {
            Trace.Assert(content.ContentType == NintendoContentMetaConstant.ContentTypeControl);

            FileNameMatchingDelegate matchingImpl = delegate (string fileName)
            {
                return fileName == "control.nacp";
            };

            var bytesList = GetFileDataFromContent(content, config, 0, matchingImpl);
            Debug.Assert(bytesList.Count == 1);

            return ApplicationControlPropertyModel.PropertyBytesToModel(bytesList.Select(x => x.Item2).Single());
        }

        public static ISource GetMemorySourceFromContent(NintendoSubmissionPackageFileSystemInfo.ContentInfo content, KeyConfiguration config, string fileName)
        {
            FileNameMatchingDelegate matchingImpl = delegate (string entryFileName)
            {
                return entryFileName == fileName;
            };

            var bytesList = GetFileDataFromContent(content, config, 0, matchingImpl);
            if (bytesList.Count() == 0)
            {
                return null;
            }
            else if (bytesList.Count() != 1)
            {
                throw new ArgumentException(string.Format("'{0}' is duplicated.", fileName));
            }
            var bytes = bytesList[0].Item2;
            return new MemorySource(bytes, 0, bytes.Length);
        }

        public static void GetAdditionalInfoForCardSizeCalculation(out long additionalSize, out List<ContentMetaLiteModel> additionalContentMetaList, NintendoSubmissionPackageFileSystemInfo fileSystemInfo)
        {
            additionalSize = 0;
            additionalContentMetaList = new List<ContentMetaLiteModel>();

            if (fileSystemInfo.OnCardAddOnContentInfo != null)
            {
                additionalContentMetaList.AddRange(fileSystemInfo.OnCardAddOnContentInfo.Application.ContentMetaList);
                if (fileSystemInfo.OnCardAddOnContentInfo.Patch != null)
                {
                    additionalContentMetaList.AddRange(fileSystemInfo.OnCardAddOnContentInfo.Patch.ContentMetaList);
                }
            }
            else if (fileSystemInfo.MultiApplicationCardInfo != null)
            {
                additionalContentMetaList.AddRange(fileSystemInfo.MultiApplicationCardInfo.Application.ContentMetaList);
                if (fileSystemInfo.MultiApplicationCardInfo.Patch != null)
                {
                    additionalContentMetaList.AddRange(fileSystemInfo.MultiApplicationCardInfo.Patch.ContentMetaList);
                }
            }
            else
            {
                foreach (var entry in fileSystemInfo.Entries)
                {
                    if (!entry.Contents.Any(x => x.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation))
                    {
                        // リーガル情報がないときには、 512 KB の下駄を履かせる
                        additionalSize += 512 * 1024;
                    }
                    if (entry.MetaType == NintendoContentMetaConstant.ContentMetaTypePatch && entry.ContentMetaInfo != null)
                    {
                        // レガシーなパッチ(creatensp --type Patch)は、ContentMetaInfo が設定されていない場合がある
                        additionalSize += (512 * 1024); // delta をマージするときに、contentmeta が膨れるのでマージン
                        var model = entry.ContentMetaInfo.Model as PatchContentMetaModel;
                        var history = model.HistoryList.Find(p => p.Type == NintendoContentMetaConstant.ContentMetaTypeApplication);
                        if (history != null)
                        {
                            additionalContentMetaList.Add(history);
                        }
                    }
                }
            }
        }
    }
    public interface IContentMetaExtendedData
    {

    }
    public class DeltaMetaExtendedData : IContentMetaExtendedData
    {
        public string Source { get; set; }
        public string Destination { get; set; }

        public long DeltaCommandSizeMax { get; set; }

        public bool WillSave { get; set; }
    }

    public class NintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        private bool CheckNeedTicket(NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, bool isProdEncryption)
        {
            return (entry.HasTicket || isProdEncryption) && TicketUtility.NeedCreateTicket(entry.MetaType);
        }

        private bool CheckNeedCardSpec(NintendoSubmissionPackageFileSystemInfo fileSystemInfo)
        {
            return fileSystemInfo.Entries.Any(entry => entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication || entry.MetaType == NintendoContentMetaConstant.ContentMetaTypePatch) || fileSystemInfo.OnCardAddOnContentInfo != null || fileSystemInfo.MultiApplicationCardInfo != null;
        }

        private UInt64 GetApplicationIdFromContentMetaRawData(byte[] contentMetaRawData)
        {
            return BitConverter.ToUInt64(contentMetaRawData, 0);
        }

        private void AddConnection(out IReadableSink outReadableSubSink, ISource source, IReadableSink outSink, long offset)
        {
            if (outSink.Size < offset + source.Size)
            {
                outSink.SetSize(offset + source.Size); // SubSink で使う範囲までサイズを更新しておく
            }

            outReadableSubSink = new ReadableSubSink(outSink, offset, source.Size);
            ConnectionList.Add(new Connection(source, outReadableSubSink));
        }

        private void AddConnection(ISource source, IReadableSink outSink, long offset)
        {
            IReadableSink sink;
            AddConnection(out sink, source, outSink, offset);
        }

        private long RegisterRootEntry(IReadableSink outSink, ref PartitionFileSystemInfo rootFsInfo, long rootFsHeaderSize, ref List<ContentHashSource> hashSources, string name, ISource source, long offset, ContentHashSource hashSource)
        {
            // PartitionFs のエントリオフセットはヘッダを除いたもの
            var partFsEntry = PartitionFileSystemInfo.EntryInfo.Make(name, (ulong)source.Size, (ulong)(offset - rootFsHeaderSize));
            rootFsInfo.entries.Add(partFsEntry);

            AddConnection(source, outSink, offset);

            hashSources.Add(hashSource);

            return offset + source.Size;
        }

        bool CheckNeedLogoPartition(NintendoSubmissionPackageFileSystemInfo.EntryInfo entry)
        {
            return entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication;
        }

        private void CheckFileSystemInfo(NintendoSubmissionPackageFileSystemInfo fileSystemInfo)
        {
            if (fileSystemInfo.IsProdEncryption)
            {
                if (fileSystemInfo.Entries.Where(x => x.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication).Any())
                {
                    Trace.Assert(fileSystemInfo.CardSize != 0 && fileSystemInfo.CardClockRate != 0);
                }
            }
        }

        private void UpdateGenerationForAddOnContent(ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, int index)
        {
            if (entry.MetaType != NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
            {
                return;
            }

            uint version;
            if (entry.ContentMetaInfo != null && entry.ContentMetaInfo.Data != null)
            {
                var contentMetaBase = new NintendoContentMetaBase(entry.MetaType, entry.ContentMetaInfo.Model);
                version = contentMetaBase.GetContentMetaDescryptor().Version;
            }
            else
            {
                var metaReader = new MetaFileReader(entry.MetaFilePath, entry.MetaType);
                var meta = metaReader.GetContentMetaList().ElementAt(index);
                version = NintendoContentMetaBase.MakeVersion(meta.Version, meta.ReleaseVersion, meta.PrivateVersion);
            }

            foreach (var content in entry.Contents)
            {
                if (content.FsInfo == null)
                {
                    continue;
                }
                var fsEntries = (content.FsInfo as NintendoContentFileSystemInfo).fsEntries;
                for (int i = 0; i < fsEntries.Count; ++i)
                {
                    var fsEntry = fsEntries[i];
                    fsEntry.generation = version;
                    fsEntries[i] = fsEntry;
                }
            }
        }

        public NintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageFileSystemInfo fileSystemInfo, KeyConfiguration config)
        {
            CheckFileSystemInfo(fileSystemInfo);

            ConnectionList = new List<Connection>();

            PartitionFileSystemMeta metaMgr = new PartitionFileSystemMeta();

            // ヘッダのサイズを先に計算
            long headerSize = 0;
            {
                var entryNameList = new List<string>();
                for (int i = 0; i < fileSystemInfo.Entries.Count; i++)
                {
                    for (int j = 0; j < fileSystemInfo.Entries[i].Contents.Count; j++)
                    {
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.nca"); // 32 文字+ 拡張子

                        if (NintendoSubmissionPackageArchiveUtils.HasExternalSparseStorage(fileSystemInfo.Entries[i].ContentMetaInfo, fileSystemInfo.Entries[i].Contents[j].ContentType))
                        {
                            entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.compaction.bin"); // 32 文字+ 拡張子
                        }
                    }
                    entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.nca"); // 32 文字+ 拡張子
                    entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.xml"); // 32 文字+ 拡張子

                    foreach (var programContent in fileSystemInfo.Entries[i].Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeProgram))
                    {
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.programinfo.xml"); // 32 文字+ 拡張子
                    }

                    // creatensp で --ticket オプションが指定されているかまたは prodencryption である場合、かつ、
                    // MetaType が Application または Patch または AddOnContent であればチケット用と証明書用のヘッダを作成する
                    if (CheckNeedTicket(fileSystemInfo.Entries[i], fileSystemInfo.IsProdEncryption))
                    {
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.tik"); // 32 文字+ 拡張子
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cert"); // 32 文字+ 拡張子
                    }
                    if (fileSystemInfo.EnableContentMetaBinaryExport)
                    {
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt"); // 32 文字+ 拡張子
                    }

                    foreach(var extra in fileSystemInfo.Entries[i].ExtraData)
                    {
                        entryNameList.Add(extra.EntryName);
                    }

                    foreach (var controlContent in fileSystemInfo.Entries[i].Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeControl))
                    {
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.nacp.xml");
                    }

                    foreach (var legalInfoContent in fileSystemInfo.Entries[i].Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation))
                    {
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.legalinfo.xml"); // 32 文字+ 拡張子
                    }

                    foreach (var htmlDocContent in fileSystemInfo.Entries[i].Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument))
                    {
                        entryNameList.Add("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.htmldocument.xml"); // 32 文字+ 拡張子
                    }
                }

                if (CheckNeedCardSpec(fileSystemInfo))
                {
                    entryNameList.Add("cardspec.xml");
                }

                if (fileSystemInfo.OnCardAddOnContentInfo != null)
                {
                    entryNameList.Add("oncardaoc.xml");
                }

                if (fileSystemInfo.MultiApplicationCardInfo != null)
                {
                    entryNameList.Add("multiapplicationcard.xml");
                }

                if (!fileSystemInfo.IsProdEncryption ||
                    (fileSystemInfo.IsProdEncryption && fileSystemInfo.AuthoringToolInfo != null))
                {
                    entryNameList.Add("authoringtoolinfo.xml");
                }

                headerSize = HashNameEntryPartitionFsHeaderSource<PartitionFileSystemMeta>.GetDummySize(entryNameList);
            }
            long currentOffset = headerSize;

            // TODO: hashSources が Entry 毎になっていない
            List<ContentHashSource> hashSources = new List<ContentHashSource>();

            // NintendoSubmissionPackageFileSystemInfo を元に、PartitionFs 相当のアーカイブを作る
            PartitionFileSystemInfo partFsInfo = new PartitionFileSystemInfo();
            partFsInfo.version = fileSystemInfo.Version;

            for (int i = 0; i < fileSystemInfo.Entries.Count; i++)
            {
                var entry = fileSystemInfo.Entries[i];
                UpdateGenerationForAddOnContent(ref entry, i);

                bool hasLegalInformation = entry.Contents.Exists(content => content.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation);
                // コンテンツ
                var contentSourceList = new List<Tuple<ISource, NintendoContentDescriptor>>();
                for (int j = 0; j < entry.Contents.Count; j++)
                {
                    var content = entry.Contents[j];

                    NintendoContentArchiveSource.BuildLog.SetContent(j, content.ContentType);

                    ISource contentSource;
                    if (content.FsInfo != null)
                    {
                        var ncaInfo = content.FsInfo as NintendoContentFileSystemInfo;
                        contentSource = new NintendoContentArchiveSource(ncaInfo, config, CheckNeedLogoPartition(entry));
                        content.SetSourceSizeCache(contentSource);
                    }
                    else if (content.Source != null)
                    {
                        if (content.Source is NintendoDeltaFragmentArchiveSource)
                        {
                            if (NintendoContentArchiveSource.BuildLog.NeedsOutputLog)
                            {
                                ((NintendoDeltaFragmentArchiveSource)content.Source).OutputBuildLog(NintendoContentArchiveSource.BuildLog);
                            }
                        }

                        contentSource = content.Source;
                    }
                    else
                    {
                        throw new NotImplementedException();
                    }

                    // ファイル名は後で修正される。文字数(32 文字+ 拡張子)だけ合わせておく
                    // PartitionFs のエントリオフセットはヘッダを除いたもの
                    var partFsEntry = PartitionFileSystemInfo.EntryInfo.Make("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.nca", (ulong)contentSource.Size, (ulong)(currentOffset - headerSize));
                    partFsInfo.entries.Add(partFsEntry);

                    // 出力用 Sink と接続
                    IReadableSink contentSink;
                    AddConnection(out contentSink, contentSource, outSink, currentOffset);

                    // コンテンツのハッシュ計算（コンテンツメタの作成にも利用）
                    // 一度 Sink に書き出したものをソースにするようにする
                    var contentHashSource = new SinkLinkedSource(contentSink, new Sha256StreamHashSource(contentSink.ToSource()));

                    // ファイル名に差し込むためにリストに保持
                    hashSources.Add(new ContentHashSource(contentHashSource, ".nca", content.ContentType));

                    currentOffset += contentSource.Size;

                    var contentInfo = new NintendoContentInfo(content.ContentType, contentSource.Size, content.KeyGeneration, content.IdOffset);
                    var contentFragmentInfo = new NintendoContentFragmentInfo();
                    if (content.Fragment != null)
                    {
                        contentFragmentInfo.FragmentIndex = content.Fragment.Index;
                        contentFragmentInfo.SourceContentId = content.Fragment.SourceContentId;
                        contentFragmentInfo.SourceSize = content.Fragment.SourceSize;
                        contentFragmentInfo.DestinationContentId = content.Fragment.DestinationContentId;
                        contentFragmentInfo.DestinationSize = content.Fragment.DestinationSize;
                        contentFragmentInfo.FragmentTargetContentType = content.Fragment.TargetContentType;
                        contentFragmentInfo.UpdateType = content.Fragment.UpdateType;
                    }

                    var contentDescriptor = new NintendoContentDescriptor();
                    contentDescriptor.ContentInfo = contentInfo;
                    contentDescriptor.ContentFragmentInfo = contentFragmentInfo;

                    contentSourceList.Add(Tuple.Create(hashSources[hashSources.Count - 1].Source, contentDescriptor));

                    if (content.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation)
                    {
                        hasLegalInformation = true;
                    }

                    // スパース情報
                    if (NintendoSubmissionPackageArchiveUtils.HasExternalSparseStorage(entry.ContentMetaInfo, content.ContentType))
                    {
                        var patchMetaModel = entry.ContentMetaInfo.Model as PatchContentMetaModel;
                        var sparseStream = patchMetaModel.SparseStorages.Where(value => value.Type == content.ContentType).Single();
                        var sparseSource = new SparseStorageArchiveSource(sparseStream);
                        currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.compaction.bin", sparseSource, currentOffset, new ContentHashSource(contentHashSource, ".compaction.bin"));
                    }
                }

                if (entry.MetaType == NintendoContentMetaConstant.ContentMetaTypeApplication && !hasLegalInformation)
                {
                    Log.Info("Legal infomation is not included. Please add legal infomation before submitting nsp to nintendo.");
                }

                // コンテンツメタと、コンテンツメタのハッシュをファイル名にするエントリ
                {
                    NintendoContentMetaBase contentMetaBase;
                    if (entry.ContentMetaInfo?.Data != null &&
                        entry.ContentMetaInfo?.Model != null)
                    {
                        contentMetaBase = new NintendoContentMetaBase(contentSourceList, entry.ContentMetaInfo.Data, entry.ContentMetaInfo.Model, fileSystemInfo.IsProdEncryption);
                    }
                    else
                    {
                        contentMetaBase = new NintendoContentMetaBase(contentSourceList, entry.MetaType, entry.MetaFilePath, entry.MetaExtendedData, i);
                    }
                    NintendoContentMetaArchiveSource contentMetaSource = new NintendoContentMetaArchiveSource(contentMetaBase, entry.KeyIndex, config, fileSystemInfo.IsProdEncryption, false, NintendoSubmissionPackageArchiveUtils.IsNoEncryptionContentIncluded(entry, config));
                    ISource contentMetaHashSource;

                    // cnmt.nca
                    {
                        var partFsEntry = PartitionFileSystemInfo.EntryInfo.Make("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.nca", (ulong)contentMetaSource.Size, (ulong)(currentOffset - headerSize));
                        partFsInfo.entries.Add(partFsEntry);

                        IReadableSink contentMetaSink;
                        AddConnection(out contentMetaSink, contentMetaSource, outSink, currentOffset);

                        // コンテンツメタのハッシュ計算
                        contentMetaHashSource = new SinkLinkedSource(contentMetaSink, new Sha256StreamHashSource(contentMetaSink.ToSource()));

                        hashSources.Add(new ContentHashSource(contentMetaHashSource, ".cnmt.nca"));

                        currentOffset += contentMetaSource.Size;
                    }

                    // cnmt.xml
                    {
                        var xmlSource = new NintendoContentMetaXmlSource(contentMetaBase, contentMetaHashSource, contentMetaSource.Size);
                        currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt.xml", xmlSource, currentOffset, new ContentHashSource(contentMetaHashSource, ".cnmt.xml"));
                    }

                    // programInfo.xml
                    foreach (var programContent in fileSystemInfo.Entries[i].Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeProgram).Select((x, idx) => new { Value = x, IdOffset = idx }))
                    {
                        var model = fileSystemInfo.Entries[i].ProgramInfo?.Models.Skip(programContent.IdOffset).FirstOrDefault();

                        ISource programInfoXmlSource;
                        if (NintendoSubmissionPackageArchiveUtils.HasSparseStorage(fileSystemInfo.Entries[i]))
                        {
                            // 元々の programinfo.xml をそのまま流用する
                            if (model == null)
                            {
                                throw new InvalidOperationException();
                            }

                            // パッチの場合に PatchToolVersion を追加する
                            if (fileSystemInfo.Entries[i].MetaType == NintendoContentMetaConstant.ContentMetaTypePatch && fileSystemInfo.IsProdEncryption == false)
                            {
                                model.SetPatchToolVersion();
                            }

                            programInfoXmlSource = new ProgramInfoXmlSource(model);
                        }
                        else
                        {
                            // main.npdm から acid, SaveDataOwnerId を読み込み
                            var npdmBinary = NintendoSubmissionPackageArchiveUtils.GetNpdmBinary(programContent.Value, config);
                            // acid 領域の読み込み
                            var acidBinary = NpdmParser.GetAcid(npdmBinary);
                            var saveDataOwnerIdList = NpdmParser.ParseSaveDataOwnerId(npdmBinary);

                            // 出力用のProgramInfoXmlインスタンスを作成
                            if (model != null)
                            {
                                // SystemProgram のみ ProgramInfo.xml 内に <Desc> がなくても許容する
                                if (fileSystemInfo.Entries[i].MetaType == NintendoContentMetaConstant.ContentMetaTypeSystemProgram)
                                {
                                    if (model.Desc == null)
                                    {
                                        model.Desc = Convert.ToBase64String(acidBinary);
                                        model.DescFlags = new DescFlags();
                                        model.DescFlags.UnqualifiedApproval = NpdmParser.GetUnqualifiedApproval(acidBinary);
                                        model.DescFlags.Production = NpdmParser.GetProductionFlag(acidBinary);
                                    }
                                }

                                // ProgramInfo 内の DescFileName をコンテンツが持つ情報に更新
                                var newDescFileName = programContent.Value.DescFileName;
                                if (!string.IsNullOrEmpty(newDescFileName))
                                {
                                    model.DescFileName = newDescFileName;
                                }

                                // 引き継いだ ProgramInfo 内の Desc と（製品化前の）プログラムコンテンツに含まれる main.npdm の値が一致することを確認
                                model.CheckAcidIntegrity(acidBinary);

                                // パッチの場合に、PatchToolVersion を追加する
                                if (fileSystemInfo.Entries[i].MetaType == NintendoContentMetaConstant.ContentMetaTypePatch && fileSystemInfo.IsProdEncryption == false)
                                {
                                    model.SetPatchToolVersion();
                                }

                                programInfoXmlSource = new ProgramInfoXmlSource(model);
                            }
                            else
                            {
                                List<ProgramInfoModel> programinfoModelList = new List<ProgramInfoModel>();
                                bool elf32Mode;

                                programinfoModelList.Add(NintendoSubmissionPackageArchiveUtils.GetProgramInfoModelFromContent(programContent.Value, config, out elf32Mode));
                                programinfoModelList.Add(NintendoSubmissionPackageArchiveUtils.GetProgramInfoModelFromLibrary(programContent.Value, config, elf32Mode));

                                List<byte[]> usefulNroHashList = NintendoSubmissionPackageArchiveUtils.GetUsefulNroHashListFromNrr(programContent.Value, config);
                                if (fileSystemInfo.Entries[i].NroDirectoryPath != null && fileSystemInfo.Entries[i].NroDirectoryPath != String.Empty)
                                {
                                    programinfoModelList.Add(NintendoSubmissionPackageArchiveUtils.GetProgramInfoModelFromNro(fileSystemInfo.Entries[i].NroDirectoryPath, elf32Mode, usefulNroHashList));
                                }
                                // nro検索結果チェック
                                if (NrrFile.GetUsefulNroHashListResult(usefulNroHashList) == false)
                                {
                                    throw new Exception("a Nro file registered with Nrr Hash Lists wasn't found. or -nro option isn't defined.\n");
                                }

                                programInfoXmlSource = new ProgramInfoXmlSource(ProgramInfoModel.Create(programinfoModelList, acidBinary, saveDataOwnerIdList, programContent.Value.DescFileName));
                            }
                        }

                        var programContentHashSource = hashSources.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeProgram).Select(x => x.Source).Skip(programContent.IdOffset).First();
                        currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.programinfo.xml", programInfoXmlSource, currentOffset, new ContentHashSource(programContentHashSource, ".programinfo.xml"));
                    }

                    // バイナリ出力
                    if (fileSystemInfo.EnableContentMetaBinaryExport)
                    {
                        var cnmtSource = contentMetaBase.GetSource();
                        currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.cnmt", cnmtSource, currentOffset, new ContentHashSource(contentMetaHashSource, ".cnmt"));
                    }

                    // アプリケーション管理データ
                    foreach (var controlContent in fileSystemInfo.Entries[i].Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeControl).Select((x, idx) => new { Value = x, IdOffset = idx }))
                    {
                        var controlContentHashSource = hashSources.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeControl).Select(x => x.Source).Skip(controlContent.IdOffset).First();

                        // 拡張データ（アイコン）
                        var targetExtraData = entry.ExtraData.Where(x => x.IdOffset == controlContent.IdOffset);
                        if (targetExtraData == null)
                        {
                            // 個別のアイコンが指定されていない場合は IdOffset が 0 のものを参照する
                            targetExtraData = entry.ExtraData.Where(x => x.IdOffset == 0);
                        }
                        foreach(var extra in targetExtraData)
                        {
                            currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, extra.EntryName, extra, currentOffset, new ContentHashSource(controlContentHashSource, ".jpg"));
                        }

                        var model = fileSystemInfo.Entries[i].ApplicationControlPropertyInfo?.Models.Skip(controlContent.IdOffset).FirstOrDefault();
                        if (model == null)
                        {
                            model = NintendoSubmissionPackageArchiveUtils.GetApplicationControlProperty(controlContent.Value, config);
                        }
                        model.Icon = IconConverter.GetApplicationControlPropertyIconEntryWithHash(targetExtraData.ToList());
                        var nacpXmlSource = new ApplicationControlPropertyXmlSource(model);

                        currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.nacp.xml", nacpXmlSource, currentOffset, new ContentHashSource(controlContentHashSource, ".nacp.xml"));
                    }

                    // リーガル情報の XML 出力
                    foreach (var legalInfoContent in fileSystemInfo.Entries[i].Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation).Select((x, idx) => new { Value = x, IdOffset = idx }))
                    {
                        var model = fileSystemInfo.Entries[i].LegalInformationInfo?.Models.Skip(legalInfoContent.IdOffset).FirstOrDefault();
                        ISource legalInfoXmlSource = (model != null) ? new LegalInformationXml(model) : new LegalInformationXml(legalInfoContent.Value, config);
                        var legalInfoContentHashSource = hashSources.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation).Select(x => x.Source).Skip(legalInfoContent.IdOffset).First();
                        currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.legalinfo.xml", legalInfoXmlSource, currentOffset, new ContentHashSource(legalInfoContentHashSource, ".legalinfo.xml"));
                    }

                    // HtmlDocument 情報の XML 出力
                    foreach (var htmlDocContent in fileSystemInfo.Entries[i].Contents.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument).Select((x, idx) => new { Value = x, IdOffset = idx }))
                    {
                        var model = fileSystemInfo.Entries[i].HtmlDocumentInfo?.Models.Skip(htmlDocContent.IdOffset).FirstOrDefault();
                        ISource htmlDocumentXmlSource = (model != null) ? new HtmlDocumentXml(model) : new HtmlDocumentXml(htmlDocContent.Value, config);

                        if (htmlDocumentXmlSource.Size != 0)
                        {
                            var htmlDocumentContentHashSource = hashSources.Where(x => x.ContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument).Select(x => x.Source).Skip(htmlDocContent.IdOffset).First();
                            currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX.htmldocument.xml", htmlDocumentXmlSource, currentOffset, new ContentHashSource(htmlDocumentContentHashSource, ".htmldocument.xml"));
                        }
                    }
                }

                // その他のエントリ
                {
                    // チケットと証明書
                    // creatensp で --ticket オプションが指定されているかまたは prodencryption である場合、かつ、
                    // MetaType が Application または Patch または AddOnContent であればチケットと証明書を作成する
                    if (CheckNeedTicket(fileSystemInfo.Entries[i], fileSystemInfo.IsProdEncryption))
                    {
                        byte? keyGenerationForTicket = null; // 外部暗号化されるコンテンツの keyGeneration
                        foreach (var content in fileSystemInfo.Entries[i].Contents.Where(x => ProdEncryptionUtils.IsContentTypeEncryptedByExternalKey(x.ContentType)))
                        {
                            if (keyGenerationForTicket.HasValue)
                            {
                                if (keyGenerationForTicket.Value != content.KeyGeneration)
                                {
                                    throw new ArgumentException("There are contents that should be encrypted by external key but have different keygeneration.");
                                }
                            }
                            else
                            {
                                keyGenerationForTicket = content.KeyGeneration;
                            }
                        }

                        UInt64 contentMetaId = 0;
                        if (fileSystemInfo.Entries[i].ContentMetaInfo == null)
                        {
                            var reader = new MetaFileReader(fileSystemInfo.Entries[i].MetaFilePath, fileSystemInfo.Entries[i].MetaType);
                            contentMetaId = reader.GetContentMetaList().ElementAt(i).Id;
                        }
                        else
                        {
                            contentMetaId = GetApplicationIdFromContentMetaRawData(fileSystemInfo.Entries[i].ContentMetaInfo.Data);
                        }

                        // チケット
                        //TODO: ここで RightsId を再計算するのではなく、fileSystemInfo に持っている計算済みの値を利用した方がよい
                        string rightsIdText = TicketUtility.CreateRightsIdText(contentMetaId, keyGenerationForTicket.HasValue ? keyGenerationForTicket.Value : (byte)0);
                        {
                            var ticket = new Ticket();
                            ticket.PublishTicket(contentMetaId, fileSystemInfo.IsProdEncryption, keyGenerationForTicket.HasValue ? keyGenerationForTicket.Value : (byte)0, config); // TODO: 外部鍵対象コンテンツがなかった場合の扱い
                            var ticketSource = new TicketSource(ticket.Data, ticket.Length);

                            currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, rightsIdText + ".tik", ticketSource, currentOffset, new ContentHashSource(null, ".tik"));
                        }

                        // 証明書
                        {
                            var ticketCertificateSource = new TicketCertificateSource(fileSystemInfo.IsProdEncryption, config);
                            currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, rightsIdText + ".cert", ticketCertificateSource, currentOffset, new ContentHashSource(null, ".cert"));
                        }
                    }
                }
            }

            // カード情報の XML 出力
            if (CheckNeedCardSpec(fileSystemInfo))
            {
                long additionalSize = 0;
                List<ContentMetaLiteModel> additionalContentMetaList = null;
                NintendoSubmissionPackageArchiveUtils.GetAdditionalInfoForCardSizeCalculation(out additionalSize, out additionalContentMetaList, fileSystemInfo);
                ISource cardSpecXmlSource = new CardSpecXmlSource(fileSystemInfo, config, additionalContentMetaList, additionalSize);
                currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "cardspec.xml", cardSpecXmlSource, currentOffset, new ContentHashSource(null, ".xml"));
            }

            // OnCardAddOnContent 用情報の XML 出力
            if (fileSystemInfo.OnCardAddOnContentInfo != null)
            {
                ISource onCardAocInfoXmlSource = new OnCardAddOnContentInfoXmlSource(fileSystemInfo.OnCardAddOnContentInfo);
                currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "oncardaoc.xml", onCardAocInfoXmlSource, currentOffset, new ContentHashSource(null, ".xml"));
            }

            // MultiApplicationCard 用情報の XML 出力
            if (fileSystemInfo.MultiApplicationCardInfo != null)
            {
                ISource multiApplicationCardInfoXmlSource = new MultiApplicationCardInfoXmlSource(fileSystemInfo.MultiApplicationCardInfo);
                currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "multiapplicationcard.xml", multiApplicationCardInfoXmlSource, currentOffset, new ContentHashSource(null, ".xml"));
            }

            // ツール情報の XML 出力
            if (!fileSystemInfo.IsProdEncryption ||
                (fileSystemInfo.IsProdEncryption && fileSystemInfo.AuthoringToolInfo != null))
            {
                ISource authoringToolInfoXmlSource = fileSystemInfo.IsProdEncryption ? new AuthoringToolInfoXmlSource(fileSystemInfo.AuthoringToolInfo) : new AuthoringToolInfoXmlSource();
                currentOffset = RegisterRootEntry(outSink, ref partFsInfo, headerSize, ref hashSources, "authoringtoolinfo.xml", authoringToolInfoXmlSource, currentOffset, new ContentHashSource(null, ".xml"));
            }

            ISource headerSource;
            headerSource = new HashNameEntryPartitionFsHeaderSource<PartitionFileSystemMeta>(hashSources, partFsInfo, headerSize);

            // 出力用 Sink と接続
            ISink headerSink = new SubSink(outSink, 0, headerSource.Size);
            ConnectionList.Add(new Connection(headerSource, headerSink));

            outSink.SetSize(currentOffset);
        }
        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }
}
