﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    [XmlRoot("ContentMetaProperties")]
    public class NintendoContentMetaPropertyList
    {
        [XmlElement("Property")]
        public NintendoContentMetaProperty[] Properties { get; set; }
    }

    public class NintendoContentMetaProperty
    {
        public class SizeInfo
        {
            [XmlElement("DownLoad")]
            public long DownLoad { get; set; }

            [XmlElement("MinimumSaveData")]
            public long? MinimumSaveData { get; set; }
            [XmlIgnore]
            public bool MinimumSaveDataSpecified { get { return MinimumSaveData != null; } }

            [XmlElement("EachUserAccountSaveData")]
            public long? EachUserAccountSaveData { get; set; }
            [XmlIgnore]
            public bool EachUserAccountSaveDataSpecified { get { return EachUserAccountSaveData != null; } }

            [XmlElement("ApplicationAreaUsedOnCard")]
            public long? ApplicationAreaUsedOnCard { get; set; }
            [XmlIgnore]
            public bool ApplicationAreaUsedOnCardSpecified { get { return ApplicationAreaUsedOnCard != null; } }

            [XmlElement("ApplicationAreaAvailableOnCard")]
            public long? ApplicationAreaAvailableOnCard { get; set; }
            [XmlIgnore]
            public bool ApplicationAreaAvailableOnCardSpecified { get { return ApplicationAreaAvailableOnCard != null; } }
        }

        [XmlElement("Id")]
        public String Id { get; set; }
        [XmlElement("Type")]
        public String Type { get; set; }
        [XmlElement("Size")]
        public SizeInfo Size { get; set; }

        // exFAT フォーマット SD カードのクラスターサイズの最大値
        private const long ContentStorageUnitSizeMax = 512 * 1024;

        // コンテンツ新規配置時のファイルシステムメタ領域の容量消費の最大値
        private const long ContentStorageFileCreationOverhead = 2 * ContentStorageUnitSizeMax;

        private const long UserAccountSaveDataMetaMargin = 1 * 1024 * 1024;

        private const long SaveDataStorageFileCreationOverhead = 16 * 1024;

        static private long RoundupMiB(long value)
        {
            long alignment = 1024 * 1024;
            return (value + (alignment - 1)) & ~(alignment - 1);
        }

        static private long GetDownloadSize(ContentMetaModel model)
        {
            long downLoadSize = 0;
            foreach (var content in model.ContentList)
            {
                /*
                  ダウンロードサイズ =
                      コンテンツ (nca) のサイズをコンテンツが置かれるストレージの最大ブロックサイズでアラインアップした値の合計 +
                      コンテンツ数 × ファイル新規作成時の容量オーバーヘッド;
                 */
                downLoadSize += (content.Size + (ContentStorageUnitSizeMax - 1)) & ~(ContentStorageUnitSizeMax - 1) ;
                downLoadSize += ContentStorageFileCreationOverhead;
            }
            return RoundupMiB(downLoadSize);
        }

        // Item1: MinimumSaveData (ユーザアカウント数が 1 のとき), Item2: EachUserAccountSaveData（アカウント 1 つ当たりの増加量）
        static private Tuple<long, long> GetSaveDataSize(NintendoSubmissionPackageFileSystemInfo.ApplicationControlPropertyInfo acpInfo)
        {
            /*
              （遊ぶのに必要な）セーブデータサイズ =
                  アカウントセーブ * ユーザーアカウント数 + 本体セーブ + BCAT セーブ;
               各セーブデータサイズは本体保存メモリー上のサイズ
               アカウントセーブデータには、アカウントセーブデータメタサイズとして 1 MB 分マージンをとる
               他のセーブデータはファイル新規作成時の容量オーバーヘッド分マージンをとる
             */
            var accountSaveDataAvailableSize = Convert.ToInt64(acpInfo.Model.UserAccountSaveDataSize, 16);
            var accountSaveDataJournalSize = Convert.ToInt64(acpInfo.Model.UserAccountSaveDataJournalSize, 16);
            var deviceSaveDataAvailableSize = Convert.ToInt64(acpInfo.Model.DeviceSaveDataSize, 16);
            var deviceSaveDataJournalSize = Convert.ToInt64(acpInfo.Model.DeviceSaveDataJournalSize, 16);
            var bcatDeliveryCacheStorageAvailableSize = Convert.ToInt64(acpInfo.Model.BcatDeliveryCacheStorageSize, 16);

            long accountSaveDataSize = 0;
            if (accountSaveDataAvailableSize != 0 &&  accountSaveDataJournalSize != 0)
            {
                accountSaveDataSize = SaveDataUtils.QuerySaveDataTotalSize(accountSaveDataAvailableSize, accountSaveDataJournalSize) + UserAccountSaveDataMetaMargin;
            }
            long deviceSaveDataSize = 0;
            if (deviceSaveDataAvailableSize != 0 &&  deviceSaveDataJournalSize != 0)
            {
                deviceSaveDataSize = SaveDataUtils.QuerySaveDataTotalSize(deviceSaveDataAvailableSize, deviceSaveDataJournalSize) + SaveDataStorageFileCreationOverhead;
            }
            long bcatDeliveryCacheStorageSize = 0;
            if (bcatDeliveryCacheStorageAvailableSize != 0)
            {
                const long BcatDeliveryCacheStorageJournalSize = 2 * 1024 * 1024;
                bcatDeliveryCacheStorageSize = SaveDataUtils.QuerySaveDataTotalSize(bcatDeliveryCacheStorageAvailableSize, BcatDeliveryCacheStorageJournalSize) + SaveDataStorageFileCreationOverhead;
            }
            return Tuple.Create(RoundupMiB(accountSaveDataSize + deviceSaveDataSize + bcatDeliveryCacheStorageSize), RoundupMiB(accountSaveDataSize));
        }

        static private long GetApplicationAreaUsedOnCardSize(NintendoSubmissionPackageFileSystemInfo.EntryInfo entryInfo, KeyConfiguration keyConfig)
        {
            var nspInfo = new NintendoSubmissionPackageFileSystemInfo();
            nspInfo.Entries.Add(entryInfo); // entry 単体のカードサイズ

            long additionalSize = 0;
            List<ContentMetaLiteModel> additionalContentMetaList = null;
            NintendoSubmissionPackageArchiveUtils.GetAdditionalInfoForCardSizeCalculation(out additionalSize, out additionalContentMetaList, nspInfo);

            var xciSizeInfo = ProdEncryptedXciArchive.CalculateXciSize(nspInfo, additionalContentMetaList, false, keyConfig);
            return xciSizeInfo.TotalSize + additionalSize - (XciInfo.UpdatePartitionLimitSize + XciInfo.HeaderReservedSize);
        }

        static private long? GetApplicationAreaAvailableOnCardSize(int cardSize, NintendoSubmissionPackageFileSystemInfo.EntryInfo entryInfo)
        {
            if (cardSize == XciInfo.InvalidRomSize)
            {
                return null;
            }
            XciUtils.CheckRomSize(cardSize);
            // TODO: CardSpec.cs と共有
            return XciUtils.GetAvailableAreaSize(cardSize) - (XciInfo.UpdatePartitionLimitSize + XciInfo.HeaderReservedSize);
        }

        static internal List<NintendoContentMetaProperty> GetListImpl(NintendoSubmissionPackageReader nspReader, KeyConfiguration keyConfig)
        {
            var properties = new List<NintendoContentMetaProperty>();
            var nspInfo = ArchiveReconstructionUtils.GetNspInfo(nspReader, keyConfig);
            foreach (var entry in nspInfo.Entries)
            {
                var property = new NintendoContentMetaProperty();
                var model = entry.ContentMetaInfo.Model;
                property.Id = model.Id;
                property.Type = model.Type;
                var sizeInfo = new SizeInfo();
                {
                    sizeInfo.DownLoad = GetDownloadSize(model);
                    if (model.Type == NintendoContentMetaConstant.ContentMetaTypeApplication || model.Type == NintendoContentMetaConstant.ContentMetaTypePatch)
                    {
                        var saveDataSize = GetSaveDataSize(entry.ApplicationControlPropertyInfo);
                        sizeInfo.MinimumSaveData = saveDataSize.Item1;
                        sizeInfo.EachUserAccountSaveData = saveDataSize.Item2;

                        sizeInfo.ApplicationAreaUsedOnCard = GetApplicationAreaUsedOnCardSize(entry, keyConfig);
                        sizeInfo.ApplicationAreaAvailableOnCard = GetApplicationAreaAvailableOnCardSize(nspInfo.CardSize, entry);
                    }
                    else if (nspInfo.OnCardAddOnContentInfo != null)
                    {
                        sizeInfo.ApplicationAreaUsedOnCard = GetApplicationAreaUsedOnCardSize(entry, keyConfig);
                    }
                }
                property.Size = sizeInfo;
                properties.Add(property);
            }
            return properties;
        }

        static public NintendoContentMetaPropertyList GetList(NintendoSubmissionPackageReader nspReader, KeyConfiguration keyConfig)
        {
            var properties = GetListImpl(nspReader, keyConfig);
            var output = new NintendoContentMetaPropertyList();
            output.Properties = properties.ToArray();
            return output;
        }
    }

    [XmlRoot("ContentMetaProperties")]
    public class NintendoContentMetaPropertyInternal // TODO: 統合
    {
        [XmlElement("ApplyDifference")]
        public bool? ApplyDifference { get; set; }
        [XmlIgnore]
        public bool ApplyDifferenceSpecified { get { return ApplyDifference != null; } }
        [XmlElement("ApplyDifferenceBefore3NUP")]
        public bool? ApplyDifferenceBefore3NUP { get; set; }
        [XmlIgnore]
        public bool ApplyDifferenceBefore3NUPSpecified { get { return ApplyDifferenceBefore3NUP != null; } }

        static public NintendoContentMetaPropertyInternal Get(NintendoSubmissionPackageReader nspReader)
        {
            var models = ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader);
            var property = new NintendoContentMetaPropertyInternal();
            property.ApplyDifference = ResultModel.IsApplyDifference(models, false);
            property.ApplyDifferenceBefore3NUP = ResultModel.IsApplyDifference(models, true);
            return property;
        }
    }
}
