﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    internal class MultiApplicationCardIndicatorNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        public MultiApplicationCardIndicatorNintendoSubmissionPackageArchive(IReadableSink outSink, List<NintendoSubmissionPackageReader> nspReaders, ulong id, int? cardSize, int? cardClockRate, byte? cardLaunchFlags, KeyConfiguration keyConfig)
        {
            var nspInfo = new NintendoSubmissionPackageFileSystemInfo();
            {
                // 構成情報
                nspInfo.MultiApplicationCardInfo = new MultiApplicationCardInfoModel();
                {
                    if (nspReaders.Any(x => ArchiveReconstructionUtils.ReadContentMetaInNsp(x).Any(y => y.Type != NintendoContentMetaConstant.ContentMetaTypeApplication && y.Type != NintendoContentMetaConstant.ContentMetaTypePatch)))
                    {
                        throw new ArgumentException("Only Application or Patch can be bundled.");
                    }

                    var applicationContentMetas = nspReaders.SelectMany(x => ArchiveReconstructionUtils.ReadContentMetaInNsp(x).Where(y => y.Type == NintendoContentMetaConstant.ContentMetaTypeApplication));
                    nspInfo.MultiApplicationCardInfo.SetApplication(applicationContentMetas.ToList());

                    var patchContentMetas = nspReaders.SelectMany(x => ArchiveReconstructionUtils.ReadContentMetaInNsp(x).Where(y => y.Type == NintendoContentMetaConstant.ContentMetaTypePatch).Select(y => y as PatchContentMetaModel));
                    if (patchContentMetas.Count() != 0)
                    {
                        nspInfo.MultiApplicationCardInfo.SetPatch(patchContentMetas.ToList());
                    }

                    nspInfo.MultiApplicationCardInfo.SetDigest();
                    nspInfo.MultiApplicationCardInfo.SetId(id);
                }

                // カード設定の外部入力
                {
                    nspInfo.CardSize = cardSize ?? 0;
                    nspInfo.CardClockRate = cardClockRate ?? 0;
                    nspInfo.CardLaunchFlags = cardLaunchFlags;
                }
            }

            var nsp = new NintendoSubmissionPackageArchive(outSink, nspInfo, keyConfig);
            ConnectionList = new List<Connection>(nsp.ConnectionList);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }

    [XmlRoot("MultiApplicationCard", IsNullable = false)]
    public class MultiApplicationCardInfoModel
    {
        [XmlElement("Application")]
        public MultiApplicationCardInfoContentMetaListModel Application { get; set; }

        [XmlElement("Patch")]
        public MultiApplicationCardInfoContentMetaListModel Patch { get; set; }
        [XmlIgnore]
        public bool PatchSpecified { get { return Patch != null; } }

        [XmlElement("Digest")]
        public string Digest { get; set; }

        [XmlElement("Id")]
        public string Id { get; set; }

        [XmlElement("ApplicationCount")]
        public int ApplicationCount { get; set; }

        public void SetApplication(List<ContentMetaModel> contentMetaList)
        {
            Application = GetContentMetaList(contentMetaList);
            // 重複した ID があるとエラー
            {
                var distinctCount = Application.ContentMetaList.Select(x => x.Id).Distinct().Count();
                if (distinctCount != Application.ContentMetaList.Count)
                {
                    throw new ArgumentException("Application IDs in multi-application card cannot be duplicated.");
                }
            }

            // Application の個数を数える
            ApplicationCount = Application.ContentMetaList.Count();
        }

        public void SetPatch(List<PatchContentMetaModel> patchContentMetaList)
        {
            // 対応するアプリケーションが含まれていないとエラー
            foreach (var patchContentMeta in patchContentMetaList)
            {
                var appContentMeta = patchContentMeta.HistoryList.Where(y => y.Type == NintendoContentMetaConstant.ContentMetaTypeApplication).Single();
                if (!Application.ContentMetaList.Any(x => x.Id == appContentMeta.Id))
                {
                    throw new ArgumentException(string.Format("There is no Application corresponds to Patch {0}.", patchContentMeta.Id));
                }
            }
            Patch = GetContentMetaList(patchContentMetaList.Select(x => x as ContentMetaModel).ToList());
            // 重複した ID があるとエラー
            {
                var distinctCount = Patch.ContentMetaList.Select(x => x.Id).Distinct().Count();
                if (distinctCount != Patch.ContentMetaList.Count)
                {
                    throw new ArgumentException("Patch IDs in multi-application card cannot be duplicated.");
                }
            }
       }

        public void SetDigest()
        {
            // Application と Patch の Digest を連結したストリームのハッシュを Digest とする
            var bytes = Application.ContentMetaList.SelectMany(x => GetStringBytes(x.Digest, 32)).ToArray();
            if (Patch != null)
            {
                bytes.Concat(Patch.ContentMetaList.SelectMany(x => GetStringBytes(x.Digest, 32)).ToArray());
            }
            var hashCalculator = new SHA256CryptoServiceProvider();
            var hash = hashCalculator.ComputeHash(bytes, 0, bytes.Length);
            Digest = BitConverter.ToString(hash, 0, hash.Length).Replace("-", string.Empty).ToLower();
        }

        public void SetId(ulong id)
        {
            // どの ContentMetaId とも被らないこと
            if (Application.ContentMetaList.Any(x => Convert.ToUInt64(x.Id, 16) == id) ||
                (Patch != null ? Patch.ContentMetaList.Any(x => Convert.ToUInt64(x.Id, 16) == id) : false))
            {
                throw new ArgumentException("ID for multi-application card must be unique.");
            }
            Id = "0x" + id.ToString("x16");
        }

        public byte[] GetBytes()
        {
            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(String.Empty, String.Empty);

            using (var memoryStream = new MemoryStream())
            {
                var sw = new StreamWriter(memoryStream, Encoding.UTF8);
                var serializer = new XmlSerializer(typeof(MultiApplicationCardInfoModel));
                serializer.Serialize(sw, this, nameSpace);
                return memoryStream.ToArray();
            }
        }

        public byte[] GetDigestBytes()
        {
            return GetStringBytes(Digest, 32);
        }

        private byte[] GetStringBytes(string str, int size)
        {
            Trace.Assert(str.Length == size * 2);
            var bytes = new byte[size];
            int j = 0;
            for (int i = 0; i < bytes.Length; i++)
            {
                bytes[i] = Convert.ToByte(str.Substring(j, 2), 16);
                j += 2;
            }
            return bytes;
        }

        private MultiApplicationCardInfoContentMetaListModel GetContentMetaList(List<ContentMetaModel> refContentMetaList)
        {
            var contentMetaList = new MultiApplicationCardInfoContentMetaListModel();
            contentMetaList.ContentMetaList = new List<MultiApplicationCardInfoContentMetaModel>();
            foreach (var refContentMeta in refContentMetaList)
            {
                var contentMeta = new MultiApplicationCardInfoContentMetaModel();
                contentMeta.Type = refContentMeta.Type;
                contentMeta.Id = refContentMeta.Id;
                contentMeta.Version = refContentMeta.Version.Value;
                contentMeta.Digest = refContentMeta.Digest;
                contentMeta.ContentList = refContentMeta.ContentList;
                switch (refContentMeta.Type)
                {
                    case NintendoContentMetaConstant.ContentMetaTypeApplication:
                    case NintendoContentMetaConstant.ContentMetaTypePatch:
                        break;
                    default:
                        throw new ArgumentException(string.Format("Not supported type {0}", contentMeta.Type));
                }
                contentMetaList.ContentMetaList.Add(contentMeta);
            }
            return contentMetaList;
        }

        public void VerifyAndReorder(out List<NintendoSubmissionPackageReader> outAppNspReaders, out List<NintendoSubmissionPackageReader> outPatchNspReaders, List<NintendoSubmissionPackageReader> inputNspReaders)
        {
            var appNspReaders = inputNspReaders.Select(x => Tuple.Create(x, ArchiveReconstructionUtils.ReadContentMetaInNsp(x).Single())).Where(x => x.Item2.Type == NintendoContentMetaConstant.ContentMetaTypeApplication);
            var patchNspReaders = inputNspReaders.Select(x => Tuple.Create(x, ArchiveReconstructionUtils.ReadContentMetaInNsp(x).Single())).Where(x => x.Item2.Type == NintendoContentMetaConstant.ContentMetaTypePatch);

            Action<bool, string> throwUnless = (condition, suffix) =>
            {
                if (!condition)
                {
                    throw new ArgumentException(string.Format("There is no consistency between multi-application card indicator and input nsp files. {0}", suffix));
                }
            };

            throwUnless(appNspReaders.Count() == Application.ContentMetaList.Count, "Application count mismatch.");
            foreach (var appDigest in appNspReaders.Select(x => x.Item2.Digest))
            {
                throwUnless(Application.ContentMetaList.Any(x => x.Digest == appDigest), "Application digest mismatch.");
            }

            outAppNspReaders = new List<NintendoSubmissionPackageReader>();
            foreach (var appId in Application.ContentMetaList.Select(x => x.Id))
            {
                outAppNspReaders.Add(appNspReaders.Single(x => x.Item2.Id == appId).Item1);
            }

            outPatchNspReaders = new List<NintendoSubmissionPackageReader>();
            if (Patch != null)
            {
                throwUnless(patchNspReaders.Count() == Patch.ContentMetaList.Count, "Patch count mismatch.");
                foreach (var patchDigest in patchNspReaders.Select(x => x.Item2.Digest))
                {
                    throwUnless(Patch.ContentMetaList.Any(x => x.Digest == patchDigest), "Patch digest mismatch.");
                }

                foreach (var patchId in Patch.ContentMetaList.Select(x => x.Id))
                {
                    outPatchNspReaders.Add(patchNspReaders.Single(x => x.Item2.Id == patchId).Item1);
                }
            }
        }
    }

    public class MultiApplicationCardInfoContentMetaListModel
    {
        [XmlElement("ContentMeta")]
        public List<MultiApplicationCardInfoContentMetaModel> ContentMetaList { get; set; }
    }

    public class MultiApplicationCardInfoContentMetaModel : ContentMetaLiteModel
    {
        override public bool ContentListSpecified { get { return false; } }

        [XmlElement("Tag")]
        public string Tag { get; set; }
        [XmlIgnore]
        public bool TagSpecified { get { return Tag != null; } }
    }

    public class MultiApplicationCardInfoXmlSource : ISource
    {
        public long Size { get; set; }
        private ISource m_source;

        public MultiApplicationCardInfoXmlSource(MultiApplicationCardInfoModel model)
        {
            var bytes = model.GetBytes();
            m_source = new MemorySource(bytes, 0, bytes.Length);
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }
}
