﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml.Serialization;
using YamlDotNet.RepresentationModel;
using Nintendo.Authoring.FileSystemMetaLibrary;
using Nintendo.Authoring.CryptoLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public enum EntryReplaceAction
    {
        Replace,
        Add,
        Delete,
    };

    public class EntryReplaceRule
    {
        public StreamSource Source { get; set; }
        public string Path { get; set; }
        public EntryReplaceAction Action { get; set; }
    }

    internal class ModifiableArchiveUtil
    {
        private static readonly string AddEntryPrefix = "add:";
        private static readonly string DelEntryPrefix = "del:";

        public static readonly string AddIconInReplaceTargetKeyword = "null.icon";
        public static readonly string AddNcaInReplaceTargetKeyword = "null.nca";
        public static readonly char[] AddRootEntryInReplaceTargetDelimiter = {'@'};

        public static List<EntryReplaceRule> ConvertToReplaceRule(List<StreamSource> inSourceList, List<string> targetEntryPathList)
        {
            if (inSourceList.Count != targetEntryPathList.Count)
            {
                throw new InvalidOperationException(string.Format("number of replace rule does not match. ({0} vs. {1})", inSourceList.Count, targetEntryPathList.Count));
            }

            var replaceRuleList = new List<EntryReplaceRule>();
            for (int i = 0; i<inSourceList.Count; i++)
            {
                var replaceRule = new EntryReplaceRule();
                replaceRule.Source = inSourceList[i];
                replaceRule.Path = targetEntryPathList[i];
                replaceRule.Action = EntryReplaceAction.Replace;
                // add 判定
                if (replaceRule.Path.StartsWith(AddEntryPrefix))
                {
                    replaceRule.Path = replaceRule.Path.Substring(AddEntryPrefix.Length);
                    replaceRule.Action = EntryReplaceAction.Add;
                }
                // delete 判定
                else if (replaceRule.Path.StartsWith(DelEntryPrefix))
                {
                    replaceRule.Path = replaceRule.Path.Substring(DelEntryPrefix.Length);
                    replaceRule.Action = EntryReplaceAction.Delete;
                }
                replaceRuleList.Add(replaceRule);
            }

            // パスの重複チェック
            var replaceTargetEntry = replaceRuleList.Select(rule => rule.Path).ToList();
            var noDupReplaceTargetEntry = new HashSet<string>(replaceTargetEntry);
            if (replaceTargetEntry.Count() != noDupReplaceTargetEntry.Count)
            {
                foreach(var entry in noDupReplaceTargetEntry)
                {
                    replaceTargetEntry.Remove(entry);
                }
                string message = "";
                foreach(var entry in replaceTargetEntry)
                {
                    message += string.Format("'{0}'\n", entry);
                }

                throw new InvalidOperationException("multiple actions are specified for one entry.\n" + message);
            }

            return replaceRuleList;
        }

        public static void CheckReplaceCount(int expectedCount, int actualCount)
        {
            if (actualCount == 0)
            {
                throw new Exception("nothing was replaced.");
            }
            else if (actualCount != expectedCount)
            {
                throw new Exception(string.Format("entry to be replaced not found (replaced {0} entries while {1} rules are specified).", actualCount, expectedCount));
            }
        }

        private class BasicFileInfo
        {
            public string Name { get; private set; }
            public long Size { get; set; }
            public long Offset { get; set; }
            public SourceInterface Source { get; set; }

            public BasicFileInfo(Tuple<string, long> fileInfo, IFileSystemArchiveReader fsReader)
            {
                Name = fileInfo.Item1;
                Size = fileInfo.Item2;
                var fileFragment = fsReader.GetFileFragmentList(Name).First();
                Offset = fileFragment.Item1;
                Source = null;
            }

            public BasicFileInfo(string name, long size, long offset, SourceInterface source)
            {
                Name = name;
                Size = size;
                Offset = offset;
                Source = source;
            }
        }

        delegate void AddEntryToFsInfo(IEnumerable<BasicFileInfo> fileInfoList);
        static private void GetFsInfo(IFileSystemArchiveReader fsReader, List<EntryReplaceRule> replaceRuleList, AddEntryToFsInfo addEntryDelegate, ref int replaceCount)
        {
            // ファイルのアライメント調整
            Func<long, long> GetRomFsAlignValue = value => BitUtility.AlignUp(value, RomFsAdfWriter.AlignmentSize);
            Func<long, long> GetPartitionFsAlignValue = value => value;
            var GetEntryAlignValue = (fsReader is RomFsFileSystemArchiveReader) ? GetRomFsAlignValue : GetPartitionFsAlignValue;

            // エントリ情報の取得
            var sortedFileInfoList = new List<Tuple<long, Tuple<string, long>>>();
            foreach (var fileInfo in fsReader.ListFileInfo())
            {
                var basicFileInfo = new BasicFileInfo(fileInfo, fsReader);  // メモリを食いそうなので必要なもの以外は読み捨てる
                sortedFileInfoList.Add(new Tuple<long, Tuple<string, long>>(basicFileInfo.Offset, fileInfo));
            }

            // オフセット計算の為にソートしておく
            sortedFileInfoList.Sort((leftA, rightB) => leftA.Item1.CompareTo(rightB.Item1));

            var fileInfoList = new List<BasicFileInfo>();

            // エントリ追加関数の定義
            Func<List<EntryReplaceRule>, Func<long>, Action<long>, int> AddRuleEntries;
            AddRuleEntries = (entryInsertRule, getInsertionOffset, updateOffsetAction) =>
            {
                int localReplaceCount = 0;
                // 複数エントリが挿入されるときの事を考えてソート
                entryInsertRule.Sort((a, b) => string.CompareOrdinal(a.Path, b.Path));
                foreach (var insertRule in entryInsertRule)
                {
                    long insertedSize = insertRule.Source.Size;
                    var insertFileInfo = new BasicFileInfo(insertRule.Path, insertedSize, getInsertionOffset(), new CliCompatibleSource(insertRule.Source));
                    fileInfoList.Add(insertFileInfo);
                    localReplaceCount++;
                    updateOffsetAction(insertedSize);
                }
                return localReplaceCount;
            };

            string lastEntryName = string.Empty;
            long entryOffset = (0 < sortedFileInfoList.Count) ? sortedFileInfoList[0].Item1 : 0;

            // エントリ情報の修正
            foreach (var fileInfoTuple in sortedFileInfoList)
            {
                var fileInfo = fileInfoTuple.Item2;
                var basicFileInfo = new BasicFileInfo(fileInfo, fsReader);

                var replaceRule = replaceRuleList.Find(rule => rule.Path == basicFileInfo.Name);
                // Source のセット
                if (replaceRule != null)
                {
                    replaceCount++;

                    // エントリ削除の場合はスキップ
                    if (replaceRule.Action == EntryReplaceAction.Delete)
                    {
                        continue;
                    }

                    if (replaceRule.Action != EntryReplaceAction.Replace)
                    {
                        throw new InvalidOperationException("detected unknown replace action.");
                    }

                    basicFileInfo.Size = replaceRule.Source.Size;
                    basicFileInfo.Source = new CliCompatibleSource(replaceRule.Source);
                }
                else
                {
                    basicFileInfo.Source = new CliCompatibleSource(new FileSystemArchvieFileSource(fsReader, basicFileInfo.Name));
                }

                // エントリ追加の場合（挿入場所かどうかチェック）
                var entryInsertRule = replaceRuleList.Where(
                    rule => (rule.Action == EntryReplaceAction.Add) &&
                        string.CompareOrdinal(lastEntryName, rule.Path) < 0 &&
                        string.CompareOrdinal(rule.Path, basicFileInfo.Name) < 0).ToList();

                replaceCount += AddRuleEntries(
                    entryInsertRule,
                    () => GetEntryAlignValue(entryOffset),
                    insertedSize => entryOffset = GetEntryAlignValue(entryOffset) + insertedSize
                );

                // オフセットの修正（mod 512 が揃うように調整 @ SIGLO-52978）
                const int AdjustOffset = 512;
                {
                    var stride = (basicFileInfo.Offset % AdjustOffset) - (entryOffset % AdjustOffset);
                    if (0 <= stride)
                    {
                        entryOffset += stride;
                    }
                    else
                    {
                        entryOffset += AdjustOffset + stride;
                    }
                    Debug.Assert((basicFileInfo.Offset % AdjustOffset) == (entryOffset % AdjustOffset));

                    basicFileInfo.Offset = entryOffset;
                }

                fileInfoList.Add(basicFileInfo);
                lastEntryName = basicFileInfo.Name;
                entryOffset += basicFileInfo.Size;
            }

            entryOffset = GetEntryAlignValue(entryOffset);

            // エントリの末尾追加（あれば）
            var entryAppendRule = replaceRuleList.Where(
                rule => (rule.Action == EntryReplaceAction.Add) && string.CompareOrdinal(lastEntryName, rule.Path) < 0).ToList();

            replaceCount += AddRuleEntries(
                entryAppendRule,
                () => entryOffset,
                insertedSize => entryOffset += GetEntryAlignValue(insertedSize)
            );

            addEntryDelegate(fileInfoList);
        }

        static private PartitionFileSystemInfo GetPartitionFsInfo(IFileSystemArchiveReader fsReader, List<EntryReplaceRule> replaceRuleList, ref int replaceCount)
        {
            var fsInfo = new PartitionFileSystemInfo();
            AddEntryToFsInfo AddEntryToPartitionFsInfo = delegate (IEnumerable<BasicFileInfo> fileInfoList)
            {
                foreach (var basicFileInfo in fileInfoList)
                {
                    var entry = new PartitionFileSystemInfo.EntryInfo();
                    entry.type = "source";
                    entry.name = basicFileInfo.Name;
                    entry.size = (ulong)basicFileInfo.Size;
                    entry.offset = (ulong)basicFileInfo.Offset;
                    entry.path = null;
                    entry.sourceInterface = basicFileInfo.Source;
                    fsInfo.entries.Add(entry);
                }
            };
            GetFsInfo(fsReader, replaceRuleList, AddEntryToPartitionFsInfo, ref replaceCount);
            return fsInfo;
        }

        static private RomFsFileSystemInfo GetRomFsInfo(IFileSystemArchiveReader fsReader, List<EntryReplaceRule> replaceRuleList, ref int replaceCount)
        {
            var fsInfo = new RomFsFileSystemInfo();
            var directories = new DirectoryList();
            AddEntryToFsInfo AddEntryToRomFsInfo = delegate (IEnumerable<BasicFileInfo> fileInfoList) {
                foreach (var basicFileInfo in fileInfoList)
                {
                    var entry = new RomFsFileSystemInfo.EntryInfo();
                    entry.type = "source";
                    entry.name = "/" + basicFileInfo.Name;
                    entry.size = (ulong)basicFileInfo.Size;
                    entry.offset = (ulong)basicFileInfo.Offset;
                    entry.path = null;
                    entry.sourceInterface = basicFileInfo.Source;

                    fsInfo.AddFileEntry(entry);

                    // 初出のディレクトリをカウント
                    directories.AddAncestors(entry.DirectoryOfEntryName);
                }
            };
            GetFsInfo(fsReader, replaceRuleList, AddEntryToRomFsInfo, ref replaceCount);
            directories.CalculateFileSystemDirectoryInfo(fsInfo);
            return fsInfo;
        }

        static private NintendoContentFileSystemInfo.EntryInfo GetReplacedFsEntry(NintendoContentArchiveReader ncaReader, int fsIndex, List<EntryReplaceRule> replaceRuleList, ref int replaceCount)
        {
            System.Diagnostics.Debug.Assert(replaceRuleList.Count > 0, "replaceRule for GetFsEntry must not be null.");

            NintendoContentArchiveFsHeaderInfo fsHeaderInfo = null;
            var fsReader = ncaReader.OpenFileSystemArchiveReader(fsIndex, ref fsHeaderInfo);

            // 世代番号は差し替え前のものに合わせておく
            var fsEntry = ArchiveReconstructionUtils.GetCommonFsEntry(fsHeaderInfo, fsIndex);

            bool isSizeChanged = replaceRuleList.Where(rule => rule.Action != EntryReplaceAction.Replace).Any();
            if(! isSizeChanged)
            {
                foreach (var replaceRule in replaceRuleList)
                {
                    var fileFragment = fsReader.GetFileFragmentList(replaceRule.Path).First();
                    var targetFileOffset = fileFragment.Item1;
                    var targetFileSize = fileFragment.Item2;
                    isSizeChanged |= (targetFileSize != replaceRule.Source.Size);
                }
            }

            if (! isSizeChanged)
            {
                // FS の body 部の解析が不要な場合
                var fsSource = new FileSystemArchvieBaseSource(fsReader);
                var fsAdaptedSourceList = new List<Tuple<ISource, long, long>>();

                foreach (var replaceRule in replaceRuleList)
                {
                    var fileFragment = fsReader.GetFileFragmentList(replaceRule.Path).First();
                    var targetFileOffset = fileFragment.Item1;
                    var targetSourceSet = new Tuple<ISource, long, long>(replaceRule.Source, targetFileOffset, replaceRule.Source.Size);
                    fsAdaptedSourceList.Add(targetSourceSet);
                    replaceCount++;
                }
                fsEntry.sourceInterface = new CliCompatibleSource(new AdaptedSource(fsSource, fsAdaptedSourceList));
            }
            else
            {
                // FS 内部のエントリ単位の解析が必要な場合（サイズが変わった場合）
                if (fsEntry.formatType == NintendoContentFileSystemMetaConstant.FormatTypeRomFs)
                {
                    var fsInfo = GetRomFsInfo(fsReader, replaceRuleList, ref replaceCount);
                    var fsSource = new RomFsArchiveSource(fsInfo);
                    fsEntry.sourceInterface = new CliCompatibleSource(fsSource);
                }
                else if (fsEntry.formatType == NintendoContentFileSystemMetaConstant.FormatTypePartitionFs)
                {
                    var fsInfo = GetPartitionFsInfo(fsReader, replaceRuleList, ref replaceCount);
                    var fsSource = new PartitionFsArchiveSource(fsInfo);
                    fsEntry.sourceInterface = new CliCompatibleSource(fsSource);
                }
            }

            return fsEntry;
        }

        static internal NintendoContentFileSystemInfo GetReplacedNcaInfo(NintendoContentArchiveReader ncaReader, string descFilePath, List<EntryReplaceRule> replaceRuleList, ref int replaceCount, UInt64 contentMetaId)
        {
            var ncaInfo = ArchiveReconstructionUtils.GetCommonNcaInfo(ncaReader, contentMetaId);
            NintendoContentAdfReader.RetrieveInfoFromDesc(ref ncaInfo, descFilePath);

            // Program コンテントの差替えは desc 必須
            if (replaceRuleList.Count > 0 && ncaInfo.contentType == (Byte)NintendoContentFileSystemMetaConstant.ContentTypeProgram)
            {
                if (descFilePath == null)
                {
                    throw new Exception("Replacing 'Program' content needs desc file.");
                }
            }

            foreach (var fsIndex in ncaInfo.existentFsIndices)
            {
                var fsPath = string.Format("fs{0}", fsIndex);
                var selectedReplaceRuleList = replaceRuleList.FindAll(rule => rule.Path.StartsWith(fsPath));
                if (selectedReplaceRuleList.Count() > 0)
                {
                    selectedReplaceRuleList = selectedReplaceRuleList.Select(rule => { rule.Path = rule.Path.Substring(fsPath.Length + 1); return rule; } ).ToList();
                    ncaInfo.fsEntries.Add(GetReplacedFsEntry(ncaReader, fsIndex, selectedReplaceRuleList, ref replaceCount));
                }
                else
                {
                    ncaInfo.fsEntries.Add(ArchiveReconstructionUtils.GetFsEntry(ncaReader, fsIndex));
                }

                // 与えられた desc に対する ACID のチェック
                if (ncaInfo.contentType == (Byte)NintendoContentFileSystemMetaConstant.ContentTypeProgram && ncaInfo.fsEntries.Last().formatType == NintendoContentFileSystemMetaConstant.FormatTypePartitionFs)
                {
                    byte[] npdmData = null;
                    var npdmRule = replaceRuleList.Where(x => x.Path == "main.npdm");

                    // main.npdm を差替える場合は差し替え後のものを見る
                    if (npdmRule.Any())
                    {
                        var npdmSource = npdmRule.Single().Source;
                        npdmData = npdmSource.PullData(0, (int)npdmSource.Size).Buffer.Array;
                    }
                    else
                    {
                        var fsReader = ncaReader.OpenFileSystemArchiveReader(fsIndex);
                        if (fsReader.ListFileInfo().Where(x => x.Item1 == "main.npdm").Any())
                        {
                            var fragmentList = fsReader.GetFileFragmentList("main.npdm");
                            var npdmSize = fragmentList.Single().Item2;
                            npdmData = fsReader.ReadFile("main.npdm", 0, npdmSize);
                        }
                    }

                    if (npdmData != null)
                    {
                        ArchiveReconstructionUtils.CheckAcid(npdmData, descFilePath);
                    }
                }
            }
            ncaInfo.GenerateExistentFsIndicesFromFsEntries();

            return ncaInfo;
        }

        // nsp のコンテンツの編集
        public static void ModifyContent(NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string descFilePath, string nroDirectoryPath, List<EntryReplaceRule> replaceRuleList, ref int replaceCount, KeyConfiguration keyConfig)
        {
            var belowRootEntryReplaceRuleList = replaceRuleList.Where(rule => ! (rule.Path.Contains("/") && rule.Path.Contains("\\"))).ToList();
            var addContentList = belowRootEntryReplaceRuleList.Where(rule => rule.Path.StartsWith(ModifiableArchiveUtil.AddNcaInReplaceTargetKeyword)).ToList();

            var dstContents = new List<NintendoSubmissionPackageFileSystemInfo.ContentInfo>();

            bool programInfoModified = false;
            bool controlInfoModified = false;
            bool htmlDocumentInfoModified = false;
            bool legalInfomationInfoModified = false;

            byte oldestKeyGeneration = NintendoContentFileSystemMetaConstant.SupportedKeyGenerationMax;

            foreach (var srcContentInfo in entry.Contents)
            {
                if (srcContentInfo.Source == null)
                {
                    // Source が未知の場合は差し換えできないのでそのまま追加
                    dstContents.Add(srcContentInfo);
                    continue;
                }

                // 差し替え元の nca の最も古い鍵世代を記録しておく
                using (var orgNcaStream = new SourceBasedStream(srcContentInfo.Source))
                using (var orgNcaReader = new NintendoContentArchiveReader(orgNcaStream, new NcaKeyGenerator(keyConfig)))
                {
                    var keyGeneration = orgNcaReader.GetKeyGeneration();
                    if (oldestKeyGeneration > keyGeneration)
                    {
                        oldestKeyGeneration = keyGeneration;
                    }
                }

                var ncaName = srcContentInfo.ContentId + ".nca";
                var selectedReplaceRuleList = replaceRuleList.Where(rule => rule.Path.StartsWith(ncaName)).ToList();
                if (selectedReplaceRuleList.Count == 0)
                {
                    dstContents.Add(srcContentInfo);
                }
                // TODO: nca の削除は非対応
                // nca 単位での差し換え
                else if (selectedReplaceRuleList.Where(rule => rule.Path == ncaName).Any())
                {
                    if (selectedReplaceRuleList.Count != 1)
                    {
                        throw new InvalidDataException("Error: attempt to replace a content and files in the content at the same time");
                    }

                    var rule = selectedReplaceRuleList.First();
                    var newContent = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();

                    if (rule.Action != EntryReplaceAction.Delete)
                    {
                        byte? newKeyGeneration;
                        // 内容のチェック
                        // 差し替え元
                        using (var orgNcaStream = new SourceBasedStream(srcContentInfo.Source))
                        using (var orgNcaReader = new NintendoContentArchiveReader(orgNcaStream, new NcaKeyGenerator(keyConfig)))
                        // 差し替え先
                        using (var newNcaReader = new NintendoContentArchiveReader(rule.Source.GetStream(), new NcaKeyGenerator(keyConfig)))
                        {
                            var orgType = orgNcaReader.GetContentType();
                            var orgProgramId = orgNcaReader.GetProgramId();
                            var orgKeyGeneration = orgNcaReader.GetKeyGeneration();
                            var orgIdOffset = orgNcaReader.GetRepresentProgramIdOffset();
                            var newType = newNcaReader.GetContentType();
                            var newProgramId = newNcaReader.GetProgramId();
                            newKeyGeneration = newNcaReader.GetKeyGeneration();
                            var newIdOffset = newNcaReader.GetRepresentProgramIdOffset();
                            if (orgType != newType)
                            {
                                throw new InvalidDataException(string.Format("Replacing nca ({0}) has different content type from target nca", ncaName));
                            }
                            if (orgProgramId != newProgramId)
                            {
                                throw new InvalidDataException(string.Format("Replacing nca ({0}) has different program id from target nca", ncaName));
                            }
                            if (orgKeyGeneration != newKeyGeneration.Value)
                            {
                                throw new InvalidDataException(string.Format("Replacing nca ({0}) has no compatibility with target nca. Please recreate the nca by tool has same version as one used for target nca.", ncaName));
                            }
                            if (orgIdOffset != newIdOffset)
                            {
                                throw new InvalidDataException(string.Format("Replacing nca ({0}) has different program index from target nca", ncaName));
                            }
                        }
                        newContent.SetSource(srcContentInfo.ContentType, srcContentInfo.IdOffset, newKeyGeneration.Value, null, rule.Source);
                        dstContents.Add(newContent);
                    }

                    if (srcContentInfo.ContentType == NintendoContentMetaConstant.ContentTypeProgram)
                    {
                        programInfoModified = true;
                    }
                    else if (srcContentInfo.ContentType == NintendoContentMetaConstant.ContentTypeControl)
                    {
                        controlInfoModified = true;
                    }
                    else if (srcContentInfo.ContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument)
                    {
                        htmlDocumentInfoModified = true;
                    }
                    else if (srcContentInfo.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation)
                    {
                        legalInfomationInfoModified = true;
                    }
                    replaceCount++;
                }
                // FS エントリ単位での差し換え
                else
                {
                    var newContent = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                    var selectedFsReplaceRuleList = selectedReplaceRuleList.Select(rule => { rule.Path = rule.Path.Substring(ncaName.Length + 1); return rule; }).ToList();
                    using (var orgNcaStream = new SourceBasedStream(srcContentInfo.Source))
                    using (var orgNcaReader = new NintendoContentArchiveReader(orgNcaStream, new NcaKeyGenerator(keyConfig)))
                    {
                        var rightsId = orgNcaReader.GetRightsId();
                        if (TicketUtility.IsValidRightsId(rightsId))
                        {
                            var hashCalculator = new HmacSha256HashCryptoDriver(HmacSha256KeyIndex.TitleKeyGenarateKey);
                            var contentMetaId = TicketUtility.GetContentMetaIdFromRightsId(rightsId);
                            orgNcaReader.SetExternalKey(ExternalContentKeyGenerator.GetNcaExternalContentKey(hashCalculator, contentMetaId, orgNcaReader.GetKeyGeneration()).Key);
                        }
                        var fsInfo = GetReplacedNcaInfo(orgNcaReader, descFilePath, selectedFsReplaceRuleList, ref replaceCount, entry.ContentMetaInfo.Model.GetUInt64Id());
                        fsInfo.partitionAlignmentType = (int)NintendoContentArchiveUtil.GetAlignmentType(entry.ContentMetaInfo.Model.Type, srcContentInfo.ContentType);
                        newContent.SetFsInfo(srcContentInfo.ContentType, null, fsInfo);
                    }
                    if (srcContentInfo.ContentType == NintendoContentMetaConstant.ContentTypeProgram)
                    {
                        // fs0 以下のファイルか、fs1/.nrr 以下の書き換えの場合のみ xml を作り直す
                        var regexStrings = new List<string>() {
                            @"fs0/.*",
                            @"fs1/\.nrr/.+(.nrr)$"
                        };
                        if (selectedReplaceRuleList.Where(rule => regexStrings.Where(pattern => Regex.IsMatch(rule.Path, pattern)).Any()).Any())
                        {
                            programInfoModified = true;
                        }
                    }
                    else if (srcContentInfo.ContentType == NintendoContentMetaConstant.ContentTypeControl)
                    {
                        controlInfoModified = true;
                    }
                    else if (srcContentInfo.ContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument)
                    {
                        htmlDocumentInfoModified = true;
                    }
                    else if (srcContentInfo.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation)
                    {
                        legalInfomationInfoModified = true;
                    }
                    dstContents.Add(newContent);
                }
            }

            if (programInfoModified)
            {
                entry.ProgramInfo = null;
                entry.NroDirectoryPath = nroDirectoryPath;
            }

            if (controlInfoModified)
            {
                entry.ApplicationControlPropertyInfo = null;
            }

            if (htmlDocumentInfoModified)
            {
                entry.HtmlDocumentInfo = null;
            }

            if (legalInfomationInfoModified)
            {
                entry.LegalInformationInfo = null;
            }

            // nca の追加（隠しコマンド）
            foreach (var addNcaRule in addContentList)
            {
                var addNcaArgs = addNcaRule.Path.Split(AddRootEntryInReplaceTargetDelimiter);
                if (addNcaArgs.Count() < 2)
                {
                    throw new Exception(string.Format("format for adding nca should be: 'null.nca@<ncm::ContentType>[@<ContentMetaId>]' ('{0}')", addNcaRule.Path));
                }
                string userInputContentType = addNcaArgs[1];
                // TODO: addNcaArgs[2] は ContentMetaId だが複数 cnmt.xml 対応の際に判断する必要がある

                var content = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                {
                    var ncaReader = new NintendoContentArchiveReader(addNcaRule.Source.GetStream(), new NcaKeyGenerator(keyConfig));
                    byte ncaContentType = ncaReader.GetContentType();
                    byte userContentType = NintendoContentAdfReader.ConvertToContentTypeByte(userInputContentType);
                    if (userContentType != ncaContentType)
                    {
                        throw new InvalidDataException(string.Format("content type mismatch - user input: '{0}', nca info: '{1}'", userContentType, ncaContentType));
                    }
                    var keyGeneration = ncaReader.GetKeyGeneration();
                    if (oldestKeyGeneration < keyGeneration)
                    {
                        throw new InvalidDataException(string.Format("Adding nca has no compatibility with target nsp. Please recreate the nca by tool has same version as one used for target nca."));
                    }
                    content.SetSource(userInputContentType, ncaReader.GetRepresentProgramIdOffset(), keyGeneration, null, addNcaRule.Source);
                }
                dstContents.Add(content);
                replaceCount++;
            }

            entry.Contents = dstContents;
        }

        // nsp のコンテンツメタの編集
        public static void ModifyContentMeta(NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, List<EntryReplaceRule> replaceRuleList, ref int replaceCount, KeyConfiguration keyConfig)
        {
            var replaceCnmtRule = replaceRuleList.Find(rule => Regex.IsMatch(rule.Path, @".*\.cnmt\.nca/fs0/.+\.cnmt$"));
            if (replaceCnmtRule == null)
            {
                return;
            }

            ulong id = 0;
            uint oldVersion = 0;
            uint newVersion = 0;
            {
                var reader = new NintendoContentMetaReader(entry.ContentMetaInfo.Data);
                id = reader.GetId();
                oldVersion = reader.GetVersion();
            }
            // コンテンツメタバイナリの差し換え
            // TODO: 不正なデータが書かれた場合にエラー
            if (replaceCnmtRule.Source.Size != entry.ContentMetaInfo.Data.Length)
            {
                throw new Exception(".cnmt file specified to be replaced is invalid.");
            }
            var data = replaceCnmtRule.Source.PullData(0, (int)replaceCnmtRule.Source.Size);
            System.Diagnostics.Debug.Assert(data.Buffer.Count == entry.ContentMetaInfo.Data.Length);
            Buffer.BlockCopy(data.Buffer.Array, data.Buffer.Offset, entry.ContentMetaInfo.Data, 0, data.Buffer.Count);
            {
                var reader = new NintendoContentMetaReader(entry.ContentMetaInfo.Data);
                var newId = reader.GetId();
                if (id != newId)
                {
                    throw new ArgumentException(string.Format("Ids of content meta are different. (oldId = {0:x16}, newID = {1:x16})", id, newId));
                }
                newVersion = reader.GetVersion();
            }
            Log.Info(string.Format("content meta (ID = {0:x16}) will be replaced : version {1} -> {2}", id, oldVersion, newVersion));
            replaceCount++;
        }

        // nsp のルートエントリの編集
        // TORIAEZU: 現状はアイコン（大）のみ対応
        public static void ModifyExtraEntry(NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, List<EntryReplaceRule> replaceRuleList, ref int replaceCount)
        {
            var belowRootEntryReplaceRuleList = replaceRuleList.Where(rule => ! (rule.Path.Contains("/") && rule.Path.Contains("\\"))).ToList();
            var addIconList = belowRootEntryReplaceRuleList.Where(rule => rule.Path.StartsWith(ModifiableArchiveUtil.AddIconInReplaceTargetKeyword)).ToList();

            var dstExtraDataList = new List<NintendoSubmissionPackageExtraData>();

            // icon の編集または削除
            foreach (var srcExtraData in entry.ExtraData)
            {
                var targetRule = replaceRuleList.Find(rule => rule.Path == srcExtraData.EntryName);
                if (targetRule != null)
                {
                    replaceCount++;
                    if (targetRule.Action == EntryReplaceAction.Delete)
                    {
                        continue;
                    }
                    var extra = new NintendoSubmissionPackageExtraData(srcExtraData.EntryName, targetRule.Source, srcExtraData.IdOffset);
                    dstExtraDataList.Add(extra);
                }
                else
                {
                    dstExtraDataList.Add(srcExtraData);
                }
            }

            var controlContentId = entry.ContentMetaInfo.Model.ContentList.Where(x => x.Type == NintendoContentMetaConstant.ContentTypeControl).Select(x => x.Id);

            // icon の追加（隠しコマンド）
            foreach (var addIconRule in addIconList)
            {
                int NumberOfIconRuleElement = 3;
                var addIconArgs = addIconRule.Path.Split(AddRootEntryInReplaceTargetDelimiter);
                if (addIconArgs.Count() < NumberOfIconRuleElement)
                {
                    throw new Exception(string.Format("format for adding icon should be: 'null.icon@<Language>@<raw|nx>' ('{0}')", addIconRule.Path));
                }
                string userInputLanguageType = addIconArgs[1];
                string userInputIconTarget = addIconArgs[2];
                if (!Enum.GetNames(typeof(ApplicationControlProperty.Language)).Contains(userInputLanguageType))
                {
                    throw new Exception(string.Format("unknown language '{0}' for icon", userInputLanguageType));
                }

                if (!controlContentId.Any())
                {
                    throw new Exception("This nsp cannot have icon because this has no control content.");
                }

                var extra = new NintendoSubmissionPackageExtraData(controlContentId.Single() + "." + ApplicationControl.GetRootIconName(userInputLanguageType, userInputIconTarget == "nx"), addIconRule.Source, 0);
                dstExtraDataList.Add(extra);
                replaceCount++;
            }

            entry.ExtraData = dstExtraDataList;
        }

        public static bool HasTicket(NintendoSubmissionPackageReader nspReader, ulong contentMetaId)
        {
            var rightsIdTextHalf = TicketUtility.CreateRightsIdText(contentMetaId, (byte)0).Substring(0, TicketUtility.RightsIdLength);
            return nspReader.ListFileInfo().Any(x => x.Item1.StartsWith(rightsIdTextHalf) && x.Item1.EndsWith(".tik"));
        }

        public static bool HasSparseStorage(NintendoSubmissionPackageFileSystemInfo fsInfo, KeyConfiguration keyConfig)
        {
            foreach (var entry in fsInfo.Entries)
            {
                // アプリ側にスパース情報が含まれているか
                foreach (var content in entry.Contents)
                {
                    using (var stream = new SourceBasedStream(content.Source))
                    using (var reader = new NintendoContentArchiveReader(stream, new NcaKeyGenerator(keyConfig)))
                    {
                        foreach (var index in reader.GetExistentFsIndices())
                        {
                            Debug.Assert(reader.HasFsInfo(index));

                            using (var header = reader.GetFsHeaderInfo(index))
                            {
                                if (header.ExistsSparseLayer())
                                {
                                    return true;
                                }
                            }
                        }
                    }
                }
                // パッチ側にスパース情報が含まれているか
                if ((entry.ContentMetaInfo?.Model as PatchContentMetaModel)?.SparseStorages?.FirstOrDefault() != null)
                {
                    return true;
                }
            }
            return false;
        }
    }

    internal class ModifiableNintendoContentArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        public ModifiableNintendoContentArchive(IReadableSink outSink, NintendoContentArchiveReader ncaReader, List<StreamSource> inSourceList, List<string> targetEntryPathList, string descFilePath, KeyConfiguration keyConfig, int alignmentType)
        {
            int replaceCount = 0;
            var replaceRuleList = ModifiableArchiveUtil.ConvertToReplaceRule(inSourceList, targetEntryPathList);
            var modifiedNcaInfo = ModifiableArchiveUtil.GetReplacedNcaInfo(ncaReader, descFilePath, replaceRuleList, ref replaceCount, ncaReader.GetProgramId());
            modifiedNcaInfo.partitionAlignmentType = alignmentType;
            ModifiableArchiveUtil.CheckReplaceCount(replaceRuleList.Count, replaceCount);

            var modifiedNca = new NintendoContentArchiveSource(modifiedNcaInfo, keyConfig);
            outSink.SetSize(modifiedNca.Size);
            ConnectionList = new List<Connection>();
            ConnectionList.Add(new Connection(modifiedNca, outSink));
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }

    internal class ModifiableNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        public ModifiableNintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageReader nspReader, List<StreamSource> inSourceList, List<string> targetEntryPathList, string descFilePath, string nroDirectoryPath, KeyConfiguration keyConfig)
        {
            var replaceRuleList = ModifiableArchiveUtil.ConvertToReplaceRule(inSourceList, targetEntryPathList);

            int replaceCount = 0;
            var modifiedNspInfo = ArchiveReconstructionUtils.GetNspInfo(nspReader, keyConfig);
            if (modifiedNspInfo.OnCardAddOnContentInfo != null)
            {
                throw new ArgumentException("To replace AddOnContent converted for card is not supported.");
            }
            if (ModifiableArchiveUtil.HasSparseStorage(modifiedNspInfo, keyConfig))
            {
                throw new ArgumentException("Modification of compacted application nsp is not supported.");
            }

            for (int i = 0; i < modifiedNspInfo.Entries.Count; i++)
            {
                var entry = modifiedNspInfo.Entries[i];
                entry.HasTicket = ModifiableArchiveUtil.HasTicket(nspReader, entry.ContentMetaInfo.Model.GetUInt64Id());

                // TODO: nca やアイコンの追加の対象の ContentMetaId を指定できるようにした際にはここで対象の entry かどうか区別する
                // TODO: patch 対象の replace をエラーに

                // コンテンツの差し換え
                ModifiableArchiveUtil.ModifyContent(entry, descFilePath, nroDirectoryPath, replaceRuleList, ref replaceCount, keyConfig);

                // コンテンツメタの差し換え
                ModifiableArchiveUtil.ModifyContentMeta(entry, replaceRuleList, ref replaceCount, keyConfig);

                // ルートエントリの差し換え
                ModifiableArchiveUtil.ModifyExtraEntry(entry, replaceRuleList, ref replaceCount);

                modifiedNspInfo.Entries[i] = entry;
            }

            ModifiableArchiveUtil.CheckReplaceCount(replaceRuleList.Count, replaceCount);

            var modifiedNsp = new NintendoSubmissionPackageArchive(outSink, modifiedNspInfo, keyConfig);
            ConnectionList = new List<Connection>(modifiedNsp.ConnectionList);
        }

        // TODO: 別クラスに移す
        public ModifiableNintendoSubmissionPackageArchive(IReadableSink outSink, NintendoSubmissionPackageReader patch, NintendoSubmissionPackageReader merging, KeyConfiguration keyConfig, bool mergeContent)
        {
            NintendoSubmissionPackageFileSystemInfo modifiedNspInfo;
            if (mergeContent)
            {
                modifiedNspInfo = ArchiveReconstructionUtils.GetDeltaMergedNspInfo(patch, merging, keyConfig);
            }
            else
            {
                modifiedNspInfo = ArchiveReconstructionUtils.GetContentMetaMergedNspInfo(patch, merging, keyConfig);
            }

            var modifiedNsp = new NintendoSubmissionPackageArchive(outSink, modifiedNspInfo, keyConfig);
            ConnectionList = new List<Connection>(modifiedNsp.ConnectionList);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }
}
