﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Text.RegularExpressions;
using Nintendo.Authoring.ImageLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class IconConverter
    {
        const string SoftwareName = "Nintendo AuthoringTool";
        const UInt32 DefaultNxIconSizeMax = 100 * 1024;
        private static void CheckRawIcon(Bitmap icon)
        {
            string commonErrorMessage = "元アイコンは解像度が 1024 * 1024 のビットマップで、24 ビット RGB のフォーマットのみが許可されています。";
            if (!(icon.PixelFormat == PixelFormat.Format24bppRgb ||
                icon.PixelFormat == PixelFormat.Format32bppRgb))
            {
                throw new ArgumentException(string.Format("{0}\n{1}\n", commonErrorMessage,
                    "ピクセルフォーマットが不正です。"));
            }

            if (!(icon.Width == 1024 && icon.Height == 1024))
            {
                throw new ArgumentException(string.Format("{0}\n{1}\n", commonErrorMessage,
                    "画像の解像度が 1024 * 1024 ではありません。"));
            }

        }

        private static void CheckNxIcon(Bitmap icon)
        {
            string commonErrorMessage = "NX 用アイコンは解像度が 256 * 256、サイズが 100KiB 以下でなければなりません。";
            if (icon.RawFormat.Guid != ImageFormat.Jpeg.Guid)
            {
                throw new ArgumentException(string.Format("{0}\n{1}\n", commonErrorMessage,
                    "このファイルは JPEG 形式ではありません。"));
            }

            if (!(icon.Width == 256 && icon.Height == 256))
            {
                throw new ArgumentException(string.Format("{0}\n{1}\n", commonErrorMessage,
                    "画像の解像度が 256 * 256 ではありません。"));
            }
        }

        private static void CheckNxIconJpeg(byte[] data, UInt32 nxIconMaxSize)
        {

            if(data.Length > nxIconMaxSize)
            {
                throw new ArgumentException(string.Format("NX 用アイコンのサイズは {0} バイト以下である必要があります。", nxIconMaxSize));
            }

            if(data.Length > DefaultNxIconSizeMax)
            {
                Console.WriteLine(string.Format("[警告] NX 用アイコンのサイズが {0} バイトを超えています。", DefaultNxIconSizeMax));
            }

            return;
        }

        private static Bitmap GetNxIconBitmap(Bitmap RawIcon, string nxIconPath)
        {
            if (nxIconPath != null)
            {
                var icon = new Bitmap(nxIconPath);
                CheckNxIcon(icon);
                return icon;
            }

            // INFO: Bilinear でいきなり 1024 -> 256 にすると劣化が顕著なので
            //       1024 -> 512 -> 256 と段階を踏む。
            var tempIcon = new Bitmap(512, 512);
            var tempGraphics = Graphics.FromImage(tempIcon);
            tempGraphics.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.Bilinear;
            tempGraphics.DrawImage(RawIcon, 0, 0, 512, 512);

            var nxIcon = new Bitmap(256, 256);
            var graphics = Graphics.FromImage(nxIcon);
            graphics.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.Bilinear;
            graphics.DrawImage(tempIcon, 0, 0, 256, 256);

            return nxIcon;
        }

        private static Bitmap GetRawIcon(string rawIconPath)
        {
            var icon = new Bitmap(rawIconPath);
            CheckRawIcon(icon);
            return icon;
        }

        private static void CorrectRgbChannel(byte[] pixel, PixelFormat format)
        {
            var unitSize = format == PixelFormat.Format24bppRgb ? 3 : 4;
            for (var offset = 0; offset < pixel.Length; offset += unitSize)
            {
                // INFO: R, G, B, (a) の R と B を入れ替える
                byte v0 = pixel[offset];
                pixel[offset] = pixel[offset + 2];
                pixel[offset + 2] = v0;
            }
        }

        private static void ThrowExceptionIfJpegResultFail(JpegStatus status)
        {
            if(status != JpegStatus.Ok)
            {
                throw new Exception(string.Format("Jpeg のエンコード処理で例外が発生しました。\n{0}", status));
            }
        }

        private static Size ToSize(SizeF sizef)
        {
            var size = new Size((int)sizef.Width, (int)sizef.Height);
            return size;
        }

        private static byte[] Encode(Bitmap icon, int quality)
        {
            using (var encoder = new JpegEncoder())
            {
                var bitmatData = icon.LockBits(new Rectangle(0, 0, icon.Width, icon.Height), ImageLockMode.ReadOnly, icon.PixelFormat);
                byte[] pixelData = new byte[icon.Height * bitmatData.Stride];
                Marshal.Copy(bitmatData.Scan0, pixelData, 0, pixelData.Length);
                icon.UnlockBits(bitmatData);

                CorrectRgbChannel(pixelData, icon.PixelFormat);
                var prepareResult = encoder.SetPixelData(pixelData, icon.PixelFormat, ToSize(icon.PhysicalDimension), 1);
                ThrowExceptionIfJpegResultFail(prepareResult);

                encoder.SetQuality(quality);
                encoder.SetSamplingRatio(SamplingRatio.Ratio_444);


                var exif = new Exif();
                exif.Software = SoftwareName;

                byte[] encodedBuffer;
                var encodeResult = encoder.Encode(out encodedBuffer, exif);
                ThrowExceptionIfJpegResultFail(encodeResult);

                return encodedBuffer;
            }

        }

        private static byte[] EncodeRawIcon(Bitmap icon)
        {
            // INFO: 98 の根拠は http://spdlybra.nintendo.co.jp/jira/browse/SPRD-2613 参照
            return Encode(icon, 98);
        }

        // INFO: NX アイコンは nxIconMaxSize 以下で一番いいやつにする。
        // 実際にエンコードしないと出力サイズがわからないので、パラメータを変えながら複数回エンコードする。
        private static byte[] EncodeNxIcon(Bitmap icon, UInt32 nxIconMaxSize)
        {
            if(icon.RawFormat.Guid == ImageFormat.Jpeg.Guid)
            {
                var rect = new Rectangle(0, 0, icon.Width, icon.Height);
                var data = icon.LockBits(rect, ImageLockMode.ReadOnly, icon.PixelFormat);
                var rawData = new byte[data.Stride * data.Height];
                Marshal.Copy(data.Scan0, rawData, 0, rawData.Length);
                return rawData;
            }

            int max = 100;
            int highQuality = max;
            int lowQuality = 0;
            UInt32 thresholdSize = nxIconMaxSize;
            byte[] encoded;
            do
            {
                encoded = Encode(icon, highQuality);
                if (encoded.Length > thresholdSize)
                {
                    highQuality = (highQuality + lowQuality) / 2;
                }
                else
                {
                    lowQuality = highQuality;
                    highQuality = (highQuality + max) / 2;
                }
            } while (highQuality != lowQuality);

            encoded = Encode(icon, highQuality);

            return encoded;
        }

        private static byte[] FixExifData(byte[] originalJpegData)
        {
            using (var jpegDecoder = new JpegDecoder())
            using (var exifExtractor = new ExifExtractor())
            using (var exifBuilder = new ExifBuilder())
            using(var jpegEncoder = new JpegEncoder())
            {
                byte[] originalExifData;
                Int32 exifDataOffset;
                var getExifResult = jpegDecoder.GetExifData(out exifDataOffset, out originalExifData, originalJpegData);
                if(getExifResult == JpegStatus.WrongFormat)
                {
                    // WrongFormat が返る場合は Exif が存在しない
                    return originalJpegData;
                }
                ThrowExceptionIfJpegResultFail(getExifResult);

                Exif exif;
                ThrowExceptionIfJpegResultFail(exifExtractor.Parse(out exif, originalExifData));

                // サムネイルを消す
                exif.Thumbnail = null;

                // exif の再構築
                ThrowExceptionIfJpegResultFail(jpegDecoder.SetJpegData(originalJpegData));
                var dimension = jpegDecoder.GetDimension();
                ThrowExceptionIfJpegResultFail(exifBuilder.SetExif(exif));
                var fixedExifData = exifBuilder.GetExifData((int)dimension.Width, (int)dimension.Height);

                // exif の置き換え
                var newJpegSize = originalJpegData.Length - originalExifData.Length + fixedExifData.Length;
                byte[] fixedJpegData = new byte[newJpegSize];
                {
                    // 1. exif 部分のバイナリを置き換え
                    // 先頭から exif の手前までを書き込み
                    int writeOffset = 0;
                    Array.Copy(originalJpegData, fixedJpegData, exifDataOffset);
                    writeOffset += exifDataOffset;

                    // 新しい exif を書き込み
                    Array.Copy(fixedExifData, 0, fixedJpegData, writeOffset, fixedExifData.Length);
                    writeOffset += fixedExifData.Length;

                    // exif 後のデータを書きこみ
                    Array.Copy(originalJpegData, exifDataOffset + originalExifData.Length, fixedJpegData, writeOffset, originalJpegData.Length - (exifDataOffset + originalExifData.Length));

                    // 2. exif を含むセグメントのサイズの値の更新
                    var sizeDiff = fixedExifData.Length - originalExifData.Length;
                    var sizeInfoIndex = exifDataOffset - 8; // INFO: http://hp.vector.co.jp/authors/VA032610/JPEGFormat/AboutExif.htm

                    var currentSize = (originalJpegData[sizeInfoIndex] << 8) + originalJpegData[sizeInfoIndex + 1];
                    var newSize = currentSize + sizeDiff;
                    fixedJpegData[sizeInfoIndex] = (byte)(newSize >> 8);
                    fixedJpegData[sizeInfoIndex + 1] = (byte)newSize;
                }

                return fixedJpegData;
            }
        }

        public static Tuple<byte[], byte[]> Convert(string rawIconPath, string nxIconPath, UInt32 nxIconMaxSize)
        {
            var rawIcon = GetRawIcon(rawIconPath);
            var encodedRawIcon = EncodeRawIcon(rawIcon);

            var nxIcon = GetNxIconBitmap(rawIcon, nxIconPath);
            byte[] encodedNxIcon;
            if(nxIcon.RawFormat.Guid != ImageFormat.Jpeg.Guid)
            {
                encodedNxIcon = EncodeNxIcon(nxIcon, nxIconMaxSize);
            }
            else
            {
                encodedNxIcon = FixExifData(File.ReadAllBytes(nxIconPath));
            }
            CheckNxIconJpeg(encodedNxIcon, nxIconMaxSize);


            return new Tuple<byte[], byte[]>(encodedRawIcon, encodedNxIcon);
        }

        public static byte[] ConvertNxIcon(string rawIconPath, string nxIconPath, UInt32 nxIconMaxSize)
        {
            var encodedIcon = Convert(rawIconPath, nxIconPath, nxIconMaxSize);

            return encodedIcon.Item2;
        }

        public static Dictionary<string, Tuple<string, string>> GetMergedIconPathMap(List<Tuple<string, string>> rawIconPathList, List<Tuple<string, string>> nxIconPathList)
        {
            var merged = new Dictionary<string, Tuple<string, string>>();
            rawIconPathList.ForEach(p => merged.Add(p.Item1, new Tuple<string, string>(p.Item2, null)));
            nxIconPathList.ForEach(p =>
            {
                var lang = p.Item1;
                var nxIconPath = p.Item2;
                if (!merged.ContainsKey(lang))
                {
                    merged.Add(lang, new Tuple<string, string>(null, null));
                }
                var pair = merged[lang];
                merged[lang] = new Tuple<string, string>(pair.Item1, nxIconPath);
            });
            return merged;
        }

        public static List<ApplicationControlProperty.Icon> GetApplicationControlPropertyIconEntryWithHash(List<NintendoSubmissionPackageExtraData> iconDataList)
        {
            List<ApplicationControlProperty.Icon> iconList = new List<ApplicationControlProperty.Icon>();
            var pattern = @"[^\.]+\.(raw|nx)\.([^\.]+)\.jpg$";

            foreach (var iconData in iconDataList)
            {
                var match = Regex.Match(iconData.EntryName, pattern);
                if (!match.Success)
                {
                    continue;
                }

                SHA256CryptoServiceProvider hashCalculator = new SHA256CryptoServiceProvider();
                byte[] hash = hashCalculator.ComputeHash(iconData.PullData(0, (int)iconData.Size).Buffer.Array, 0, (int)iconData.Size);
                var hashStr = BitConverter.ToString(hash, 0, hash.Length).Replace("-", string.Empty).ToLower().Substring(0, hash.Length);

                var index = iconList.FindIndex(entry => entry.Language == match.Groups[2].Value);
                if (index < 0)
                {
                    var icon = new ApplicationControlProperty.Icon() { IconPath = null, NxIconPath = null };
                    iconList.Add(icon);
                    index = iconList.Count() - 1;
                }
                iconList[index].Language = match.Groups[2].Value;
                if (match.Groups[1].Value == "nx")
                {
                    iconList[index].NxIconHash = hashStr;
                }
                else
                {
                    iconList[index].RawIconHash = hashStr;
                }
            }

            if (iconList.Count == 0)
            {
                return null;
            }

            return iconList;
        }
    }
}
