﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    [XmlRoot("HtmlDocument", IsNullable = false)]
    public class HtmlDocumentXmlModel
    {
        [XmlElement("AccessibleUrls", IsNullable = false)]
        public AccessibleUrlsModel AccessibleUrls { get; set; }

        public byte[] GetBytes()
        {
            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(String.Empty, String.Empty);

            using (var dst = new MemoryStream())
            {
                var sw = new StreamWriter(dst, Encoding.UTF8);
                var serializer = new XmlSerializer(typeof(HtmlDocumentXmlModel));
                serializer.Serialize(sw, this, nameSpace);
                return dst.ToArray();
            }
        }
    }

    [XmlRoot("AccessibleUrls", IsNullable = false)]
    public class AccessibleUrlsModel
    {
        [XmlElement("Url", IsNullable = false)]
        public List<string> Url { get; set; }
    }

    public class HtmlDocumentXml : ISource
    {
        public long Size { get; private set; }
        private ISource m_source;

        public HtmlDocumentXml(NintendoSubmissionPackageFileSystemInfo.ContentInfo content, KeyConfiguration config)
        {
            Trace.Assert(content.ContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument);

            HtmlDocumentXmlModel model = new HtmlDocumentXmlModel();
            model.AccessibleUrls = new AccessibleUrlsModel();
            model.AccessibleUrls.Url = new List<string>();

            var source = NintendoSubmissionPackageArchiveUtils.GetMemorySourceFromContent(content, config, "accessible-urls/accessible-urls.txt");
            if (source != null && source.Size > 0)
            {
                var bytesData = source.PullData(0, (int)source.Size).Buffer.Array;
                using (var src = new MemoryStream(bytesData.Length))
                {
                    src.Write(bytesData, 0, bytesData.Length);
                    src.Seek(0, SeekOrigin.Begin);

                    var sr = new StreamReader(src, Encoding.UTF8);

                    while(!sr.EndOfStream)
                    {
                        model.AccessibleUrls.Url.Add(sr.ReadLine());
                    }
                }
            }

            SetModel(model);
        }

        public HtmlDocumentXml(HtmlDocumentXmlModel model)
        {
            SetModel(model);
        }

        private void SetModel(HtmlDocumentXmlModel model)
        {
            var xmlBytes = model.GetBytes();
            m_source = new MemorySource(xmlBytes, 0, xmlBytes.Length);
            Size = m_source.Size;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }
    }
}
