﻿using Nintendo.Authoring.FileSystemMetaLibrary;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.Serialization;

namespace Nintendo.Authoring.AuthoringLibrary
{

    public class NintendoDeltaFragmentArchiveSource : ISource
    {
        public long Size { get; private set; }
        private string TargetContentType { get; set; }
        private ISource FragmentSource { get; set; }

        private ISource m_source;

        internal NintendoDeltaFragmentArchiveSource(ISource fragment, UInt64 programId, int keyEncryptionKeyIndex, byte keyGeneration, uint nspGeneration, uint fragmentIndexOnNsp, KeyConfiguration config, bool isProdEncryption, string targetContentType)
        {
            FragmentSource = fragment;

            // fragment をファイルとして含む PartitionFsArchiveSource を作成
            ISource partFsSource;
            {
                var partFsInfo = new PartitionFileSystemInfo();
                partFsInfo.version = 0;
                var entry = new PartitionFileSystemInfo.EntryInfo();
                entry.type = "source";
                entry.name = "fragment";
                entry.offset = 0;
                entry.size = (UInt64)fragment.Size;
                partFsInfo.entries.Add(entry);

                PartitionFileSystemMeta partFsMetaMgr = new PartitionFileSystemMeta();
                List<ConcatenatedSource.Element> partFsElements = new List<ConcatenatedSource.Element>();
                {
                    byte[] buffer = partFsMetaMgr.Create(partFsInfo);
                    ConcatenatedSource.Element headerElement = new ConcatenatedSource.Element(
                        new MemorySource(buffer, 0, buffer.Length), "meta", 0);
                    ConcatenatedSource.Element bodyElement = new ConcatenatedSource.Element(
                        fragment, "body", headerElement.Source.Size);
                    partFsElements.Add(headerElement);
                    partFsElements.Add(bodyElement);
                }
                partFsSource = new ConcatenatedSource(partFsElements);
            }

            // fragment をファイルとして含む NintendoContentArchive を作成
            ISource ncaSource;
            {
                NintendoContentFileSystemInfo ncaInfo = new NintendoContentFileSystemInfo();
                ncaInfo.distributionType = NintendoContentFileSystemMetaConstant.DistributionTypeDownload;
                ncaInfo.contentType = (byte)NintendoContentArchiveContentType.Data;
                ncaInfo.keyGeneration = (byte)keyGeneration;
                ncaInfo.programId = programId;
                ncaInfo.contentIndex = 0;
                ncaInfo.keyAreaEncryptionKeyIndex = (byte)keyEncryptionKeyIndex;
                ncaInfo.isProdEncryption = isProdEncryption;
                ncaInfo.partitionAlignmentType = (int)NintendoContentArchiveUtil.GetAlignmentType(NintendoContentMetaConstant.ContentMetaTypeDelta, NintendoContentMetaConstant.ContentTypeDeltaFragment);

                NintendoContentFileSystemInfo.EntryInfo fsEntry = new NintendoContentFileSystemInfo.EntryInfo();
                fsEntry.type = "source";
                fsEntry.formatType = NintendoContentFileSystemMetaConstant.FormatTypePartitionFs;
                fsEntry.sourceInterface = new CliCompatibleSource(partFsSource);
                fsEntry.version = 2;
                fsEntry.hashType = (byte)NintendoContentArchiveHashType.Auto;
                fsEntry.encryptionType = (byte)NintendoContentArchiveEncryptionType.Auto;
                fsEntry.generation = nspGeneration;
                fsEntry.secureValue = fragmentIndexOnNsp;
                fsEntry.partitionIndex = 0;

                ncaInfo.fsEntries.Add(fsEntry);
                ncaInfo.GenerateExistentFsIndicesFromFsEntries();
                ncaSource = new NintendoContentArchiveSource(ncaInfo, config);
            }

            m_source = ncaSource;
            Size = m_source.Size;
            TargetContentType = targetContentType;
        }

        public ByteData PullData(long offset, int size)
        {
            return m_source.PullData(offset, size);
        }

        public SourceStatus QueryStatus()
        {
            return m_source.QueryStatus();
        }

        internal void OutputBuildLog(NintendoContentArchiveBuildLog buildLog)
        {
            if (buildLog != null)
            {
                if (FragmentSource is SubSource)
                {
                    var subSource = (SubSource)FragmentSource;

                    ISource subSourceChild;
                    long offset;
                    subSource.GetInternalInformation(out subSourceChild, out offset);
                    if (subSourceChild is DeltaArchiveSource)
                    {
                        var deltaSource = (DeltaArchiveSource)subSourceChild;
                        deltaSource.OutputBuildLog(TargetContentType, offset, subSource.Size, buildLog);
                    }
                }
            }
        }
    }

    public class DeltaFragmentContentGenerator
    {
        private static readonly long DefaultSplitSize = 128 << 20;
        private static readonly int MaxProgramFragmentCount = 100;  // Program コンテンツの差分の分割最大数
        private static readonly int MaxOtherDeltaFragmentCount = 4; // Program 以外の、差分更新時の分割最大数
        private static readonly int MaxDataFragmentCount = 4;       // 丸抱えする場合の、分割最大数
        public UInt64 ProgramId { get; set; }
        private KeyConfiguration m_Config;
        private bool m_IsProdEncryption;
        private Dictionary<ulong, byte> m_ContentMetaKeyGeneration;

        private NintendoSubmissionPackageReader m_sourceReader;
        private NintendoSubmissionPackageReader m_destinationReader;

        public DeltaFragmentContentGenerator(KeyConfiguration config, Dictionary<ulong, byte> contentMetaKeyGeneration, bool isProdEncryption)
        {
            m_Config = config;
            m_ContentMetaKeyGeneration = contentMetaKeyGeneration;
            m_IsProdEncryption = isProdEncryption;
        }

        public void Dispose()
        {
            if (m_sourceReader != null)
            {
                m_sourceReader.Dispose();
            }
            if (m_destinationReader != null)
            {
                m_destinationReader.Dispose();
            }
        }
        private static long CalculateSplitSize(long deltaSize, int maxFragmentCount)
        {
            // コンテンツサイズと最大フラグメント数から、分割サイズを決める
            // 1 コマンドの最長サイズが 8MB (DeltaArchive::DeltaCommandSizeMax) であり、それを加味する
            var splitSize = DefaultSplitSize;
            while ((splitSize - DeltaArchive.DeltaCommandSizeMax) * maxFragmentCount < deltaSize)
            {
                splitSize += 64;
            }
            return splitSize;

        }

        private static List<ISource> GetDeltaFragments(ISource source, ISource destination, string outputPath, long deltaCommandSizeMax, bool willSave, int maxFragmentCount)
        {
            var deltaFragments = new List<ISource>();

            if (source == null)
            {
                source = new MemorySource(new byte[0], 0, 0);
            }

            var sourceInfo = DeltaInfo.ReadSourceInfo(source);
            IList<DeltaCommand> deltaCommands = DeltaInfo.ExtractDeltaCommands(source, destination, deltaCommandSizeMax, DeltaArchive.DeltaCommandCompareSize, null);
            DeltaInfo.CombineNeighbor(deltaCommands, DeltaArchive.DeltaCommandMergeSeekSizeMax, deltaCommandSizeMax);

            var deltaSource = new DeltaArchiveSource(sourceInfo, destination, (IReadOnlyList<DeltaCommand>)deltaCommands);

            if (willSave)
            {
                using (var file = File.Create(outputPath))
                {
                    const int LimitSize = 32 * 1024 * 1024;
                    long offset = 0;
                    while (offset < deltaSource.Size)
                    {
                        var readableSize = (int)Math.Min(deltaSource.Size - offset, LimitSize);
                        var data = deltaSource.PullData(offset, readableSize);
                        file.Write(data.Buffer.Array, data.Buffer.Offset, data.Buffer.Count);
                        offset += readableSize;
                    }
                }
            }

            {
                var splitSize = CalculateSplitSize(deltaSource.Size, maxFragmentCount);
                long offset = 0;
                while (offset < deltaSource.Size)
                {
                    var endOffset = deltaSource.FindNearestEndOffset(offset + splitSize);
                    deltaFragments.Add(new SubSource(
                        deltaSource,
                        offset,
                        endOffset - offset));
                    offset = endOffset;
                }
            }

            return deltaFragments;
        }

        private static List<ISource> GetDataFragments(ISource destination, long splitSize)
        {
            var delta = destination;
            var deltaSize = delta.Size; // サイズ取得のために、一度 source / destination を一通りなめる必要がある

            var list = new List<ISource>();
            for(long offset = 0; offset < deltaSize; offset += splitSize )
            {
                var size = Math.Min(splitSize, (deltaSize - offset));
                list.Add(new SubSource(delta, offset, size));
            }
            return list;
        }
        public static List<PatchContentMetaModel> ReadPatchContentMetaInNsp(NintendoSubmissionPackageReader reader)
        {
            var list = new List<PatchContentMetaModel>();

            var metas = reader.ListFileInfo().FindAll(p => p.Item1.EndsWith(".cnmt.xml"));
            foreach (var meta in metas)
            {
                using (var ms = new MemoryStream((int)meta.Item2))
                {
                    var xmlData = reader.ReadFile(meta.Item1, 0, meta.Item2);
                    ms.Write(xmlData, 0, xmlData.Length);
                    ms.Seek(0, SeekOrigin.Begin);
                    list.Add(new XmlSerializer(typeof(PatchContentMetaModel)).Deserialize(ms) as PatchContentMetaModel);
                }
            }
            return list;
        }
        public static string GetNcaName(ContentModel model)
        {
            if(model.Type == NintendoContentMetaConstant.ContentTypeMeta)
            {
                return string.Format("{0}.cnmt.nca", model.Id);
            }
            else
            {
                return string.Format("{0}.nca", model.Id);
            }
        }
        public static List<Pair<ContentModel, ContentModel>> FindMatchings(ContentMetaModel source, ContentMetaModel destination)
        {
            // sourceNca / destinationNca のファイル名のペアを見つける
            // 対応は contentType, idOffset によってみつける
            // destination にしかない場合は、source に null が入る
            // source にしかない場合はリストに追加されない
            var list = new List<Pair<ContentModel, ContentModel>>();
            foreach(var content in destination.ContentList)
            {
                if (content.Type == NintendoContentMetaConstant.ContentTypeDeltaFragment)
                {
                    continue;
                }
                var pair = source.ContentList.FindAll(p => p.Type == content.Type && p.IdOffset == content.IdOffset);
                if(pair.Count() > 1 )
                {
                    throw new FormatException("Multiple contents matching");
                }
                if(pair.Count() == 1)
                {
                    list.Add(new Pair<ContentModel, ContentModel>(pair.First(), content));
                }
                else
                {
                    // count == 0
                    list.Add(new Pair<ContentModel, ContentModel>(null, content));
                }
            }
            return list;
        }
        private NintendoSubmissionPackageFileSystemInfo.ContentInfo GetContentInfo(ContentModel sourceModel, ContentModel destinationModel, ISource source, int fragmentIndex, int keyIndex, byte keyGeneration, uint generation, uint fragmentIndexOnNsp, string updateType)
        {
            var contentInfo = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
            contentInfo.SetSource(NintendoContentMetaConstant.ContentTypeDeltaFragment, (byte)0, keyGeneration, null, new NintendoDeltaFragmentArchiveSource(source, ProgramId, keyIndex, keyGeneration, generation, fragmentIndexOnNsp, m_Config, m_IsProdEncryption, destinationModel.Type));
            var fragmentInfo = new NintendoSubmissionPackageFileSystemInfo.FragmentInfo();
            {
                fragmentInfo.TargetContentType = destinationModel.Type;
                fragmentInfo.UpdateType = updateType;
                fragmentInfo.Index = (UInt16) fragmentIndex;

                if (sourceModel == null)
                {
                    fragmentInfo.SourceContentId = "00000000000000000000000000000000";
                    fragmentInfo.SourceSize = 0;
                }
                else
                {
                    fragmentInfo.SourceContentId = sourceModel.Id;
                    fragmentInfo.SourceSize = sourceModel.Size;
                }

                fragmentInfo.DestinationContentId = destinationModel.Id;
                fragmentInfo.DestinationSize = destinationModel.Size;
            }
            contentInfo.SetFragment(fragmentInfo);
            return contentInfo;
        }

        public List<NintendoSubmissionPackageFileSystemInfo.ContentInfo> GetContentInfo(string outputDir, string sourceNsp, string destinationNsp, int keyIndex, long deltaCommandSizeMax, bool saveDelta)
        {
            var list = new List<NintendoSubmissionPackageFileSystemInfo.ContentInfo>();

            var sourceReader = new NintendoSubmissionPackageReader(sourceNsp);
            var destinationReader = new NintendoSubmissionPackageReader(destinationNsp);
            {
                var sourceMetas = ReadPatchContentMetaInNsp(sourceReader);
                var destinationMetas = ReadPatchContentMetaInNsp(destinationReader);
                this.ProgramId = Convert.ToUInt64(destinationMetas[0].Id, 16);

                foreach(var destinationMeta in destinationMetas)
                {
                    // 更新先コンテントメタに対応する元コンテントメタを見つける
                    var sourceMeta = sourceMetas.FindAll(p => p.Id == destinationMeta.Id);

                    // 現時点では 1:1 で対応するコンテントメタがない場合はエラーとする
                    if(sourceMeta.Count() != 1)
                    {
                        throw new ArgumentException("Invalid nsp pair");
                    }
                    // コンテントメタにひもづくコンテンツ群の、マッチングを行う
                    var matchings = FindMatchings(sourceMeta[0], destinationMeta);
                    foreach(var matching in matchings)
                    {
                        var destinationNca = new FileSystemArchvieFileSource(destinationReader, GetNcaName(matching.second));

                        // マッチングしたコンテントのペアから、fragment 群を生成する
                        List<ISource> fragments;
                        string updateType = null;
                        if (matching.first == null)
                        {
                            fragments = GetDataFragments(destinationNca, CalculateSplitSize(destinationNca.Size, MaxDataFragmentCount));
                            updateType = "Create";
                        }
                        else if (ArchiveReconstructionUtils.IsPatchTargetNca(matching.second.Type))
                        {
                            var sourceNca = new FileSystemArchvieFileSource(sourceReader, GetNcaName(matching.first));
                            var maxFragmentCount = (matching.second.Type == NintendoContentMetaConstant.ContentTypeProgram) ? MaxProgramFragmentCount : MaxOtherDeltaFragmentCount;
                            fragments = GetDeltaFragments(sourceNca, destinationNca, outputDir + "\\" + matching.second.Type + ".delta", deltaCommandSizeMax, saveDelta, maxFragmentCount);
                            updateType = "ApplyAsDelta";
                        }
                        else
                        {
                            fragments = GetDataFragments(destinationNca, CalculateSplitSize(destinationNca.Size, MaxDataFragmentCount));
                            updateType = "Overwrite";
                        }
                        for(var i = 0; i < fragments.Count(); ++i)
                        {
                            list.Add(GetContentInfo(matching.first, matching.second, fragments[i], i, keyIndex, m_ContentMetaKeyGeneration[this.ProgramId], destinationMeta.Version.Value, (uint)list.Count(), updateType));
                        }
                    }
                }

                m_sourceReader = sourceReader;
                m_destinationReader = destinationReader;

                return list;
            }
        }

        // warning CA2202 (System.ObjectDisposedException の回避への対応)
        static private void OpenFileStreamWriter(string path, Action<StreamWriter> doStreamWriter)
        {
            Stream stream = null;
            try
            {
                stream = new FileStream(path, FileMode.Create, FileAccess.Write);
                using (var streamWriter = new StreamWriter(stream))
                {
                    stream = null;
                    doStreamWriter?.Invoke(streamWriter);
                }
            }
            finally
            {
                if (stream != null)
                {
                    stream.Dispose();
                }
            }
        }

        public static void MakeDeltaMetaFromDestination(string metaPath, string destination)
        {
            using (var destinationReader = new NintendoSubmissionPackageReader(destination))
            {
                var meta = ReadPatchContentMetaInNsp(destinationReader);
                var applicationId = Convert.ToUInt64( meta[0].ApplicationId, 16);
                var deltaId = IdConverter.ConvertToDeltaId(applicationId);
                var version = meta[0].Version;

                var metaModel = new DeltaMetaRootModel();
                metaModel.Delta = new DeltaMetaModel();
                metaModel.Delta.SetId(deltaId);
                metaModel.Delta.Version = version.ToString();

                OpenFileStreamWriter(metaPath, sw =>
                {
                    var nameSpace = new XmlSerializerNamespaces();
                    nameSpace.Add(String.Empty, String.Empty);

                    var serializer = new XmlSerializer(typeof(DeltaMetaRootModel));
                    serializer.Serialize(sw, metaModel, nameSpace);
                });
            }
        }
    }
    public class DeltaMetaModel
    {
        [XmlElement("Version")]
        public string Version { get; set; }

        [XmlElement("Id")]
        public string Id { get; set; }

        public void SetId(UInt64 id)
        {
            Id = "0x" + id.ToString("x16");
        }
    }
    [XmlRoot("Meta")]
    public class DeltaMetaRootModel
    {
        [XmlElement("Delta")]
        public DeltaMetaModel Delta { get; set; }
    }

}
