﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Linq;
using System.Security.Cryptography;
using System.Xml;
using System.Xml.Linq;
using Nintendo.Authoring.CryptoLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    /// <summary>
    /// オーサリング処理全体に関わる設定のクラス
    /// </summary>
    public class AuthoringConfiguration
    {
        public DebugConfiguration DebugConfig { get; set; }

        // KeyConfig
        private KeyConfiguration m_KeyConfig = new KeyConfiguration();
        private string m_KeyConfigFilePath;

        public string KeyConfigFilePath
        {
            set
            {
                m_KeyConfigFilePath = value;
                m_KeyConfig.SetKey(m_KeyConfigFilePath);
            }
            get
            {
                return m_KeyConfigFilePath;
            }
        }

        public KeyConfiguration GetKeyConfiguration()
        {
            return m_KeyConfig;
        }

        public AuthoringConfiguration()
        {
            DebugConfig = new DebugConfiguration();
        }
    }

    public class RsaKey
    {
        public byte[] KeyModulus { get; private set; }
        public byte[] KeyPublicExponent;
        public byte[] KeyPrivateExponent;

        public RsaKey(byte[] keyModulus, byte[] keyPublicExponent, byte[] keyPrivateExponent)
        {
            KeyModulus = keyModulus;
            KeyPublicExponent = keyPublicExponent;
            KeyPrivateExponent = keyPrivateExponent;
        }
    }

    public class AesKey
    {
        public byte[] Key { get; private set; }
        public AesKey(byte[] key)
        {
            Key = key;
        }
    }

    public class DebugConfiguration
    {
        public bool EnableContentMetaBinaryExport { get; set; }
    }

    public enum EncryptedXciKeyIndex
    {
        Xcie = Rsa2048OaepSha256KeyIndex.XcieKeyArea,
        Xcir = Rsa2048OaepSha256KeyIndex.XcirKeyArea
    }

    public class KeyConfiguration
    {
        public EncryptedXciKeyIndex KeyIndexForEncryptedXci { get; set; } = EncryptedXciKeyIndex.Xcie;

        private List<Tuple<int, byte, AesKey>> m_KeyAreaEncryptionKeyList; // index, generation, key
        private List<Tuple<int, byte, AesKey>> m_KeyAreaEncryptionKeyHwList; // index, generation, key
        private RsaKey m_NcaHeader1SignKey;
        private RsaKey m_AcidSignKey;
        private RsaKey m_NrrCertificateSignKey;
        private List<Tuple<byte, AesKey>> m_ProdKeyAreaEncryptionKeyList; // generation, key
        private AesKey m_ProdNcaHeaderEncryptionKek;
        private RsaKey m_ProdXciHeaderSignKey;
        private AesKey m_ProdXciInitialDataEncryptionKey;
        private AesKey m_ProdTitleKeyGenarateKey;
        private List<Tuple<byte, AesKey>> m_ProdETicketCommonKeyList; // generation, key
        private RsaKey m_ProdETicketSignKey;

        public KeyConfiguration()
        {
            m_KeyAreaEncryptionKeyList = new List<Tuple<int, byte, AesKey>>();
            {
                var keyIndex0 = new AesKey(new byte[16] { 0x3a, 0x7c, 0x3e, 0x38, 0x4a, 0x8f, 0x22, 0xff, 0x4b, 0x21, 0x57, 0x19, 0xb7, 0x81, 0xad, 0x0c });
                var keyIndex2 = new AesKey(new byte[16] { 0xce, 0x3b, 0x2a, 0x68, 0x07, 0x9d, 0x99, 0xb1, 0x44, 0x8d, 0x69, 0xdc, 0x5e, 0x94, 0x2e, 0x48 });
                var keyIndex3 = new AesKey(new byte[16] { 0xdb, 0x04, 0x79, 0xba, 0x2d, 0x53, 0x95, 0xb2, 0x7c, 0x71, 0xbd, 0x6a, 0xfe, 0x5c, 0x3e, 0xc7 });
                var keyIndex4 = new AesKey(new byte[16] { 0xe6, 0xda, 0x78, 0x56, 0xd2, 0x13, 0xf0, 0xaa, 0xcb, 0x2b, 0x34, 0xa7, 0x1a, 0x87, 0xb6, 0xfb });
                var keyIndex5 = new AesKey(new byte[16] { 0x02, 0x74, 0x06, 0xfd, 0xf5, 0x96, 0x32, 0xb8, 0x8b, 0xd0, 0x53, 0xaa, 0xf3, 0x06, 0xd3, 0x9b });
                m_KeyAreaEncryptionKeyList.Add(Tuple.Create(0, (byte)0, keyIndex0));
                m_KeyAreaEncryptionKeyList.Add(Tuple.Create(0, (byte)1, keyIndex0)); // gen0 == gen1
                m_KeyAreaEncryptionKeyList.Add(Tuple.Create(0, (byte)2, keyIndex2));
                m_KeyAreaEncryptionKeyList.Add(Tuple.Create(0, (byte)3, keyIndex3));
                m_KeyAreaEncryptionKeyList.Add(Tuple.Create(0, (byte)4, keyIndex4));
                m_KeyAreaEncryptionKeyList.Add(Tuple.Create(0, (byte)5, keyIndex5));
            }

            m_KeyAreaEncryptionKeyHwList = new List<Tuple<int, byte, AesKey>>();
            {
                var keyIndex0Hw = new AesKey(new byte[16] { 0xc9, 0x0c, 0xe9, 0xff, 0x2a, 0xfb, 0xcd, 0xe9, 0xfd, 0x6a, 0x0d, 0xfe, 0x58, 0xd4, 0xe8, 0xc5 });
                var keyIndex2Hw = new AesKey(new byte[16] { 0x99, 0xc4, 0x40, 0xff, 0xc8, 0x9c, 0x74, 0xeb, 0x4b, 0x35, 0x73, 0x95, 0x7c, 0x19, 0x7a, 0xa9 });
                var keyIndex3Hw = new AesKey(new byte[16] { 0x8c, 0x60, 0xad, 0x7f, 0xb4, 0x3b, 0xed, 0xe5, 0xf8, 0x14, 0xb4, 0x3c, 0x05, 0x03, 0x8f, 0x82 });
                var keyIndex4Hw = new AesKey(new byte[16] { 0x36, 0x5a, 0x6f, 0xf2, 0xa6, 0x5e, 0x43, 0x8a, 0x12, 0x3e, 0x5b, 0x84, 0x9c, 0xb0, 0xf0, 0x91 });
                var keyIndex5Hw = new AesKey(new byte[16] { 0x24, 0x55, 0xfa, 0x8c, 0x65, 0x27, 0xf2, 0xdd, 0x9e, 0xbc, 0xa3, 0xe8, 0x41, 0xb8, 0x0d, 0xf1 });
                m_KeyAreaEncryptionKeyHwList.Add(Tuple.Create(0, (byte)0, keyIndex0Hw));
                m_KeyAreaEncryptionKeyHwList.Add(Tuple.Create(0, (byte)1, keyIndex0Hw)); // gen0 == gen1
                m_KeyAreaEncryptionKeyHwList.Add(Tuple.Create(0, (byte)2, keyIndex2Hw));
                m_KeyAreaEncryptionKeyHwList.Add(Tuple.Create(0, (byte)3, keyIndex3Hw));
                m_KeyAreaEncryptionKeyHwList.Add(Tuple.Create(0, (byte)4, keyIndex4Hw));
                m_KeyAreaEncryptionKeyHwList.Add(Tuple.Create(0, (byte)5, keyIndex5Hw));
            }

            m_ProdKeyAreaEncryptionKeyList = new List<Tuple<byte, AesKey>>();
            m_ProdETicketCommonKeyList = new List<Tuple<byte, AesKey>>();
        }

        private static Tuple<AesKey,AesKey> LoadAesXtsKeyConfig(XElement xml, string keyName)
        {
            XElement key;
            try
            {
                key = xml.Elements(keyName).Single();
            }
            catch  (System.InvalidOperationException)
            {
                return null;
            }
            var key1 = Convert.FromBase64String(key.Elements("Key1").Single().Value);
            var key2 = Convert.FromBase64String(key.Elements("Key2").Single().Value);
            return Tuple.Create(new AesKey(key1), new AesKey(key2));
        }

        private static AesKey LoadAesKeyConfig(XElement xml, string keyName)
        {
            XElement key;
            try
            {
                key = xml.Elements(keyName).Single();
            }
            catch (System.InvalidOperationException)
            {
                return null;
            }
            var keyValue = new AesKey(Convert.FromBase64String(key.Elements("Key").Single().Value));
            return keyValue;
        }

        private static RsaKey LoadRsaKeyConfig(XElement xml, string keyName)
        {
            XElement key;
            try
            {
                key = xml.Elements(keyName).Single();
            }
            catch  (System.InvalidOperationException)
            {
                return null;
            }
            var modulus = Convert.FromBase64String(key.Elements("Modulus").Single().Value);
            var publicExponent = Convert.FromBase64String(key.Elements("PublicExponent").Single().Value);
            var privateExponent = Convert.FromBase64String(key.Elements("PrivateExponent").Single().Value);
            return new RsaKey(modulus, publicExponent, privateExponent);
        }

        public void SetKey(string keyConfigFilePath)
        {
            var xml = XElement.Load(keyConfigFilePath);

            try
            {
                foreach(var keys in xml.Elements("KeyAreaEncryptionKeys").Elements("Key"))
                {
                    var keyIndex = Convert.ToInt32(keys.Attribute("index").Value);
                    var keyGeneration = Convert.ToByte(keys.Attribute("generation").Value);
                    var keyValue = Convert.FromBase64String(keys.Value);
                    m_KeyAreaEncryptionKeyList.Add(Tuple.Create(keyIndex, keyGeneration, new AesKey(keyValue)));
                }

                foreach (var keys in xml.Elements("KeyAreaEncryptionKeysHw").Elements("Key"))
                {
                    var keyIndex = Convert.ToInt32(keys.Attribute("index").Value);
                    var keyGeneration = Convert.ToByte(keys.Attribute("generation").Value);
                    var keyValue = Convert.FromBase64String(keys.Value);
                    m_KeyAreaEncryptionKeyHwList.Add(Tuple.Create(keyIndex, keyGeneration, new AesKey(keyValue)));
                }

                foreach (var keys in xml.Elements("ProdKeyAreaEncryptionKey").Elements("Key"))
                {
                    var keyGeneration = Convert.ToByte(keys.Attribute("generation").Value);
                    var keyValue = Convert.FromBase64String(keys.Value);
                    m_ProdKeyAreaEncryptionKeyList.Add(Tuple.Create(keyGeneration, new AesKey(keyValue)));
                }
                m_ProdNcaHeaderEncryptionKek = LoadAesKeyConfig(xml, "ProdNcaHeaderEncryptionKek");

                m_NcaHeader1SignKey = LoadRsaKeyConfig(xml, "NcaHeader1SignKey");
                m_AcidSignKey = LoadRsaKeyConfig(xml, "NcaAcidSignKey");
                m_NrrCertificateSignKey = LoadRsaKeyConfig(xml, "NcaNrrCertificateSignKey");

                m_ProdXciHeaderSignKey = LoadRsaKeyConfig(xml, "ProdXciHeaderSignKey");
                m_ProdXciInitialDataEncryptionKey = LoadAesKeyConfig(xml, "ProdXciInitialDataEncryptionKey");

                m_ProdTitleKeyGenarateKey = LoadAesKeyConfig(xml, "ProdTitleKeyGenarateKey");
                foreach (var keys in xml.Elements("ProdETicketCommonKey").Elements("Key"))
                {
                    var keyGeneration = Convert.ToByte(keys.Attribute("generation").Value);
                    var keyValue = Convert.FromBase64String(keys.Value);
                    m_ProdETicketCommonKeyList.Add(Tuple.Create(keyGeneration, new AesKey(keyValue)));
                }
                m_ProdETicketSignKey = LoadRsaKeyConfig(xml, "ProdETicketSignKey");
            }
            catch
            {
                throw new ArgumentException(String.Format("Failed to load key configuration from {0}.", keyConfigFilePath));
            }
        }

        public byte[][] GetKeyAreaEncryptionKeys(byte ncaKeyGeneration) // NcaReader に入力する Index = 1 以降の鍵リスト
        {
            int indexMax = 2; // fssystem::NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount - 1
            var keys = new byte[indexMax][];
            for (uint i = 0; i < indexMax; ++i)
            {
                try
                {
                    var key = GetKeyAreaEncryptionKey(i + 1, ncaKeyGeneration).Key;
                    keys[i] = key;
                }
                catch
                {
                    keys[i] = null;
                }
            }
            return keys;
        }

        public AesKey GetKeyAreaEncryptionKey(uint keyIndex, byte ncaKeyGeneration)
        {
            bool foundIndex = false;
            foreach(var keyInfo in m_KeyAreaEncryptionKeyList)
            {
                if (!foundIndex && keyInfo.Item1 == keyIndex)
                {
                    foundIndex = true;
                }

                if (keyInfo.Item1 == keyIndex && keyInfo.Item2 == ncaKeyGeneration)
                {
                    return keyInfo.Item3;
                }
            }

            if (foundIndex)
            {
                // 無効な鍵世代を使おうとしている
                return GetRandomAesKey();
            }

            throw new ArgumentException(String.Format("A key config file is required for key index {0}. Use --keyconfig option.", keyIndex));
        }

        public AesKey GetKeyAreaEncryptionKeyHw(uint keyIndex, byte ncaKeyGeneration)
        {
            bool foundIndex = false;
            foreach (var keyInfo in m_KeyAreaEncryptionKeyHwList)
            {
                if (!foundIndex && keyInfo.Item1 == keyIndex)
                {
                    foundIndex = true;
                }

                if (keyInfo.Item1 == keyIndex && keyInfo.Item2 == ncaKeyGeneration)
                {
                    return keyInfo.Item3;
                }
            }

            if (foundIndex)
            {
                // 無効な鍵世代を使おうとしている
                return GetRandomAesKey();
            }

            throw new ArgumentException(String.Format("A key config file is required for key index {0}. Use --keyconfig option.", keyIndex));
        }

        public AesKey GetProdKeyAreaEncryptionKey(byte ncaKeyGeneration)
        {
            foreach (var keyInfo in m_ProdKeyAreaEncryptionKeyList)
            {
                if (keyInfo.Item1 == ncaKeyGeneration)
                {
                    return keyInfo.Item2;
                }
            }
            return null;
        }

        public RsaKey GetNcaHeader1SignKey()
        {
            return m_NcaHeader1SignKey;
        }

        public RsaKey GetAcidSignKey()
        {
            return m_AcidSignKey;
        }

        public AesKey GetProdNcaHeaderEncryptionKek()
        {
            return m_ProdNcaHeaderEncryptionKek;
        }

        public RsaKey GetNrrCertificateSignKey()
        {
            return m_NrrCertificateSignKey;
        }

        public RsaKey GetProdXciHeaderSignKey()
        {
            return m_ProdXciHeaderSignKey;
        }

        public AesKey GetProdXciInitialDataEncryptionKey()
        {
            return m_ProdXciInitialDataEncryptionKey;
        }

        public AesKey GetProdTitleKeyGenarateKey()
        {
            return m_ProdTitleKeyGenarateKey;
        }

        public AesKey GetProdETicketCommonKey(byte ncaKeyGeneration)
        {
            foreach (var keyInfo in m_ProdETicketCommonKeyList)
            {
                if (keyInfo.Item1 == ncaKeyGeneration)
                {
                    return keyInfo.Item2;
                }
            }
            return null;
        }

        public RsaKey GetProdETicketSignKey()
        {
            return m_ProdETicketSignKey;
        }

        public AesKey GetRandomAesKey()
        {
            var rng = new System.Security.Cryptography.RNGCryptoServiceProvider();
            var key = new Byte[16];
            rng.GetBytes(key);
            return new AesKey(key);
        }

        public RsaKey GetRandomRsaKey()
        {
            const int RetryCountMax = 10;
            const int RetryWaitMilliSec = 10000;
            int retryCount = 0;
            while (true)
            {
                try
                {
                    var rsaServiceProvider = new RSACryptoServiceProvider(2048);
                    var param = rsaServiceProvider.ExportParameters(true);
                    return new RsaKey(param.Modulus, param.Exponent, param.D);
                }
                catch
                {
                    Log.Info(string.Format("Retry #{0} GetRandomRsaKey() after {1} msec.", retryCount, RetryWaitMilliSec));
                    System.Threading.Thread.Sleep(RetryWaitMilliSec);
                    retryCount++;
                    if (retryCount == RetryCountMax)
                    {
                        throw;
                    }
                }
            }
        }
    }
}
