﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;
using Nintendo.ApplicationControlProperty;

namespace Nintendo.Authoring.AuthoringLibrary
{
    // TODO: 全体的に整理（patch 向け nspInfo の生成を後処理に patch 関連の余分なコードを削除）
    public class ArchiveReconstructionUtils
    {
        internal static bool IsPatchTargetNca(string ncmContentType)
        {
            return ncmContentType == NintendoContentMetaConstant.ContentTypeProgram
                || ncmContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument;
        }

        internal static bool IsPatchTargetPartition(string ncmContentType, NintendoContentArchivePartitionType partitionType)
        {
            // データ、HtmlDocument のみパッチ化
            return (ncmContentType == NintendoContentMetaConstant.ContentTypeProgram && partitionType == NintendoContentArchivePartitionType.Data)
                || ncmContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument;
        }

        internal static bool IsSparseTargetPartition(string ncmContentType, NintendoContentArchivePartitionType partitionType)
        {
            // データ、コード、HtmlDocument のみスパース化
            return (ncmContentType == NintendoContentMetaConstant.ContentTypeProgram &&
                (partitionType == NintendoContentArchivePartitionType.Data || partitionType == NintendoContentArchivePartitionType.Code)) ||
                ncmContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument;
        }

        internal static bool WillPatchIncludePartition(string ncmContentType, NintendoContentArchivePartitionType partitionType)
        {
            // プログラムの Logo は含まない
            if (ncmContentType == NintendoContentMetaConstant.ContentTypeProgram && partitionType == NintendoContentArchivePartitionType.Logo)
            {
                return false;
            }
            return true;
        }

        internal static bool IsPatchTargetPartitionForAssersion(NintendoContentArchiveContentType contentType, NintendoContentArchivePartitionType partitionType)
        {
            // データ、HtmlDocument のみパッチ化
            return (contentType == NintendoContentArchiveContentType.Program && partitionType == NintendoContentArchivePartitionType.Data)
                || contentType == NintendoContentArchiveContentType.Manual;
        }

        static internal NintendoContentFileSystemInfo.EntryInfo GetCommonFsEntry(NintendoContentArchiveFsHeaderInfo fsHeaderInfo, int fsIndex)
        {
            var fsEntry = new NintendoContentFileSystemInfo.EntryInfo();
            fsEntry.version = fsHeaderInfo.GetVersion();
            fsEntry.hashType = fsHeaderInfo.GetHashType();
            fsEntry.encryptionType = fsHeaderInfo.GetEncryptionType();
            fsEntry.secureValue = fsHeaderInfo.GetSecureValue();
            fsEntry.generation = fsHeaderInfo.GetGeneration();
            fsEntry.partitionIndex = fsIndex;
            fsEntry.patchInfo = fsHeaderInfo.GetPatchInfo();
            fsEntry.hashTargetOffset = fsHeaderInfo.GetHashTargetOffset();
            fsEntry.existsSparseLayer = fsHeaderInfo.ExistsSparseLayer();

            // source は GetFsEntry で埋める TODO: replace の整理後統合
            fsEntry.sourceInterface = null;

            if (fsEntry.patchInfo != null)
            {
                fsEntry.formatType = fsHeaderInfo.GetFormatTypeWithPatch();
                // 後処理で FsInfo を設定しないとエラー
                fsEntry.type = "format";
            }
            else
            {
                fsEntry.formatType = fsHeaderInfo.GetFormatType();
                fsEntry.type = "source";
            }

            return fsEntry;
        }

        static internal NintendoContentFileSystemInfo.EntryInfo GetFsEntry(NintendoContentArchiveReader ncaReader, int fsIndex)
        {
            NintendoContentArchiveFsHeaderInfo fsHeaderInfo = ncaReader.GetFsHeaderInfo(fsIndex);
            var fsEntry = GetCommonFsEntry(fsHeaderInfo, fsIndex);

            if (fsEntry.patchInfo != null)
            {
                // sourceInterface に DifferenceSource + IndirectTable + AesCtrExTable を保持しておく
                var readers = ncaReader.OpenFsDataAndTableStorageArchiveReader(fsHeaderInfo);
                var dataSource = new StorageArchiveSource(readers.data);
                var tableSource = new StorageArchiveSource(readers.table);
                var elements = new List<ConcatenatedSource.Element>();
                elements.Add(new ConcatenatedSource.Element(dataSource, "dataAndIndirect", 0));
                elements.Add(new ConcatenatedSource.Element(tableSource, "aesCtrEx", dataSource.Size));
                fsEntry.sourceInterface = new CliCompatibleSource(new ConcatenatedSource(elements));
            }
            else
            {
                var fsReader = ncaReader.OpenFileSystemArchiveReader(fsIndex);
                var fsSource = new FileSystemArchvieBaseSource(fsReader);
                fsEntry.sourceInterface = new CliCompatibleSource(fsSource);
            }

            return fsEntry;
        }

        static private NintendoContentFileSystemInfo.EntryInfo GetPatchFsEntry(string ncmContentType, NintendoContentArchiveReader originalNcaReader, NintendoContentArchiveReader currentNcaReader, int fsIndex)
        {
            NintendoContentArchiveFsHeaderInfo fsHeaderInfo = null;
            var fsReader = currentNcaReader.OpenFileSystemArchiveReader(fsIndex, ref fsHeaderInfo);
            var fsEntry = GetCommonFsEntry(fsHeaderInfo, fsIndex);

            fsEntry.startOffset = (ulong)currentNcaReader.GetFsStartOffset(fsIndex);
            fsEntry.endOffset = (ulong)currentNcaReader.GetFsEndOffset(fsIndex);

            var currentPartitionType = (NintendoContentArchivePartitionType)fsIndex;
            if (IsPatchTargetPartition(ncmContentType, currentPartitionType))
            {
                IStorageArchiveReader originalSource = null;
                IFileSystemArchiveReader originalFsReader = null;
                NintendoContentArchiveFsHeaderInfo originalHeaderInfo = null;
                if (originalNcaReader != null && originalNcaReader.HasFsInfo(fsIndex))
                {
                    // original の FsEntry が存在
                    originalFsReader = originalNcaReader.OpenFileSystemArchiveReader(fsIndex, ref originalHeaderInfo);
                    originalSource = originalNcaReader.OpenFsDataStorageArchiveReader(originalHeaderInfo);
                }

                var binaryMatchHintPair = PatchConstructionUtility.GetHintsFromNcaFiles(originalNcaReader, currentNcaReader, fsIndex);
                var patchInfo = new FileSystemArchviePatchSource.PatchInfo()
                {
                    currentHeader = fsHeaderInfo.GetRawData(),
                    currentStorageReader = currentNcaReader.OpenFsDataStorageArchiveReader(fsHeaderInfo),
                    currentFsReader = fsReader,
                    originalHeader = originalHeaderInfo != null ? originalHeaderInfo.GetRawData() : null,
                    originalStorageReader = originalSource,
                    originalFsReader = originalFsReader,
                    binaryMatchHintPaths = binaryMatchHintPair.Item1,
                    binaryMatchHints = binaryMatchHintPair.Item2,
                };

                var fsSource = new FileSystemArchviePatchSource(patchInfo);
                fsEntry.sourceInterface = fsSource;
                fsEntry.formatType = "PatchFs";

                // 暗号化方式を AesCtrEx に設定
                fsEntry.encryptionType = (byte)NintendoContentArchiveEncryptionType.AesCtrEx;
            }
            else
            {
                var fsSource = new FileSystemArchvieBaseSource(fsReader);
                fsEntry.sourceInterface = new CliCompatibleSource(fsSource);

            }

            return fsEntry;
        }

        static private NintendoContentFileSystemInfo.EntryInfo GetIscPatchFsEntry(string ncmContentType, NintendoContentArchiveReader previousNcaReader, NintendoContentArchiveReader currentNcaReader, int fsIndex, uint nspGeneration)
        {
            // パッチ対象かどうか
            var isPatchFsIndex = IsPatchTargetPartition(ncmContentType, (NintendoContentArchivePartitionType)fsIndex);

            IFileSystemArchiveReader fsReader = null;
            NintendoContentArchiveFsHeaderInfo fsHeaderInfo = null;

            if (isPatchFsIndex)
            {
                fsHeaderInfo = currentNcaReader.GetFsHeaderInfo(fsIndex);
            }
            else
            {
                fsReader = currentNcaReader.OpenFileSystemArchiveReader(fsIndex, ref fsHeaderInfo);
            }

            var fsEntry = GetCommonFsEntry(fsHeaderInfo, fsIndex);

            fsEntry.startOffset = (ulong)currentNcaReader.GetFsStartOffset(fsIndex);
            fsEntry.endOffset = (ulong)currentNcaReader.GetFsEndOffset(fsIndex);

            // patchInfo の取得
            if (isPatchFsIndex)
            {
                if (0 < fsEntry.generation)
                {
                    // OptimizedPatch の最適化には未対応
                    throw new Exception("Current patch is already optimized.");
                }

                byte[] previousRawEntryInfo;
                long previousFsStartOffset;
                ISource previousSource;
                ISource previousAesCtrExTable;

                if (previousNcaReader != null && previousNcaReader.HasFsInfo(fsIndex))
                {
                    // previous に 対応する FsEntry が存在
                    var previousFsHeaderInfo = previousNcaReader.GetFsHeaderInfo(fsIndex);
                    previousRawEntryInfo = previousFsHeaderInfo.GetRawData();
                    previousFsStartOffset = previousNcaReader.GetFsStartOffset(fsIndex);
                    var readers = previousNcaReader.OpenFsDataAndTableStorageArchiveReader(previousFsHeaderInfo);
                    previousSource = new StorageArchiveSource(readers.data);
                    previousAesCtrExTable = new StorageArchiveSource(readers.table);
                }
                else
                {
                    previousRawEntryInfo = null;
                    previousFsStartOffset = 0;
                    previousSource = null;
                    previousAesCtrExTable = null;
                }

                var patchInfo = new FileSystemArchiveIscPatchSource.IscPatchInfo()
                {
                    currentRawEntryInfo = fsHeaderInfo.GetRawData(),
                    currentFsStartOffset = currentNcaReader.GetFsStartOffset(fsIndex),
                    currentSource = new StorageArchiveSource(currentNcaReader.OpenFsDataStorageArchiveReader(fsHeaderInfo)),
                    previousRawEntryInfo = previousRawEntryInfo,
                    previousFsStartOffset = previousFsStartOffset,
                    previousSource = previousSource,
                    previousAesCtrExTable = previousAesCtrExTable
                };

                var fsSource = new FileSystemArchiveIscPatchSource(patchInfo);
                fsEntry.sourceInterface = fsSource;
                fsEntry.formatType = "IscPatchFs";

                // 暗号化方式を AesCtrEx に設定
                fsEntry.encryptionType = (byte)NintendoContentArchiveEncryptionType.AesCtrEx;
            }
            else
            {
                var fsSource = new FileSystemArchvieBaseSource(fsReader);
                fsEntry.sourceInterface = new CliCompatibleSource(fsSource);
            }

            if (NintendoContentArchiveUtil.NeedsAesCtrGeneration(fsEntry) && previousNcaReader != null)
            {
                fsEntry.generation = nspGeneration + 1;
            }

            return fsEntry;
        }

        static internal NintendoContentFileSystemInfo GetCommonNcaInfo(NintendoContentArchiveReader ncaReader, UInt64 contentMetaId)
        {
            var ncaInfo = new NintendoContentFileSystemInfo();
            ncaInfo.distributionType = ncaReader.GetDistributionType();
            ncaInfo.contentType = ncaReader.GetContentType();
            ncaInfo.keyGeneration = ncaReader.GetKeyGeneration();
            ncaInfo.programId = ncaReader.GetProgramId();
            ncaInfo.contentIndex = ncaReader.GetContentIndex();
            ncaInfo.existentFsIndices = ncaReader.GetExistentFsIndices();
            ncaInfo.sdkAddonVersion = ncaReader.GetSdkAddonVersion();
            ncaInfo.headerEncryptionType = ncaReader.GetHeaderEncryptionType();
            ncaInfo.contentMetaId = contentMetaId;

            // 以下は仮の値であり、必要に応じて埋め直す必要がある
            ncaInfo.rightsId = ncaReader.GetRightsId();
            ncaInfo.keyAreaEncryptionKeyIndex = ncaReader.GetKeyIndex();
            ncaInfo.isProdEncryption = false;
            ncaInfo.isHardwareEncryptionKeyEmbedded = ncaReader.IsHardwareEncryptionKeyEmbedded();
            ncaInfo.partitionAlignmentType = (int)NintendoContentArchivePartitionAlignmentType.AlignSectorOrEncryptionBlockSize;

            return ncaInfo;
        }

        static internal NintendoContentFileSystemInfo GetProdNcaInfo(NintendoContentArchiveReader ncaReader, UInt64 contentMetaId)
        {
            var ncaInfo = GetCommonNcaInfo(ncaReader, contentMetaId);
            if (ncaInfo.headerEncryptionType == (byte)NintendoContentArchiveHeaderEncryptionType.None)
            {
                throw new ArgumentException("This nsp cannot be prod-encrypted because it is created without encryption.");
            }

            // 製品化向けに再設定
            ncaInfo.rightsId = null;
            ncaInfo.keyAreaEncryptionKeyIndex = 0;
            ncaInfo.isProdEncryption = true;
            ncaInfo.isHardwareEncryptionKeyEmbedded = false;

            foreach (var fsIndex in ncaInfo.existentFsIndices)
            {
                ncaInfo.fsEntries.Add(GetFsEntry(ncaReader, fsIndex));
            }
            ncaInfo.GenerateExistentFsIndicesFromFsEntries();

            return ncaInfo;
        }

        static internal NintendoContentFileSystemInfo GetNcaInfo(NintendoContentArchiveReader ncaReader, UInt64 contentMetaId)
        {
            var ncaInfo = GetCommonNcaInfo(ncaReader, contentMetaId);

            foreach (var fsIndex in ncaInfo.existentFsIndices)
            {
                ncaInfo.fsEntries.Add(GetFsEntry(ncaReader, fsIndex));
            }
            ncaInfo.GenerateExistentFsIndicesFromFsEntries();

            return ncaInfo;
        }

        static internal NintendoContentFileSystemInfo GetPatchedNcaInfo(string ncmContentType, NintendoContentArchiveReader originalNcaReader, NintendoContentArchiveReader currentNcaReader, string descFilePath, string contentMetaType, UInt64 contentMetaId)
        {
            var ncaInfo = GetCommonNcaInfo(currentNcaReader, contentMetaId);
            if (ncaInfo.headerEncryptionType == (byte)NintendoContentArchiveHeaderEncryptionType.None)
            {
                throw new ArgumentException("The nsp created without encryption is not supported for making patch.");
            }
            if (contentMetaId != 0 && (ncmContentType == NintendoContentMetaConstant.ContentTypeProgram || ncmContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument))
            {
                ncaInfo.rightsId = TicketUtility.CreateRightsId(contentMetaId, ncaInfo.keyGeneration);
            }
            ncaInfo.keyAreaEncryptionKeyIndex = currentNcaReader.GetKeyIndex();
            ncaInfo.isProdEncryption = false;
            ncaInfo.isHardwareEncryptionKeyEmbedded = currentNcaReader.IsHardwareEncryptionKeyEmbedded();
            ncaInfo.partitionAlignmentType = (int)NintendoContentArchiveUtil.GetAlignmentType(NintendoContentMetaConstant.ContentMetaTypePatch, ncmContentType);

            if (descFilePath != null)
            {
                NintendoContentAdfReader.RetrieveInfoFromDesc(ref ncaInfo, descFilePath);
            }
            else
            {
                // Program コンテントは desc 必須
                if (ncaInfo.contentType == (Byte)NintendoContentFileSystemMetaConstant.ContentTypeProgram)
                {
                    throw new Exception("Replacing 'Program' content needs desc file.");
                }
            }

            foreach (int fsIndex in ncaInfo.existentFsIndices)
            {
                if (ArchiveReconstructionUtils.WillPatchIncludePartition(ncmContentType, (NintendoContentArchivePartitionType)fsIndex))
                {
                    ncaInfo.fsEntries.Add(GetPatchFsEntry(ncmContentType, originalNcaReader, currentNcaReader, fsIndex));
                }
            }

            return ncaInfo;
        }

        static internal List<SparseStorageArchiveStream> RetrieveSparseStorageStream(NintendoSubmissionPackageReader nspReader, KeyConfiguration keyConfig)
        {
            if (nspReader != null)
            {
                var nspInfo = GetNspInfo(nspReader, keyConfig);
                var patchList = nspInfo.Entries.Where(value => value.ContentMetaInfo.Model.Type == NintendoContentMetaConstant.ContentMetaTypePatch);
                if (0 < patchList.Count())
                {
                    return (patchList.Single().ContentMetaInfo.Model as PatchContentMetaModel).SparseStorages;
                }
            }
            return null;
        }

        static internal void AttachSparseStorageStream(string contentType, List<SparseStorageArchiveStream> streams, NintendoContentArchiveReader ncaReader)
        {
            if (streams != null)
            {
                var stream = streams.Find(value => value.Type == contentType);
                if (stream != null)
                {
                    stream.Attach(ncaReader);
                }
            }
        }

        static internal NintendoContentFileSystemInfo GetIscPatchedNcaInfo(string ncmContentType, NintendoContentArchiveReader previousNcaReader, NintendoContentArchiveReader currentNcaReader, string descFilePath, uint nspGeneration, string contentMetaType, UInt64 contentMetaId, byte ncaKeyGeneration)
        {
            var ncaInfo = GetCommonNcaInfo(currentNcaReader, contentMetaId);
            if (ncaInfo.headerEncryptionType == (byte)NintendoContentArchiveHeaderEncryptionType.None)
            {
                throw new ArgumentException("The nsp created without encryption is not supported for making patch.");
            }
            if (contentMetaId != 0 && (ncmContentType == NintendoContentMetaConstant.ContentTypeProgram || ncmContentType == NintendoContentMetaConstant.ContentTypeHtmlDocument))
            {
                ncaInfo.rightsId = TicketUtility.CreateRightsId(contentMetaId, ncaKeyGeneration);
            }
            ncaInfo.keyAreaEncryptionKeyIndex = currentNcaReader.GetKeyIndex();
            ncaInfo.isProdEncryption = false;
            ncaInfo.isHardwareEncryptionKeyEmbedded = currentNcaReader.IsHardwareEncryptionKeyEmbedded();
            ncaInfo.partitionAlignmentType = (int)NintendoContentArchiveUtil.GetAlignmentType(NintendoContentMetaConstant.ContentMetaTypePatch, ncmContentType);

            // パッチ最適化の際、前 ver. のパッチのプログラムコンテンツの鍵世代を引き継ぐ
            ncaInfo.keyGeneration = ncaKeyGeneration;

            if (descFilePath != null)
            {
                NintendoContentAdfReader.RetrieveInfoFromDesc(ref ncaInfo, descFilePath);
            }
            else
            {
                // Program コンテントは desc 必須
                if (ncaInfo.contentType == (Byte)NintendoContentFileSystemMetaConstant.ContentTypeProgram)
                {
                    throw new Exception("Replacing 'Program' content needs desc file.");
                }
            }

            foreach (int fsIndex in ncaInfo.existentFsIndices)
            {
                if (ArchiveReconstructionUtils.WillPatchIncludePartition(ncmContentType, (NintendoContentArchivePartitionType)fsIndex))
                {
                    ncaInfo.fsEntries.Add(GetIscPatchFsEntry(ncmContentType, previousNcaReader, currentNcaReader, fsIndex, nspGeneration));
                }
            }

            return ncaInfo;
        }

        static public ModelType ReadXml<ModelType>(NintendoSubmissionPackageReader nspReader, string fileName, long fileSize)
        {
            var xmlData = nspReader.ReadFile(fileName, 0, fileSize);
            return ReadXml<ModelType>(xmlData);
        }

        static public ModelType ReadXml<ModelType>(byte[] xmlData)
        {
            var serializer = new XmlSerializer(typeof(ModelType));
            ModelType model;
            using (var ms = new MemoryStream(xmlData.Length))
            {
                ms.Write(xmlData, 0, xmlData.Length);
                ms.Seek(0, SeekOrigin.Begin);
                model = (ModelType)serializer.Deserialize(ms);
            }
            return model;
        }

        static private void GetCommonContentMetaInfo(ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string contentMetaFileName, ContentMetaModel model, NintendoSubmissionPackageReader nspReader, KeyConfiguration keyConfig)
        {
            var metaNcaReader = nspReader.OpenNintendoContentArchiveReader(contentMetaFileName, new NcaKeyGenerator(keyConfig));

            // XML 情報を生成
            {
                var metaFsReader = metaNcaReader.OpenFileSystemArchiveReader(0);
                var metaFileInfo = metaFsReader.ListFileInfo().Single();
                entry.ContentMetaInfo = new NintendoSubmissionPackageFileSystemInfo.ContentMetaInfo(
                    metaFsReader.ReadFile(metaFileInfo.Item1, 0, metaFileInfo.Item2), model
                );
            }

            // 生成処理で使えるように metaType も取得しておく
            {
                var reader = new NintendoContentMetaReader(entry.ContentMetaInfo.Data);
                entry.MetaType = reader.GetType();
            }

            // スパース情報を生成
            if (entry.MetaType == NintendoContentMetaConstant.ContentMetaTypePatch)
            {
                var patchMeta = entry.ContentMetaInfo.Model as PatchContentMetaModel;
                patchMeta.SparseStorages = new List<SparseStorageArchiveStream>();

                var fileInfoList = nspReader.ListFileInfo();

                foreach (var content in patchMeta.ContentList)
                {
                    var fileInfo = fileInfoList.Find(value => value.Item1 == content.Id + ".compaction.bin");
                    if (fileInfo != null)
                    {
                        patchMeta.SparseStorages.Add(new SparseStorageArchiveStream(content.Type, nspReader.ReadFile(fileInfo.Item1, 0, fileInfo.Item2)));
                    }
                }
            }

            // TODO: metaType に応じて正しく引き継ぐ
        }

        static public ContentMetaModel ReadModel(NintendoSubmissionPackageReader nspReader, string fileName, Int64 fileSize)
        {
            var xmlData = nspReader.ReadFile(fileName, 0, fileSize);
            return ReadModel(xmlData);
        }

        static public ContentMetaModel ReadModel(byte[] xmlData)
        {
            var model = ReadXml<ContentMetaModel>(xmlData);

            // metaType に応じて開き直し
            switch (model.Type)
            {
                case NintendoContentMetaConstant.ContentMetaTypeApplication:
                    model = ReadXml<ApplicationContentMetaModel>(xmlData);
                    break;
                case NintendoContentMetaConstant.ContentMetaTypePatch:
                    model = ReadXml<PatchContentMetaModel>(xmlData);
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                    model = ReadXml<AddOnContentContentMetaModel>(xmlData);
                    break;
                case NintendoContentMetaConstant.ContentMetaTypeDelta:
                    model = ReadXml<DeltaContentMetaModel>(xmlData);
                    break;
            }
            return model;
        }

        private delegate NintendoSubmissionPackageFileSystemInfo.ContentInfo GetContentInfoDelegate(ContentModel contentInfo, string contentMetaType, UInt64 contentMetaId);
        private delegate void GetContentMetaInfoDelegate(ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string contentMetaFileName, ContentMetaModel model);

        static private NintendoSubmissionPackageFileSystemInfo GetNspInfo(NintendoSubmissionPackageReader nspReader,
            KeyConfiguration keyConfig, GetContentInfoDelegate getContentInfoImpl, GetContentMetaInfoDelegate getContentMetaInfoImpl)
        {
            var reconstructedNspInfo = new NintendoSubmissionPackageFileSystemInfo();
            reconstructedNspInfo.Version = 0; // nspReader.GetFormatVersion(); 今はヘッダ生成時に version を使っていない

            var cnmtXmlList = new List<Tuple<string, long>>();

            foreach (var fileInfo in nspReader.ListFileInfo())
            {
                var fileName = fileInfo.Item1;
                var fileSize = fileInfo.Item2;

                // *** cnmt.xml に紐づくもの
                if (fileName.EndsWith(".cnmt.xml"))
                {
                    cnmtXmlList.Add(fileInfo);
                }

                // *** nsp 直下エントリ

                // authoringtoolinfo.xml
                if (fileName == "authoringtoolinfo.xml")
                {
                    reconstructedNspInfo.AuthoringToolInfo = ReadXml<AuthoringToolInfoModel>(nspReader, fileName, fileSize);
                }

                // カード情報の引き継ぎ
                if (fileName == "cardspec.xml")
                {
                    var cardSpec = ReadXml<CardSpecModel>(nspReader, fileName, fileSize);
                    reconstructedNspInfo.CardSize = Convert.ToInt32(cardSpec.Size, 10);
                    reconstructedNspInfo.CardClockRate = Convert.ToInt32(cardSpec.ClockRate, 10);
                    reconstructedNspInfo.AutoSetSize = cardSpec.AutoSetSize;
                    reconstructedNspInfo.AutoSetClockRate = cardSpec.AutoSetClockRate;
                    reconstructedNspInfo.CardLaunchFlags = cardSpec.LaunchFlags;
                }

                // oncardaoc.xml
                if (fileName == "oncardaoc.xml")
                {
                    reconstructedNspInfo.OnCardAddOnContentInfo = ReadXml<OnCardAddOnContentInfoModel>(nspReader, fileName, fileSize);
                }

                // multiapplicationcard.xml
                if (fileName == "multiapplicationcard.xml")
                {
                    reconstructedNspInfo.MultiApplicationCardInfo = ReadXml<MultiApplicationCardInfoModel>(nspReader, fileName, fileSize);
                }
            }

            // *** cnmt.xml に紐づくもの
            foreach (var cnmtFileInfo in cnmtXmlList)
            {
                var fileName = cnmtFileInfo.Item1;
                var fileSize = cnmtFileInfo.Item2;

                var model = ReadXml<ContentMetaModel>(nspReader, fileName, fileSize);

                // metaType に応じて開き直し
                switch (model.Type)
                {
                    case NintendoContentMetaConstant.ContentMetaTypeApplication:
                        model = ReadXml<ApplicationContentMetaModel>(nspReader, fileName, fileSize);
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypePatch:
                        model = ReadXml<PatchContentMetaModel>(nspReader, fileName, fileSize);
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                        model = ReadXml<AddOnContentContentMetaModel>(nspReader, fileName, fileSize);
                        break;
                }

                // コンテンツメタ(XML) に紐づくコンテンツをエントリに追加
                var entry = new NintendoSubmissionPackageFileSystemInfo.EntryInfo();

                entry.Contents = new List<NintendoSubmissionPackageFileSystemInfo.ContentInfo>();
                foreach (var contentInfo in model.ContentList)
                {
                    if(contentInfo.Type == NintendoContentMetaConstant.ContentTypeMeta)
                    {
                        continue;
                    }

                    var content = getContentInfoImpl(contentInfo, model.Type, model.GetUInt64Id());
                    entry.Contents.Add(content);

                    var programInfoXmls = nspReader.ListFileInfo().Where(x => x.Item1 == string.Format("{0}.programinfo.xml", contentInfo.Id));
                    if (programInfoXmls.Any())
                    {
                        var programInfoXml = programInfoXmls.Single();
                        var programInfoModel = ReadXml<ProgramInfoModel>(nspReader, programInfoXml.Item1, programInfoXml.Item2);
                        if (entry.ProgramInfo == null)
                        {
                            entry.ProgramInfo = new NintendoSubmissionPackageFileSystemInfo.ProgramInfo(programInfoModel);
                        }
                        else
                        {
                            entry.ProgramInfo.Models.Add(programInfoModel);
                        }
                    }
                    var htmlDocumentInfoXmls = nspReader.ListFileInfo().Where(x => x.Item1 == string.Format("{0}.htmldocument.xml", contentInfo.Id));
                    if (htmlDocumentInfoXmls.Any())
                    {
                        var htmlDocumentInfoXml = htmlDocumentInfoXmls.Single();
                        var htmlDocumentXmlModel = ReadXml<HtmlDocumentXmlModel>(nspReader, htmlDocumentInfoXml.Item1, htmlDocumentInfoXml.Item2);
                        if (entry.HtmlDocumentInfo == null)
                        {
                            entry.HtmlDocumentInfo = new NintendoSubmissionPackageFileSystemInfo.HtmlDocumentInfo(htmlDocumentXmlModel);
                        }
                        else
                        {
                            entry.HtmlDocumentInfo.Models.Add(htmlDocumentXmlModel);
                        }
                    }
                    var legalInformationInfoXmls = nspReader.ListFileInfo().Where(x => x.Item1 == string.Format("{0}.legalinfo.xml", contentInfo.Id));
                    if (legalInformationInfoXmls.Any())
                    {
                        var legalInformationInfoXml = legalInformationInfoXmls.Single();
                        var legalInformationInfoXmlModel = ReadXml<LegalInformationModel>(nspReader, legalInformationInfoXml.Item1, legalInformationInfoXml.Item2);
                        if (entry.LegalInformationInfo == null)
                        {
                            entry.LegalInformationInfo = new NintendoSubmissionPackageFileSystemInfo.LegalInformationInfo(legalInformationInfoXmlModel);
                        }
                        else
                        {
                            entry.LegalInformationInfo.Models.Add(legalInformationInfoXmlModel);
                        }
                    }
                    var acpXmls = nspReader.ListFileInfo().Where(x => x.Item1 == string.Format("{0}.nacp.xml", contentInfo.Id));
                    if (acpXmls.Any())
                    {
                        var acpXml = acpXmls.Single();
                        var acpModel = ReadXml<ApplicationControlPropertyModel>(nspReader, acpXml.Item1, acpXml.Item2);
                        if (entry.ApplicationControlPropertyInfo == null)
                        {
                            entry.ApplicationControlPropertyInfo = new NintendoSubmissionPackageFileSystemInfo.ApplicationControlPropertyInfo(acpModel);
                        }
                        else
                        {
                            entry.ApplicationControlPropertyInfo.Models.Add(acpModel);
                        }

                        var iconEntries = nspReader.ListFileInfo().FindAll(x => x.Item1.StartsWith(acpXml.Item1.Replace(".nacp.xml", "")) && x.Item1.EndsWith(".jpg"));
                        var iconList = iconEntries.Select(x =>
                        {
                            var data = nspReader.ReadFile(x.Item1, 0, x.Item2);
                            var extra = new NintendoSubmissionPackageExtraData(
                                x.Item1,
                                new MemorySource(data, 0, data.Length),
                                contentInfo.IdOffset);
                            return extra;
                        }).ToList();
                        iconList.RemoveAll(x => x == null);

                        entry.ExtraData.AddRange(iconList);
                    }
                }

                // コンテンツメタの引き継ぎ
                {
                    var contentMetaFileName = Path.GetFileNameWithoutExtension(fileName) + ".nca";
                    getContentMetaInfoImpl(ref entry, contentMetaFileName, model);
                }

                reconstructedNspInfo.Entries.Add(entry);
            }

            return reconstructedNspInfo;
        }

        // NCA 再構築を選べる版。コンテンツメタは作り直す
        static internal NintendoSubmissionPackageFileSystemInfo GetNspInfo(NintendoSubmissionPackageReader nspReader, KeyConfiguration keyConfig, bool reconstructNca = false)
        {
            var keyGenerator = new NcaKeyGenerator(keyConfig);

            GetContentInfoDelegate getContentInfoImpl = delegate (ContentModel contentInfo, string contentMetaType, UInt64 contentMetaId)
            {
                var content = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                var ncaName = contentInfo.Id + ".nca";
                if (reconstructNca)
                {
                    var ncaReader = nspReader.OpenNintendoContentArchiveReader(ncaName, new NcaKeyGenerator(keyConfig));
                    TicketUtility.SetExternalKey(ref ncaReader, nspReader);
                    var fsInfo = GetNcaInfo(ncaReader, contentMetaId);
                    fsInfo.partitionAlignmentType = (int)NintendoContentArchiveUtil.GetAlignmentType(contentMetaType, contentInfo.Type);
                    content.SetFsInfo(contentInfo.Type, contentInfo.Id, fsInfo);
                }
                else
                {
                    using (var ncaReader = nspReader.OpenNintendoContentArchiveReader(ncaName, keyGenerator))
                    {
                        content.SetSource(contentInfo.Type, ncaReader.GetRepresentProgramIdOffset(), ncaReader.GetKeyGeneration(), contentInfo.Id, new FileSystemArchvieFileSource(nspReader, ncaName));
                    }
                }
                return content;
            };

            GetContentMetaInfoDelegate getContentMetaInfoImpl = delegate (ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string contentMetaFileName, ContentMetaModel model)
            {
                GetCommonContentMetaInfo(ref entry, contentMetaFileName, model, nspReader, keyConfig);
            };

            var nspInfo = GetNspInfo(nspReader, null, getContentInfoImpl, getContentMetaInfoImpl);

            return nspInfo;
        }

        // NCA を再構築しない版。製品化済み NSP を読み込む際に利用
        static internal NintendoSubmissionPackageFileSystemInfo GetProdNspInfo(NintendoSubmissionPackageReader nspReader)
        {
            GetContentInfoDelegate getContentInfoImpl = delegate (ContentModel contentInfo, string contentMetaType, UInt64 contentMetaId)
            {
                var content = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                var ncaName = contentInfo.Id + ".nca";
                content.SetSource(contentInfo.Type, contentInfo.IdOffset, NintendoContentFileSystemMetaConstant.InvalidKeyGeneration, contentInfo.Id, new FileSystemArchvieFileSource(nspReader, ncaName));
                return content;
            };

            GetContentMetaInfoDelegate getContentMetaInfoImpl = delegate (ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string contentMetaFileName, ContentMetaModel model)
            {
                // do nothing
            };

            var nspInfo = GetNspInfo(nspReader, null, getContentInfoImpl, getContentMetaInfoImpl);
            nspInfo.IsProdEncryption = true;

            return nspInfo;
        }

        static internal NintendoSubmissionPackageFileSystemInfo GetProdNspInfo(NintendoSubmissionPackageReader nspReader, KeyConfiguration keyConfig)
        {
            GetContentInfoDelegate getContentInfoImpl = delegate (ContentModel contentInfo, string contentMetaType, UInt64 contentMetaId)
            {
                var content = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                var ncaName = contentInfo.Id + ".nca";
                var ncaReader = nspReader.OpenNintendoContentArchiveReader(ncaName, new NcaKeyGenerator(keyConfig));
                TicketUtility.SetExternalKey(ref ncaReader, nspReader);
                var fsInfo = GetProdNcaInfo(ncaReader, contentMetaId);
                fsInfo.partitionAlignmentType = (int)NintendoContentArchiveUtil.GetAlignmentType(contentMetaType, contentInfo.Type);
                content.SetFsInfo(contentInfo.Type, null, fsInfo);
                return content;
            };

            GetContentMetaInfoDelegate getContentMetaInfoImpl = delegate (ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string contentMetaFileName, ContentMetaModel model)
            {
                GetCommonContentMetaInfo(ref entry, contentMetaFileName, model, nspReader, keyConfig);
            };

            var nspInfo = GetNspInfo(nspReader, keyConfig, getContentInfoImpl, getContentMetaInfoImpl);
            nspInfo.IsProdEncryption = true;

            return nspInfo;
        }

        static internal NintendoSubmissionPackageFileSystemInfo GetProdNspInfo(List<NintendoSubmissionPackageReader> nspReaders, KeyConfiguration keyConfig)
        {
            Trace.Assert(nspReaders.Count >= 1);
            var mergedNspInfo = GetProdNspInfo(nspReaders.First(), keyConfig);
            foreach (var nspReader in nspReaders.Skip(1))
            {
                var nspInfo = GetProdNspInfo(nspReader, keyConfig);
                mergedNspInfo.Entries.AddRange(nspInfo.Entries);
                if (nspInfo.KeepGeneration.HasValue && !mergedNspInfo.KeepGeneration.HasValue)
                {
                    mergedNspInfo.KeepGeneration = nspInfo.KeepGeneration;
                }
            }
            return mergedNspInfo;
        }

        public static List<ContentMetaModel> ReadContentMetaInNsp(string nspFilePath)
        {
            using (var nspReader = new NintendoSubmissionPackageReader(nspFilePath))
            {
                return ReadContentMetaInNsp(nspReader);
            }
        }

        public static List<ContentMetaModel> ReadContentMetaInNsp(NintendoSubmissionPackageReader reader)
        {
            var list = new List<ContentMetaModel>();

            var metas = reader.ListFileInfo().FindAll(p => p.Item1.EndsWith(".cnmt.xml"));
            foreach (var meta in metas)
            {
                list.Add(ReadModel(reader, meta.Item1, meta.Item2));
            }
            return list;
        }
        private static CardSpecModel ReadCardSpecInNsp(NintendoSubmissionPackageReader reader)
        {
            var spec = reader.ListFileInfo().Find(p => p.Item1 == "cardspec.xml");
            return ReadXml<CardSpecModel>(reader, spec.Item1, spec.Item2);
        }
        private static ApplicationControlPropertyModel ReadApplicationControlPropertyInNsp(NintendoSubmissionPackageReader reader)
        {
            var spec = reader.ListFileInfo().Find(p => p.Item1.EndsWith(".nacp.xml"));
            return ReadXml<ApplicationControlPropertyModel>(reader, spec.Item1, spec.Item2);
        }

        private static void VerifySameApplicationSpecified(PatchContentMetaModel currentModel, PatchContentMetaModel previousModel)
        {
            var current = currentModel.HistoryList.Where(p => p.Type == NintendoContentMetaConstant.ContentMetaTypeApplication).Single();
            var previous = previousModel.HistoryList.Find(p => p.Type == NintendoContentMetaConstant.ContentMetaTypeApplication && p.Id == current.Id);

            bool success = previous != null;
            if (success)
            {
                success &= (current.Version == previous.Version);
                success &= (current.ContentList.Count == previous.ContentList.Count);
                foreach (var content in current.ContentList)
                {
                    success &= previous.ContentList.Any(p => p.Id == content.Id && p.Size == content.Size && p.Type == content.Type);
                }
            }
            if (!success)
            {
                throw new ArgumentException("Original application of previous is not same as that of current");
            }
        }

        static internal NintendoSubmissionPackageFileSystemInfo GetContentMetaMergedNspInfo(NintendoSubmissionPackageReader patch, NintendoSubmissionPackageReader merging, KeyConfiguration keyConfig)
        {
            var mergingMeta = ReadContentMetaInNsp(merging).Find(p => new List<string>{ NintendoContentMetaConstant.ContentMetaTypeApplication, NintendoContentMetaConstant.ContentMetaTypePatch, NintendoContentMetaConstant.ContentMetaTypeDelta }.Contains(p.Type));
            var keyGenerator = new NcaKeyGenerator(keyConfig);

            // コンテントメタ以外に特に修正は行わない
            GetContentInfoDelegate getContentInfoImpl = delegate (ContentModel contentInfo, string contentMetaType, UInt64 contentMetaId)
            {
                var content = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                var ncaName = contentInfo.Id + ".nca";
                using (var ncaReader = patch.OpenNintendoContentArchiveReader(ncaName, keyGenerator))
                {
                    content.SetSource(contentInfo.Type, ncaReader.GetRepresentProgramIdOffset(), ncaReader.GetKeyGeneration(), contentInfo.Id, new FileSystemArchvieFileSource(patch, ncaName));
                }
                return content;
            };

            GetContentMetaInfoDelegate getContentMetaInfoImpl = delegate (ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string contentMetaFileName, ContentMetaModel model)
            {
                var patchModel = model as PatchContentMetaModel;
                if(patchModel == null)
                {
                    throw new ArgumentException("Specified nsp is not patch");
                }

                switch(mergingMeta.Type)
                {
                    case NintendoContentMetaConstant.ContentMetaTypeApplication:
                        {
                            // history として、アプリケーションのコンテント情報を追加する
                            var history = new PatchHistoryModel();
                            history.Type = mergingMeta.Type;
                            history.Id = mergingMeta.Id;
                            history.Version = mergingMeta.Version.Value;
                            history.Digest = mergingMeta.Digest;
                            history.ContentList = mergingMeta.ContentList;

                            patchModel.HistoryList.Add(history);
                        }
                        break;
                     case NintendoContentMetaConstant.ContentMetaTypePatch:
                        {
                            var mergingPatchMetaModel = mergingMeta as PatchContentMetaModel;

                            // history として、アプリケーションのコンテント情報を追加する
                            var history = new PatchHistoryModel();
                            history.Type = mergingPatchMetaModel.Type;
                            history.Id = mergingPatchMetaModel.Id;
                            history.Version = mergingPatchMetaModel.Version.Value;
                            history.Digest = mergingPatchMetaModel.Digest;
                            history.ContentList = mergingPatchMetaModel.ContentList.Where(p => p.Type != NintendoContentMetaConstant.ContentTypeDeltaFragment).ToList();

                            patchModel.HistoryList.Add(history);

                            // パッチが持っている history を追加する
                            mergingPatchMetaModel.HistoryList.ForEach(p => patchModel.HistoryList.Add(p));

                            // パッチが持っている deltaHistory を追加する
                            mergingPatchMetaModel.DeltaHistoryList.ForEach(p => patchModel.DeltaHistoryList.Add(p));
                        }
                        break;
                    case NintendoContentMetaConstant.ContentMetaTypeDelta:
                        {
                            // delta の持っている情報を、delta に追加する
                            // TODO: meta の delta 情報、および fragment 情報を取り除く
                            // とりあえず delta だけ取り除いて、fragment は残しておく (index 保持のため)
                            var mergingDeltaMetaModel = mergingMeta as DeltaContentMetaModel;

                            var delta = new PatchDeltaModel();
                            delta.ContentList = mergingDeltaMetaModel.ContentList;

                            delta.Source = mergingDeltaMetaModel.Source;
                            delta.Destination = mergingDeltaMetaModel.Destination;
                            delta.FragmentSetList = mergingDeltaMetaModel.FragmentSetList.Where(p => p.FragmentTargetContentType != NintendoContentMetaConstant.ContentTypeMeta).ToList();

                            patchModel.DeltaList.Add(delta);
                            patchModel.DeltaHistoryList.Add(delta.GetHistory());
                        }
                        break;

                }
                patchModel.Sort();

                GetCommonContentMetaInfo(ref entry, contentMetaFileName, model, patch, keyConfig);

                // パッチはデフォルトでチケット有り
                entry.HasTicket = true;
            };

            var nspInfo = GetNspInfo(patch, keyConfig, getContentInfoImpl, getContentMetaInfoImpl);
            return nspInfo;
        }

        static internal NintendoSubmissionPackageFileSystemInfo GetDeltaMergedNspInfo(NintendoSubmissionPackageReader patch, NintendoSubmissionPackageReader delta, KeyConfiguration keyConfig)
        {
            var mergingMeta = ReadContentMetaInNsp(delta).Find(p => p.Type == NintendoContentMetaConstant.ContentMetaTypeDelta);
            var contentIds = new List<string>();
            var keyGenerator = new NcaKeyGenerator(keyConfig);

            GetContentInfoDelegate getContentInfoImpl = delegate (ContentModel contentInfo, string contentMetaType, UInt64 contentMetaId)
            {
                contentIds.Add(contentInfo.Id);
                var content = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                var ncaName = contentInfo.Id + ".nca";
                using (var ncaReader = patch.OpenNintendoContentArchiveReader(ncaName, keyGenerator))
                {
                    content.SetSource(contentInfo.Type, ncaReader.GetRepresentProgramIdOffset(), ncaReader.GetKeyGeneration(), contentInfo.Id, new FileSystemArchvieFileSource(patch, ncaName));
                }
                return content;
            };
            GetContentInfoDelegate getContentInfoOfDeltaImpl = delegate (ContentModel contentInfo, string contentMetaType, UInt64 contentMetaId)
            {
                var content = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                var ncaName = contentInfo.Id + ".nca";
                using (var ncaReader = delta.OpenNintendoContentArchiveReader(ncaName, keyGenerator))
                {
                    content.SetSource(contentInfo.Type, ncaReader.GetRepresentProgramIdOffset(), ncaReader.GetKeyGeneration(), contentInfo.Id, new FileSystemArchvieFileSource(delta, ncaName));
                }
                return content;
            };

            GetContentMetaInfoDelegate getContentMetaInfoImpl = delegate (ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string contentMetaFileName, ContentMetaModel model)
            {
                var patchModel = model as PatchContentMetaModel;
                if (patchModel == null)
                {
                    throw new ArgumentException("Specified nsp is not patch");
                }

                {
                    // delta の持っている情報を追加する
                    var mergingDeltaMetaModel = mergingMeta as DeltaContentMetaModel;

                    var deltaModel = new PatchDeltaModel();
                    deltaModel.ContentList = mergingDeltaMetaModel.ContentList;

                    deltaModel.Source = mergingDeltaMetaModel.Source;
                    deltaModel.Destination = mergingDeltaMetaModel.Destination;
                    deltaModel.FragmentSetList = mergingDeltaMetaModel.FragmentSetList.Where(p => p.FragmentTargetContentType != NintendoContentMetaConstant.ContentTypeMeta).ToList();

                    patchModel.DeltaList.Add(deltaModel);
                    patchModel.DeltaHistoryList.Add(deltaModel.GetHistory());
                }
                patchModel.Sort();

                GetCommonContentMetaInfo(ref entry, contentMetaFileName, model, patch, keyConfig);

                // パッチはデフォルトでチケット有り
                entry.HasTicket = true;
            };

            var nspInfo = GetNspInfo(patch, keyConfig, getContentInfoImpl, getContentMetaInfoImpl);
            nspInfo.SkipSizeCheck = true;

            // delta のコンテントを追加する、とりあえず Entries == 1 を前提
            foreach (var contentInfo in mergingMeta.ContentList)
            {
                if (contentInfo.Type == NintendoContentMetaConstant.ContentTypeMeta)
                {
                    continue;
                }
                if (contentIds.Contains(contentInfo.Id))
                {
                    continue;
                }
                var content = getContentInfoOfDeltaImpl(contentInfo, null, 0); // metaType, id は未使用
                nspInfo.Entries[0].Contents.Add(content);
                contentIds.Add(contentInfo.Id);
            }
            return nspInfo;
        }

        static internal void CheckSaveDataSize(ApplicationControlPropertyModel original, ApplicationControlPropertyModel current)
        {
            var orgUserAccountSaveDataSize        = Convert.ToUInt64(original.UserAccountSaveDataSize, 16);
            var orgUserAccountSaveDataJournalSize = Convert.ToUInt64(original.UserAccountSaveDataJournalSize, 16);
            var orgDeviceSaveDataSize             = Convert.ToUInt64(original.DeviceSaveDataSize, 16);
            var orgDeviceSaveDataJournalSize      = Convert.ToUInt64(original.DeviceSaveDataJournalSize, 16);
            var orgBcatDeliveryCacheStorageSize   = Convert.ToUInt64(original.BcatDeliveryCacheStorageSize, 16);
            var orgTemporaryStorageSize           = Convert.ToUInt64(original.TemporaryStorageSize, 16);
            var orgCacheStorageDataSize           = Convert.ToUInt64(original.CacheStorageSize, 16);
            var orgCacheStorageJournalSize        = Convert.ToUInt64(original.CacheStorageJournalSize, 16);
            var orgUserAccountSaveDataSizeMax     = Convert.ToUInt64(original.UserAccountSaveDataSizeMax, 16);
            var orgUserAccountSaveDataJournalSizeMax = Convert.ToUInt64(original.UserAccountSaveDataJournalSizeMax, 16);
            var orgCacheStorageDataAndJournalSizeMax = Convert.ToUInt64(original.CacheStorageDataAndJournalSizeMax, 16);
            var orgCacheStorageIndexMax           = Convert.ToUInt64(original.CacheStorageIndexMax, 16);

            var curUserAccountSaveDataSize        = Convert.ToUInt64(current.UserAccountSaveDataSize, 16);
            var curUserAccountSaveDataJournalSize = Convert.ToUInt64(current.UserAccountSaveDataJournalSize, 16);
            var curDeviceSaveDataSize             = Convert.ToUInt64(current.DeviceSaveDataSize, 16);
            var curDeviceSaveDataJournalSize      = Convert.ToUInt64(current.DeviceSaveDataJournalSize, 16);
            var curBcatDeliveryCacheStorageSize   = Convert.ToUInt64(current.BcatDeliveryCacheStorageSize, 16);
            var curTemporaryStorageSize           = Convert.ToUInt64(current.TemporaryStorageSize, 16);
            var curCacheStorageDataSize           = Convert.ToUInt64(current.CacheStorageSize, 16);
            var curCacheStorageJournalSize        = Convert.ToUInt64(current.CacheStorageJournalSize, 16);
            var curUserAccountSaveDataSizeMax     = Convert.ToUInt64(current.UserAccountSaveDataSizeMax, 16);
            var curUserAccountSaveDataJournalSizeMax = Convert.ToUInt64(current.UserAccountSaveDataJournalSizeMax, 16);
            var curCacheStorageDataAndJournalSizeMax = Convert.ToUInt64(current.CacheStorageDataAndJournalSizeMax, 16);
            var curCacheStorageIndexMax           = Convert.ToUInt64(current.CacheStorageIndexMax, 16);

            Func<ulong, ulong, ulong, ulong, bool> checkExtend = (orgSaveDataSize, orgSaveDataJournalSize, curSaveDataSize, curSaveDataJournalSize) =>
            {
                if (orgSaveDataSize > curSaveDataSize || orgSaveDataJournalSize > curSaveDataJournalSize)
                {
                    // size shrink
                    return false;
                }
                else if (orgSaveDataSize == curSaveDataSize && orgSaveDataJournalSize == curSaveDataJournalSize)
                {
                    // no change
                    return true;
                }
                else if (orgSaveDataSize == 0 && orgSaveDataJournalSize == 0)
                {
                    // new save
                    return true;
                }
                else
                {
                    // extend
                    const ulong ExtensionUnitSize = 1 * 1024 * 1024;
                    if (curSaveDataSize % ExtensionUnitSize != 0 || curSaveDataJournalSize % ExtensionUnitSize != 0)
                    {
                        // 拡張が一方だけであっても、データ領域・ジャーナル領域ともにサイズが 1 MiB の倍数であることを要求する
                        throw new ArgumentException("Save data extension requires that the data and journal size is multiple of 1 MiB.");
                    }
                    return true;
                }
            };

            Action<string, ulong, ulong> checkLE = (str, orgSize, curSize) =>
            {
                if (orgSize > curSize)
                {
                    throw new ArgumentException(string.Format("{0} can't be smaller than original.", str));
                }
            };

            if (original.SaveDataOwnerId != current.SaveDataOwnerId)
            {
                throw new ArgumentException("SaveDataOwnerId can't be changed by patch.");
            }

            bool saveCheck = true;
            saveCheck &= checkExtend(orgUserAccountSaveDataSize, orgUserAccountSaveDataJournalSize, curUserAccountSaveDataSize, curUserAccountSaveDataJournalSize);
            saveCheck &= checkExtend(orgDeviceSaveDataSize, orgDeviceSaveDataJournalSize, curDeviceSaveDataSize, curDeviceSaveDataJournalSize);
            saveCheck &= checkExtend(orgBcatDeliveryCacheStorageSize, 0, curBcatDeliveryCacheStorageSize, 0);

            if (!saveCheck)
            {
                throw new ArgumentException("Save data size shrink from original application is detected. Currently this feature is not supported.");
            }

            // キャッシュストレージからインデックス付きキャッシュストレージへの移行
            if (!checkExtend(orgCacheStorageDataSize, orgCacheStorageJournalSize, curCacheStorageDataSize, curCacheStorageJournalSize))
            {
                if (curCacheStorageDataSize != 0 || curCacheStorageJournalSize != 0 || curCacheStorageDataAndJournalSizeMax == 0)
                {
                    // 単なる縮小やキャッシュストレージの利用取りやめはできない
                    throw new ArgumentException("Save data size shrink from original application is detected. Currently this feature is not supported.");
                }

                if (curCacheStorageDataAndJournalSizeMax < orgCacheStorageDataSize + orgCacheStorageJournalSize)
                {
                    // 各インデックスのキャッシュストレージサイズの最大値が移行前のサイズを下回ることは許可しない
                    throw new ArgumentException("Switing to IndexedCacheStorage from CachcStorage requires that CacheStorageDataAndJournalSizeMax be larger than sum of CacheStorageSize and CacheStorageJournalSize for original application.");
                }
            }

            checkLE("TemporaryStorageSize", orgTemporaryStorageSize, curTemporaryStorageSize);
            checkLE("UserAccountSaveDataSizeMax", orgUserAccountSaveDataSizeMax, curUserAccountSaveDataSizeMax);
            checkLE("UserAccountSaveDataJournalSizeMax", orgUserAccountSaveDataJournalSizeMax, curUserAccountSaveDataJournalSizeMax);
            checkLE("CacheStorageDataAndJournalSizeMax", orgCacheStorageDataAndJournalSizeMax, curCacheStorageDataAndJournalSizeMax);
            checkLE("CacheStorageIndexMax", orgCacheStorageIndexMax, curCacheStorageIndexMax);
        }

        static internal NintendoSubmissionPackageFileSystemInfo GetPatchedNspInfo(NintendoSubmissionPackageReader originalNspReader, NintendoSubmissionPackageReader currentNspReader, NintendoSubmissionPackageReader previousNspReader, string metaFilePath, string descFilePath, KeyConfiguration keyConfig)
        {
            if (!ReadContentMetaInNsp(originalNspReader).All(p => p.Type == NintendoContentMetaConstant.ContentMetaTypeApplication))
            {
                throw new ArgumentException("Original is not Application");
            }
            if (!ReadContentMetaInNsp(currentNspReader).All(p => p.Type == NintendoContentMetaConstant.ContentMetaTypeApplication))
            {
                throw new ArgumentException("Current is not Application");
            }
            if (previousNspReader != null && !ReadContentMetaInNsp(previousNspReader).All(p => p.Type == NintendoContentMetaConstant.ContentMetaTypePatch))
            {
                throw new ArgumentException("Previous is not Patch");
            }

            CheckSaveDataSize(ReadApplicationControlPropertyInNsp(originalNspReader), ReadApplicationControlPropertyInNsp(currentNspReader));

            {
                var originalCardSpec = ReadCardSpecInNsp(originalNspReader);
                var currentCardSpec = ReadCardSpecInNsp(currentNspReader);

                // autoset かつ、カードサイズが異なる場合はここで warning
                if (currentCardSpec.AutoSetSize && originalCardSpec.Size != currentCardSpec.Size)
                {
                    Log.Warning("Card sizes differ between original nsp and current nsp.");
                }
                // autoset かつ、カード速度が異なる場合はここで warning
                if (currentCardSpec.AutoSetClockRate && originalCardSpec.ClockRate != currentCardSpec.ClockRate)
                {
                    Log.Warning("Card clock rates differ between original nsp and current nsp.");
                }
            }

            var sparseStorages = RetrieveSparseStorageStream(previousNspReader, keyConfig);

            var nsp = new NintendoSubmissionPackageComparer(originalNspReader, currentNspReader, keyConfig);

            UInt64? patchId = null;
            if (!string.IsNullOrEmpty(metaFilePath))
            {
                var contentMetaBase = new NintendoContentMetaBase(NintendoContentMetaConstant.ContentMetaTypePatch, metaFilePath);
                patchId = contentMetaBase.GetContentMetaDescryptor().Id;
            }

            GetContentInfoDelegate getContentInfoImpl = delegate (ContentModel contentInfo, string contentMetaType, UInt64 contentMetaId)
            {
                var currentNcaName = contentInfo.Id + ".nca";
                NintendoContentArchiveReader currentNcaReader = null;
                NintendoContentArchiveReader originalNcaReader = null;

                if (!nsp.FindOriginal(currentNcaName, out originalNcaReader, out currentNcaReader))
                {
                    if (currentNcaReader == null)
                    {
                        throw new FileNotFoundException("nca file not found.", currentNcaName);
                    }
                    // originalNcaReader は null を許す
                }

                var content = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                contentMetaType = NintendoContentMetaConstant.ContentMetaTypePatch;
                if (patchId != null)
                {
                    contentMetaId = patchId.Value;
                }
                else
                {
                    contentMetaId = IdConverter.ConvertToPatchId(contentMetaId);
                }

                // スパース情報の設定
                AttachSparseStorageStream(contentInfo.Type, sparseStorages, originalNcaReader);

                content.SetFsInfo(contentInfo.Type, null, GetPatchedNcaInfo(contentInfo.Type, originalNcaReader, currentNcaReader, descFilePath, contentMetaType, contentMetaId));
                return content;
            };

            GetContentMetaInfoDelegate getContentMetaInfoImpl = delegate (ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string contentMetaFileName, ContentMetaModel model)
            {
                GetCommonContentMetaInfo(ref entry, contentMetaFileName, model, currentNspReader, keyConfig);

                if (entry.MetaType != NintendoContentMetaConstant.ContentMetaTypeApplication)
                {
                    return;
                }

                NintendoContentMetaBase contentMetaBase;

                // アプリケーションメタからパッチメタにコンバート
                if (string.IsNullOrEmpty(metaFilePath))
                {
                    contentMetaBase = new NintendoContentMetaBase(NintendoContentMetaConstant.ContentMetaTypePatch, entry.ContentMetaInfo.Model);
                }
                // 指定のメタファイルを読み込む
                else
                {
                    contentMetaBase = new NintendoContentMetaBase(NintendoContentMetaConstant.ContentMetaTypePatch, metaFilePath);
                }

                // バージョンのチェック
                var originalMetaModel = ReadContentMetaInNsp(originalNspReader).Find(metaModel => metaModel.Type == NintendoContentMetaConstant.ContentMetaTypeApplication);
                if (originalMetaModel == null)
                {
                    throw new Exception("The application meta is not included in the original nsp.");
                }
                if (!originalMetaModel.Version.HasValue || contentMetaBase.GetContentMetaDescryptor().Version <= originalMetaModel.Version)
                {
                    throw new Exception(string.Format("The release version number ({0}) in the .nmeta file of current nsp file should be larger than that of original nsp ({1}).", contentMetaBase.GetContentMetaDescryptor().Version >> 16, originalMetaModel.Version >> 16));
                }

                var patchModel = (PatchContentMetaModel)(new NintendoContentMetaXmlSource(contentMetaBase, null, 0)).Model;
                patchModel.HistoryList = new List<PatchHistoryModel>();
                patchModel.DeltaHistoryList = new List<PatchDeltaHistoryModel>();
                patchModel.DeltaList = new List<PatchDeltaModel>();

                // history として、元アプリのコンテント情報を追加する
                var history = new PatchHistoryModel();
                history.Type = originalMetaModel.Type;
                history.Id = originalMetaModel.Id;
                history.Digest = originalMetaModel.Digest;
                history.Version = originalMetaModel.Version.Value;
                history.ContentList = originalMetaModel.ContentList;

                patchModel.HistoryList.Add(history);
                patchModel.Sort();

                // スパース情報をそのまま引き継ぎ
                patchModel.SparseStorages = sparseStorages;

                // パッチメタに置き換え
                entry.MetaType = NintendoContentMetaConstant.ContentMetaTypePatch;
                entry.MetaFilePath = null;
                entry.ContentMetaInfo = new NintendoSubmissionPackageFileSystemInfo.ContentMetaInfo(contentMetaBase.GetBytes(), patchModel);

                // パッチはデフォルトでチケット有り
                entry.HasTicket = true;
            };

            return GetNspInfo(currentNspReader, keyConfig, getContentInfoImpl, getContentMetaInfoImpl);
        }

        static internal NintendoSubmissionPackageFileSystemInfo GetIscPatchedNspInfo(NintendoSubmissionPackageReader previousNspReader, NintendoSubmissionPackageReader currentNspReader, string descFilePath, KeyConfiguration keyConfig)
        {
            if (!ReadContentMetaInNsp(previousNspReader).All(p => p.Type == NintendoContentMetaConstant.ContentMetaTypePatch))
            {
                throw new ArgumentException("Previous is not Patch");
            }
            if (!ReadContentMetaInNsp(currentNspReader).All(p => p.Type == NintendoContentMetaConstant.ContentMetaTypePatch))
            {
                // makepatch の時には、Current には Application を指定するが、最適化のタイミングでは Current は Patch である
                throw new ArgumentException("Current is not Patch");
            }

            CheckSaveDataSize(ReadApplicationControlPropertyInNsp(previousNspReader), ReadApplicationControlPropertyInNsp(currentNspReader));

            var nsp = new NintendoSubmissionPackageComparer(previousNspReader, currentNspReader, keyConfig);
            var keyGenerator = new NcaKeyGenerator(keyConfig);

            GetContentInfoDelegate getContentInfoImpl = delegate (ContentModel contentInfo, string contentMetaType, UInt64 contentMetaId)
            {
                var currentNcaName = contentInfo.Id + ".nca";
                NintendoContentArchiveReader currentNcaReader = null;
                NintendoContentArchiveReader previousNcaReader = null;

                if (!nsp.FindOriginal(currentNcaName, out previousNcaReader, out currentNcaReader))
                {
                    if (currentNcaReader == null)
                    {
                        throw new FileNotFoundException("nca file not found.", currentNcaName);
                    }
                    // previousNcaReader は null を許す
                }

                // 前 ver. パッチのプログラムコンテンツの鍵世代を取得
                // マルチプログラムアプリケーションのパッチで追加されたプログラムの場合はその鍵世代を利用
                var previousProgramNcaReader = GetSpecifiedContentArchiveReader(previousNspReader, contentMetaId, NintendoContentMetaConstant.ContentTypeProgram, contentInfo.IdOffset, keyGenerator);
                var keyGeneration = previousProgramNcaReader != null ? previousProgramNcaReader.GetKeyGeneration() : currentNcaReader.GetKeyGeneration();

                var content = new NintendoSubmissionPackageFileSystemInfo.ContentInfo();
                content.SetFsInfo(contentInfo.Type, null, GetIscPatchedNcaInfo(contentInfo.Type, previousNcaReader, currentNcaReader, descFilePath, previousNspReader.GetAesCtrGeneration(keyGenerator), contentMetaType, contentMetaId, keyGeneration));
                return content;
            };

            GetContentMetaInfoDelegate getContentMetaInfoImpl = delegate (ref NintendoSubmissionPackageFileSystemInfo.EntryInfo entry, string contentMetaFileName, ContentMetaModel model)
            {
                GetCommonContentMetaInfo(ref entry, contentMetaFileName, model, currentNspReader, keyConfig);
                System.Diagnostics.Debug.Assert(entry.MetaType != NintendoContentMetaConstant.ContentMetaTypeApplication);

                {
                    var patchModel = model as PatchContentMetaModel;
                    var mergingPatchModel = ReadContentMetaInNsp(previousNspReader).Find(p => new List<string> { NintendoContentMetaConstant.ContentMetaTypeApplication, NintendoContentMetaConstant.ContentMetaTypePatch, NintendoContentMetaConstant.ContentMetaTypeDelta }.Contains(p.Type)) as PatchContentMetaModel;

                    VerifySameApplicationSpecified(patchModel, mergingPatchModel);

                    // history を再設定
                    patchModel.HistoryList = new List<PatchHistoryModel>();
                    var history = new PatchHistoryModel();
                    history.Type = mergingPatchModel.Type;
                    history.Id = mergingPatchModel.Id;
                    history.Version = mergingPatchModel.Version.Value;
                    history.Digest = mergingPatchModel.Digest;
                    history.ContentList = mergingPatchModel.ContentList.Where(p => p.Type != NintendoContentMetaConstant.ContentTypeDeltaFragment).ToList();

                    patchModel.HistoryList.Add(history);

                    // パッチが持っている history を追加する
                    mergingPatchModel.HistoryList.ForEach(p => patchModel.HistoryList.Add(p));

                    // パッチが持っている deltaHistory を追加する
                    mergingPatchModel.DeltaHistoryList.ForEach(p => patchModel.DeltaHistoryList.Add(p));

                    patchModel.Sort();
                }

                // パッチはデフォルトでチケット有り
                entry.HasTicket = true;
            };

            return GetNspInfo(currentNspReader, keyConfig, getContentInfoImpl, getContentMetaInfoImpl);
        }

        static internal PartitionFileSystemInfo GetMergedNspInfo(List<NintendoSubmissionPackageReader> nspReaderList)
        {
            long offset = 0;

            var mergedNspInfo = new PartitionFileSystemInfo();
            mergedNspInfo.version = 0;

            foreach (var nspReader in nspReaderList)
            {
                foreach (var fileInfo in nspReader.ListFileInfo())
                {
                    var fileName = fileInfo.Item1;
                    var fileSize = fileInfo.Item2;

                    // ファイル名が同じものはスキップ（先にリストアップされたものが優先）
                    if (mergedNspInfo.entries.Where(x => x.name == fileName).Any())
                    {
                        continue;
                    }

                    var entry = PartitionFileSystemInfo.EntryInfo.Make(fileName, (ulong)fileSize, (ulong)offset);
                    entry.type = "source";
                    entry.sourceInterface = new CliCompatibleSource(new FileSystemArchvieFileSource(nspReader, fileName));

                    mergedNspInfo.entries.Add(entry);

                    offset += fileSize;
                }
            }

            return mergedNspInfo;
        }

        static internal PartitionFileSystemInfo GetRootEntryRemovedNspInfo(NintendoSubmissionPackageReader nspReader, List<string> regexStrings)
        {
            long offset = 0;

            var removedNspInfo = new PartitionFileSystemInfo();
            removedNspInfo.version = 0;

            foreach (var fileInfo in nspReader.ListFileInfo())
            {
                var fileName = fileInfo.Item1;
                var fileSize = fileInfo.Item2;

                if (regexStrings.Where(x => Regex.IsMatch(fileName, x)).Any())
                {
                    continue;
                }

                var entry = PartitionFileSystemInfo.EntryInfo.Make(fileName, (ulong)fileSize, (ulong)offset);
                entry.type = "source";
                entry.sourceInterface = new CliCompatibleSource(new FileSystemArchvieFileSource(nspReader, fileName));

                removedNspInfo.entries.Add(entry);

                offset += fileSize;
            }

            return removedNspInfo;
        }

        static public string GetSpecifiedContentId(NintendoSubmissionPackageReader nspReader, UInt64 contentMetaId, string contentType, byte idOffset)
        {
            foreach (var cnmtXml in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt.xml")))
            {
                var cnmtXmlModel = ReadXml<ContentMetaModel>(nspReader, cnmtXml.Item1, cnmtXml.Item2);
                if (cnmtXmlModel.GetUInt64Id() != contentMetaId)
                {
                    continue;
                }
                foreach (var contentInfo in cnmtXmlModel.ContentList.Where(x => x.Type == contentType && x.IdOffset == idOffset))
                {
                    return contentInfo.Id;
                }
            }
            return null;
        }

        static internal NintendoContentArchiveReader GetSpecifiedContentArchiveReader(NintendoSubmissionPackageReader nspReader, UInt64 contentMetaId, string contentType, byte idOffset, IKeyGenerator keyGenerator)
        {
            var contentId = GetSpecifiedContentId(nspReader, contentMetaId, contentType, idOffset);
            if (contentId == null)
            {
                return null;
            }
            var ncaReader = nspReader.OpenNintendoContentArchiveReader(contentId + ".nca", keyGenerator);
            TicketUtility.SetExternalKey(ref ncaReader, nspReader);
            return ncaReader;
        }

        static internal void CheckAcid(byte[] npdmData, string descFilePath)
        {
            Trace.Assert(npdmData != null);
            var acidBinary = NintendoContentAdfReader.GetAcidFromDesc(descFilePath);
            var npdmAcidBinary = NpdmParser.GetAcid(npdmData);
            if (!acidBinary.SequenceEqual(npdmAcidBinary) || acidBinary.Length != npdmAcidBinary.Length)
            {
                throw new ArgumentException("Wrong desc file. Acid in the desc file and main.npdm don't match.");
            }
        }

        static internal PartitionFileSystemInfo GetSplitNspInfo(NintendoSubmissionPackageReader nspReader, ulong contentMetaId, uint version)
        {
            long offset = 0;

            var splitNspInfo = new PartitionFileSystemInfo();
            splitNspInfo.version = 0;

            var contentIds = new List<string>();
            foreach (var cnmtXml in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt.xml")))
            {
                var cnmtXmlModel = ArchiveReconstructionUtils.ReadXml<ContentMetaModel>(nspReader, cnmtXml.Item1, cnmtXml.Item2);
                if (cnmtXmlModel.GetUInt64Id() != contentMetaId || cnmtXmlModel.Version != version)
                {
                    continue;
                }

                foreach (var contentInfo in cnmtXmlModel.ContentList)
                {
                    contentIds.Add(contentInfo.Id);
                }
            }

            if (contentIds.Count == 0)
            {
                throw new ArgumentException(string.Format("There is no content that matches contentMetaId: {0}, version: {1}", contentMetaId, version));
            }

            var rightsIdTextHalf = TicketUtility.CreateRightsIdText(contentMetaId, (byte)0).Substring(0, TicketUtility.RightsIdLength);

            foreach (var fileInfo in nspReader.ListFileInfo())
            {
                var fileName = fileInfo.Item1;
                var fileSize = fileInfo.Item2;

                // 抽出対象のファイルでない場合スキップ
                if (fileName != "authoringtoolinfo.xml" &&
                    fileName != "cardspec.xml" &&
                    !fileName.StartsWith(rightsIdTextHalf) &&
                    !contentIds.Any(x => fileName.StartsWith(x)))
                {
                    continue;
                }

                // ファイル名が同じものはスキップ（先にリストアップされたものが優先）
                if (splitNspInfo.entries.Where(x => x.name == fileName).Any())
                {
                    continue;
                }

                var entry = PartitionFileSystemInfo.EntryInfo.Make(fileName, (ulong)fileSize, (ulong)offset);
                entry.type = "source";
                entry.sourceInterface = new CliCompatibleSource(new FileSystemArchvieFileSource(nspReader, fileName));

                splitNspInfo.entries.Add(entry);

                offset += fileSize;
            }

            return splitNspInfo;
        }
    }

    public class NintendoSubmissionPackageComparer
    {
        public class ListItem
        {
            public string Name { get; private set; } = string.Empty;
            public string NcmContentType { get; private set; } = string.Empty;
            public byte IdOffset { get; private set; }

            public ListItem(string name, string ncmContentType, byte idOffset)
            {
                Name = name;
                NcmContentType = ncmContentType;
                IdOffset = idOffset;
            }
        }

        private NintendoSubmissionPackageReader m_originalReader;
        private NintendoSubmissionPackageReader m_currentReader;
        private KeyConfiguration m_keyConfig;
        private List<ListItem> m_originalList;
        private List<ListItem> m_currentList;

        public NintendoSubmissionPackageComparer(NintendoSubmissionPackageReader originalReader, NintendoSubmissionPackageReader currentReader, KeyConfiguration keyConfig)
        {
            m_originalReader = originalReader;
            m_currentReader = currentReader;
            m_keyConfig = keyConfig;
            m_originalList = EnumerateListItem(originalReader);
            m_currentList = EnumerateListItem(currentReader);
        }

        private static List<ListItem> EnumerateListItem(NintendoSubmissionPackageReader reader)
        {
            if (reader == null)
            {
                return null;
            }
            var originalCnmt = reader.ListFileInfo().Find(t => t.Item1.EndsWith(".cnmt.xml"));
            var originalMeta = ArchiveReconstructionUtils.ReadModel(reader, originalCnmt.Item1, originalCnmt.Item2);
            var list = new List<ListItem>();
            reader.ListFileInfo().ForEach(tuple =>
            {
                if (Path.GetExtension(tuple.Item1) == ".nca")
                {
                    var content = originalMeta.ContentList.Find(m => m.Id == tuple.Item1.Substring(0, tuple.Item1.IndexOf(".")));
                    if (content != null)
                    {
                        list.Add(new ListItem(tuple.Item1, content.Type, content.IdOffset));
                    }
                }
            });
            return list;
        }

        public bool FindOriginal(string currentFileName, out NintendoContentArchiveReader originalNcaReader, out NintendoContentArchiveReader currentNcaReader)
        {
            originalNcaReader = null;
            currentNcaReader = null;

            var currentContent = m_currentList.Find(item => item.Name == currentFileName);
            if (currentContent == null)
            {
                return false;
            }

            var keyGenerator = new NcaKeyGenerator(m_keyConfig);

            currentNcaReader = m_currentReader.OpenNintendoContentArchiveReader(currentFileName, keyGenerator);
            TicketUtility.SetExternalKey(ref currentNcaReader, m_currentReader);

            if (currentContent.NcmContentType != null)
            {
                // パッチ化しないものは original は調査しない
                if (!ArchiveReconstructionUtils.IsPatchTargetNca(currentContent.NcmContentType))
                {
                    return true;
                }
            }

            // 順番に検索
            foreach (var originalContent in m_originalList)
            {
                if (currentContent.NcmContentType == originalContent.NcmContentType && currentContent.IdOffset == originalContent.IdOffset)
                {
                    var originalFileName = originalContent.Name;
                    Console.WriteLine($"Pair: {currentContent.Name} - {originalContent.Name}");
                    originalNcaReader = m_originalReader.OpenNintendoContentArchiveReader(originalFileName, keyGenerator);
                    TicketUtility.SetExternalKey(ref originalNcaReader, m_originalReader);
                    return true;
                }
            }

            return false;
        }

        public bool FindPatch(string originalFileName, out NintendoContentArchiveReader originalNcaReader, out NintendoContentArchiveReader patchNcaReader)
        {
            originalNcaReader = null;
            patchNcaReader = null;

            var originalContent = m_originalList.Find(item => item.Name == originalFileName);
            if (originalContent == null)
            {
                return false;
            }

            var keyGenerator = new NcaKeyGenerator(m_keyConfig);

            originalNcaReader = m_originalReader.OpenNintendoContentArchiveReader(originalFileName, keyGenerator);
            TicketUtility.SetExternalKey(ref originalNcaReader, m_originalReader);

            // 順番に検索
            foreach (var patchContent in m_currentList)
            {
                if (originalContent.NcmContentType == patchContent.NcmContentType && originalContent.IdOffset == patchContent.IdOffset)
                {
                    var patchFileName = patchContent.Name;
                    Console.WriteLine($"Pair: {patchContent.Name} - {originalContent.Name}");
                    patchNcaReader = m_currentReader.OpenNintendoContentArchiveReader(patchFileName, keyGenerator);
                    TicketUtility.SetExternalKey(ref patchNcaReader, m_currentReader);
                    return true;
                }
            }

            return false;
        }
    }
}
