﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "InitialImage/systemInitializer_InitialImage.h"
#include "Utility/systemInitializer_IBlockStorage.h"
#include "Utility/systemInitializer_IFile.h"
#include "Partition/systemInitializer_Partition.h"
#include <vector>
#include <Partition/systemInitializer_GptHolder.h>

struct SystemIntializerOption
{
    bool enableOverwritingProductionInfo;
    bool enableEraseEmmc;

    static SystemIntializerOption Make(bool enableOverwritingProductionInfo)
    {
        SystemIntializerOption option = {enableOverwritingProductionInfo};
        return option;
    }

    static SystemIntializerOption Make(bool enableOverwritingProductionInfo, bool enableEraseEmmc)
    {
        SystemIntializerOption option = {enableOverwritingProductionInfo, enableEraseEmmc};
        return option;
    }
};

class SystemInitializer
{
public:
    SystemInitializer() : m_IsInitialized(false) {}

    nn::Result Initialize(IBlockStorage *pBlockStorage, IFile *ImageFile, SystemIntializerOption option);

    nn::Result InitializeSystem();

    nn::Result PrintPartitionInfo();

private:
    nn::Result ProcessCommands(const InitialImageCommandPartition &commandPartition);
    nn::Result ProcessCommand(const InitialImageCommand &command);
    nn::Result WritePartitionTable(int64_t mbrIndex, int64_t gptIndex, int64_t gptPartitionIndex, int64_t partitionAddresses);
    nn::Result WritePartitionImage(int64_t destinationGptIndex, int64_t sourceIndex, int64_t offset);
    nn::Result WriteSparsePartitionImage(int64_t destinationGptIndex, int64_t sourceIndex, int64_t offset);
    nn::Result WriteFsPartitionImage(const char* destitinationPartitionName, int64_t sourceIndex, int64_t offset);
    nn::Result WriteFsSparsePartitionImage(const char* destitinationPartitionName, int64_t sourceIndex, int64_t offset);
    nn::Result WriteFsPartition(const char* partitionName, int64_t offset, void* buffer, size_t bufferSize);
    nn::Result WriteBatchedBootPartitions(const char* bctPartitionName, int64_t sourceBctIndex, int64_t bctOffset, const char* blPartitionName, int64_t sourceBlIndex, int64_t blOffset, int64_t secureInfoIndex);
    nn::Result EraseEmmcWithEscapingData(int64_t targetPartition1, int64_t targetPartition2);
    nn::Result InitializeZero(const char* targetPartitionName);
    nn::Result EnableUpdatingFromBootImagePackage(int64_t target);
    nn::Result WriteDatabase(const char* destitinationPartitionName, int64_t sourceIndex);


    nn::Result ReadMbrFromStorage(Mbr *pOut);
    nn::Result ReadPrimaryGptFromStorage(GptHeader *pOut);
    nn::Result ReadAlternateGptFromStorage(GptHeader *pOut);
    nn::Result ReadGptPartitionsFromStorage(GptPartitionEntry *pOutArray, int numOutArray, const GptHeader &gptHeader);

    nn::Result EnsureProtectProductionInfo(bool allowOverwritingProductionInfo, int64_t gptIndex, int64_t gptPartitionIndex, int64_t partitionAddresses);
    nn::Result HasCommandsWriteProductionInfo(bool *pOut, const InitialImageCommandPartition &commandPartition, GptHolder *pOldGpt, std::vector<int> *pOldProductionInfoPartitionIndexes, GptHolder *pNewGpt, std::vector<int> *pNewProductionInfoPartitionIndexes, std::vector<PartitionAddress> *pPartitionAddresses);
    nn::Result HasUpdateProductionInfoPartition(bool *pOut, GptHolder *pOldGpt, std::vector<int> *pOldProductionInfoPartitionIndexes, GptHolder *pNewGpt, std::vector<int> *pNewProductionInfoPartitionIndexes, std::vector<PartitionAddress> *pPartitionAddresses);
    nn::Result ReadGptFromImage(GptHolder *pOut, int64_t gptIndex, int64_t gptPartitionIndex);
    nn::Result UpdateSecureInfo(int64_t index);
    nn::Result UpdateSecureInfo(void *pOutBctBuffer);
    nn::Result UpdateSecureInfo(void *pOutBctBuffer, nn::fs::IStorage *pStorage);
    nn::Result UpdateSecureInfo(void *pOutBctBuffer, void *pSourceEksBuffer);
    nn::Result ReadBct(void* bct, size_t size, nn::fs::IStorage *pStorage, int64_t index);
    nn::Result ReadEks(void* eks, size_t size, nn::fs::IStorage *pStorage, int64_t index);
    nn::Result ReadBootLoaderVersion(int32_t *pOut, void* buffer, size_t size);
    int32_t CalculateEksIndex(int32_t bootLoaderVersion);
    nn::Result WriteBct(nn::fs::IStorage *pStorage, int64_t index, void *buffer, size_t size);

    void GetProductionInfoPartitionIndexes(std::vector<int> *pOut, GptHolder *pGpt);

    nn::Result ReadPartitionAddresses(std::vector<PartitionAddress> *pOut, int64_t partitionAddressIndex);

    bool VerifyGptPartition(const GptHeader &gptHeader, const GptPartitionEntry partitions[]);

    bool m_IsInitialized;
    IBlockStorage *m_pBlockStorage;
    IFile *m_pImageFile;
    InitialImage m_InitialImage;
    SystemIntializerOption m_Option;
    GptHolder m_NewGpt;
};

nn::Result WriteInitialImageFromFile(IBlockStorage *pBlockStorage, IFile *pImageFile, SystemIntializerOption option);
