﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "systemInitializer_StorageUtility.h"
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>

class BlockStorageFromIStorage : public IBlockStorage
{
public:
    explicit BlockStorageFromIStorage(std::unique_ptr<nn::fs::IStorage> pFsStorage) : m_pFsStorage(std::move(pFsStorage)) {}
    virtual ~BlockStorageFromIStorage() {}
    virtual uint64_t GetTotalSize();
    virtual uint64_t GetBlockSize();
    virtual uint64_t GetTotalBlocks();
    virtual nn::Result Write(uint64_t blockAddress, uint64_t numBlocks, const void* pData, size_t dataSize);
    virtual nn::Result Read(void* pOutBuffer, size_t bufferSize, uint64_t blockAddress, uint64_t numBlocks);
private:
    std::unique_ptr<nn::fs::IStorage> m_pFsStorage;
};

uint64_t BlockStorageFromIStorage::GetBlockSize()
{
    return 512;
}

uint64_t BlockStorageFromIStorage::GetTotalBlocks()
{
    int64_t totalSize;
    NN_ABORT_UNLESS(m_pFsStorage->GetSize(&totalSize).IsSuccess());
    return totalSize / GetBlockSize();
}

uint64_t BlockStorageFromIStorage::GetTotalSize()
{
    int64_t totalSize;
    NN_ABORT_UNLESS(m_pFsStorage->GetSize(&totalSize).IsSuccess());
    return totalSize;
}

nn::Result BlockStorageFromIStorage::Write(uint64_t blockAddress, uint64_t numBlocks, const void* pData, size_t dataSize)
{
    NN_UTILTOOL_LOG_DEBUG("Write: address=%016llx, size=%zd", static_cast<uint32_t>(blockAddress) * GetBlockSize(), dataSize);
    NN_UTILTOOL_RESULT_DO(
        m_pFsStorage->Write(static_cast<uint32_t>(blockAddress) * GetBlockSize(), pData, dataSize));

    NN_RESULT_SUCCESS;
}

nn::Result BlockStorageFromIStorage::Read(void* pOutBuffer, size_t bufferSize, uint64_t blockAddress, uint64_t numBlocks)
{
    NN_UTILTOOL_RESULT_DO(
        m_pFsStorage->Read(static_cast<uint32_t>(blockAddress) * GetBlockSize(), pOutBuffer, bufferSize));

    NN_RESULT_SUCCESS;
}

std::unique_ptr<IBlockStorage> MakeBlockStorageFrom(std::unique_ptr<nn::fs::IStorage> storage)
{
    return std::unique_ptr<IBlockStorage>(new BlockStorageFromIStorage(std::move(storage)));
}

class SubStorage : public nn::fs::IStorage
{
    NN_DISALLOW_COPY(SubStorage);

private:
    std::unique_ptr<nn::fs::IStorage> m_pBaseStorage;
    const int64_t m_Offset;
    const int64_t m_Size;

public:
    SubStorage(std::unique_ptr<nn::fs::IStorage> pBaseStorage, int64_t offset, int64_t size) NN_NOEXCEPT
        : m_pBaseStorage(std::move(pBaseStorage))
        , m_Offset(offset)
        , m_Size(size)
    {
    }

    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( offset < 0 || m_Size < static_cast<int64_t>(offset + size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        return m_pBaseStorage->Read(m_Offset + offset, buffer, size);
    }

    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( offset < 0 || m_Size < static_cast<int64_t>(offset + size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        return m_pBaseStorage->Write(m_Offset + offset, buffer, size);
    }

    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->Flush();
    }

    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        *outValue = m_Size;
        NN_RESULT_SUCCESS;
    }

};

std::unique_ptr<nn::fs::IStorage> MakeSubBlockStorage(std::unique_ptr<nn::fs::IStorage> storage, int64_t offset, int64_t size)
{
    return std::unique_ptr<nn::fs::IStorage>(new SubStorage(std::move(storage), offset, size));
}

