﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Utility/systemInitializer_SaveData.h"
#include "InitialImage/systemInitializer_InitialImage.h"

#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/fs.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_AccessLogPrivate.h>

#include <nn/ncm/ncm_ContentManagerSaveDataInfo.h>
#include <ns_Config.h>

#include <string>

namespace
{
    struct SaveDataInfo
    {
        nn::fs::SystemSaveDataId saveDataId;
        nn::fs::UserId userId;
        uint64_t saveDataOwnerId;
        int64_t saveDataSize;
        int64_t journalSize;
        int64_t saveDataFlag;
    };

    const SaveDataInfo SupportedSaveDataInfos[] =
    {
        {
            nn::ncm::BuildInSystemSystemSaveDataId,
            nn::fs::InvalidUserId,
            0x0000000000000000,  // ncm の programId(0x0100000000000002) と ownerId(0x0000000000000000) は違う(SIGLO-76586)
            nn::ncm::BuildInSystemSystemSaveDataSize,
            nn::ncm::BuildInSystemSystemSaveDataJournalSize,
            nn::ncm::BuildInSystemSystemSaveDataFlag
        },
        {
            nn::ncm::BuildInUserSystemSaveDataId,
            nn::fs::InvalidUserId,
            0x0000000000000000,  // ncm の programId(0x0100000000000002) と ownerId(0x0000000000000000) は違う(SIGLO-76586)
            nn::ncm::BuildInUserSystemSaveDataSize,
            nn::ncm::BuildInUserSystemSaveDataJournalSize,
            nn::ncm::BuildInUserSystemSaveDataFlag
        },
        {
            nn::ns::srv::ApplicationRecordSaveDataId,
            nn::fs::InvalidUserId,
            0x010000000000001F,
            nn::ns::srv::ApplicationRecordSaveDataSize,
            nn::ns::srv::ApplicationRecordSaveDataJournalSize,
            nn::ns::srv::ApplicationRecordSaveDataFlag
        },
    };

    const size_t NumSupportedSaveDataInfos = sizeof(SupportedSaveDataInfos) / sizeof(SupportedSaveDataInfos[0]);

    bool HasSaveDataInfo(nn::fs::SystemSaveDataId saveDataId)
    {
        for (int i = 0; i < NumSupportedSaveDataInfos; i++)
        {
            if (SupportedSaveDataInfos[i].saveDataId == saveDataId)
            {
                return true;
            }
        }

        return false;
    }

    const SaveDataInfo &GetSaveDataInfo(nn::fs::SystemSaveDataId saveDataId)
    {
        for (int i = 0; i < NumSupportedSaveDataInfos; i++)
        {
            if (SupportedSaveDataInfos[i].saveDataId == saveDataId)
            {
                return SupportedSaveDataInfos[i];
            }
        }

        NN_ABORT("Found no saveDataInfo");
    }
}

struct SavedFileInfo
{
    static const size_t PathSize = 48;

    uint64_t saveId;
    int64_t fileSize;
    char filePath[PathSize];
};

class SaveDataArchiveExtractor final
{
public:
    SaveDataArchiveExtractor(InitialImage *pInitialImage, nn::fs::SaveDataSpaceId, int64_t sourceIndex);
    nn::Result Extract();
private:

    nn::Result Read(void *buffer, size_t size);
    nn::Result EnsureMountSaveData(const char* mountName, nn::fs::SaveDataSpaceId spaceId, uint64_t saveId);
    std::string GetDirectoryName(std::string path);
    nn::Result EnsureCreateDirectory(std::string path, int limit = 2);
    nn::Result EnsureEmptyFile(std::string path, int64_t fileSize);
    nn::Result WriteFile(std::string destPath, int64_t fileSize);

    template<typename ValueType>
    nn::Result ReadValue(ValueType *pOut);

    InitialImage *m_pInitialImage;
    InitialImageInfo::PartitionInfo m_Info;
    int64_t m_Offset;
    const int64_t SourceIndex;
    const nn::fs::SaveDataSpaceId TargetStorage;
};

SaveDataArchiveExtractor::SaveDataArchiveExtractor(InitialImage *pInitialImage, nn::fs::SaveDataSpaceId targetStorage, int64_t sourceIndex)
    : m_pInitialImage(pInitialImage),
    m_Offset(0),
    SourceIndex(sourceIndex),
    TargetStorage(targetStorage)
{
    m_Info = m_pInitialImage->GetPartitionInfo(SourceIndex);
}

nn::Result SaveDataArchiveExtractor::Extract()
{
    int32_t numArchives;

    NN_UTILTOOL_RESULT_DO(
        ReadValue(&numArchives));

    NN_UTILTOOL_LOG_DEBUG("Number of Savedata Files: %lld", numArchives);

    for (int i = 0; i < numArchives; i++)
    {
        SavedFileInfo info;

        NN_UTILTOOL_RESULT_DO(
            ReadValue(&info));

        NN_UTILTOOL_LOG_DEBUG("SavedFile[%d]", i);
        NN_UTILTOOL_LOG_DEBUG("  storageId: %d", TargetStorage);
        NN_UTILTOOL_LOG_DEBUG("  id:   0x%016llx", info.saveId);
        NN_UTILTOOL_LOG_DEBUG("  size: %lld bytes", info.fileSize);
        NN_UTILTOOL_LOG_DEBUG("  path: %s", info.filePath);

        const char* SystemSaveMountName = "tmpsys";

        NN_UTILTOOL_RESULT_DO(
            EnsureMountSaveData(SystemSaveMountName, TargetStorage, info.saveId));

        NN_UTIL_SCOPE_EXIT{
            nn::fs::Unmount(SystemSaveMountName);
        };

        std::string destPath = std::string("tmpsys:/") + info.filePath;

        NN_UTILTOOL_LOG_DEBUG("ensure directory for %s", destPath.c_str());

        NN_UTILTOOL_RESULT_DO(
            EnsureCreateDirectory(GetDirectoryName(destPath)));

        NN_UTILTOOL_LOG_DEBUG("write file to %s", destPath.c_str());

        int64_t expectedOffset = m_Offset + info.fileSize;

        NN_UTILTOOL_RESULT_DO(
            WriteFile(destPath, info.fileSize));

        NN_ABORT_UNLESS(expectedOffset == m_Offset);

        NN_UTILTOOL_RESULT_DO(
            nn::fs::CommitSaveData(SystemSaveMountName));
    }

    NN_UTILTOOL_LOG_DEBUG("Success to extract savedata archive");

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataArchiveExtractor::EnsureMountSaveData(const char* mountName, nn::fs::SaveDataSpaceId spaceId, uint64_t saveId)
{
    NN_ABORT_UNLESS(HasSaveDataInfo(saveId), "Unsupported SaveData");

    auto result = nn::fs::MountSystemSaveData(
        mountName,
        spaceId,
        saveId);

    if (result <= nn::fs::ResultTargetNotFound())
    {
        auto saveDataInfo = GetSaveDataInfo(saveId);

        NN_UTILTOOL_RESULT_DO(
            nn::fs::CreateSystemSaveData(
                spaceId,
                saveId,
                saveDataInfo.userId,
                saveDataInfo.saveDataOwnerId,
                saveDataInfo.saveDataSize,
                saveDataInfo.journalSize,
                saveDataInfo.saveDataFlag));

        NN_UTILTOOL_RESULT_DO(
            nn::fs::MountSystemSaveData(
                mountName,
                spaceId,
                saveId));
    }
    else if (result.IsFailure())
    {
        NN_UTILTOOL_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataArchiveExtractor::Read(void *outBuffer, size_t size)
{
    size_t readSize;
    NN_UTILTOOL_RESULT_DO(
        m_pInitialImage->ReadPartition(&readSize, SourceIndex, m_Offset, outBuffer, size));

    NN_ABORT_UNLESS_EQUAL(readSize, size);

    m_Offset += readSize;

    NN_RESULT_SUCCESS;
}

template<typename ValueType>
nn::Result SaveDataArchiveExtractor::ReadValue(ValueType *pOut)
{
    NN_UTILTOOL_RESULT_DO(
        Read(pOut, sizeof(ValueType)));

    NN_RESULT_SUCCESS;
}

std::string SaveDataArchiveExtractor::GetDirectoryName(std::string path)
{
    auto slash = path.rfind('/');
    auto backSlash = path.rfind('\\');

    NN_ABORT_UNLESS(slash != std::string::npos || backSlash != std::string::npos);

    if (slash == std::string::npos)
    {
        return path.substr(0, backSlash);
    }
    else if (backSlash == std::string::npos)
    {
        return path.substr(0, slash);
    }
    else
    {
        return path.substr(0, std::max(slash, backSlash));
    }
}

nn::Result SaveDataArchiveExtractor::EnsureCreateDirectory(std::string path, int limit)
{
    NN_ABORT_UNLESS(0 < limit);

    if (path.length() != 0 && path[path.length() - 1] == ':')
    {
        NN_RESULT_SUCCESS;
    }

    nn::fs::DirectoryEntryType entryType;
    auto result = nn::fs::GetEntryType(&entryType, path.c_str());
    if (result <= nn::fs::ResultPathNotFound())
    {
        NN_UTILTOOL_RESULT_DO(
            EnsureCreateDirectory(GetDirectoryName(path), limit - 1));
        NN_UTILTOOL_RESULT_DO(nn::fs::CreateDirectory(path.c_str()));
    }
    else if (result.IsFailure())
    {
        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataArchiveExtractor::WriteFile(std::string destPath, int64_t fileSize)
{
    nn::fs::FileHandle file;

    NN_UTILTOOL_RESULT_DO(
        EnsureEmptyFile(destPath, fileSize));

    NN_UTILTOOL_RESULT_DO(nn::fs::OpenFile(&file, destPath.c_str(), nn::fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT{
        nn::fs::CloseFile(file);
    };

    int64_t restSize = fileSize;
    int64_t offset = 0;
    const int64_t BatchSize = 1024;
    static uint8_t buffer[BatchSize];

    while (0 < restSize)
    {
        int64_t currentWriteSize = std::min(restSize, BatchSize);

        NN_UTILTOOL_RESULT_DO(
            Read(buffer, currentWriteSize));
        NN_UTILTOOL_RESULT_DO(
            nn::fs::WriteFile(file, offset, buffer, currentWriteSize, nn::fs::WriteOption::MakeValue(0)));

        offset += currentWriteSize;
        restSize -= currentWriteSize;
    }

    NN_UTILTOOL_RESULT_DO(
        nn::fs::FlushFile(file));

    NN_RESULT_SUCCESS;
}

nn::Result SaveDataArchiveExtractor::EnsureEmptyFile(std::string path, int64_t fileSize)
{
    nn::fs::DirectoryEntryType entryType;
    auto result = nn::fs::GetEntryType(&entryType, path.c_str());
    if (result.IsSuccess() && entryType == nn::fs::DirectoryEntryType_File)
    {
        NN_UTILTOOL_RESULT_DO(
            nn::fs::DeleteFile(path.c_str()));
    }

    NN_UTILTOOL_RESULT_DO(nn::fs::CreateFile(path.c_str(), fileSize));

    NN_RESULT_SUCCESS;
}

nn::Result WriteNcmDatabase(InitialImage *pInitialImage, nn::fs::SaveDataSpaceId targetStorage, int64_t sourceIndex)
{
    SaveDataArchiveExtractor extractor(pInitialImage, targetStorage, sourceIndex);

    NN_UTILTOOL_RESULT_DO(
        extractor.Extract());

    NN_RESULT_SUCCESS;
}
