﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

const size_t CRC_TABLE_SIZE = 256;

//----------------------------------------------------------------------------
// CRC-32
//----------------------------------------------------------------------------

//! @brief    CRC-32 を計算するためのクラスです。
//!
//! SystemInitializer で必要なため一時的に SystemInitializer 内に配置します。
//! 次の段階で nn::util 名前空間等に移動します。
//!
class Crc32
{

public:

    //! @brief  デフォルトコンストラクタです。
    Crc32();

    //! @brief CRC-32 ハッシュ計算を行うためのコンテキストを初期化します。
    //! @param[in]    context    コンテキストの初期値。通常はデフォルト値を使います。
    void InitializeContext(uint32_t context = CRC32_STANDARD_INIT);

    //! @brief CRC-32 ハッシュ値を入力データで更新します。
    //! @param[in]    input    入力データへのポインタ
    //! @param[in]    length   入力データのサイズ
    void Update(const void* input, size_t length);

    //! @brief 最終的な CRC-32 ハッシュ計算結果を取得します。
    //! @return 計算結果
    uint32_t GetHash();

    /*!
        @brief

        CRC-32 を計算するための一連の関数呼び出しをまとめて行い、計算結果を取得します。

        @param[in]    input   入力データへのポインタ
        @param[in]    length  入力データのサイズ
        @param[in]    context    コンテキストの初期値。通常はデフォルト値を使います。

        @return 計算結果
        */
    static uint32_t Calculate(const void* input, size_t length, uint32_t context = CRC32_STANDARD_INIT);

private:
    //! @brief 生成多項式です。
    static const uint32_t CRC32_STANDARD_POLY = 0xedb88320;  // ビット反転するものは生成多項式も反転

    //! @brief コンテキストの初期値です。
    static const uint32_t CRC32_STANDARD_INIT = 0xffffffff;

    //! @brief ハッシュ計算を行うためのコンテキスト構造体です。
    typedef uint32_t Context;

    //! @brief ハッシュ計算を行うためのテーブルの構造体です。
    struct Table
    {
        uint32_t      table[CRC_TABLE_SIZE];
    };

    //! @brief ハッシュ計算を行うためのコンテキストです。
    Context m_Context;

    //! @brief ハッシュ計算を行うためのテーブルです。
    Table   m_Table;


    //! @brief ハッシュ計算を行うためのテーブルを初期化します。
    //! @param[in]    poly    テーブルを初期化するための生成多項式
    void InitializeTable(uint32_t poly);

};
