﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_Bis.h>
#include <memory>
#include <vector>
#include <string>

struct MbrPartitionRecord
{
    uint8_t bootIndicator;
    uint8_t startingChs[3];
    uint8_t osType;
    uint8_t endingChs[3];
    uint8_t startingLba[4];
    uint8_t sizeInLba[4];
};

struct Mbr
{
    static const int NUM_PARTITIONS = 4;

    uint8_t bootCode[440];
    char diskSignature[4];
    uint8_t unused[2];
    MbrPartitionRecord partitions[NUM_PARTITIONS];
    uint8_t signature[2];
};

struct Guid
{
    uint32_t data1;
    uint16_t data2;
    uint16_t data3;
    uint8_t data4[8];

    static Guid Make(
        uint32_t data1,
        uint16_t data2,
        uint16_t data3,
        uint8_t data4_0,
        uint8_t data4_1,
        uint8_t data4_2,
        uint8_t data4_3,
        uint8_t data4_4,
        uint8_t data4_5,
        uint8_t data4_6,
        uint8_t data4_7)
    {
        Guid ret = {data1, data2, data3, {data4_0, data4_1, data4_2, data4_3, data4_4, data4_5, data4_6, data4_7}};
        return ret;
    }

    static bool IsEqual(const Guid &a, const Guid &b)
    {
        return std::memcmp(&a, &b, sizeof(a)) == 0;
    }
};

struct GptHeader
{
    char signature[8];
    uint32_t revision;
    uint32_t headerSize;
    uint32_t headerCrc32;
    uint32_t reserved;
    int64_t myLba;
    int64_t alternateLba;
    int64_t firstUsableLba;
    int64_t lastUsableLba;
    Guid diskGuid;
    int64_t partitionEntryLba;
    int32_t numberOfPartitionEntries;
    int32_t sizeOfPartitionEntry;
    uint32_t partitionEntryArrayCrc32;
    uint8_t reserved2[512 - 92];
};
NN_STATIC_ASSERT(sizeof(GptHeader) == 512);


struct GptPartitionEntry
{
    Guid partitionTypeGuid;
    Guid uniquePartitionGuid;
    int64_t startingLba;
    int64_t endingLba;
    uint64_t attributes;
    uint16_t partitionName[36];

    int64_t GetBlocks() { return endingLba - startingLba; }
};
NN_STATIC_ASSERT(sizeof(GptPartitionEntry) == 128);

struct PartitionAddress
{
    enum AddressType
    {
        AddressType_Absolute,
        AddressType_Relative,
        AddressType_Keep
    };

    int type;
    int basePartitionIndex;
    int64_t address;
    int64_t size;
};

const int GPT_DEFAULT_PARTITION_ENTRIES = 128;

class IBlockStorage;

nn::Result ReadMbrFromStorage(Mbr *pOut, IBlockStorage *pStorage);
nn::Result ReadGptFromStorage(GptHeader *pOut, IBlockStorage *pStorage, uint64_t lba = 1);
nn::Result ReadGptFromStorage(GptHeader *pOut, nn::fs::IStorage *pStorage, int64_t address = 512);
nn::Result ReadGptPartitionsFromStorage(GptPartitionEntry *pOutArray, int numOutArray, const GptHeader &gptHeader, IBlockStorage *pStorage);
nn::Result ReadGptPartitionsFromStorage(GptPartitionEntry *pOutArray, int numOutArray, const GptHeader &gptHeader, nn::fs::IStorage *pStorage);

uint64_t CalculatePartitionSize(const GptPartitionEntry &partitionEntry, IBlockStorage *pStorage);
std::string GetPartitionName(const GptPartitionEntry &partitionEntry);
uint64_t CalculateNumberOfSectors(uint64_t bytes, uint64_t sectorSize);
void UpdateCrc32InGptHeader(GptHeader *pInOut, const GptPartitionEntry *pPartitions);
void ReCalculateLba(GptHeader *pGpt, GptPartitionEntry *pPartitions, uint64_t firstUsableLba, uint64_t lastUsableLba, uint64_t lastLba, uint64_t sectorSize);
void CalculatePartitionAddress(GptHeader *pOutGpt, GptPartitionEntry *pOutPartitions, GptHeader *pPreviousGpt, GptPartitionEntry *pPreviousPartitions, int numPartitionAddress, PartitionAddress *pPartitionAddresses, uint64_t storageSize, uint64_t sectorSize);
void MakeAlternateGpt(GptHeader *pOutHeader, const GptHeader &primaryHeader, const GptPartitionEntry *pPrimaryPartitions, uint64_t sectorSize);
nn::Result WritePartitionTable(IBlockStorage *pBlockStorage, const Mbr &mbr, const GptHeader &gptHeader, const GptHeader &alternateGptHeader, const GptPartitionEntry *partitions, int numPartitions);
bool IsProductionInfoPartitionGuid(const Guid &guid);
bool IsProductionInfoFsPartitionId(const char * fsPartitionName);
bool IsProductionInfoFsPartitionId(nn::fs::BisPartitionId partitionId);
bool HasProductionInfoPartition(const GptHeader &gptHeader, const GptPartitionEntry partitions[]);
void GetProductionInfoPartitionIndexes(int *pOutNum, int* pOutArray, size_t arraySize, const GptHeader &gptHeader, const GptPartitionEntry partitions[]);

void PrintGuid(const Guid guid);
void PrintMbr(const Mbr &mbr);
void PrintGptHeader(const GptHeader &gptHeader);
void PrintGptPartitionEntry(GptPartitionEntry entry);
void PrintGptInfo(const GptHeader &gptHeader, GptPartitionEntry *pEntries);

class IFile;

nn::Result DumpMbr(IFile *pFile, const char* filename, const Mbr &mbr);
nn::Result DumpGptHeader(IFile *pFile, const char* filename, const GptHeader &gptHeader);
nn::Result DumpGptPartitions(IFile *pFile, const char* filename, const GptPartitionEntry *pGptPartitions, int numPartitions);
