﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "systemInitializer_Partition.h"
#include "../Utility/systemInitializer_Crc32.h"
#include "../Utility/systemInitializer_IBlockStorage.h"
#include "../Utility/systemInitializer_IFile.h"
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <algorithm>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/fs/fs_Bis.h>
#include <Utility/systemInitializer_FsBisPartition.h>

nn::Result ReadMbrFromStorage(Mbr *pOut, IBlockStorage *pStorage)
{
    NN_UTILTOOL_RESULT_DO(pStorage->Read(pOut, sizeof(Mbr), 0, 1));

    NN_RESULT_SUCCESS;
}

nn::Result ReadGptFromStorage(GptHeader *pOut, IBlockStorage *pStorage, uint64_t lba)
{
    NN_UTILTOOL_RESULT_DO(pStorage->Read(pOut, sizeof(GptHeader), lba, 1));

    NN_RESULT_SUCCESS;
}

nn::Result ReadGptFromStorage(GptHeader *pOut, nn::fs::IStorage *pStorage, int64_t address)
{
    NN_UTILTOOL_RESULT_DO(pStorage->Read(address, pOut, sizeof(GptHeader)));

    NN_RESULT_SUCCESS;
}


uint64_t CalculatePartitionSize(const GptPartitionEntry &partitionEntry, IBlockStorage *pStorage)
{
    return (partitionEntry.endingLba - partitionEntry.startingLba) * pStorage->GetBlockSize();
}

std::string GetPartitionName(const GptPartitionEntry &partitionEntry)
{
    char buffer[64];
    NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == nn::util::ConvertStringUtf16NativeToUtf8(buffer, 64, reinterpret_cast<const uint16_t*>(partitionEntry.partitionName), 36));

    return std::string(buffer);
}

uint64_t CalculateNumberOfSectors(uint64_t bytes, uint64_t sectorSize)
{
    return nn::util::align_up(bytes, static_cast<size_t>(sectorSize)) / sectorSize;
}

uint64_t CalculateLba(uint64_t address, uint64_t sectorSize)
{
    NN_ABORT_UNLESS(nn::util::is_aligned(address, sectorSize));
    return address / sectorSize;
}

uint64_t CalculatePartitionEntrySectors(const GptHeader &gptHeader, uint64_t sectorSize)
{
    return CalculateNumberOfSectors(
        gptHeader.sizeOfPartitionEntry *
        std::max(GPT_DEFAULT_PARTITION_ENTRIES, gptHeader.numberOfPartitionEntries), sectorSize);
}

uint64_t CalculateGptHeaderAndPartitionEntrySectors(const GptHeader &gptHeader, uint64_t sectorSize)
{
    uint64_t headerSectors = CalculateNumberOfSectors(sizeof(GptHeader), sectorSize);
    uint64_t partitionEntrySectors = CalculatePartitionEntrySectors(gptHeader, sectorSize);

    return headerSectors + partitionEntrySectors;
}

nn::Result ReadGptPartitionsFromStorage(
    GptPartitionEntry *pOutArray,
    int numOutArray,
    const GptHeader &gptHeader,
    IBlockStorage *pStorage)
{
    int num = std::min(numOutArray, gptHeader.numberOfPartitionEntries);
    int64_t readSectors = CalculateNumberOfSectors(
        gptHeader.sizeOfPartitionEntry * num,
        pStorage->GetBlockSize());
    int64_t startingLba = gptHeader.partitionEntryLba;

    NN_UTILTOOL_RESULT_DO(pStorage->Read(
        pOutArray,
        sizeof(GptPartitionEntry) * numOutArray,
        startingLba,
        readSectors));

    NN_RESULT_SUCCESS;
}

nn::Result ReadGptPartitionsFromStorage(GptPartitionEntry *pOutArray, int numOutArray, const GptHeader &gptHeader, nn::fs::IStorage *pStorage)
{
    NN_ABORT_UNLESS(nn::util::is_aligned(numOutArray, 4));
    int num = nn::util::align_up(std::min(numOutArray, gptHeader.numberOfPartitionEntries), 4);

    NN_UTILTOOL_RESULT_DO(pStorage->Read(
        gptHeader.partitionEntryLba * 512,
        pOutArray,
        sizeof(GptPartitionEntry) * num));

    NN_RESULT_SUCCESS;
}

void UpdateCrc32InGptHeader(GptHeader *pInOut, const GptPartitionEntry *pPartitions)
{
    pInOut->headerCrc32 = 0;

    Crc32 crc32;
    pInOut->partitionEntryArrayCrc32 = crc32.Calculate(
        pPartitions,
        sizeof(GptPartitionEntry) * pInOut->numberOfPartitionEntries);
    pInOut->headerCrc32 = crc32.Calculate(pInOut, pInOut->headerSize);
}

void ReCalculateLba(
    GptHeader *pGpt,
    GptPartitionEntry *pPartitions,
    uint64_t firstUsableLba,
    uint64_t lastUsableLba,
    uint64_t lastLba,
    uint64_t sectorSize)
{
    pGpt->firstUsableLba = firstUsableLba;
    pGpt->lastUsableLba = lastUsableLba;
    pGpt->myLba = 1ULL;
    pGpt->alternateLba = lastLba;

    uint64_t currentLba = firstUsableLba;
    for(int i = 0; i < pGpt->numberOfPartitionEntries; i++)
    {
        GptPartitionEntry &partition = pPartitions[i];
        uint64_t partitionSize = partition.endingLba - partition.startingLba;
        uint64_t numSectors = CalculateNumberOfSectors(partitionSize, sectorSize);
        partition.startingLba = currentLba;
        partition.endingLba = currentLba + numSectors - 1;
        currentLba += numSectors;
    }
}

int FindPartition(
    int64_t startLba,
    const uint16_t *name,
    GptHeader *pGpt,
    GptPartitionEntry *partitions)
{
    for ( int i = 0; i < pGpt->numberOfPartitionEntries; i++ )
    {
        if( startLba <= partitions[i].startingLba && 0 == memcmp(
            partitions[i].partitionName,
            name,
            sizeof(partitions[i].partitionName)))
        {
            return i;
        }
    }

    return -1;
}

void CalculatePartitionAddress(
    GptHeader *pOutGpt,
    GptPartitionEntry *pOutPartitions,
    GptHeader *pPreviousGpt,
    GptPartitionEntry *pPreviousPartitions,
    int numPartitionAddress,
    PartitionAddress *pPartitionAddresses,
    uint64_t storageSize,
    uint64_t sectorSize)
{
    NN_ABORT_UNLESS(0 < numPartitionAddress);
    NN_ABORT_UNLESS(
        numPartitionAddress == pOutGpt->numberOfPartitionEntries,
        "numPartitionAddress(%d) == pOutGpt->numberOfPartitionEntries(%d)",
        numPartitionAddress, pOutGpt->numberOfPartitionEntries);

    pOutGpt->firstUsableLba = 1 + CalculateGptHeaderAndPartitionEntrySectors(*pOutGpt, sectorSize);
    pOutGpt->lastUsableLba = CalculateLba(storageSize, sectorSize)
        - CalculateGptHeaderAndPartitionEntrySectors(*pOutGpt, sectorSize) - 1;
    pOutGpt->alternateLba = CalculateLba(storageSize, sectorSize) - 1;
    pOutGpt->myLba = 1ULL;

    int64_t currentLba = pOutGpt->firstUsableLba;

    for(int i = 0; i < numPartitionAddress; i++)
    {
        GptPartitionEntry &partition = pOutPartitions[i];
        PartitionAddress &partitionAddress = pPartitionAddresses[i];

        switch(partitionAddress.type)
        {
        case PartitionAddress::AddressType_Absolute:
            {
                NN_UTILTOOL_LOG_DEBUG("Partition[%d].type == Absolute", i);
                NN_UTILTOOL_LOG_DEBUG("Partition[%d].address = %08llx", i, partitionAddress.address);
                NN_UTILTOOL_LOG_DEBUG("Partition[%d].size    = %08llx", i, partitionAddress.size);
                partition.startingLba = CalculateLba(partitionAddress.address, sectorSize);
                partition.endingLba = CalculateLba(partitionAddress.address +  partitionAddress.size, sectorSize) - 1;
                NN_ABORT_UNLESS(currentLba <= partition.startingLba && partition.endingLba <= pOutGpt->lastUsableLba,
                    "currentLba(0x%llx) <= partition.startingLba(0x%llx) && partition.endingLba(0x%llx) <= pOutGpt->lastUsableLba(0x%llx)",
                    currentLba, partition.startingLba, partition.endingLba, pOutGpt->lastUsableLba);
                currentLba = partition.endingLba + 1;
            }
            break;
        case PartitionAddress::AddressType_Relative:
            {
                NN_UTILTOOL_LOG_DEBUG("Partition[%d].type == Relative", i);
                NN_UTILTOOL_LOG_DEBUG("Partition[%d].basePartition = %d", i, partitionAddress.basePartitionIndex);
                NN_UTILTOOL_LOG_DEBUG("Partition[%d].address = %08llx", i, partitionAddress.address);
                NN_UTILTOOL_LOG_DEBUG("Partition[%d].size    = %08llx", i, partitionAddress.size);
                NN_ABORT_UNLESS(partitionAddress.basePartitionIndex < i);
                partition.startingLba = CalculateLba(
                    (pOutPartitions[partitionAddress.basePartitionIndex].endingLba + 1) * sectorSize + partitionAddress.address,
                    sectorSize);
                partition.endingLba = CalculateLba(
                    partition.startingLba * sectorSize + partitionAddress.size,
                    sectorSize) - 1;
                NN_ABORT_UNLESS(currentLba <= partition.startingLba && partition.endingLba <= pOutGpt->lastUsableLba,
                    "currentLba(0x%llx) <= partition.startingLba(0x%llx) && partition.endingLba(0x%llx) <= pOutGpt->lastUsableLba(0x%llx)",
                    currentLba, partition.startingLba, partition.endingLba, pOutGpt->lastUsableLba);
                currentLba = partition.endingLba + 1;
            }
            break;
        case PartitionAddress::AddressType_Keep:
            {
                NN_UTILTOOL_LOG_DEBUG("Partition[%d].type == Keep", i);
                NN_ABORT_UNLESS(partitionAddress.basePartitionIndex == i);
                int partitionIndex = FindPartition(
                    currentLba,
                    partition.partitionName,
                    pPreviousGpt,
                    pPreviousPartitions);
                NN_ABORT_UNLESS(0 <= partitionIndex, "partitionIndex: %d", partitionIndex);
                NN_ABORT_UNLESS(0 == memcmp(
                    partition.partitionName,
                    pPreviousPartitions[partitionIndex].partitionName,
                    sizeof(pPreviousPartitions[partitionIndex].partitionName)));
                partition = pPreviousPartitions[partitionIndex];
                NN_ABORT_UNLESS(currentLba <= partition.startingLba && partition.endingLba <= pOutGpt->lastUsableLba,
                    "currentLba(0x%llx) <= partition.startingLba(0x%llx) && partition.endingLba(0x%llx) <= pOutGpt->lastUsableLba(0x%llx)",
                    currentLba, partition.startingLba, partition.endingLba, pOutGpt->lastUsableLba);
                currentLba = partition.endingLba + 1;
            }
            break;
        default:
            NN_ABORT("Unknown address type: %d", partitionAddress.type);
        }

        NN_UTILTOOL_LOG_DEBUG("Partition[%d].name == %ls", i, partition.partitionName);
        NN_UTILTOOL_LOG_DEBUG("Partition[%d].startingLba == %08llx()", i, partition.startingLba);
        NN_UTILTOOL_LOG_DEBUG("Partition[%d].endingLba == %08llx()", i, partition.endingLba);
    }
}

void MakeAlternateGpt(
    GptHeader *pOutHeader,
    const GptHeader &primaryHeader,
    const GptPartitionEntry *pPrimaryPartitions,
    uint64_t sectorSize)
{
    *pOutHeader = primaryHeader;
    uint64_t primaryHeaderLba = primaryHeader.myLba;
    uint64_t alternateHeaderLba = primaryHeader.alternateLba;
    pOutHeader->myLba = alternateHeaderLba;
    pOutHeader->alternateLba = primaryHeaderLba;
    pOutHeader->partitionEntryLba = pOutHeader->myLba - CalculatePartitionEntrySectors(primaryHeader, sectorSize);
    UpdateCrc32InGptHeader(pOutHeader, pPrimaryPartitions);
}

nn::Result WritePartitionTable(
    IBlockStorage *pBlockStorage,
    const Mbr &mbr,
    const GptHeader &gptHeader,
    const GptHeader &alternateGptHeader,
    const GptPartitionEntry *partitions,
    int numPartitions)
{
    uint64_t partitionEntrySize = static_cast<uint64_t>(gptHeader.sizeOfPartitionEntry) * gptHeader.numberOfPartitionEntries;
    uint64_t partitionEntrySectors = CalculateNumberOfSectors(partitionEntrySize, pBlockStorage->GetBlockSize());
    size_t partitionBufferSize = sizeof(GptPartitionEntry) * numPartitions;
    NN_UTILTOOL_RESULT_DO(pBlockStorage->Write(0, 1, &mbr, sizeof(Mbr)));
    NN_UTILTOOL_RESULT_DO(pBlockStorage->Write(gptHeader.myLba, 1, &gptHeader, sizeof(GptHeader)));
    NN_UTILTOOL_RESULT_DO(pBlockStorage->Write(gptHeader.partitionEntryLba, partitionEntrySectors, partitions, partitionBufferSize));
    NN_UTILTOOL_RESULT_DO(pBlockStorage->Write(alternateGptHeader.myLba, 1, &alternateGptHeader, sizeof(GptHeader)));
    NN_UTILTOOL_RESULT_DO(pBlockStorage->Write(alternateGptHeader.partitionEntryLba, partitionEntrySectors, partitions, partitionBufferSize));

    nn::fs::InvalidateBisCache();

    NN_RESULT_SUCCESS;
}

bool IsProductionInfoPartitionGuid(const Guid &guid)
{
    const Guid guids[] =
    {
        Guid::Make(0x98109e25, 0x64e2, 0x4c95, 0x8a, 0x77, 0x41, 0x49, 0x16, 0xf5, 0xbc, 0xeb)
    };

    for (Guid protectedGuid : guids)
    {
        if(Guid::IsEqual(guid, protectedGuid))
        {
            return true;
        }
    }

    return false;
}

bool IsProductionInfoFsPartitionId(const char * fsPartitionName)
{
    return IsProductionInfoFsPartitionId(ConvertToBisPartitionId(fsPartitionName));
}

bool IsProductionInfoFsPartitionId(nn::fs::BisPartitionId partitionId)
{
    const nn::fs::BisPartitionId partitionIds[] =
    {
        nn::fs::BisPartitionId::CalibrationBinary
    };

    for (nn::fs::BisPartitionId id : partitionIds)
    {
        if(id == partitionId)
        {
            return true;
        }
    }

    return false;
}

bool HasProductionInfoPartition(const GptHeader &gptHeader, const GptPartitionEntry partitions[])
{
    for( int i = 0; i < gptHeader.numberOfPartitionEntries; i++)
    {
        const auto partition = partitions[i];
        if(IsProductionInfoPartitionGuid(partition.partitionTypeGuid))
        {
            return true;
        }
    }

    return false;
}

void GetProductionInfoPartitionIndexes(int *pOutNum, int* pOutArray, size_t arraySize, const GptHeader &gptHeader, const GptPartitionEntry partitions[])
{
    *pOutNum = 0;
    for( int i = 0; i < gptHeader.numberOfPartitionEntries; i++)
    {
        const auto &partition = partitions[i];
        if(IsProductionInfoPartitionGuid(partition.partitionTypeGuid))
        {
            if(*pOutNum < static_cast<int>(arraySize))
            {
                pOutArray[*pOutNum] = i;
                *pOutNum += 1;
            }
            else
            {
                NN_ABORT("array size is too small");
            }
        }
    }
}

void PrintGuid(const Guid guid)
{
    NN_UTILTOOL_LOG_VERBOSE(
        "%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X",
        guid.data1,
        guid.data2,
        guid.data3,
        guid.data4[0], guid.data4[1],
        guid.data4[2], guid.data4[3],
        guid.data4[4], guid.data4[5],
        guid.data4[6], guid.data4[7]);
}

void PrintMbr(const Mbr &mbr)
{
    NN_UTILTOOL_LOG_VERBOSE("");
    NN_UTILTOOL_LOG_VERBOSE("MBR info");
    NN_UTILTOOL_LOG_VERBOSE("========");
    NN_UTILTOOL_LOG_VERBOSE("StructureSize: %d", sizeof(Mbr));
    NN_UTILTOOL_LOG_VERBOSE("DiskSignature: %02x%02x%02x%02x",
        mbr.diskSignature[0],mbr.diskSignature[1],mbr.diskSignature[2],mbr.diskSignature[3]);
    NN_UTILTOOL_LOG_VERBOSE("Signature: %02x%02x", mbr.signature[0], mbr.signature[1]);
    NN_UTILTOOL_LOG_VERBOSE("Partitions:");
    for(int i=0; i < Mbr::NUM_PARTITIONS; i++)
    {
        const MbrPartitionRecord &partition = mbr.partitions[i];
        NN_UTILTOOL_LOG_VERBOSE("  - Index: %d", i);
        NN_UTILTOOL_LOG_VERBOSE("    BootIndicator: %02x", partition.bootIndicator);
        NN_UTILTOOL_LOG_VERBOSE("    StartingChs: %02x%02x%02x",
            partition.startingChs[0],partition.startingChs[1],partition.startingChs[2]);
        NN_UTILTOOL_LOG_VERBOSE("    OsType: %02x", partition.osType);
        NN_UTILTOOL_LOG_VERBOSE("    EndingChs: %02x%02x%02x",
            partition.endingChs[0], partition.endingChs[1], partition.endingChs[2]);
        NN_UTILTOOL_LOG_VERBOSE("    StartingLba: %02x%02x%02x%02x",
            partition.startingLba[0], partition.startingLba[1], partition.startingLba[2], partition.startingLba[3]);
        NN_UTILTOOL_LOG_VERBOSE("    SizeInLba: %02x%02x%02x%02x",
            partition.sizeInLba[0],partition.sizeInLba[1],partition.sizeInLba[2],partition.sizeInLba[3]);
    }
}

void PrintGptHeader(const GptHeader &gptHeader)
{
    const char* s = gptHeader.signature;

    NN_UTILTOOL_LOG_VERBOSE("");
    NN_UTILTOOL_LOG_VERBOSE("GptHeader info");
    NN_UTILTOOL_LOG_VERBOSE("==============");
    NN_UTILTOOL_LOG_VERBOSE("StructureSize: %d", sizeof(GptHeader));
    NN_UTILTOOL_LOG_VERBOSE("Signature: '%c%c%c%c%c%c%c%c'", s[0], s[1], s[2], s[3], s[4], s[5], s[6], s[7]);
    NN_UTILTOOL_LOG_VERBOSE("Revision: %08x", gptHeader.revision);
    NN_UTILTOOL_LOG_VERBOSE("HeaderSize: %d", gptHeader.headerSize);
    NN_UTILTOOL_LOG_VERBOSE("HeaderCrc32: %08x", gptHeader.headerCrc32);
    NN_UTILTOOL_LOG_VERBOSE("MyLba: 0x%08llx", gptHeader.myLba);
    NN_UTILTOOL_LOG_VERBOSE("AlternateLba: 0x%08llx", gptHeader.alternateLba);
    NN_UTILTOOL_LOG_VERBOSE("FirstUsableLba: 0x%08llx", gptHeader.firstUsableLba);
    NN_UTILTOOL_LOG_VERBOSE("LastUsableLba: 0x%08llx", gptHeader.lastUsableLba);
    NN_UTILTOOL_LOG_VERBOSE("DiskGuid: "); PrintGuid(gptHeader.diskGuid);
    NN_UTILTOOL_LOG_VERBOSE("PartitionEntryLba: %08llx", gptHeader.partitionEntryLba);
    NN_UTILTOOL_LOG_VERBOSE("NumberOfPartitionEntries: %d", gptHeader.numberOfPartitionEntries);
    NN_UTILTOOL_LOG_VERBOSE("SizeOfPartitionEntry: %d", gptHeader.sizeOfPartitionEntry);
    NN_UTILTOOL_LOG_VERBOSE("PartitionEntryArrayCrc32: %08x", gptHeader.partitionEntryArrayCrc32);
}

void PrintGptPartitionEntry(GptPartitionEntry entry)
{
    NN_UTILTOOL_LOG_VERBOSE("PartitionTypeGuid: ");
    PrintGuid(entry.partitionTypeGuid);
    NN_UTILTOOL_LOG_VERBOSE("UniquePartitionGuid: ");
    PrintGuid(entry.uniquePartitionGuid);
    NN_UTILTOOL_LOG_VERBOSE("StartingLba: 0x%08llx", entry.startingLba);
    NN_UTILTOOL_LOG_VERBOSE("EndingLba: 0x%08llx", entry.endingLba);
    NN_UTILTOOL_LOG_VERBOSE("Attributes: 0x%08llx", entry.attributes);

    char buffer[64];
    NN_ABORT_UNLESS(nn::util::CharacterEncodingResult_Success == nn::util::ConvertStringUtf16NativeToUtf8(buffer, 64, reinterpret_cast<const uint16_t*>(entry.partitionName), 36));

    NN_UTILTOOL_LOG_VERBOSE("PartitionName: %s", buffer);
}

void PrintGptInfo(const GptHeader &gptHeader, GptPartitionEntry *pEntries)
{
    PrintGptHeader(gptHeader);
    for(int i = 0; i < gptHeader.numberOfPartitionEntries; i++)
    {
        NN_UTILTOOL_LOG_VERBOSE("");
        NN_UTILTOOL_LOG_VERBOSE("Partition Index: %d", i);
        NN_UTILTOOL_LOG_VERBOSE("-------------------");
        PrintGptPartitionEntry(pEntries[i]);
    }
    NN_UTILTOOL_LOG_VERBOSE("");
}

nn::Result DumpMbr(IFile *pFile, const char* filename, const Mbr &mbr)
{
    NN_UTILTOOL_RESULT_DO(pFile->OpenWrite(filename));
    NN_UTIL_SCOPE_EXIT { pFile->Close(); };

    NN_UTILTOOL_RESULT_DO(
        pFile->Write(0, &mbr, sizeof(mbr), true));

    NN_RESULT_SUCCESS;
}

nn::Result DumpGptHeader(IFile *pFile, const char* filename, const GptHeader &gptHeader)
{
    NN_UTILTOOL_RESULT_DO(pFile->OpenWrite(filename));
    NN_UTIL_SCOPE_EXIT { pFile->Close(); };

    NN_UTILTOOL_RESULT_DO(
        pFile->Write(0, &gptHeader, sizeof(gptHeader), true));

    NN_RESULT_SUCCESS;
}

nn::Result DumpGptPartitions(
    IFile *pFile,
    const char* filename,
    const GptPartitionEntry *pGptPartitions,
    int numPartitions)
{
    NN_UTILTOOL_RESULT_DO(pFile->OpenWrite(filename));
    NN_UTIL_SCOPE_EXIT { pFile->Close(); };

    NN_UTILTOOL_RESULT_DO(
        pFile->Write(0, pGptPartitions, sizeof(GptPartitionEntry) * numPartitions, true));

    NN_RESULT_SUCCESS;
}
