﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_Host.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include "Utility/systemInitializer_Allocator.h"
#include <nn/utilTool/utilTool_CommandFramework.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include <nn/utilTool/utilTool_StringUtility.h>
#include "ListExportablePartitions.h"
#include "ExportPartition.h"
#include "ListFilesInFatPartition.h"
#include "ExportFilesInFatPartition.h"
#include <cstdlib>

extern "C" void nninitStartup()
{
}

class ExportPartitionCommand
{
public:
    ExportPartitionCommand() :
        m_ListExportable(false),
        m_OutputSize(0),
        m_BeginAddress("0"),
        m_ListFiles(false),
        m_ExportFiles(false)
    {}

    const char* Name()
    {
        return "ExportPartition";
    }

    const char* Summary()
    {
        return "Export a partition from the build-in-storage of a target.";
    }

    const char* Description()
    {
        return "";
    }

    void Arguments(nn::utilTool::SingleCommandInterface &commandInterface)
    {
        commandInterface.AddKeywordArgument(&m_OutputPath,   'o',         "output", "Output path of image file.a", false);
        commandInterface.AddKeywordArgument(&m_OutputType,   "type",      "Output type. (See --list)", false);
        commandInterface.AddKeywordArgument(&m_TargetName,   "name",      "Target partition name.", false);
        commandInterface.AddKeywordArgument(&m_OutputSize,   "size",      "Output size.", false);
        commandInterface.AddKeywordArgument(&m_BeginAddress, "begin",     "Begin address of target partition. example: '--begin ABCD0123'", false);
        commandInterface.AddFlagArgument(&m_ListExportable,  "list",      "List exportable partitions.");
        commandInterface.AddFlagArgument(&m_ListFiles,       "file-list", "Export file list of fat partition.");
        commandInterface.AddFlagArgument(&m_ExportFiles,     "file-copy", "Export files of fat partition.");
    }

    bool ValidateArgumentsAsExportCommand()
    {
        if(m_OutputPath == "")
        {
            NN_UTILTOOL_LOG_ERROR("Output path(--output) is not found.");
            return false;
        }

        return true;
    }

    int Main()
    {
        nn::fs::InitializeWithMultiSessionForTargetTool();
        nn::fs::SetAllocator(Allocate, Deallocate);

        if(m_ListExportable)
        {
            NN_UTILTOOL_LOG_VERBOSE("List exportable partitions.");

            nn::Result result = ListExportablePartitions();

            if( result.IsFailure() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed to list exportable partitions. Result(module=%d, desc=%d)", result.GetModule(), result.GetDescription());
                return 1;
            }

            return 0;
        }
        else if(m_ListFiles)
        {
            if( !ValidateArgumentsAsExportCommand() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed to export because arguments is invalid.");
                return 1;
            }

            NN_UTILTOOL_LOG_VERBOSE("Start dump file list of fat partition.");
            NN_UTILTOOL_LOG_VERBOSE("output path: %s", m_OutputPath.c_str());

            NN_UTILTOOL_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::MountHostRoot());

            nn::Result result = ListFilesInFatPartition(m_OutputPath, m_OutputType, m_TargetName);
            if( result.IsFailure() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed to list files. Result(module=%d, desc=%d)", result.GetModule(), result.GetDescription());
                return 1;
            }

            return 0;
        }
        else if(m_ExportFiles)
        {
            if( !ValidateArgumentsAsExportCommand() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed to export because arguments is invalid.");
                return 1;
            }

            NN_UTILTOOL_LOG_VERBOSE("Start export files of fat partition.");
            NN_UTILTOOL_LOG_VERBOSE("output path: %s", m_OutputPath.c_str());

            NN_UTILTOOL_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::MountHostRoot());

            nn::Result result = ExportFilesInFatPartition(m_OutputPath, m_OutputType, m_TargetName);
            if( result.IsFailure() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed to list files. Result(module=%d, desc=%d)", result.GetModule(), result.GetDescription());
                return 1;
            }

            return 0;
        }
        else
        {
            if( !ValidateArgumentsAsExportCommand() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed to export because arguments is invalid.");
                return 1;
            }

            NN_UTILTOOL_LOG_VERBOSE("Start export partition.");
            NN_UTILTOOL_LOG_VERBOSE("output path: %s", m_OutputPath.c_str());

            NN_UTILTOOL_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::MountHostRoot());

            int64_t beginAddress;

            if(m_BeginAddress == "")
            {
                beginAddress = 0;
            }
            else
            {
                beginAddress = nn::utilTool::StrToInt64(m_BeginAddress.c_str(), nullptr, 16);
            }

            nn::Result result = ExportPartition(m_OutputPath, m_OutputType, m_TargetName, m_OutputSize, beginAddress);
            if( result.IsFailure() )
            {
                NN_UTILTOOL_LOG_ERROR("Failed to export partition. Result(module=%d, desc=%d)", result.GetModule(), result.GetDescription());
                return 1;
            }

            return 0;
        }
    }

private:
    std::string m_OutputPath;
    std::string m_OutputType;
    std::string m_TargetName;
    bool m_ListExportable;
    int64_t m_OutputSize;
    std::string m_BeginAddress;
    bool m_ListFiles;
    bool m_ExportFiles;
};

extern "C" void nnMain()
{
    nn::utilTool::BasicCommandFramework<ExportPartitionCommand>::Run(
        nn::os::GetHostArgc(),
        const_cast<const char**>(nn::os::GetHostArgv())
    );
}

