﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ExportFilesInFatPartition.h"
#include "ListFilesInFatPartition.h"
#include "ExportPartition.h"
#include "ListExportablePartitions.h"
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Bis.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <vector>
#include <functional>

#include "Utility/systemInitializer_FsFile.h"
#include "Utility/systemInitializer_FsBisPartition.h"
#include "Utility/systemInitializer_StorageUtility.h"
#include "Utility/systemInitializer_FileSystemUtility.h"

namespace
{
    const int64_t WritingBufferSize = 4 * 1024;
    NN_ALIGNAS(4096) static uint8_t s_WritingBuffer[WritingBufferSize];

    nn::Result CopyFile(FsFile &output, FsFile &input)
    {
        int64_t fileSize;

        NN_UTILTOOL_RESULT_DO(
            input.GetSize(&fileSize));

        int64_t restBytes = fileSize;
        uint64_t currentOffset = 0;

        NN_UTILTOOL_LOG_PROGRESS("Copy file: size = %lld", restBytes);

        while(0 < restBytes)
        {
            size_t writeChunkSize = static_cast<size_t>(std::min(restBytes, static_cast<int64_t>(WritingBufferSize)));

            NN_UTILTOOL_LOG_PROGRESS("Write %zu bytes. Progress: %lld/%lld.", writeChunkSize, currentOffset, fileSize);

            size_t readSize;
            NN_UTILTOOL_RESULT_DO(
                input.Read(&readSize, currentOffset, s_WritingBuffer, writeChunkSize));
            NN_ABORT_UNLESS(readSize == writeChunkSize);

            NN_UTILTOOL_RESULT_DO(
                output.Write(currentOffset, s_WritingBuffer, writeChunkSize, false));

            restBytes -= writeChunkSize;
            currentOffset += writeChunkSize;
        }

        NN_UTILTOOL_LOG_VERBOSE("Finish to export %lld bytes.", fileSize);

        NN_RESULT_SUCCESS;
    }

    nn::Result CopyFile(std::string outputPath, std::string inputPath)
    {
        NN_UTILTOOL_LOG_DEBUG("copy file '%s' -> '%s'", inputPath.c_str(), outputPath.c_str());

        bool fileExist;
        NN_UTILTOOL_RESULT_DO(
            FsFile::Exists(&fileExist, inputPath.c_str()));

        if(!fileExist)
        {
            NN_UTILTOOL_LOG_WARNING("file not found '%s'", inputPath.c_str());
            NN_RESULT_SUCCESS;
        }

        bool outputFileExist;
        NN_UTILTOOL_RESULT_DO(
            FsFile::Exists(&outputFileExist, outputPath.c_str()));

        if(!outputFileExist)
        {
            NN_UTILTOOL_LOG_DEBUG("create file: '%s'", outputPath.c_str());
            NN_UTILTOOL_RESULT_DO(
                FsFile::Create(outputPath.c_str()));
        }

        FsFile outputFile;
        FsFile inputFile;

        NN_UTILTOOL_RESULT_DO(
            outputFile.OpenWrite(outputPath.c_str()));
        NN_UTILTOOL_RESULT_DO(
            inputFile.OpenRead(inputPath.c_str()));

        NN_UTILTOOL_RESULT_DO(
            CopyFile(outputFile, inputFile));

        NN_UTILTOOL_RESULT_DO(
            outputFile.Flush());

        NN_RESULT_SUCCESS;
    }
}

nn::Result ExportFilesInFatPartition(std::string outputPath, std::string type, std::string name)
{
    if (type != "fs-partition")
    {
        NN_UTILTOOL_LOG_ERROR("No support paritition type(support only fs-partition). type='%s'", type.c_str());
        NN_UTILTOOL_RESULT_THROW(nn::fs::ResultPartitionNotFound());
    }

    bool hasPartition;
    NN_UTILTOOL_RESULT_DO(
        HasExportablePartition(&hasPartition, type, name));

    if (!hasPartition)
    {
        NN_UTILTOOL_LOG_ERROR("Found no partitions(type='%s', name='%s').", type.c_str(), name.c_str());
        NN_UTILTOOL_RESULT_THROW(nn::fs::ResultPartitionNotFound());
    }

    bool exist;
    NN_UTILTOOL_RESULT_DO(
        DirectoryExists(&exist, outputPath.c_str()));

    if(exist)
    {
        NN_UTILTOOL_RESULT_THROW(nn::fs::ResultPathAlreadyExists());
    }

    NN_UTILTOOL_LOG_DEBUG("image opened");

    auto bisPartitionId = ConvertToBisPartitionId(name);
    NN_UTILTOOL_RESULT_DO(nn::fs::MountBis(bisPartitionId, nullptr));

    auto mountName = std::string(nn::fs::GetBisMountName(bisPartitionId)) + ":/";

    std::function<nn::Result (std::string, nn::fs::DirectoryEntryType)> copyFunction = [&outputPath, &mountName] (std::string fileName, nn::fs::DirectoryEntryType type) -> nn::Result {
        switch(type)
        {
            case nn::fs::DirectoryEntryType_File:
                {
                    std::string sourcePath = fileName;
                    std::string destinationPath = outputPath + "/" + &fileName[mountName.length()];
                    NN_UTILTOOL_LOG_VERBOSE("copy file: %s -> %s", sourcePath.c_str(), destinationPath.c_str());
                    NN_UTILTOOL_RESULT_DO(
                        CopyFile(destinationPath, sourcePath));
                    break;
                }
            case nn::fs::DirectoryEntryType_Directory:
                {
                    std::string sourcePath = fileName;
                    std::string destinationPath = outputPath + "/" + &fileName[mountName.length()];
                    NN_UTILTOOL_LOG_VERBOSE("copy directory: %s -> %s", sourcePath.c_str(), destinationPath.c_str());
                    NN_UTILTOOL_RESULT_DO(nn::fs::CreateDirectory(destinationPath.c_str()));
                    break;
                }
            default:
                {
                    NN_ABORT();
                }
        }

        NN_RESULT_SUCCESS;
    };

    NN_UTILTOOL_RESULT_DO(EnumerateDirectoryEntries(mountName, copyFunction));

    NN_RESULT_SUCCESS;
}
