﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>

#include <nn/os/os_Thread.h>
#include <nn/os/os_MessageQueue.h>
#include <nn/os/os_MultipleWaitApi.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/spl/impl/spl_ApiImpl.h>
#include <nn/spl/spl_Result.h>
#include "spl_SecureMonitorManager.h"

namespace nn { namespace spl {

    namespace
    {


    }

    void SecureMonitorManager::Initialize() NN_NOEXCEPT
    {
        for( int i = 0; i < impl::AesKeySlotCount; ++i )
        {
            m_AesKeySlotOwners[i] = nullptr;
        }
        impl::Initialize();
    }

    Result SecureMonitorManager::GetConfig(Bit64* pOut, ConfigItem key) NN_NOEXCEPT
    {
        return impl::GetConfig(pOut, key);
    }

    Result SecureMonitorManager::GetPackage2Hash(
        void*       pOutBuffer,
        size_t      bufferSize) NN_NOEXCEPT
    {
        return impl::GetPackage2Hash(
            pOutBuffer,
            bufferSize);
    }

    Result SecureMonitorManager::ModularExponentiate(
            void*       pResultBuffer,
            size_t      resultBufferSize,
            const void* pBase,
            size_t      baseSize,
            const void* pExponent,
            size_t      exponentSize,
            const void* pModulus,
            size_t      modulusSize ) NN_NOEXCEPT
    {
        return impl::ModularExponentiate(
            pResultBuffer,
            resultBufferSize,
            pBase,
            baseSize,
            pExponent,
            exponentSize,
            pModulus,
            modulusSize );
    }


    Result SecureMonitorManager::GenerateAesKek(
        AccessKey*          pAccessKey,
        const void*         pKeySource,
        size_t              keySourceSize,
        int                 generation,
        int                 option ) NN_NOEXCEPT
    {
        return impl::GenerateAesKek(pAccessKey, pKeySource, keySourceSize, generation, option);
    }

    Result SecureMonitorManager::LoadAesKey(
        int                 slotIndex,
        const void*         pOwner,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        auto result = TestAesKeySlot(slotIndex, pOwner);
        if( result.IsFailure() )
        {
            return result;
        }

        return impl::LoadAesKey(slotIndex, accessKey, pKeySource, keySourceSize);
    }

    Result SecureMonitorManager::GenerateAesKey(
        void*               pOutBuffer,
        size_t              outBufferSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        return impl::GenerateAesKey(pOutBuffer, outBufferSize, accessKey, pKeySource, keySourceSize);
    }

    Result SecureMonitorManager::DecryptDeviceUniqueData(
        void*               pOutBuffer,
        size_t              outBufferSize,
        const void*         pData,
        size_t              dataSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        return impl::DecryptDeviceUniqueData(
            pOutBuffer,
            outBufferSize,
            pData,
            dataSize,
            accessKey,
            pKeySource,
            keySourceSize);
    }

    Result SecureMonitorManager::ReencryptDeviceUniqueData(
        void*               pOutBuffer,
        size_t              outBufferSize,
        const void*         pData,
        size_t              dataSize,
        const AccessKey&    accessKeyForDecryption,
        const void*         pKeySourceForDecryption,
        size_t              keySourceSizeForDecryption,
        const AccessKey&    accessKeyForEncryption,
        const void*         pKeySourceForEncryption,
        size_t              keySourceSizeForEncryption,
        Bit32               option ) NN_NOEXCEPT
    {
        return impl::ReencryptDeviceUniqueData(
            pOutBuffer,
            outBufferSize,
            pData,
            dataSize,
            accessKeyForDecryption,
            pKeySourceForDecryption,
            keySourceSizeForDecryption,
            accessKeyForEncryption,
            pKeySourceForEncryption,
            keySourceSizeForEncryption,
            option);
    }

    Result SecureMonitorManager::SetConfig(
        ConfigItem  key,
        Bit64       value ) NN_NOEXCEPT
    {
        return impl::SetConfig(key, value);
    }

    Result  SecureMonitorManager::GenerateRandomBytes(
        void*       pOutBuffer,
        size_t      bufferSize ) NN_NOEXCEPT
    {
        return impl::GenerateRandomBytes(pOutBuffer, bufferSize);
    }

    Result  SecureMonitorManager::DecryptAndStoreGcKey(
        const void*         pData,
        size_t              dataSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        return impl::DecryptAndStoreGcKey(
            pData,
            dataSize,
            accessKey,
            pKeySource,
            keySourceSize);
    }

    Result  SecureMonitorManager::DecryptGcMessage(
        int*        pOutResultSize,
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize,
        const void* pLabelDigest,
        size_t      labelDigestSize) NN_NOEXCEPT
    {
        return impl::DecryptGcMessage(
            pOutResultSize,
            pResultBuffer,
            resultBufferSize,
            pCipher,
            cipherSize,
            pModulus,
            modulusSize,
            pLabelDigest,
            labelDigestSize);
    }

    Result  SecureMonitorManager::DecryptAndStoreSslClientCertKey(
        const void*         pData,
        size_t              dataSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        return impl::DecryptAndStoreSslClientCertKey(
            pData,
            dataSize,
            accessKey,
            pKeySource,
            keySourceSize);
    }

    Result  SecureMonitorManager::ModularExponentiateWithSslClientCertKey(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize) NN_NOEXCEPT
    {
        return impl::ModularExponentiateWithSslClientCertKey(
            pResultBuffer,
            resultBufferSize,
            pCipher,
            cipherSize,
            pModulus,
            modulusSize);
    }

    Result  SecureMonitorManager::DecryptAndStoreDrmDeviceCertKey(
        const void*         pData,
        size_t              dataSize,
        const AccessKey&    accessKey,
        const void*         pKeySource,
        size_t              keySourceSize ) NN_NOEXCEPT
    {
        return impl::DecryptAndStoreDrmDeviceCertKey(
            pData,
            dataSize,
            accessKey,
            pKeySource,
            keySourceSize);
    }

    Result  SecureMonitorManager::ModularExponentiateWithDrmDeviceCertKey(
        void*       pResultBuffer,
        size_t      resultBufferSize,
        const void* pCipher,
        size_t      cipherSize,
        const void* pModulus,
        size_t      modulusSize) NN_NOEXCEPT
    {
        return impl::ModularExponentiateWithDrmDeviceCertKey(
            pResultBuffer,
            resultBufferSize,
            pCipher,
            cipherSize,
            pModulus,
            modulusSize);
    }

    Result  SecureMonitorManager::IsDevelopment(
        bool* pIsDevelopment) NN_NOEXCEPT
    {
        return impl::IsDevelopment(pIsDevelopment);
    }

    Result SecureMonitorManager::GenerateSpecificAesKey(
        void*       pOutBuffer,
        size_t      outBufferSize,
        const void* pSource,
        size_t      sourceSize,
        int         generation,
        Bit32       purpose  ) NN_NOEXCEPT
    {
        return impl::GenerateSpecificAesKey(pOutBuffer, outBufferSize, pSource, sourceSize, generation, purpose);
    }

    Result SecureMonitorManager::DecryptAesKey(
        void*       pOutBuffer,
        size_t      outBufferSize,
        const void* pSource,
        size_t      sourceSize,
        int         generation,
        Bit32       option) NN_NOEXCEPT
    {
        return impl::DecryptAesKey(pOutBuffer, outBufferSize, pSource, sourceSize, generation, option);
    }

    Result SecureMonitorManager::ComputeCtr(
        void*       pOutBuffer,
        size_t      outBufferSize,
        int         slotIndex,
        const void* pOwner,
        const void* pInBuffer,
        size_t      inBufferSize,
        const void* pInitialCounter,
        size_t      initialCounterSize ) NN_NOEXCEPT
    {
        auto result = TestAesKeySlot(slotIndex, pOwner);
        if( result.IsFailure() )
        {
            return result;
        }

        return impl::ComputeCtr(pOutBuffer, outBufferSize, slotIndex, pInBuffer, inBufferSize, pInitialCounter, initialCounterSize);
    }

    Result SecureMonitorManager::ComputeCmac(
        void*       pOutBuffer,
        size_t      outBufferSize,
        int         slotIndex,
        const void* pOwner,
        const void* pInBuffer,
        size_t      inBufferSize ) NN_NOEXCEPT
    {
        auto result = TestAesKeySlot(slotIndex, pOwner);
        if( result.IsFailure() )
        {
            return result;
        }

        return impl::ComputeCmac(pOutBuffer, outBufferSize, slotIndex, pInBuffer, inBufferSize);
    }

    Result SecureMonitorManager::LoadEsDeviceKey(
        const void*      pData,
        size_t           dataSize,
        const AccessKey& accessKey,
        const void*      pKeySource,
        size_t           keySourceSize) NN_NOEXCEPT
    {
        return impl::LoadEsDeviceKey(pData, dataSize, accessKey, pKeySource, keySourceSize);
    }

    Result SecureMonitorManager::PrepareEsTitleKey(
        AccessKey*       pAccessKey,
        const void*      pCipher,
        size_t           cipherSize,
        const void*      pModulus,
        size_t           modulusSize,
        const void*      pLabelDigest,
        size_t           labelDigestSize,
        int              generation) NN_NOEXCEPT
    {
        return impl::PrepareEsTitleKey(
            pAccessKey,
            pCipher,
            cipherSize,
            pModulus,
            modulusSize,
            pLabelDigest,
            labelDigestSize,
            generation);
    }

    Result SecureMonitorManager::PrepareEsArchiveKey(
        AccessKey*       pAccessKey,
        const void*      pCipher,
        size_t           cipherSize,
        const void*      pModulus,
        size_t           modulusSize,
        const void*      pLabelDigest,
        size_t           labelDigestSize,
        int              generation) NN_NOEXCEPT
    {
        return impl::PrepareEsArchiveKey(
            pAccessKey,
            pCipher,
            cipherSize,
            pModulus,
            modulusSize,
            pLabelDigest,
            labelDigestSize,
            generation);
    }

    Result SecureMonitorManager::PrepareCommonEsTitleKey(
        AccessKey*       pAccessKey,
        const void*      pKeySource,
        size_t           keySourceSize,
        int              generation) NN_NOEXCEPT
    {
        return impl::PrepareCommonEsTitleKey(
            pAccessKey,
            pKeySource,
            keySourceSize,
            generation);
    }

    Result SecureMonitorManager::LoadPreparedAesKey(
        int              slotIndex,
        const void*      pOwner,
        const AccessKey& accessKey) NN_NOEXCEPT
    {
        auto result = TestAesKeySlot(slotIndex, pOwner);
        if( result.IsFailure() )
        {
            return result;
        }

        return impl::LoadPreparedAesKey(slotIndex, accessKey);
    }

    Result SecureMonitorManager::AllocateAesKeySlot(
        int*        pSlotIndex,
        const void* pOwner ) NN_NOEXCEPT
    {
        int slotIndex;
        auto result = impl::AllocateAesKeySlot(&slotIndex);
        if( result.IsFailure() )
        {
            return result;
        }

        m_AesKeySlotOwners[slotIndex] = pOwner;
        *pSlotIndex = slotIndex;
        return ResultSuccess();
    }

    Result SecureMonitorManager::DeallocateAesKeySlot(
        int         slotIndex,
        const void* pOwner ) NN_NOEXCEPT
    {
        auto result = TestAesKeySlot(slotIndex, pOwner);
        if( result.IsFailure() )
        {
            return result;
        }

        m_AesKeySlotOwners[slotIndex] = nullptr;
        return impl::DeallocateAesKeySlot(slotIndex);
    }

    void SecureMonitorManager::DeallocateAesKeySlots(
        const void* pOwner) NN_NOEXCEPT
    {
        for( int i = 0; i < impl::AesKeySlotCount; ++i )
        {
            if( m_AesKeySlotOwners[i] == pOwner )
            {
                m_AesKeySlotOwners[i] = nullptr;
                impl::DeallocateAesKeySlot(i);
            }
        }
    }

    nn::os::SystemEvent* SecureMonitorManager::GetAesKeySlotAvailableEvent() NN_NOEXCEPT
    {
        return impl::GetAesKeySlotAvailableEvent();
    }

    Result SecureMonitorManager::TestAesKeySlot(
        int         slotIndex,
        const void* pOwner ) NN_NOEXCEPT
    {
        if( ! (0 <= slotIndex && slotIndex < impl::AesKeySlotCount) )
        {
            return ResultInvalidKeySlot();
        }
        if( ! (m_AesKeySlotOwners[slotIndex] == pOwner) )
        {
            return ResultInvalidKeySlot();
        }

        return ResultSuccess();
    }

    Result SecureMonitorManager::SetBootReason(
        BootReasonValue bootReason) NN_NOEXCEPT
    {
        return impl::SetBootReason(bootReason);
    }

    Result SecureMonitorManager::GetBootReason(
        BootReasonValue* pOut) NN_NOEXCEPT
    {
        return impl::GetBootReason(pOut);
    }

}}  // namespace nn::spl
