﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/ldr/ldr_ProcessManagerApi.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/os/os_Thread.h>
#include <nn/os/os_MessageQueue.h>
#include <nn/os/os_MultipleWaitApi.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/pm/pm_Result.h>
#include <nn/spl/spl_Api.h>

#include <nn/fs/fs_ProgramRegistry.h>
#include <nn/sm/sm_ManagerApi.h>

#include "pm_ShellInterfaceServer.h"
#include "pm_ProcessInfo.h"
#include "pm_ProcessList.h"
#include "pm_Start1st.h"
#include "pm_StartSafeMode.h"
#include "pm_Spec.h"

namespace nn { namespace pm {

    namespace
    {
        typedef LockedAccessor<ProcessList> ProcessListAccessor;

        struct LaunchProcessInfo
        {
            os::ProcessId*          pProcessIdStore;
            ncm::ProgramLocation    location;
            int32_t                 flags;
        };

        class LaunchProcessAsyncInfo
        {
        private:
            LaunchProcessInfo   m_Info;
            Result              m_Result;

            os::Event           m_RequestEvent;
            os::Event           m_ReplyEvent;
            os::Mutex           m_RequestLock;

        public:
            LaunchProcessAsyncInfo() NN_NOEXCEPT
                : m_RequestEvent(os::EventClearMode_AutoClear)
                , m_ReplyEvent(os::EventClearMode_AutoClear)
                , m_RequestLock(false)
            {
            }

            void InitializeMultiWaitHolderWithRequestEvent(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
            {
                os::InitializeMultiWaitHolder(pHolder, m_RequestEvent.GetBase());
            }
            bool TestAndClearRequest() NN_NOEXCEPT
            {
                return m_RequestEvent.TryWait();
            }
            os::Event* GetRequestEvent() NN_NOEXCEPT
            {
                return &m_RequestEvent;
            }

            const LaunchProcessInfo& GetInfo() const NN_NOEXCEPT
            {
                return m_Info;
            }

            void Reply(Result result) NN_NOEXCEPT
            {
                m_Result = result;
                m_ReplyEvent.Signal();
            }

            Result CallAsync(nn::os::ProcessId* pOut, nn::ncm::ProgramLocation location, std::int32_t flags) NN_NOEXCEPT
            {
                std::lock_guard<os::Mutex> lock(m_RequestLock);

                m_Info.pProcessIdStore = pOut;
                m_Info.location = location;
                m_Info.flags = flags;

                m_RequestEvent.Signal();
                m_ReplyEvent.Wait();

                return m_Result;
            }
        };

        os::ThreadType              g_Thread;
        NN_ALIGNAS(4096)    Bit8    g_Stack[16 * 1024];
        NN_ALIGNAS(64)      Bit8    g_ProcessInfoBuffer[sizeof(ProcessInfo) * ProcessCountMax];

        LockedObject<ProcessList>   g_ProcessList;
        LockedObject<ProcessList>   g_ExitList;

        LaunchProcessAsyncInfo      g_LaunchProcessInfo;
        os::SystemEvent             g_ProcessEvent;
        os::SystemEvent             g_DebuggerEvent;
        os::SystemEvent             g_DebuggerEventCreateApplication;
        std::atomic<ncm::ProgramId> g_HookTarget;
        std::atomic<bool>           g_HookApplication(false);
        bool                        g_Start1stExecuted = false;


        nn::Result StartProcessImpl(ProcessInfo* pInfo, int priority, int idealCore, size_t stackSize) NN_NOEXCEPT
        {
            Result result = nn::svc::StartProcess(pInfo->GetHandle(), priority, idealCore, stackSize);
            if( result.IsSuccess() )
            {
                pInfo->SetState(svc::ProcessState_Running);
            }

            return result;
        }

        void CleanupProcess(ProcessInfo* ppi) NN_NOEXCEPT
        {
            nn::fs::UnregisterProgram(ppi->GetId().value);
            nn::sm::UnregisterProcess(ppi->GetId());
            ldr::UnpinProgram(ppi->GetPinId());
            svc::CloseHandle(ppi->GetHandle());
        }

        void CleanupProcessInfo(ProcessListAccessor* pList, ProcessInfo* ppi) NN_NOEXCEPT
        {
            (*pList)->Remove(ppi);
            delete ppi;
        }

        int MakeCreateProcessFlags(int flags)
        {
            int loaderFlags = 0;

            if( (flags & LaunchProgramFlags_NotifyException) != 0 || (flags & LaunchProgramFlags_NotStart) == 0 )
            {
                // NotifyException フラグが付いている場合だけでなく、
                // NotStart が付いていない場合（デバッガからの起動ではない場合）も、
                // クラッシュレポート作成のため強制的に EnableException を付与する
                loaderFlags |= ldr::CreateProcessFlags_EnableException;
            }
            if( (flags & LaunchProgramFlags_DisableAslr) != 0 )
            {
                loaderFlags |= ldr::CreateProcessFlags_DisableAslr;
            }

            return loaderFlags;
        }

        bool IsApplicationRunning() NN_NOEXCEPT
        {
            ProcessListAccessor list(&g_ProcessList);

            for( auto& e : *list )
            {
                if( e.IsApplication() )
                {
                    return true;
                }
            }

            return false;
        }


        nn::Result LaunchProgramImpl(os::MultiWaitType* pmw, nn::os::ProcessId* pProcessId, const ncm::ProgramLocation& location, int32_t flags) NN_NOEXCEPT
        {
            nn::ldr::ProgramInfo pi;
            nn::Result result;

            // 起動対象のプログラム情報取得
            NN_RESULT_DO(nn::ldr::GetProgramInfo(&pi, location));

            // 起動対象がアプリケーションで、かつ既にアプリケーションが起動していたらエラー
            if (((pi.flags & nn::ldr::ProgramInfoFlag_ProgramTypeMask) == nn::ldr::ProgramInfoFlag_Application) && IsApplicationRunning())
            {
                return ResultApplicationRunning();
            }

            // npdm に記載されている ProgramId で ProgramLocation を訂正 // TODO: 削除
            ncm::ProgramLocation correctedLocation = location;
            correctedLocation.programId.value = pi.programId;

            ldr::PinId pinId;
            NN_RESULT_DO(nn::ldr::PinProgram(&pinId, correctedLocation));

            // リソースリミットを取得
            WaitResource(pi);
            auto resourceLimit = GetResourceLimit(pi);

            // プロセス作成
            nn::svc::Handle handle;
            int loaderFlags = MakeCreateProcessFlags(flags);
            result = nn::ldr::CreateProcess(&handle, pinId, loaderFlags, resourceLimit);
            if( result.IsFailure() )
            {
                nn::ldr::UnpinProgram(pinId);
                return result;
            }

            // ProcessId 取得
            nn::os::ProcessId processId;
            nn::svc::GetProcessId(&processId.value, handle);

            // 管理対象に登録
            ProcessInfo* pNew = new ProcessInfo(handle, processId, pinId, correctedLocation);

            {
                ProcessListAccessor list(&g_ProcessList);
                list->push_back(*pNew);
                pNew->LinkToMultiWait(pmw);
            }

            Bit8* pSacd = pi.capabilities;
            Bit8* pSac  = pSacd + pi.sacdSize;
            Bit8* pFacd = pSac + pi.sacSize;
            Bit8* pFac  = pFacd + pi.facdSize;

            // FS にプログラム情報を登録
            NN_RESULT_DO(nn::fs::RegisterProgram(processId.value, correctedLocation.programId.value, correctedLocation.storageId, pFac, pi.facSize, pFacd, pi.facdSize));

            // SM にプログラム情報を登録
            NN_RESULT_DO(nn::sm::RegisterProcess(processId, pSacd, pi.sacdSize, pSac, pi.sacSize));

            // アプリケーション
            if( (pi.flags & ldr::ProgramInfoFlag_Application) != 0 )
            {
                pNew->SetApplication();
            }

            // 起動通知
            if( (flags & LaunchProgramFlags_NotifyStart) && (pi.flags & ldr::ProgramInfoFlag_EnableDebug) != 0 )
            {
                pNew->SetNotifyRunningRequire();
            }

            // 終了通知
            if( (flags & LaunchProgramFlags_NotifyExit) != 0 )
            {
                pNew->SetNotifyExitRequire();
            }

            // デバッグ通知
            if( (flags & LaunchProgramFlags_NotifyDebug) != 0 && (pi.flags & ldr::ProgramInfoFlag_EnableDebug) != 0 )
            {
                pNew->SetNotifyDebugRequire();
            }

            // Hook のチェック
            if( correctedLocation.programId == g_HookTarget )
            {
                g_DebuggerEvent.Signal();
                g_HookTarget = ncm::ProgramId::GetInvalidId();
            }
            else if( pNew->IsApplication() && g_HookApplication )
            {
                g_DebuggerEventCreateApplication.Signal();
                g_HookApplication = false;
            }
            else
            {
                // プロセス開始
                if( (flags & LaunchProgramFlags_NotStart) == 0 )
                {
                    result = StartProcessImpl(pNew, pi.threadPriority, pi.idealCoreNumber, pi.stackSize);
                    if( result.IsFailure() )
                    {
                        pNew->UnlinkFromMultiWait();

                        {
                            ProcessListAccessor list(&g_ProcessList);
                            CleanupProcess(pNew);
                            CleanupProcessInfo(&list, pNew);
                        }

                        return result;
                    }
                }
            }

            *pProcessId = processId;

            return nn::ResultSuccess();
        }

        Result OnQueueEntry(os::MultiWaitType* pmw, const LaunchProcessInfo& q) NN_NOEXCEPT
        {
            return LaunchProgramImpl(pmw, q.pProcessIdStore, q.location, q.flags);
        }

        void OnProcessEvent(ProcessListAccessor* pList, ProcessInfo* ppi) NN_NOEXCEPT
        {
            int64_t value;

            svc::ResetSignal(ppi->GetHandle());
            svc::GetProcessInfo(&value, ppi->GetHandle(), svc::ProcessInfoType_State);

            auto currState = static_cast<svc::ProcessState>(value);
            auto prevState = ppi->GetState();

            // 別スレッドでリクエスト処理が走るので先に状態遷移等を行っておき
            // あとで状態ごとの処理を行う
            ppi->SetState(currState);

            if( prevState == svc::ProcessState_WaitAttach )
            {
                if( currState != prevState )
                {
                    ppi->ClearExceptionWaiting();
                }
            }


            switch(currState)
            {
            case svc::ProcessState_Initializing:     // 初期化中
                break;

            case svc::ProcessState_PreAttached:      // 初期化中に Attach
                break;

            case svc::ProcessState_Running:          // 実行中
                {
                    if( ppi->IsNotifyDebugRequired() )
                    {
                        ppi->SetDebugState(false);
                        ppi->SetDebugStateChanged();
                        g_ProcessEvent.Signal();
                    }
                    else if( ppi->IsNotifyRunningRequired() )
                    {
                        ppi->SetRunningStateChanged();
                        ppi->ClearNotifyRunningRequire();
                        g_ProcessEvent.Signal();
                    }
                }
                break;

            case svc::ProcessState_WaitAttach:       // JIT デバッグ発生
                {
                    // JIT デバッグフラグを立てて通知する

                    ppi->SetJitFlags();
                    g_ProcessEvent.Signal();
                }
                break;

            case svc::ProcessState_Attached:         // デバッグ中 動作中
                {
                    // デバッグ状態を更新して通知する

                    if( ppi->IsNotifyDebugRequired() )
                    {
                        ppi->SetDebugState(false);
                        ppi->SetDebugStateChanged();
                        g_ProcessEvent.Signal();
                    }
                }
                break;

            case svc::ProcessState_Breaked:          // デバッグ中 停止中
                {
                    // デバッグ状態を更新して通知する

                    if( ppi->IsNotifyDebugRequired() )
                    {
                        ppi->SetDebugState(true);
                        ppi->SetDebugStateChanged();
                        g_ProcessEvent.Signal();
                    }
                }
                break;

            case svc::ProcessState_Terminating:      // 終了開始
                break;

            case svc::ProcessState_Terminated:       // 終了
                {
                    // 監視対象から外す
                    ppi->UnlinkFromMultiWait();

                    // リソースを解放する
                    CleanupProcess(ppi);


                    // 通知 or ProcessInfo を解放する

                    if( ppi->IsNotifyExitRequired() )
                    {
                        // exitList に繋ぎかえる
                        (*pList)->Remove(ppi);

                        {
                            ProcessListAccessor exitList(&g_ExitList);
                            exitList->push_back(*ppi);
                        }

                        g_ProcessEvent.Signal();
                    }
                    else
                    {
                        // 通知が不要なら ProcessInfo も解放
                        CleanupProcessInfo(pList, ppi);
                    }
                }
                break;

            default:
                NN_ABORT();
            }
        }

        void ThreadBody(void*) NN_NOEXCEPT
        {
            os::MultiWaitHolderType rHolder;
            os::MultiWaitType mw;

            os::InitializeMultiWait(&mw);
            g_LaunchProcessInfo.InitializeMultiWaitHolderWithRequestEvent(&rHolder);
            os::LinkMultiWaitHolder(&mw, &rHolder);

            for(;;)
            {
                auto pSignaled = os::WaitAny(&mw);

                if( pSignaled == &rHolder )
                {
                    if( g_LaunchProcessInfo.TestAndClearRequest() )
                    {
                        auto result = OnQueueEntry(&mw, g_LaunchProcessInfo.GetInfo());
                        g_LaunchProcessInfo.Reply(result);
                    }
                }
                else
                {
                    ProcessListAccessor list(&g_ProcessList);

                    OnProcessEvent(
                        &list,
                        reinterpret_cast<ProcessInfo*>(
                            os::GetMultiWaitHolderUserData(pSignaled)));
                }
            }
        }

    }

    ProcessManager* ProcessManager::g_pInstance;

    void ProcessManager::Initialize() NN_NOEXCEPT
    {
        Result result;
        NN_SDK_ASSERT(g_HookTarget.is_lock_free());

        ProcessInfo::InitializeHeap(g_ProcessInfoBuffer, sizeof(g_ProcessInfoBuffer));

        new(&g_ProcessEvent) os::SystemEvent(os::EventClearMode_AutoClear, true);
        new(&g_DebuggerEvent) os::SystemEvent(os::EventClearMode_ManualClear, true);
        new(&g_DebuggerEventCreateApplication) os::SystemEvent(os::EventClearMode_ManualClear, true);

        InitializeSpec();

        result = os::CreateThread(
                    &g_Thread,
                    ThreadBody,
                    nullptr,
                    g_Stack,
                    sizeof(g_Stack),
                    NN_SYSTEM_THREAD_PRIORITY(pm, ProcessTrack));
        NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
        os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(pm, ProcessTrack));

        os::StartThread(&g_Thread);

        result = sm::InitializeForManager();
        NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
    }

    // 指定のプログラムを起動する
    nn::Result ProcessManager::LaunchProgram(nn::os::ProcessId* pOut, const ncm::ProgramLocation& location, std::int32_t flags) NN_NOEXCEPT
    {
        return g_LaunchProcessInfo.CallAsync(pOut, location, flags);
    }

    // ProcessId で指定されるプロセスを強制的に終了させる
    nn::Result ProcessManager::TerminateProcess(nn::os::ProcessId id) NN_NOEXCEPT
    {
        {
            ProcessListAccessor list(&g_ProcessList);

            auto pFound = list->FindById(id);
            if( pFound == NULL )
            {
                return ResultProcessNotFound();
            }

            return nn::svc::TerminateProcess(pFound->GetHandle());
        }
    }

    // ProgramId で指定されるプログラムから起動されたプロセスを全て強制的に終了させる
    nn::Result ProcessManager::TerminateProgram(nn::ncm::ProgramId id) NN_NOEXCEPT
    {
        {
            ProcessListAccessor list(&g_ProcessList);

            for( auto i = list->begin(); i != list->end(); ++i )
            {
                if( i->GetProgramId() == id )
                {
                    Result r = svc::TerminateProcess(i->GetHandle());
                    if( r.IsFailure() )
                    {
                        return r;
                    }
                }
            }
        }

        return ResultSuccess();
    }

    // shell が必要とするプロセス状態が変化した時にシグナルされる Event のハンドルを取得する
    nn::Result ProcessManager::GetProcessEventHandle(nn::svc::Handle* pOut) NN_NOEXCEPT
    {
        *pOut = svc::Handle(g_ProcessEvent.GetReadableHandle());

        return ResultSuccess();
    }

    nn::Result ProcessManager::GetProcessEventInfo(nn::pm::ProcessEventInfo* pOut) NN_NOEXCEPT
    {
        {
            ProcessListAccessor list(&g_ProcessList);

            for( auto i = list->begin(); i != list->end(); ++i )
            {
                if( i->IsStarted() && i->IsRunningStateChanged() )
                {
                    i->ClearRunningStateChanged();
                    pOut->event     = ProcessEvent_Started;
                    pOut->processId = i->GetId();
                    return ResultSuccess();
                }
                if( i->IsDebugStateChanged() )
                {
                    i->ClearDebugStateChanged();
                    pOut->event     = i->IsBreaked() ? ProcessEvent_DebugBreaked: ProcessEvent_DebugRunning;
                    pOut->processId = i->GetId();
                    return ResultSuccess();
                }
                if( i->IsExceptionOccured() )
                {
                    i->ClearExceptionOccured();
                    pOut->event     = ProcessEvent_Exception;
                    pOut->processId = i->GetId();
                    return ResultSuccess();
                }
            }
        }
        {
            ProcessListAccessor exitList(&g_ExitList);

            if( ! exitList->empty() )
            {
                auto& pi = exitList->front();

                pOut->event     = ProcessEvent_Exit;
                pOut->processId = pi.GetId();

                CleanupProcessInfo(&exitList, &pi);
                return ResultSuccess();
            }
        }

        pOut->event             = ProcessEvent_None;
        pOut->processId.value   = 0;
        return ResultSuccess();
    }

    // ProcessManager が保持している JIT デバッグフラグがセットされているプロセスの
    // ProcessId のリストを取得する
    nn::Result ProcessManager::GetExceptionProcessIdList(int* pStoredCount, os::ProcessId* pBuffer, int bufferEntryCount) NN_NOEXCEPT
    {
        int count = 0;

        if( bufferEntryCount > 0 )
        {
            ProcessListAccessor list(&g_ProcessList);

            for( ProcessList::iterator i = list->begin(); i != list->end(); ++i )
            {
                if( i->IsExceptionWaiting() )
                {
                    pBuffer[count] = i->GetId();
                    count++;

                    if( count >= bufferEntryCount )
                    {
                        break;
                    }
                }
            }
        }

        *pStoredCount = count;

        return ResultSuccess();
    }

    // ProcessId で指定されるプロセスの実行を開始する
    nn::Result ProcessManager::StartProcess(os::ProcessId id) NN_NOEXCEPT
    {
        Result result;

        {
            ProcessListAccessor list(&g_ProcessList);

            auto pFound = list->FindById(id);
            if( pFound == NULL )
            {
                return ResultProcessNotFound();
            }

            if( pFound->IsStarted() )
            {
                return ResultAlreadyStarted();
            }

            nn::ldr::ProgramInfo pi;

            // 起動対象のプログラム情報取得
            result = nn::ldr::GetProgramInfo(&pi, pFound->GetProgramLocation());
            if( result.IsFailure() )
            {
                return result;
            }

            result = StartProcessImpl(pFound, pi.threadPriority, pi.idealCoreNumber, pi.stackSize);
        }

        return result;
    }

    // ProgramId で指定されるプログラムから起動されたプロセスのうち
    // いずれか一つのプロセスの ProcessId を取得する
    nn::Result ProcessManager::GetProcessId(os::ProcessId* pOut, ncm::ProgramId programId) NN_NOEXCEPT
    {
        {
            ProcessListAccessor list(&g_ProcessList);

            auto pFound = list->FindByProgramId(programId);
            if( pFound == NULL )
            {
                return ResultProcessNotFound();
            }

            *pOut = pFound->GetId();
        }

        return ResultSuccess();
    }

    // アプリケーションの ProcessId を取得する
    nn::Result ProcessManager::GetApplicationProcessId(os::ProcessId* pOut) NN_NOEXCEPT
    {
        {
            ProcessListAccessor list(&g_ProcessList);

            for( auto& e : *list )
            {
                if( e.IsApplication() )
                {
                    *pOut = e.GetId();
                    return ResultSuccess();
                }
            }
        }

        return ResultProcessNotFound();
    }

    // ProgramId で指定されるプログラムからプロセスを起動しようとした場合に
    // デバッガがアタッチできるようにするようにする
    nn::Result ProcessManager::HookToCreateProcess(os::NativeHandle* pOut, ncm::ProgramId programId) NN_NOEXCEPT
    {
        *pOut = os::InvalidNativeHandle;

        ncm::ProgramId invalidId = ncm::ProgramId::GetInvalidId();
        if( ! g_HookTarget.compare_exchange_strong(invalidId, programId) )
        {
            return ResultHookBusy();
        }

        *pOut = g_DebuggerEvent.GetReadableHandle();

        return ResultSuccess();
    }

    nn::Result ProcessManager::HookToCreateApplicationProcess(os::NativeHandle* pOut) NN_NOEXCEPT
    {
        bool expectedValue = false;
        if( ! g_HookApplication.compare_exchange_strong(expectedValue, true) )
        {
            return ResultHookBusy();
        }

        *pOut = g_DebuggerEventCreateApplication.GetReadableHandle();

        return ResultSuccess();
    }

    nn::Result ProcessManager::ClearHook(int hookTypes) NN_NOEXCEPT
    {
        if( (hookTypes & HookType_ProgramId) != 0 )
        {
            g_HookTarget = ncm::ProgramId::GetInvalidId();
        }
        if( (hookTypes & HookType_Application) != 0 )
        {
            g_HookApplication = false;
        }

        return ResultSuccess();
    }

    void ProcessManager::NotifyBootFinished() NN_NOEXCEPT
    {
        // 二回目以降は無視
        if ( !g_Start1stExecuted )
        {
#if defined(NN_PM_BUILD_TYPE_SAFE_MODE)
            nn::pm::StartSafeMode();
#else
            nn::pm::Start1st();
#endif
            g_Start1stExecuted = true;
        }
    }

    nn::Result ProcessManager::BoostSystemMemoryResourceLimit(int64_t boostSize) NN_NOEXCEPT
    {
        return pm::BoostSystemMemoryResourceLimit(boostSize);
    }

    nn::Result ProcessManager::GetProgramId(ncm::ProgramId* pOut, nn::os::ProcessId id) NN_NOEXCEPT
    {
        {
            ProcessListAccessor list(&g_ProcessList);

            auto pFound = list->FindById(id);
            if( pFound == NULL )
            {
                return ResultProcessNotFound();
            }

            *pOut = pFound->GetProgramId();
        }

        return ResultSuccess();
    }

}}  // namespace nn::pm
