﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <random>

#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/ldr/ldr_Types.h>
#include <nn/ldr/ldr_Result.h>
#include <nn/util/util_BitUtil.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_MemoryMapSelect.h>

#include "ldr_RoManager.h"
#include "ldr_DataTypes.h"
#include "ldr_AutoClose.h"
#include "ldr_Random.h"

namespace nn { namespace ldr {

    const PinId RoManager::InvalidPinId = {};
    RoManager         g_RoManager;

    //
    // RoManager
    //
    bool RoManager::Allocate(PinId* pOutPinId, const ncm::ProgramLocation& location) NN_NOEXCEPT
    {
        // 互換性のため初期化しておく
        *pOutPinId = InvalidPinId;
        auto pFound = AllocateProcessInfo();
        if (pFound == nullptr)
        {
            return false;
        }

        m_PinId++;
        std::memset(pFound, 0, sizeof(*pFound));
        pFound->programLocation = location;
        pFound->pinId.value = m_PinId;
        pFound->isUsed = true;
        *pOutPinId = pFound->pinId;
        return true;
    }

    bool RoManager::Free(PinId pinId) NN_NOEXCEPT
    {
        auto pFound = FindProcessInfo(pinId);
        if( pFound == nullptr )
        {
            return false;
        }

        pFound->isUsed = false;

        for( int i = 0; i < NsoCount; ++i )
        {
            pFound->nsoInfo[i].isUsed = false;
        }

        return true;
    }

    void RoManager::RegisterProcess(
            PinId                           pinId,
            os::ProcessId                   processId,
            Bit64                           programId,
            bool                            is64BitAddressSpace) NN_NOEXCEPT
    {
        auto pFound = FindProcessInfo(pinId);
        if( pFound == nullptr )
        {
            return;
        }

        pFound->processId = processId;
        pFound->programId = programId;
    }

    bool RoManager::GetProgramLocation(ncm::ProgramLocation* outValue, PinId pinId) NN_NOEXCEPT
    {
        auto pFound = FindProcessInfo(pinId);
        if( pFound == NULL )
        {
            return false;
        }

        *outValue = pFound->programLocation;
        return true;
    }

    void RoManager::AddNso(PinId id, const Bit8* pModuleId, uint64_t address, uint64_t size) NN_NOEXCEPT
    {
        auto pFound = FindProcessInfo(id);
        if( pFound == nullptr )
        {
            return;
        }

        auto pInfo = AllocateNsoInfo(pFound);
        if ( pInfo == nullptr )
        {
            return;
        }

        std::memcpy(&pInfo->moduleId, pModuleId, sizeof(pInfo->moduleId));
        pInfo->moduleAddress = address;
        pInfo->moduleSize = size;
        pInfo->isUsed = true;
    }

    bool RoManager::GetProcessModuleInfo(
            int*                            pOutCount,
            ModuleInfo*                     pBuffer,
            int                             num,
            os::ProcessId                   processId) NN_NOEXCEPT
    {
        auto pFound = FindProcessInfo(processId);
        if( pFound == nullptr )
        {
            return false;
        }

        int count = 0;
        for( int i = 0; i < NsoCount && count < num; ++i )
        {
            if( !pFound->nsoInfo[i].isUsed )
            {
                continue;
            }

            ModuleInfo* pInfo = &pBuffer[count++];
            NN_STATIC_ASSERT(sizeof(pInfo->moduleId) == sizeof(pFound->nsoInfo[i].moduleId));
            std::memcpy(pInfo->moduleId, &pFound->nsoInfo[i].moduleId, sizeof(pInfo->moduleId));
            pInfo->address = pFound->nsoInfo[i].moduleAddress;
            pInfo->size = pFound->nsoInfo[i].moduleSize;
        }

        *pOutCount = count;

        return true;
    }

    RoManager::ProcessInfo* RoManager::AllocateProcessInfo() NN_NOEXCEPT
    {
        for( int i = 0; i < RoManager::ProcessCount; ++i )
        {
            if( !m_Processes[i].isUsed )
            {
                return &m_Processes[i];
            }
        }

        return nullptr;
    }

    RoManager::ProcessInfo* RoManager::FindProcessInfo(PinId pinId) NN_NOEXCEPT
    {
        for( int i = 0; i < RoManager::ProcessCount; ++i )
        {
            if( m_Processes[i].isUsed && m_Processes[i].pinId == pinId )
            {
                return &m_Processes[i];
            }
        }

        return nullptr;
    }

    RoManager::ProcessInfo* RoManager::FindProcessInfo(os::ProcessId processId) NN_NOEXCEPT
    {
        for( int i = 0; i < RoManager::ProcessCount; ++i )
        {
            if( m_Processes[i].isUsed && m_Processes[i].processId == processId )
            {
                return &m_Processes[i];
            }
        }

        return nullptr;
    }

    RoManager::ProcessInfo* RoManager::FindProcessInfo(Bit64 programId) NN_NOEXCEPT
    {
        for( int i = 0; i < RoManager::ProcessCount; ++i )
        {
            if( m_Processes[i].isUsed && m_Processes[i].programId == programId )
            {
                return &m_Processes[i];
            }
        }

        return nullptr;
    }

    RoManager::NsoInfo* RoManager::AllocateNsoInfo(ProcessInfo* pInfo) NN_NOEXCEPT
    {
        for( int i = 0; i < NsoCount; ++i )
        {
            if( !pInfo->nsoInfo[i].isUsed )
            {
                return &pInfo->nsoInfo[i];
            }
        }

        return nullptr;
    }

}}  // nn::ldr
