﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ldr/ldr_ProcessManagerApi.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/ldr/ldr_Types.h>
#include <nn/ldr/ldr_Result.h>
#include <climits>

namespace nn { namespace ldr {

    class Loader
    {
    public:
        static void Initialize() NN_NOEXCEPT;

        nn::Result CreateProcess(nn::sf::Out<nn::sf::NativeHandle> pOut, PinId pinId, int flags, nn::sf::NativeHandle&& resourceLimit) NN_NOEXCEPT
        {
            nn::svc::Handle handle;
            auto x = CreateProcess(&handle, pinId, flags, nn::svc::Handle(resourceLimit.GetOsHandle()));
            pOut.Set(nn::sf::NativeHandle(handle.operator nnHandle().value, true));
            return x;
        }
        nn::Result GetProgramInfo(nn::sf::Out<nn::ldr::ProgramInfo> outStorage, const ncm::ProgramLocation& location) NN_NOEXCEPT
        {
            return GetProgramInfo(outStorage.GetPointer(), location);
        }
        nn::Result SetProgramArgument(ncm::ProgramId programId, const nn::sf::InArray<nn::Bit8>& argument, std::uint32_t size) NN_NOEXCEPT
        {
            return SetProgramArgument(programId, reinterpret_cast<const void*>(argument.GetData()), argument.GetLength());
        }
        nn::Result FlushArguments() NN_NOEXCEPT;
        nn::Result PinProgram(nn::sf::Out<nn::ldr::PinId> outId, const ncm::ProgramLocation& programLocation) NN_NOEXCEPT
        {
            return PinProgram(outId.GetPointer(), programLocation);
        }
        nn::Result UnpinProgram(PinId id) NN_NOEXCEPT;
        nn::Result FlushArguments2() NN_NOEXCEPT
        {
            return FlushArguments();
        }
        nn::Result SetProgramArgument2(ncm::ProgramId programId, const nn::sf::InArray<nn::Bit8>& argument, std::uint32_t size) NN_NOEXCEPT
        {
            return SetProgramArgument(programId, reinterpret_cast<const void*>(argument.GetData()), argument.GetLength());
        }
        nn::Result GetProcessModuleInfo(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ldr::ModuleInfo>& outModules, nn::os::ProcessId id) NN_NOEXCEPT
        {
            if (outModules.GetLength() > INT_MAX)
            {
                return nn::ldr::ResultInvalidSize();
            }

            return GetProcessModuleInfo(outCount.GetPointer(), outModules.GetData(), outModules.GetLength(), id);
        }

    private:
        nn::Result CreateProcess(nn::svc::Handle* pOut, PinId pinId, int flags, nn::svc::Handle resourceLimit) NN_NOEXCEPT;
        nn::Result GetProgramInfo(nn::ldr::ProgramInfo* pOut, const ncm::ProgramLocation& location) NN_NOEXCEPT;
        nn::Result SetProgramArgument(ncm::ProgramId programId, const void* pArgument, size_t size) NN_NOEXCEPT;
        nn::Result PinProgram(PinId* pOutId, const ncm::ProgramLocation& programLocation) NN_NOEXCEPT;
        nn::Result GetProcessModuleInfo(int* pOutCount, ldr::ModuleInfo* pBuffer, int num, nn::os::ProcessId id) NN_NOEXCEPT;
    };


}}  // namespace nn::ldr
