﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/util/util_BitPack.h>
#include <nn/ldr/ldr_Result.h>
#include <nn/ldr/ldr_Types.h>

#include "ldr_CapabilityTest.h"

namespace nn { namespace ldr {

    namespace
    {
        const util::BitPack32 NumberNotFound = { 0 };

        struct CapabilityNo3
        {
            int     coreNumberMax;
            int     coreNumberMin;
            int     priorityHigh;
            int     priorityLow;

            typedef util::BitPack32::Field<                  4,  6, int> PriorityLow;
            typedef util::BitPack32::Field<  PriorityLow::Next,  6, int> PriorityHigh;
            typedef util::BitPack32::Field< PriorityHigh::Next,  8, int> CoreNumberMin;
            typedef util::BitPack32::Field<CoreNumberMin::Next,  8, int> CoreNumberMax;
        };
        struct CapabilityNo4
        {
            typedef util::BitPack32::Field<                  5, 24, Bit32> Bits;
            typedef util::BitPack32::Field<         Bits::Next,  3, int> Index;
        };
        struct CapabilityNo6
        {
            typedef util::BitPack32::Field<                  7, 24, Bit32> Address24;
            typedef util::BitPack32::Field<    Address24::Next,  1, bool> Flag;
        };
        struct CapabilityNo7
        {
            typedef util::BitPack32::Field<                  8, 24, Bit32> Address24;
        };
        struct CapabilityNo11
        {
            typedef util::BitPack32::Field<                 12, 10, int> Number0;
            typedef util::BitPack32::Field<      Number0::Next, 10, int> Number1;
        };
        struct CapabilityNo13
        {
            enum ProgramTypes
            {
                ProgramTypes_Undefined,
                ProgramTypes_Application,
                ProgramTypes_Applet,
            };

            typedef util::BitPack32::Field<                 14,  3, int> ProgramType;
        };
        struct CapabilityNo14
        {
            typedef util::BitPack32::Field<                 15,  4, int> Minor;
            typedef util::BitPack32::Field<        Minor::Next, 13, int> Major;
        };
        struct CapabilityNo15
        {
            typedef util::BitPack32::Field<                 16, 10, int> HandleCountMax;
        };
        struct CapabilityNo16
        {
            typedef util::BitPack32::Field<                 17,  1, bool> AllowDebug;
            typedef util::BitPack32::Field<   AllowDebug::Next,  1, bool> PermitForceDebug;
        };


        bool operator ==(const util::BitPack32& a, const util::BitPack32& b)
        {
            return a.storage == b.storage;
        }

        bool IsCapabilityNumber(util::BitPack32 e, int no) NN_NOEXCEPT
        {
            Bit32 mask = (~0u >> (31 - no));
            Bit32 test = (mask >> 1);
            return ((e.storage & mask) == test);
        }
        void ParseCapabilityNo3(CapabilityNo3* p, util::BitPack32 e) NN_NOEXCEPT
        {
            p->coreNumberMax = e.Get<CapabilityNo3::CoreNumberMax>();
            p->coreNumberMin = e.Get<CapabilityNo3::CoreNumberMin>();
            p->priorityHigh  = e.Get<CapabilityNo3::PriorityHigh>();
            p->priorityLow   = e.Get<CapabilityNo3::PriorityLow>();
        }
        util::BitPack32 FindNumber(const util::BitPack32* pKcd, const int kcdCount, int no) NN_NOEXCEPT
        {
            for( int kcdi = 0; kcdi < kcdCount; ++kcdi )
            {
                util::BitPack32 d = pKcd[kcdi];

                if( IsCapabilityNumber(d, no) )
                {
                    return d;
                }
            }

            return NumberNotFound;
        }

        // No.3
        // スレッド優先度とコア番号
        // 複数指定 不可
        bool TestNo3(util::BitPack32 c, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            auto d = FindNumber(pKcd, kcdCount, 3);
            if( d == NumberNotFound )
            {
                return false;
            }

            CapabilityNo3 value;
            ParseCapabilityNo3(&value, c);

            CapabilityNo3 limit;
            ParseCapabilityNo3(&limit, d);

            // 優先度は値と価値が逆
            if( ! ( (value.priorityLow  <= limit.priorityLow)
                 && (value.priorityHigh >= limit.priorityHigh)
                 && (value.priorityHigh <= value.priorityLow) ))
            {
                return false;
            }

            if( ! ( (value.coreNumberMin >= limit.coreNumberMin)
                 && (value.coreNumberMax <= limit.coreNumberMax)
                 && (value.coreNumberMin <= value.coreNumberMax) ))
             {
                return false;
             }

            return true;
        }

        // No.4
        // システムコール番号
        // 複数指定 可
        bool TestNo4(util::BitPack32 c, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            int valueIndex  = c.Get<CapabilityNo4::Index>();
            Bit32 valueBits = c.Get<CapabilityNo4::Bits>();

            for( int kcdi = 0; kcdi < kcdCount; ++kcdi )
            {
                util::BitPack32 d = pKcd[kcdi];

                if( IsCapabilityNumber(d, 4) )
                {
                    int limitIndex = d.Get<CapabilityNo4::Index>();
                    int limitBits  = d.Get<CapabilityNo4::Bits>();

                    if( (limitIndex == valueIndex)
                     && ((valueBits & limitBits) == valueBits) )
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        // No.6
        // 物理アドレス範囲マップ
        // 複数指定 可
        // ペア指定
        bool TestNo6(util::BitPack32 c0, util::BitPack32 c1, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            if( ! IsCapabilityNumber(c1, 6) )
            {
                return false;
            }

            uint32_t valueAddress24     = c0.Get<CapabilityNo6::Address24>();
            uint32_t valueSize24        = c1.Get<CapabilityNo6::Address24>();
            bool     valueIsReadOnly    = c0.Get<CapabilityNo6::Flag>();
            bool     valueIsStatic      = c1.Get<CapabilityNo6::Flag>();

            if( valueSize24 >= (1u << 20) )
            {
                return false;
            }

            for( int kcdi = 0; kcdi < kcdCount; ++kcdi )
            {
                util::BitPack32 d0 = pKcd[kcdi];

                if( IsCapabilityNumber(d0, 6) )
                {
                    ++kcdi;
                    if( ! (kcdi < kcdCount) )
                    {
                        return false;
                    }

                    util::BitPack32 d1 = pKcd[kcdi];
                    if( ! IsCapabilityNumber(d1, 6) )
                    {
                        return false;
                    }

                    uint32_t limitAddress24     = d0.Get<CapabilityNo6::Address24>();
                    uint32_t limitSize24        = d1.Get<CapabilityNo6::Address24>();
                    bool     limitIsReadOnly    = d0.Get<CapabilityNo6::Flag>();
                    bool     limitIsStatic      = d1.Get<CapabilityNo6::Flag>();

                    if( limitSize24 >= (1u << 20) )
                    {
                        continue;
                    }

                    if( (limitIsReadOnly == valueIsReadOnly) && (limitIsStatic == valueIsStatic) )
                    {
                        uint32_t valueAddressEnd24 = valueAddress24 + valueSize24;
                        uint32_t limitAddressEnd24 = limitAddress24 + limitSize24;

                        if( (valueAddress24    >= limitAddress24   )
                         && (valueAddressEnd24 <= limitAddressEnd24)
                         && (valueAddress24    <= valueAddressEnd24) )
                        {
                            return true;
                        }
                    }
                }
            }

            return false;
        }
        // No.7
        // 物理アドレス単ページマップ
        // 複数指定 可
        bool TestNo7(util::BitPack32 c, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            for( int kcdi = 0; kcdi < kcdCount; ++kcdi )
            {
                util::BitPack32 d = pKcd[kcdi];

                if( IsCapabilityNumber(d, 7) )
                {
                    if( c.storage == d.storage )
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        // No.11
        // 割り込み番号
        // 複数指定 可
        bool TestNo11Half(int number, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            if( number == 1023 )
            {
                return true;
            }

            for( int kcdi = 0; kcdi < kcdCount; ++kcdi )
            {
                util::BitPack32 d = pKcd[kcdi];

                if( IsCapabilityNumber(d, 11) )
                {
                    int limitNumber0 = d.Get<CapabilityNo11::Number0>();
                    int limitNumber1 = d.Get<CapabilityNo11::Number1>();

                    if( (limitNumber0 == 1023) && (limitNumber1 == 1023) )
                    {
                        return true;
                    }
                    if( (number == limitNumber0) || (number == limitNumber1) )
                    {
                        return true;
                    }
                }
            }

            return false;
        }
        bool TestNo11(util::BitPack32 c, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            int cNumber0 = c.Get<CapabilityNo11::Number0>();
            int cNumber1 = c.Get<CapabilityNo11::Number1>();

            return TestNo11Half(cNumber0, pKcd, kcdCount)
                && TestNo11Half(cNumber1, pKcd, kcdCount);
        }

        // No.13
        // 雑多なパラメータ
        // 複数指定 不可
        bool TestNo13(util::BitPack32 c, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            return FindNumber(pKcd, kcdCount, 13).storage == c.storage;
        }

        // No.14
        // カーネルバージョン
        // 複数指定 不可
        bool TestNo14(util::BitPack32 c, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            return FindNumber(pKcd, kcdCount, 14).storage == c.storage;
        }

        // No.15
        // ハンドルテーブルサイズ
        // 複数指定 不可
        bool TestNo15(util::BitPack32 c, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            auto d = FindNumber(pKcd, kcdCount, 15);
            if( d == NumberNotFound )
            {
                return false;
            }

            int valueHandleCountMax = c.Get<CapabilityNo15::HandleCountMax>();
            int limitHandleCountMax = d.Get<CapabilityNo15::HandleCountMax>();

            return valueHandleCountMax <= limitHandleCountMax;
        }

        // No.16
        // フラグ
        // 複数指定 不可
        bool TestNo16(util::BitPack32 c, const util::BitPack32* pKcd, const int kcdCount) NN_NOEXCEPT
        {
            return (FindNumber(pKcd, kcdCount, 16).storage & c.storage) == c.storage;
        }
    }

    Result TestCapability(
                const util::BitPack32*  pKcd,
                int                     kcdCount,
                const util::BitPack32*  pKc,
                int                     kcCount ) NN_NOEXCEPT
    {
        for( int kci = 0; kci < kcCount; ++kci )
        {
            util::BitPack32 c = pKc[kci];

#define NN_LDR_TEST_NO(c, no, pKcd, kcdCount)           \
    if( IsCapabilityNumber(c, no) )                     \
    {                                                   \
        if( ! TestNo ## no (c, pKcd, kcdCount) )        \
        {                                               \
            return ResultInvalidCapabilityNo ## no ();  \
        }                                               \
    }

            NN_LDR_TEST_NO(c, 3, pKcd, kcdCount)
            else NN_LDR_TEST_NO(c, 4, pKcd, kcdCount)
            else if( IsCapabilityNumber(c, 6) )
            {
                ++kci;
                if( ! (kci < kcCount) )
                {
                    return ResultInvalidCapabilityNo6();
                }

                util::BitPack32 c1 = pKc[kci];

                if( ! TestNo6(c, c1, pKcd, kcdCount) )
                {
                    return ResultInvalidCapabilityNo6();
                }
            }
            else NN_LDR_TEST_NO(c, 7, pKcd, kcdCount)
            else NN_LDR_TEST_NO(c, 11, pKcd, kcdCount)
            else NN_LDR_TEST_NO(c, 13, pKcd, kcdCount)
            else NN_LDR_TEST_NO(c, 14, pKcd, kcdCount)
            else NN_LDR_TEST_NO(c, 15, pKcd, kcdCount)
            else NN_LDR_TEST_NO(c, 16, pKcd, kcdCount)
            else if( c.storage != ~0u )
            {
                return ResultUnknownCapabilityNo();
            }

#undef NN_LDR_TEST_NO
        }

        return ResultSuccess();
    }

    Bit16 MakeProgramInfoFlag(
                const util::BitPack32*  pKc,
                int                     kcCount ) NN_NOEXCEPT
    {
        Bit16 flags = 0;

        for( int kci = 0; kci < kcCount; ++kci )
        {
            util::BitPack32 c = pKc[kci];

            if( IsCapabilityNumber(c, 13) )
            {
                if( c.Get<CapabilityNo13::ProgramType>() == CapabilityNo13::ProgramTypes_Application )
                {
                    flags |= ldr::ProgramInfoFlag_Application;
                }
                else if( c.Get<CapabilityNo13::ProgramType>() == CapabilityNo13::ProgramTypes_Applet )
                {
                    flags |= ldr::ProgramInfoFlag_Applet;
                }
            }
            if( IsCapabilityNumber(c, 16) )
            {
                if( c.Get<CapabilityNo16::AllowDebug>() )
                {
                    flags |= ldr::ProgramInfoFlag_EnableDebug;
                }
            }
        }

        return flags;
    }



}}  // nn::ldr
