﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace fs {


class SlotCacheAccessor : public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(SlotCacheAccessor);
    NN_DISALLOW_MOVE(SlotCacheAccessor);

public:
    SlotCacheAccessor(int slotIndex, std::unique_lock<nn::os::Mutex>&& lock)
        : m_SlotIndex(slotIndex)
        , m_Lock(std::move(lock))
    {
    }

    int GetSlotIndex()
    {
        return m_SlotIndex;
    }

private:
    const int m_SlotIndex;
    std::unique_lock<nn::os::Mutex> m_Lock;
};

class SlotCacheEntry : public nn::util::IntrusiveListBaseNode<SlotCacheEntry>
{
    NN_DISALLOW_COPY(SlotCacheEntry);
    NN_DISALLOW_MOVE(SlotCacheEntry);

public:
    explicit SlotCacheEntry(int slotIndex)
        : m_SlotIndex(slotIndex)
        , m_Key2(-1)
    {
        memset(m_Key1, 0, KeySize);
    }

    bool Contains(const void* pKey1, size_t key1Size, int key2)
    {
        NN_SDK_ASSERT(key1Size == KeySize);
        NN_UNUSED(key1Size);
        return key2 == m_Key2 && memcmp(pKey1, m_Key1, KeySize) == 0;
    }

    int GetSlotIndex()
    {
        return m_SlotIndex;
    }

    void SetKey(const void* pKey1, size_t key1Size, int key2)
    {
        NN_SDK_ASSERT(key1Size == KeySize);
        NN_UNUSED(key1Size);
        memcpy(m_Key1, pKey1, KeySize);
        m_Key2 = key2;
    }

public:
    static const int KeySize = 16;

private:
    const int m_SlotIndex;
    char m_Key1[KeySize];
    int m_Key2;
};

class SlotCache
{
    NN_DISALLOW_COPY(SlotCache);
    NN_DISALLOW_MOVE(SlotCache);

public:
    SlotCache() NN_NOEXCEPT
        : m_Mutex(false)
    {
    }

    Result AllocateHighPriority(std::unique_ptr<SlotCacheAccessor>* ppOutValue, const void* pKey1, size_t key1Size, int key2)
    {
        return AllocateFromLru(ppOutValue, &m_HighPriorityMruList, pKey1, key1Size, key2);
    }

    Result AllocateLowPriority(std::unique_ptr<SlotCacheAccessor>* ppOutValue, const void* pKey1, size_t key1Size, int key2)
    {
        return AllocateFromLru(ppOutValue, &m_LowPriorityMruList, pKey1, key1Size, key2);
    }

    Result Find(std::unique_ptr<SlotCacheAccessor>* ppOutValue, const void* pKey1, size_t key1Size, int key2)
    {
        std::unique_lock<nn::os::Mutex> uniqueLock(m_Mutex);
        SlotCacheEntryList* listArray[2] = { &m_HighPriorityMruList, &m_LowPriorityMruList };

        for (auto pList : listArray)
        {
            for (auto it = pList->begin(); it != pList->end(); ++it)
            {
                if (it->Contains(pKey1, key1Size, key2))
                {
                    std::unique_ptr<SlotCacheAccessor> accessor(new SlotCacheAccessor(it->GetSlotIndex(), std::move(uniqueLock)));
                    NN_RESULT_THROW_UNLESS(accessor != nullptr, fs::ResultAllocationMemoryFailed());
                    *ppOutValue = std::move(accessor);

                    UpdateMru(pList, it);
                    NN_RESULT_SUCCESS;
                }
            }
        }

        return fs::ResultTargetNotFound();
    }

    void AddEntry(SlotCacheEntry* pEntry)
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        m_LowPriorityMruList.push_front(*pEntry);
    }

private:
    typedef nn::util::IntrusiveList<SlotCacheEntry, nn::util::IntrusiveListBaseNodeTraits<SlotCacheEntry>> SlotCacheEntryList;

private:
    void UpdateMru(SlotCacheEntryList* pList, SlotCacheEntryList::iterator iterator)
    {
        auto* pEntry = &(*iterator);
        pList->erase(iterator);
        pList->push_front(*pEntry);
    }

    //! Low/High から LRU エントリを pop して pDstList の MRU に入れつつ鍵値を格納して返却する
    Result AllocateFromLru(std::unique_ptr<SlotCacheAccessor>* ppOutValue, SlotCacheEntryList* pDstList, const void* pKey1, size_t key1Size, int key2)
    {
        std::unique_lock<nn::os::Mutex> uniqueLock(m_Mutex);

        auto* pSrcList = !m_LowPriorityMruList.empty() ? &m_LowPriorityMruList
                                                       : &m_HighPriorityMruList;
        NN_SDK_ASSERT(!pSrcList->empty());

        auto it = pSrcList->rbegin();

        std::unique_ptr<SlotCacheAccessor> accessor(new SlotCacheAccessor(it->GetSlotIndex(), std::move(uniqueLock)));
        NN_RESULT_THROW_UNLESS(accessor != nullptr, fs::ResultAllocationMemoryFailed());
        *ppOutValue = std::move(accessor);

        it->SetKey(pKey1, key1Size, key2);

        auto* pEntry = &(*it);
        pSrcList->pop_back();
        pDstList->push_front(*pEntry);
        NN_RESULT_SUCCESS;
    }

private:
    SlotCacheEntryList m_HighPriorityMruList;
    SlotCacheEntryList m_LowPriorityMruList;
    nn::os::Mutex m_Mutex;
};


}}
