﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/crypto.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssrv/fssrv_NcaCryptoConfiguration.h>
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageCreator.h>
#include <nn/fssrv/fscreator/fssrv_EncryptedFileSystemCreator.h>
#include <nn/spl/spl_Api.h>
#include <nn/os.h>
#include <nn/util/util_Optional.h>
#include "fs_CryptoConfiguration.h"
#include "fs_SlotCache.h"

using namespace nn::fs;
using namespace nn::fssrv;
using namespace nn::fssrv::fscreator;
using namespace nn::fssystem;
using namespace nn::spl;

namespace {
    const int KeySize = 16;

    NcaCryptoConfiguration g_NcaCryptoConfiguration;
    nn::fssrv::SaveDataTransferCryptoConfiguration g_SaveDataTransferCryptoConfiguration;

    static const unsigned char g_KeyAreaEncryptionKeySourceAppli[]  = { 0x7F, 0x59, 0x97, 0x1E, 0x62, 0x9F, 0x36, 0xA1, 0x30, 0x98, 0x06, 0x6F, 0x21, 0x44, 0xC3, 0x0D};
    static const unsigned char g_KeyAreaEncryptionKeySourceOcean[]  = { 0x32, 0x7D, 0x36, 0x08, 0x5A, 0xD1, 0x75, 0x8D, 0xAB, 0x4E, 0x6F, 0xBA, 0xA5, 0x55, 0xD8, 0x82};
    static const unsigned char g_KeyAreaEncryptionKeySourceSystem[] = { 0x87, 0x45, 0xF1, 0xBB, 0xA6, 0xBE, 0x79, 0x64, 0x7D, 0x04, 0x8B, 0xA6, 0x7B, 0x5F, 0xDA, 0x4A};
    static const unsigned char g_HeaderEncryptionKeySource[]        = { 0x1F, 0x12, 0x91, 0x3A, 0x4A, 0xCB, 0xF0, 0x0D, 0x4C, 0xDE, 0x3A, 0xF6, 0xD5, 0x23, 0x88, 0x2A};

    AccessKey g_SaveDataMacAccessKey = {{0}};
    AccessKey g_SaveDataOnExternalStorageMacAccessKey = {{0}};

    SlotCache g_SlotCache;
    const int SlotCacheEntryCount = 3;
    nn::util::optional<SlotCacheEntry> g_SlotCacheEntry[SlotCacheEntryCount];

    void ComputeXor(void* pDst, const void* pSrc1, const void* pSrc2, size_t size)
    {
        for (int i = 0; i < KeySize; i++)
        {
            static_cast<char*>(pDst)[i] = static_cast<const char*>(pSrc1)[i] ^ static_cast<const char*>(pSrc2)[i];
        }
    }

    AccessKey& GetNcaKekAccessKey(int keyTypeValue)
    {
        static AccessKey s_NcaKekAccessKeyArray[KeyAreaEncryptionKeyCount] = {{{0}}};
        static AccessKey s_NcaHeaderKekAccessKeyArray = {{0}};
        static AccessKey s_InvalidNcaKekAccessKeyArray = {{0}};

        if (keyTypeValue > static_cast<int>(KeyType::NcaHeaderKey) || IsInvalidKeyTypeValue(keyTypeValue))
        {
            return s_InvalidNcaKekAccessKeyArray;
        }
        else if (keyTypeValue == static_cast<int>(KeyType::NcaHeaderKey))
        {
            return s_NcaHeaderKekAccessKeyArray;
        }
        else
        {
            return s_NcaKekAccessKeyArray[keyTypeValue];
        }
    }


    const unsigned char g_SignedSystemPartitionSignKeyPublicModulusDev[] =
    {
        0xac, 0xda, 0x94, 0x3a, 0xad, 0xc3, 0x4d, 0x69, 0xff, 0x94, 0xc4, 0x74, 0xb0, 0x69, 0xe6, 0xce,
        0x3a, 0x23, 0x03, 0x7a, 0xce, 0xd7, 0xd3, 0xbd, 0x13, 0xbe, 0x63, 0x02, 0x1b, 0xca, 0x85, 0x5e,
        0x61, 0x51, 0xaa, 0x08, 0xf5, 0xd0, 0x86, 0xec, 0xf5, 0x72, 0x9e, 0xe8, 0x6b, 0x8d, 0x0a, 0xfe,
        0x91, 0x8a, 0xe4, 0x6f, 0xf5, 0xc2, 0xda, 0xdb, 0xc2, 0x37, 0x58, 0xb2, 0xd5, 0x83, 0xaa, 0xdc,
        0x99, 0x08, 0x82, 0x88, 0x36, 0x0d, 0x90, 0x63, 0xf3, 0xb4, 0x3e, 0x08, 0xd2, 0xb5, 0x5a, 0x90,
        0x8d, 0x6a, 0xce, 0xdc, 0xdd, 0x4f, 0x6e, 0x77, 0x7b, 0x1e, 0xab, 0x80, 0xac, 0x0c, 0x4b, 0xc5,
        0xac, 0xb3, 0xd7, 0x1a, 0xba, 0x15, 0xe0, 0x38, 0x9e, 0x80, 0x58, 0x29, 0xc4, 0xa0, 0xee, 0x0a,
        0xb1, 0xac, 0xa8, 0xa8, 0x42, 0x24, 0x2a, 0x13, 0x91, 0xc7, 0xd9, 0xe4, 0xb7, 0x1c, 0x63, 0x45,
        0xf5, 0x30, 0x6f, 0xfd, 0x59, 0x30, 0x91, 0xd1, 0x99, 0x76, 0x0e, 0x59, 0xbf, 0x98, 0xed, 0x96,
        0x32, 0xb9, 0x02, 0x03, 0xe0, 0x84, 0x37, 0xe5, 0xac, 0x93, 0x0a, 0xd9, 0xee, 0xe2, 0xc3, 0x52,
        0xf4, 0xfb, 0xc1, 0xf3, 0xa2, 0xaa, 0x33, 0x72, 0x35, 0x37, 0x9d, 0x2e, 0xe9, 0xc3, 0x55, 0xc7,
        0x63, 0xcc, 0x01, 0x19, 0x7b, 0xc1, 0xf4, 0x16, 0x0f, 0x36, 0xdf, 0x6f, 0x88, 0x7f, 0x2e, 0xcc,
        0x4f, 0xa5, 0xa9, 0x58, 0x47, 0x06, 0xf7, 0xc7, 0x8c, 0x4f, 0xeb, 0x3d, 0x80, 0xac, 0x7f, 0xa0,
        0xf0, 0xac, 0xe7, 0xbe, 0x53, 0x08, 0x2c, 0x53, 0x4e, 0x22, 0x2a, 0xb2, 0xc7, 0x7b, 0x6e, 0xab,
        0xc1, 0x61, 0xb7, 0xa1, 0x11, 0x6e, 0xd1, 0x14, 0x01, 0x9f, 0x5d, 0x1b, 0x92, 0x05, 0x2c, 0x43,
        0xbc, 0x8e, 0xaa, 0x9b, 0x2c, 0x2a, 0x2f, 0xec, 0x24, 0xec, 0x63, 0x5e, 0x3e, 0xdd, 0x57, 0x4f,
    };

    const unsigned char g_SignedSystemPartitionSignKeyPublicModulusProd[] =
    {
        0x95, 0xdd, 0x29, 0x57, 0xc5, 0x3a, 0x4e, 0xca, 0x99, 0xcc, 0xda, 0xd0, 0x54, 0x06, 0x3f, 0x77,
        0xd7, 0xe5, 0xa4, 0x53, 0x17, 0x93, 0xe5, 0xfc, 0x21, 0x49, 0x19, 0x0f, 0xbb, 0xa4, 0xca, 0x7e,
        0x32, 0xcc, 0xdc, 0x5d, 0x2a, 0x1a, 0x8d, 0x75, 0x51, 0xe5, 0x7d, 0x3e, 0xd9, 0x46, 0xb9, 0x13,
        0x6b, 0xf7, 0xca, 0xf9, 0x7b, 0x10, 0xb8, 0x54, 0x35, 0x36, 0x1e, 0x3a, 0x69, 0xff, 0xbb, 0x50,
        0x90, 0x78, 0x39, 0xd8, 0xef, 0xb5, 0x18, 0x5d, 0xa3, 0xc1, 0x3b, 0xf7, 0xd7, 0xe9, 0x3d, 0x0c,
        0xc2, 0x0a, 0x68, 0x87, 0x83, 0x69, 0xb8, 0xa4, 0xb1, 0xae, 0xa7, 0xe8, 0x13, 0x8d, 0x09, 0x84,
        0x2e, 0xc2, 0xfc, 0x35, 0xcc, 0xa4, 0x44, 0x9d, 0x81, 0x28, 0x0c, 0x3a, 0x76, 0xe1, 0x8a, 0x44,
        0x75, 0x87, 0x02, 0xde, 0x65, 0xfa, 0xef, 0x2f, 0x2b, 0x39, 0x81, 0xe3, 0x5b, 0x43, 0x7f, 0x65,
        0xd4, 0xc9, 0xbb, 0x95, 0x00, 0x49, 0x91, 0xa6, 0x81, 0xf9, 0xa5, 0xf8, 0x6f, 0x85, 0x1b, 0xf1,
        0x9a, 0xb8, 0xae, 0x51, 0xfa, 0x74, 0x6a, 0xf6, 0x7e, 0xdf, 0x66, 0x7d, 0x22, 0x69, 0x71, 0x6a,
        0xfd, 0x8d, 0x0a, 0xe2, 0x5b, 0xb4, 0x44, 0xd4, 0x74, 0x6d, 0x88, 0x0f, 0x3d, 0x66, 0x54, 0x67,
        0x65, 0xb7, 0xc9, 0xda, 0x14, 0x6c, 0xfc, 0x29, 0x13, 0xd5, 0xa0, 0xf7, 0xf8, 0x3c, 0xc2, 0xbc,
        0xe8, 0x06, 0x6a, 0x4b, 0x4d, 0x7b, 0xcd, 0xa8, 0xac, 0xd8, 0x93, 0x4b, 0xab, 0x45, 0xd3, 0xa0,
        0x99, 0xe9, 0x9c, 0x28, 0xa6, 0xe0, 0x9d, 0xb1, 0xb3, 0xbf, 0xa8, 0x10, 0xfc, 0x4b, 0x56, 0x40,
        0x3d, 0x66, 0xc9, 0xa3, 0xb6, 0x60, 0x1a, 0xb4, 0x5a, 0x69, 0x18, 0x41, 0xcd, 0xcd, 0xb2, 0x82,
        0x55, 0x2a, 0xef, 0x1a, 0xad, 0x09, 0xc1, 0x47, 0x9a, 0x07, 0x5e, 0x26, 0xa0, 0x5d, 0x1f, 0xad,
    };

}


void SetUpKekAccessKeys() NN_NOEXCEPT
{
    // nca
    {
        const int Option = 0x00000000; // 共通鍵

        // ncaKeyArea
        for (uint8_t i = 0; i < NcaCryptoConfiguration::KeyGenerationMax; i++)
        {
            // 未対応の鍵世代分はエラーとなるためエラーハンドリングはしない
            GenerateAesKek(&GetNcaKekAccessKey(GetKeyTypeValue(0, i)), g_KeyAreaEncryptionKeySourceAppli,  KeySize, i, Option);
            GenerateAesKek(&GetNcaKekAccessKey(GetKeyTypeValue(1, i)), g_KeyAreaEncryptionKeySourceOcean,  KeySize, i, Option);
            GenerateAesKek(&GetNcaKekAccessKey(GetKeyTypeValue(2, i)), g_KeyAreaEncryptionKeySourceSystem, KeySize, i, Option);
        }

        // header
        NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKek(&GetNcaKekAccessKey(static_cast<int>(KeyType::NcaHeaderKey)), g_HeaderEncryptionKeySource, KeySize, 0, Option));
    }

    // save
    {
        const unsigned char KeySourceForSaveDataMacKek[KeySize] = { 0xd8, 0x9c, 0x23, 0x6e, 0xc9, 0x12, 0x4e, 0x43, 0xc8, 0x2b, 0x03, 0x87, 0x43, 0xf9, 0xcf, 0x1b};
        const int KeyGeneration = 0;
        const int Option = 0x00000001; // 固有鍵

        NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKek(&g_SaveDataMacAccessKey, KeySourceForSaveDataMacKek, KeySize, KeyGeneration, Option));
    }

    // save on external storage
    {
        const unsigned char KeySourceForSaveDataOnExternalStorageMacKek[KeySize] = { 0x04, 0x89, 0xef, 0x5d, 0x32, 0x6e, 0x1a, 0x59, 0xc4, 0xb7, 0xab, 0x8c, 0x36, 0x7a, 0xab, 0x17 };
        const int KeyGeneration = 0;
        const int Option = 0x00000000; // 共通鍵

        NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKek(&g_SaveDataOnExternalStorageMacAccessKey, KeySourceForSaveDataOnExternalStorageMacKek, KeySize, KeyGeneration, Option));
    }

    for (int i = 0; i < SlotCacheEntryCount; i++)
    {
        int slotIndex;
        NN_ABORT_UNLESS_RESULT_SUCCESS(AllocateAesKeySlot(&slotIndex));
        g_SlotCacheEntry[i].emplace(slotIndex);
        g_SlotCache.AddEntry(&g_SlotCacheEntry[i].value());
    }
}


//! nca の内部鍵を復号
void GenerateNcaKey(void* pOutKey, size_t outKeySize, const void* pEncryptedKey, size_t encryptedKeySize, int keyTypeValue, const NcaCryptoConfiguration& cryptoConfiguration) NN_NOEXCEPT
{
    NN_UNUSED(cryptoConfiguration);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(pOutKey, outKeySize, GetNcaKekAccessKey(keyTypeValue), pEncryptedKey, encryptedKeySize));
}

void DecryptAesCtr(void* pDst, size_t dstSize, int keyTypeValue, const void* pEncryptedKey, size_t encryptedKeySize, const void* pIv, size_t ivSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    std::unique_ptr<SlotCacheAccessor> accessor;

    auto result = g_SlotCache.Find(&accessor, pEncryptedKey, encryptedKeySize, keyTypeValue);
    if (ResultTargetNotFound::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_SlotCache.AllocateHighPriority(&accessor, pEncryptedKey, encryptedKeySize, keyTypeValue));
        NN_ABORT_UNLESS_RESULT_SUCCESS(LoadAesKey(accessor->GetSlotIndex(), GetNcaKekAccessKey(keyTypeValue), pEncryptedKey, encryptedKeySize));
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(ComputeCtr(pDst, dstSize, accessor->GetSlotIndex(), pSrc, srcSize, pIv, ivSize));
}

void DecryptAesCtrForPreparedKey(void* pDst, size_t dstSize, int keyTypeValue, const void* pEncryptedKey, size_t encryptedKeySize, const void* pIv, size_t ivSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    NN_UNUSED(keyTypeValue);

    std::unique_ptr<SlotCacheAccessor> accessor;

    auto result = g_SlotCache.Find(&accessor, pEncryptedKey, encryptedKeySize, static_cast<int>(KeyType::NcaExternalKey));
    if (ResultTargetNotFound::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_SlotCache.AllocateHighPriority(&accessor, pEncryptedKey, encryptedKeySize, static_cast<int>(KeyType::NcaExternalKey)));

        nn::spl::AccessKey accessKey;
        NN_ABORT_UNLESS(encryptedKeySize == sizeof(accessKey));
        memcpy(&accessKey, pEncryptedKey, sizeof(accessKey));
        NN_ABORT_UNLESS_RESULT_SUCCESS(LoadPreparedAesKey(accessor->GetSlotIndex(), accessKey));
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(ComputeCtr(pDst, dstSize, accessor->GetSlotIndex(), pSrc, srcSize, pIv, ivSize));
}

void InvalidateHwAesKey() NN_NOEXCEPT
{
    char InvalidKey[KeySize] = {0};
    for(int i=0; i < SlotCacheEntryCount; i++)
    {
        std::unique_ptr<SlotCacheAccessor> accessor;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_SlotCache.AllocateHighPriority(&accessor, InvalidKey, KeySize, -1 - i));
    }
}

const NcaCryptoConfiguration* GetNcaCryptoConfiguration(bool isProd) NN_NOEXCEPT
{
    memcpy(&g_NcaCryptoConfiguration, GetNcaDefaultCryptoConfiguration(isProd), sizeof(NcaCryptoConfiguration));
    memcpy(g_NcaCryptoConfiguration.keyAreaEncryptionKeySource[0], g_KeyAreaEncryptionKeySourceAppli,  sizeof(g_KeyAreaEncryptionKeySourceAppli));
    memcpy(g_NcaCryptoConfiguration.keyAreaEncryptionKeySource[1], g_KeyAreaEncryptionKeySourceOcean,  sizeof(g_KeyAreaEncryptionKeySourceOcean));
    memcpy(g_NcaCryptoConfiguration.keyAreaEncryptionKeySource[2], g_KeyAreaEncryptionKeySourceSystem, sizeof(g_KeyAreaEncryptionKeySourceSystem));

    memcpy(g_NcaCryptoConfiguration.headerEncryptionKeySource, g_HeaderEncryptionKeySource, sizeof(g_HeaderEncryptionKeySource));

    g_NcaCryptoConfiguration.pGenerateKey = GenerateNcaKey;
    g_NcaCryptoConfiguration.pDecryptAesCtr = DecryptAesCtr;
    g_NcaCryptoConfiguration.pDecryptAesCtrForExternalKey = DecryptAesCtrForPreparedKey;
    g_NcaCryptoConfiguration.isAvailableRawHeader = !isProd;

    return &g_NcaCryptoConfiguration;
}

const unsigned char* GetSignedSystemPartitionSignKeyPublicModulus(bool isProd) NN_NOEXCEPT
{
    return isProd ? g_SignedSystemPartitionSignKeyPublicModulusProd : g_SignedSystemPartitionSignKeyPublicModulusDev;
}

nn::fssrv::SaveDataTransferCryptoConfiguration* GetSaveDataTransferCryptoConfiguration(bool isProd) NN_NOEXCEPT
{
    const unsigned char SignKeyModulusDev[] =
    {
        0xc4, 0xf4, 0x07, 0x81, 0x50, 0x6f, 0x02, 0x95, 0x84, 0xa6, 0x00, 0x19, 0x26, 0x18, 0xf3, 0xde,
        0xf8, 0xde, 0x09, 0xa5, 0x0a, 0x7d, 0x7b, 0xfd, 0x32, 0x7d, 0x11, 0x54, 0x37, 0x9d, 0x1f, 0xdd,
        0x65, 0xcd, 0xdc, 0x58, 0x83, 0x50, 0xd6, 0x9a, 0xf9, 0x74, 0x8c, 0x7f, 0x1a, 0xb6, 0x79, 0xd7,
        0x69, 0xf6, 0x4a, 0x38, 0xa8, 0x74, 0xa7, 0x23, 0x48, 0x43, 0xa6, 0x80, 0x54, 0x0f, 0x88, 0xc6,
        0x05, 0x68, 0xc0, 0x7f, 0x29, 0xd6, 0x6d, 0x62, 0x7b, 0x15, 0x83, 0x64, 0xdc, 0xc7, 0x4b, 0x74,
        0x07, 0xb9, 0xa3, 0x61, 0x34, 0x91, 0x61, 0xfe, 0x63, 0xb1, 0x9d, 0xc8, 0x4d, 0x7b, 0x6e, 0x1d,
        0xf9, 0x02, 0x60, 0xc6, 0xd3, 0x43, 0x77, 0xfb, 0xcd, 0x74, 0xaf, 0xe6, 0x3e, 0xbb, 0x5f, 0xa4,
        0x7a, 0xa3, 0x25, 0x6c, 0x9b, 0xf0, 0xf3, 0x4f, 0xab, 0xdc, 0xe6, 0x84, 0xa3, 0x36, 0x81, 0x2f,
        0x6f, 0x2d, 0x1a, 0x77, 0x4c, 0x6c, 0x24, 0x85, 0xf0, 0x4a, 0x8a, 0xe5, 0xb5, 0x58, 0xb3, 0xe9,
        0xd9, 0x17, 0x67, 0x7c, 0xff, 0x9a, 0xfd, 0x7f, 0x60, 0xab, 0x62, 0xf8, 0x23, 0x16, 0xe3, 0x3e,
        0xd6, 0x07, 0x6c, 0xae, 0x80, 0x49, 0x76, 0xd6, 0x2a, 0x25, 0x9e, 0x58, 0x1d, 0xfc, 0xe1, 0x71,
        0xde, 0x73, 0xe5, 0xed, 0x9d, 0xab, 0x47, 0x23, 0x6a, 0xdb, 0x93, 0xf3, 0xab, 0xe7, 0x2b, 0x65,
        0x40, 0xf4, 0xba, 0xa0, 0x95, 0xf1, 0x8e, 0xfc, 0xa7, 0xe7, 0x96, 0x10, 0x9c, 0xbe, 0x33, 0xb9,
        0x53, 0x0a, 0xb4, 0xcc, 0xea, 0x4b, 0x4c, 0xca, 0xb0, 0x9d, 0x0a, 0x88, 0x38, 0x25, 0xa1, 0xbc,
        0xf3, 0x72, 0xc6, 0x7c, 0x0b, 0x27, 0x56, 0x04, 0x86, 0x51, 0x48, 0x7c, 0x2f, 0xf6, 0x66, 0x33,
        0x58, 0xf3, 0xea, 0x74, 0xba, 0xfc, 0x24, 0x6a, 0x22, 0x54, 0xaa, 0x8a, 0x8e, 0x05, 0xe0, 0x43,
    };

    const unsigned char SignKeyModulusProd[] =
    {
        0xbf, 0x18, 0x4b, 0xa5, 0xdb, 0x61, 0x2b, 0xe0, 0x94, 0x77, 0x01, 0x2e, 0x19, 0x62, 0x40, 0x87,
        0xba, 0x77, 0x72, 0xb0, 0xbd, 0xee, 0x3c, 0xde, 0x85, 0x3c, 0x32, 0xa2, 0x28, 0x6e, 0xb9, 0x34,
        0x4b, 0x11, 0x84, 0x38, 0x08, 0x74, 0x2f, 0xf6, 0x16, 0x68, 0x13, 0x66, 0xea, 0x6a, 0x07, 0x5d,
        0xb9, 0xdc, 0x8a, 0x00, 0x6a, 0x33, 0xdb, 0xa5, 0xc7, 0xa6, 0x12, 0x08, 0x04, 0x0c, 0xed, 0x26,
        0x1a, 0xde, 0xc3, 0xd0, 0x85, 0xa0, 0x0f, 0x72, 0xad, 0x27, 0x4d, 0x9e, 0x8b, 0x78, 0x35, 0x86,
        0x34, 0x7f, 0x2c, 0xa5, 0xaf, 0x19, 0x18, 0x22, 0xc9, 0x09, 0x9e, 0xfa, 0xd7, 0x2d, 0xa7, 0xed,
        0x2c, 0x8a, 0x75, 0xfc, 0x0c, 0xfc, 0xcf, 0xb9, 0x25, 0xb1, 0x1a, 0x36, 0xa9, 0xb0, 0x18, 0xbc,
        0x3f, 0x49, 0x31, 0x7e, 0x0e, 0xc5, 0xa4, 0x89, 0xb6, 0xfb, 0xdd, 0x8f, 0xd7, 0x62, 0xb1, 0xff,
        0x0e, 0xc1, 0x8f, 0x03, 0xd3, 0xae, 0x4c, 0x2c, 0x46, 0x23, 0x46, 0xaa, 0x01, 0xb9, 0xdc, 0xa5,
        0x84, 0xdc, 0x34, 0xc9, 0x58, 0xb9, 0xf0, 0xab, 0xb6, 0x7f, 0xb7, 0x88, 0x6d, 0x9d, 0xa1, 0xcb,
        0xfe, 0x96, 0xb4, 0x3f, 0x00, 0x33, 0x73, 0xa5, 0x52, 0x35, 0x40, 0x9b, 0x31, 0xc4, 0xcb, 0x6d,
        0x9c, 0xb6, 0x03, 0xb4, 0x2a, 0xab, 0x8c, 0xdf, 0x3d, 0xd3, 0x64, 0x36, 0x26, 0x56, 0x36, 0xea,
        0xdb, 0x29, 0x2a, 0xe4, 0xc7, 0xa4, 0xe2, 0x8c, 0x4a, 0xb9, 0xf5, 0x35, 0x36, 0x17, 0x8f, 0x89,
        0x33, 0x87, 0x9e, 0x88, 0x40, 0x4e, 0x38, 0x3c, 0xde, 0x41, 0x61, 0x9c, 0x1c, 0x22, 0x0e, 0x14,
        0x4a, 0x41, 0x4f, 0x3a, 0x20, 0xd2, 0x94, 0x99, 0x61, 0xa2, 0x38, 0x0e, 0xfe, 0x85, 0xf0, 0x1b,
        0x26, 0x85, 0xd1, 0xe8, 0x0a, 0x3d, 0x69, 0x23, 0xa4, 0x97, 0xfc, 0x0a, 0x86, 0xdc, 0xf3, 0xb3,
    };


    const unsigned char SignKeyModulusDevForCloudBackup[] =
    {
        0xaa, 0xe5, 0x70, 0x4d, 0x2e, 0x19, 0xe8, 0xd7, 0x18, 0x99, 0x66, 0x59, 0xe7, 0xf5, 0x66, 0x9d,
        0x56, 0x38, 0xfb, 0x38, 0xaf, 0x7c, 0xa5, 0xf2, 0x34, 0x9d, 0x32, 0x9f, 0x6a, 0x73, 0x6b, 0xcb,
        0xbd, 0xd3, 0x96, 0x82, 0xe4, 0xeb, 0x66, 0xfb, 0x76, 0x7d, 0x79, 0x44, 0xb6, 0xe9, 0xae, 0xf9,
        0xc5, 0xee, 0x34, 0xf4, 0xdf, 0xe1, 0xc0, 0x94, 0x54, 0xc1, 0xd6, 0x26, 0xdf, 0x07, 0x90, 0x70,
        0x46, 0xa7, 0x16, 0x45, 0x32, 0x83, 0xb3, 0x54, 0xdf, 0xca, 0xf4, 0xc7, 0x18, 0xde, 0xed, 0xe1,
        0xdb, 0xb3, 0x2a, 0x4d, 0x0f, 0x57, 0xde, 0x2d, 0xb5, 0x01, 0x7a, 0xde, 0xac, 0xff, 0x73, 0x83,
        0x4f, 0x3e, 0x1f, 0xa1, 0xc1, 0x15, 0x6f, 0xfa, 0xd6, 0x77, 0xa3, 0x84, 0x87, 0x7b, 0xc2, 0xb9,
        0xb1, 0xe3, 0x6e, 0x23, 0x4b, 0x9d, 0x6f, 0x5e, 0x88, 0x71, 0xdb, 0x36, 0x4c, 0xa8, 0x0a, 0xc0,
        0x80, 0x7c, 0x68, 0x89, 0x64, 0xd1, 0x4a, 0x14, 0x43, 0x44, 0x18, 0x84, 0xbc, 0xc5, 0xca, 0xd3,
        0x63, 0x0b, 0xa2, 0xdf, 0x94, 0xe7, 0x3d, 0x8b, 0x56, 0x2e, 0x19, 0x7c, 0x1e, 0xdc, 0x8d, 0xe0,
        0xcd, 0x81, 0xad, 0xda, 0xcf, 0x09, 0xa5, 0x63, 0xda, 0x48, 0xa4, 0x98, 0xcf, 0x31, 0x27, 0xd7,
        0xd2, 0x2c, 0x0d, 0x0b, 0x21, 0x34, 0xdf, 0xcb, 0x97, 0x11, 0xdf, 0xce, 0xc7, 0x8c, 0xb1, 0xca,
        0x72, 0x06, 0x5b, 0xd5, 0x5e, 0x2e, 0x60, 0xb3, 0xca, 0x1e, 0x30, 0x36, 0x94, 0x5e, 0xec, 0xab,
        0x72, 0xb6, 0x3d, 0x93, 0x0c, 0xf8, 0x78, 0x88, 0x15, 0xd0, 0x60, 0x11, 0xb0, 0x03, 0xd3, 0x54,
        0x53, 0x8e, 0x89, 0x70, 0xe2, 0xf3, 0xb9, 0x67, 0xd3, 0xc5, 0x8f, 0x39, 0xdd, 0x6d, 0x53, 0x47,
        0xb7, 0x49, 0x12, 0x43, 0x06, 0x54, 0x9a, 0x40, 0x6e, 0x89, 0xd7, 0x24, 0x5c, 0xbd, 0x82, 0x91,
    };

    // TODO: クラウド用 Prod 公開鍵に差し替え
    const unsigned char SignKeyModulusProdForCloudBackup[] =
    {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };


    g_SaveDataTransferCryptoConfiguration.pGenerateAesKey = GenerateAesKeyForSaveDataTransfer;
    g_SaveDataTransferCryptoConfiguration.pGenerateRandom = GenerateRandomForSaveDataTransfer;

    {
        auto* pSignKeyModulus = isProd ? SignKeyModulusProd : SignKeyModulusDev;
        memcpy(g_SaveDataTransferCryptoConfiguration.tokenSignKeyModulus, pSignKeyModulus, g_SaveDataTransferCryptoConfiguration.TokenSignKeyModulusSize);
    }
    {
        auto* pSignKeyModulus = isProd ? SignKeyModulusProdForCloudBackup : SignKeyModulusDevForCloudBackup;
        memcpy(g_SaveDataTransferCryptoConfiguration.kspSignKeyModulus, pSignKeyModulus, g_SaveDataTransferCryptoConfiguration.KspSignKeyModulusSize);
    }

    return &g_SaveDataTransferCryptoConfiguration;
}

int GetKeyGeneration(int requestedKeyGeneration)
{
    if (requestedKeyGeneration >= 0)
    {
        return requestedKeyGeneration;
    }
    else
    {
        nn::Bit64 keyGeneration;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GetConfig(&keyGeneration, ConfigItem_DeviceUniqueKeyGeneration));
        return static_cast<int>(keyGeneration);
    }
}

void GetBisEncryptionKey(BuiltInStorageCreator::Key* pOutValueKey, int keyCount, BisPartitionEncryptionKeyId id, int requestedKeyGeneration) NN_NOEXCEPT
{
    int keyGeneration = GetKeyGeneration(requestedKeyGeneration);

    int idIndex = static_cast<int>(id);

    NN_ABORT_UNLESS( idIndex >= 0 );
    NN_ABORT_UNLESS( idIndex < BisPartitionEncryptionKeyIdCount );
    NN_ABORT_UNLESS( keyCount == 2 );

    if( id == BisPartitionEncryptionKeyId::Calibration )
    {
        const unsigned char CalibrationPartitionEncryptionKeySource[2][KeySize] =
        {
            {0xf8, 0x3f, 0x38, 0x6e, 0x2c, 0xd2, 0xca, 0x32, 0xa8, 0x9a, 0xb9, 0xaa, 0x29, 0xbf, 0xc7, 0x48, },
            {0x7d, 0x92, 0xb0, 0x3a, 0xa8, 0xbf, 0xde, 0xe1, 0xa7, 0x4c, 0x3b, 0x6e, 0x35, 0xcb, 0x71, 0x06, },
        };

        const nn::Bit32 Purpose[2] =
        {
            nn::spl::KeyPurpose_CalibrationDataAreaEncryption0,
            nn::spl::KeyPurpose_CalibrationDataAreaEncryption1,
        };

        for(int i=0; i<2; i++)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateSpecificAesKey(pOutValueKey[i].value, KeySize, CalibrationPartitionEncryptionKeySource[i], KeySize, keyGeneration, Purpose[i]));
        }
    }
    else
    {
        const unsigned char BisEncryptionKeySourceForKek[KeySize] =
        {
            0x34, 0xc1, 0xa0, 0xc4, 0x82, 0x58, 0xf8, 0xb4, 0xfa, 0x9e, 0x5e, 0x6a, 0xda, 0xfc, 0x7e, 0x4f,
        };

        NN_STATIC_ASSERT(BisPartitionEncryptionKeyIdCount == 4);
        const unsigned char BisPartitionEncryptionKeySources[BisPartitionEncryptionKeyIdCount][2][KeySize] =
        {
            { // Calibration (unused)
                {0},
                {0},
            },
            { // SafeMode
                {0x41, 0x00, 0x30, 0x49, 0xdd, 0xcc, 0xc0, 0x65, 0x64, 0x7a, 0x7e, 0xb4, 0x1e, 0xed, 0x9c, 0x5f, },
                {0x44, 0x42, 0x4e, 0xda, 0xb4, 0x9d, 0xfc, 0xd9, 0x87, 0x77, 0x24, 0x9a, 0xdc, 0x9f, 0x7c, 0xa4, },
            },
            { // UserSystemProperEncryption
                {0x52, 0xc2, 0xe9, 0xeb, 0x09, 0xe3, 0xee, 0x29, 0x32, 0xa1, 0x0c, 0x1f, 0xb6, 0xa0, 0x92, 0x6c, },
                {0x4d, 0x12, 0xe1, 0x4b, 0x2a, 0x47, 0x4c, 0x1c, 0x09, 0xcb, 0x03, 0x59, 0xf0, 0x15, 0xf4, 0xe4, },
            },
            { // UserSystem = UserSystemProperEncryption
                {0x52, 0xc2, 0xe9, 0xeb, 0x09, 0xe3, 0xee, 0x29, 0x32, 0xa1, 0x0c, 0x1f, 0xb6, 0xa0, 0x92, 0x6c, },
                {0x4d, 0x12, 0xe1, 0x4b, 0x2a, 0x47, 0x4c, 0x1c, 0x09, 0xcb, 0x03, 0x59, 0xf0, 0x15, 0xf4, 0xe4, },
            },
        };

        AccessKey accessKey;
        const bool IsRecoveryKeyRedirection = nn::spl::GetConfigBool(nn::spl::ConfigItem_IsRecoveryBoot) && (id == BisPartitionEncryptionKeyId::UserSystem);
        int option = IsRecoveryKeyRedirection ? 0x00000004  // リカバリモード共通鍵 // TODO: spl_Api.h
                                              : 0x00000001; // 固有鍵

        NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKek(&accessKey, BisEncryptionKeySourceForKek, KeySize, keyGeneration, option));
        NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(pOutValueKey[0].value, KeySize, accessKey, BisPartitionEncryptionKeySources[idIndex][0], KeySize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(pOutValueKey[1].value, KeySize, accessKey, BisPartitionEncryptionKeySources[idIndex][1], KeySize));
    }
}

void GetSdCardEncryptionKey(EncryptedFileSystemCreator::Key* pOutValueKey, int keyCount, EncryptedFileSystemCreator::KeyId id, const nn::fs::EncryptionSeed& seed) NN_NOEXCEPT
{
    int idIndex = static_cast<int>(id);

    NN_ABORT_UNLESS_GREATER_EQUAL(idIndex, 0);
    NN_ABORT_UNLESS_LESS(idIndex, EncryptedFileSystemCreator::KeyIdCount);
    NN_ABORT_UNLESS_EQUAL(keyCount, 2);

    {
        const unsigned char SdCardEncryptionKeySourceForKek[KeySize] =
        {
            0x88, 0x35, 0x8d, 0x9c, 0x62, 0x9b, 0xa1, 0xa0, 0x01, 0x47, 0xdb, 0xe0, 0x62, 0x1b, 0x54, 0x32,
        };

        const unsigned char SdCardEncryptionKeySources[EncryptedFileSystemCreator::KeyIdCount][2][KeySize] =
        {
            { // ContentStorage
                { 0x24, 0x49, 0xb7, 0x22, 0x72, 0x67, 0x03, 0xa8, 0x19, 0x65, 0xe6, 0xe3, 0xea, 0x58, 0x2f, 0xdd, },
                { 0x9a, 0x95, 0x15, 0x17, 0xb1, 0x6e, 0x8f, 0x7f, 0x1f, 0x68, 0x26, 0x31, 0x52, 0xea, 0x29, 0x6a, },
            },
            { // SaveData
                { 0x58, 0x41, 0xa2, 0x84, 0x93, 0x5b, 0x56, 0x27, 0x8b, 0x8e, 0x1f, 0xc5, 0x18, 0xe9, 0x9f, 0x2b, },
                { 0x67, 0xc7, 0x93, 0xf0, 0xf2, 0x4f, 0xde, 0xd0, 0x75, 0x49, 0x5d, 0xca, 0x00, 0x6d, 0x99, 0xc2, },
            },
        };

        AccessKey accessKey;
        const int KeyGeneration = 0;
        int option = 0; // 共通鍵

        // seed を XOR で混ぜ込む
        char keySource[2][KeySize];
        ComputeXor(keySource[0], seed.value, SdCardEncryptionKeySources[idIndex][0], KeySize);
        ComputeXor(keySource[1], seed.value, SdCardEncryptionKeySources[idIndex][1], KeySize);

        NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKek(&accessKey, SdCardEncryptionKeySourceForKek, KeySize, KeyGeneration, option));
        NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(pOutValueKey[0].value, KeySize, accessKey, keySource[0], KeySize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(pOutValueKey[1].value, KeySize, accessKey, keySource[1], KeySize));
    }
}

void GenerateDeviceUniqueMacForSaveData(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize) NN_NOEXCEPT
{
    const auto MacSize = nn::crypto::Aes128CmacGenerator::MacSize;
    NN_SDK_REQUIRES(macBufferSize >= MacSize);

    memset(macBuffer, 0, macBufferSize);

    const unsigned char KeySourceForSaveDataMac[KeySize] = {0xe4, 0xcd, 0x3d, 0x4a, 0xd5, 0x0f, 0x74, 0x28, 0x45, 0xa4, 0x87, 0xe5, 0xa0, 0x63, 0xea, 0x1f};

    std::unique_ptr<SlotCacheAccessor> accessor;
    auto result = g_SlotCache.Find(&accessor, KeySourceForSaveDataMac, KeySize, static_cast<int>(KeyType::SaveDataDeviceUniqueMac));
    if (ResultTargetNotFound::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_SlotCache.AllocateLowPriority(&accessor, KeySourceForSaveDataMac, KeySize, static_cast<int>(KeyType::SaveDataDeviceUniqueMac)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(LoadAesKey(accessor->GetSlotIndex(), g_SaveDataMacAccessKey, KeySourceForSaveDataMac, KeySize));
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(ComputeCmac(macBuffer, MacSize, accessor->GetSlotIndex(), dataBuffer, dataBufferSize));
}

void GenerateSeedUniqueMacForSaveData(void* macBuffer, size_t macBufferSize, const void* dataBuffer, size_t dataBufferSize, const void* seedBuffer, size_t seedBufferSize) NN_NOEXCEPT
{
    const auto MacSize = nn::crypto::Aes128CmacGenerator::MacSize;
    NN_SDK_REQUIRES(macBufferSize >= MacSize);
    NN_SDK_REQUIRES(seedBufferSize == KeySize);

    memset(macBuffer, 0, macBufferSize);

    const unsigned char KeySourceForSaveDataMac[KeySize] = { 0x6f, 0x64, 0x59, 0x47, 0xc5, 0x61, 0x46, 0xf9, 0xff, 0xa0, 0x45, 0xd5, 0x95, 0x33, 0x29, 0x18 };
    char keySource[KeySize];
    ComputeXor(keySource, seedBuffer, KeySourceForSaveDataMac, KeySize);

    std::unique_ptr<SlotCacheAccessor> accessor;
    auto result = g_SlotCache.Find(&accessor, KeySourceForSaveDataMac, KeySize, static_cast<int>(KeyType::SaveDataSeedUniqueMac));
    if (ResultTargetNotFound::Includes(result))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_SlotCache.AllocateLowPriority(&accessor, keySource, KeySize, static_cast<int>(KeyType::SaveDataSeedUniqueMac)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(LoadAesKey(accessor->GetSlotIndex(), g_SaveDataOnExternalStorageMacAccessKey, keySource, KeySize));
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(ComputeCmac(macBuffer, MacSize, accessor->GetSlotIndex(), dataBuffer, dataBufferSize));
}

void GenerateRandomForSaveData(void* pData, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateRandomBytes(pData, size));
}
void GenerateRandomForSdCardEncryption(void* pData, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateRandomBytes(pData, size));
}

void GenerateRandomForSaveDataTransfer(void* pData, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GenerateRandomBytes(pData, size));
}
void GenerateAesKeyForSaveDataTransfer(void* pBuffer, size_t bufferSize, nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex keyIndex, const void* pKeySource, size_t keySourceSize) NN_NOEXCEPT
{
    auto index = static_cast<int>(keyIndex);
    NN_ABORT_UNLESS_GREATER_EQUAL(index, 0);
    NN_ABORT_UNLESS_LESS(index, static_cast<int>(nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::Count));
    NN_ABORT_UNLESS_EQUAL(bufferSize, static_cast<size_t>(KeySize));

    AccessKey accessKey;
    const int KeyGeneration = 4;
    int option = 0; // 共通鍵
    const char KeySourceForKekForSaveDataTransfer[4][KeySize] =
    {
        { 0x04, 0x9b, 0x89, 0x88, 0xec, 0x11, 0x26, 0xd2, 0xd8, 0x5f, 0xfb, 0xb1, 0xef, 0xcd, 0x81, 0x73 }, // savedata transfer token
        { 0x06, 0x8b, 0x69, 0x0b, 0xdb, 0x23, 0x87, 0xd2, 0x9f, 0xf2, 0xbb, 0x5d, 0x5f, 0x18, 0x31, 0x55 }, // savedata transfer data
        { 0x3a, 0x76, 0xae, 0x4b, 0x00, 0xce, 0x0b, 0x38, 0x6d, 0xf6, 0xe7, 0xbb, 0xb2, 0xee, 0xe8, 0x8e }, // cloud backup token
        { 0x45, 0x38, 0x4a, 0x98, 0xc7, 0x3a, 0xd7, 0x16, 0x98, 0x91, 0xaf, 0x85, 0x99, 0x8b, 0x30, 0xe9 }, // cloud backup data
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKek(&accessKey, KeySourceForKekForSaveDataTransfer[index], KeySize, KeyGeneration, option));

    switch (keyIndex)
    {
        case nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::SaveDataTransferToken:
        {
            const char KeySourceForKeyForSaveDataTransferToken[KeySize] = { 0xa2, 0x6a, 0xb1, 0x24, 0xfd, 0xca, 0x9e, 0xd9, 0x4b, 0xfc, 0x71, 0x82, 0x43, 0x72, 0x1e, 0x2a };
            NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(pBuffer, bufferSize, accessKey, KeySourceForKeyForSaveDataTransferToken, KeySize));
            return;
        }
        case nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::SaveDataTransfer:
        {
            NN_ABORT_UNLESS_EQUAL(keySourceSize, static_cast<size_t>(KeySize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(pBuffer, bufferSize, accessKey, pKeySource, keySourceSize));
            return;
        }
        case nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpToken:
        {
            const char KeySourceForKeyForCloudBackUpToken[KeySize] = { 0x36, 0xe7, 0x9e, 0xa7, 0xe1, 0x5d, 0x00, 0xae, 0x0e, 0x04, 0x4b, 0x53, 0x98, 0x58, 0xab, 0x12 };
            NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(pBuffer, bufferSize, accessKey, KeySourceForKeyForCloudBackUpToken, KeySize));
            return;
        }
        case nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::CloudBackUpData:
        {
            NN_ABORT_UNLESS_EQUAL(keySourceSize, static_cast<size_t>(KeySize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(GenerateAesKey(pBuffer, bufferSize, accessKey, pKeySource, keySourceSize));
            return;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }

}


