﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/socket/private/session.h>
#include <nn/socket/private/thread.h>

// Private bsd headers
#include <sys/socket.h>
#include <siglo/thread.h>

#include "detail/bsdsocket_CreateClientByDfc.h"

namespace nn     {
namespace socket {
namespace detail {
namespace        {

// Core BSD stack memory
uint8_t g_CoreStackMemoryPool[4 * 1024 * 1024] NN_ALIGNAS(os::StackRegionAlignment);

/*
 * Server class
 */
template<int sessionCountMax, const char* servicePortName, int coreStackPermissionGroupId, int threadCount, typename MyServerOptions>
class SocketServer
    : public nn::sf::HipcSimpleAllInOneServerManager<sessionCountMax, 1, MyServerOptions>
{
private:
#ifdef NN_SDK_BUILD_DEBUG
    static const size_t  SessionThreadStackSize = 1024 * 16;
#else
    static const size_t  SessionThreadStackSize = 1024 * 8;
#endif
    typedef nn::sf::HipcSimpleAllInOneServerManager<sessionCountMax, 1, MyServerOptions> MyServer;
    NetworkThread m_SessionThread[threadCount];
    uint8_t       m_SessionThreadStack[threadCount][SessionThreadStackSize] NN_ALIGNAS(os::StackRegionAlignment);

    static void SessionThreadEntry(void* arg) NN_NOEXCEPT
    {
        reinterpret_cast<SocketServer*>(arg)->LoopAuto();
    }

    virtual Result OnNeedsToAccept(int portIndex, typename MyServer::AcceptTarget* pPort) NN_NOEXCEPT NN_OVERRIDE final
    {
        NN_UNUSED(portIndex);

        // create a new service object for each client
        return this->AcceptImpl(pPort, detail::CreateClientByDfc(coreStackPermissionGroupId));
    }

public:
    void Initialize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(this->InitializePort(0, sessionCountMax, servicePortName));
        for (int i = 0; i < threadCount; i++)
        {
            int result = CreateNetworkThread(
                                &m_SessionThread[i],
                                SessionThreadEntry,
                                this,
                                m_SessionThreadStack[i],
                                SessionThreadStackSize,
                                NN_SYSTEM_THREAD_PRIORITY(socket, CoreStackIpcServer),
                                -1);
            NN_ABORT_UNLESS(result == 0);
            SetNetworkThreadNamePointer(&m_SessionThread[i], NN_SYSTEM_THREAD_NAME(socket, CoreStackIpcServer));
            StartNetworkThread(&m_SessionThread[i]);
        }
    }
    void Wait() NN_NOEXCEPT
    {
        boottimebin.sec = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetSeconds();

        for (int i = 0; i < threadCount; i++)
        {
            WaitNetworkThread(&m_SessionThread[i]);
        }
        // Should not reach here since not all threads exit
    }
};


/*
 * Configure and instantiate User server
 */
struct UserServerOption
{
    // maximum number of client process
    static const int SubDomainCountMax = 16;

    // maximum number of service objects per client process
    static const int ObjectInSubDomainCountMax = 33;

    // buffer size for pointer transfers
    static const size_t PointerTransferBufferSize = 4 * 1024;
};
SocketServer <
    64,                          // sessionCountMax
    SocketServiceUserPortName,   // servicePortName
    1,                           // core stack permission group: typical user
    22,                          // threadCount
    UserServerOption             // HipcSimpleAllInOneServerManager Options
    > g_UserServer;

/*
 * Configure and instantiate System server
 */
struct SystemServerOption
{
    // maximum number of client process
    static const int SubDomainCountMax = 16;

    // maximum number of service objects per client process
    static const int ObjectInSubDomainCountMax = 16;

    // buffer size for pointer transfers
    static const size_t PointerTransferBufferSize = 4 * 1024;
};
SocketServer <
    64,                          // sessionCountMax
    SocketServiceSystemPortName, // servicePortName
    0,                           // core stack permission group: system
    11,                          // threadCount
    SystemServerOption           // HipcSimpleAllInOneServerManager Options
    > g_SystemServer;


}} // end detail and anonymous namespaces

void InitializeServer()
NN_NOEXCEPT
{
    // Initialize core BSD stack
    NetworkInit(detail::g_CoreStackMemoryPool, sizeof(detail::g_CoreStackMemoryPool));

    // Initialize User server
    nn::socket::detail::g_SystemServer.Initialize();

    // Initialize User server
    nn::socket::detail::g_UserServer.Initialize();
}

void Wait()
NN_NOEXCEPT
{
    nn::socket::detail::g_UserServer.Wait();
    nn::socket::detail::g_SystemServer.Wait();
}

}}
