﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// USB eth configuration manager
// will be replaced by NIFM

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>
#include <nn/eth/eth_EthClient.h>
#include <nn/socket/socket_Api.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>

#include "bsdsocket_Ethernet.config.h"

namespace nn     {
namespace eth    {
namespace client {

const size_t ClientStackSize = (16 * 1024);

class Client
{
private:
    uint8_t                                    m_Stack[ClientStackSize] NN_ALIGNAS(nn::os::ThreadStackAlignment);
    nn::os::MultiWaitType                      m_MultiWait;
    nn::os::MultiWaitHolderType                m_InterfaceGroupHolder;
    nn::os::ThreadType                         m_ClientThread;
    nn::eth::client::InterfaceGroupHandler*    m_EthernetInterfaceGroup;
    bool                                       m_Running;
    nn::eth::InterfaceList                     m_CurrentInterfaceList;

    // Link group or event handler into multi wait
    template <typename T>
    void SetupWaitHolder(nn::os::MultiWaitHolderType* holder, T* impl)
    NN_NOEXCEPT
    {
        nn::os::InitializeMultiWaitHolder(holder,  reinterpret_cast<nn::os::SystemEventType*>(impl->GetSystemEventPointer()));
        nn::os::SetMultiWaitHolderUserData(holder, reinterpret_cast<uintptr_t>(impl));
        nn::os::LinkMultiWaitHolder(&m_MultiWait, holder);
    }

    // Handle adapter insertion/removal
    void HandleInterfaceGroupEvent()
    NN_NOEXCEPT
    {
        nn::Result result;
        nn::eth::InterfaceList removedList;
        nn::eth::InterfaceList addedList;
        nn::eth::InterfaceList newList;

        m_EthernetInterfaceGroup->ClearEvent();

        result = m_EthernetInterfaceGroup->GetInterfaceList(&newList);
        NN_ABORT_UNLESS(result.IsSuccess());

        // parse and update local interface list
        m_EthernetInterfaceGroup->ParseInterfaceList(&newList, &m_CurrentInterfaceList, &addedList, &removedList);
        m_CurrentInterfaceList = newList;

        // handle removals
        for (uint32_t i = 0; i < removedList.adapterCount; i++)
        {
            NN_SDK_LOG("Removed:\n"
                   "[%s]model:    %08x\n"
                   "[%s]revision: %08x\n"
                   "[%s]mac:      %02x:%02x:%02x:%02x:%02x:%02x\n",
                    removedList.adapterInfo[i].interfaceName,
                    removedList.adapterInfo[i].interfaceModel,
                    removedList.adapterInfo[i].interfaceName,
                    removedList.adapterInfo[i].interfaceRevision,
                    removedList.adapterInfo[i].interfaceName,
                    removedList.adapterInfo[i].interfaceMacAddress[0],
                    removedList.adapterInfo[i].interfaceMacAddress[1],
                    removedList.adapterInfo[i].interfaceMacAddress[2],
                    removedList.adapterInfo[i].interfaceMacAddress[3],
                    removedList.adapterInfo[i].interfaceMacAddress[4],
                    removedList.adapterInfo[i].interfaceMacAddress[5]
            );

            // deallocations will be done inside link state handler on
            // an event containing ResultInterfaceRemoved() result
        }

        // handle additions
        for (uint32_t i = 0; i < addedList.adapterCount; i++)
        {
            nn::eth::client::InterfaceHandler* ethernetInterface;
            nn::os::MultiWaitHolderType*          ethernetInterfaceHolder;

            NN_SDK_LOG("Added:\n"
                   "[%s]model:    %08x\n"
                   "[%s]revision: %08x\n"
                   "[%s]mac:      %02x:%02x:%02x:%02x:%02x:%02x\n",
                    addedList.adapterInfo[i].interfaceName,
                    addedList.adapterInfo[i].interfaceModel,
                    addedList.adapterInfo[i].interfaceName,
                    addedList.adapterInfo[i].interfaceRevision,
                    addedList.adapterInfo[i].interfaceName,
                    addedList.adapterInfo[i].interfaceMacAddress[0],
                    addedList.adapterInfo[i].interfaceMacAddress[1],
                    addedList.adapterInfo[i].interfaceMacAddress[2],
                    addedList.adapterInfo[i].interfaceMacAddress[3],
                    addedList.adapterInfo[i].interfaceMacAddress[4],
                    addedList.adapterInfo[i].interfaceMacAddress[5]
            );

            // create interface handler for new interface,
            // add its event to multi wait

            ethernetInterface = new nn::eth::client::InterfaceHandler;
            NN_ABORT_UNLESS(ethernetInterface != nullptr);

            result = ethernetInterface->Initialize(m_EthernetInterfaceGroup, &addedList.adapterInfo[i], nn::os::EventClearMode_ManualClear);
            NN_ABORT_UNLESS(result.IsSuccess());

            ethernetInterfaceHolder = new nn::os::MultiWaitHolderType;
            NN_ABORT_UNLESS(ethernetInterfaceHolder != nullptr);

            SetupWaitHolder<nn::eth::client::InterfaceHandler>(ethernetInterfaceHolder, ethernetInterface);

            // request auto negotiation
            result = ethernetInterface->SetMediaType(nn::eth::MediaType_AUTO);
            NN_ABORT_UNLESS(result.IsSuccess());

        }
    }

    // Configure IP
    void ConfigureInterface(const char* interfaceName, bool enable)
    {
        nn::Result result;

        if (enable)
        {
            nn::bsdsocket::cfg::IfSettings ifcfg;
            ifcfg = bsdsocket::cfg::DefaultIfSettings;

            #if STATIC_IP
            // static ip
            ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
            nn::socket::InetAton(STATIC_IP_ADDR, &ifcfg.u.modeStatic.addr);
            nn::socket::InetAton(STATIC_GW_ADDR, &ifcfg.u.modeStatic.gatewayAddr);
            nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
            ifcfg.u.modeStatic.broadcastAddr.s_addr =
                (ifcfg.u.modeStatic.addr.s_addr & ifcfg.u.modeStatic.subnetMask.s_addr) |
                    ~ifcfg.u.modeStatic.subnetMask.s_addr;
            nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
            nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);
            #else
            // dhcp
            ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
            #endif

            result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
            if (result.IsFailure())
            {
                NN_SDK_LOG("failed to configure interface %s - %d:%d\n",
                            interfaceName,
                            result.GetModule(),
                            result.GetDescription());
            }
        } else {
            nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
        }
    }

    // Handle link state changes
    void HandleInterfaceMediaEvent(nn::os::MultiWaitHolderType* holder)
    NN_NOEXCEPT
    {
        nn::eth::MediaType current;
        nn::eth::MediaType requested;
        nn::Result result;
        uint32_t   eventCounter;

        nn::eth::client::InterfaceHandler* ethernetInterface =
                reinterpret_cast<nn::eth::client::InterfaceHandler*>
                                    (nn::os::GetMultiWaitHolderUserData(holder));

        // clear this before processing so that if
        // event happens while processing is in progress
        // we would not miss it.
        ethernetInterface->ClearEvent();

        if (ethernetInterface->GetResult() <= ResultInterfaceRemoved())
        {
            NN_SDK_LOG("[%s] Adapter removed\n", ethernetInterface->GetInterfaceName());
            ConfigureInterface(ethernetInterface->GetInterfaceName(), false);
            nn::os::UnlinkMultiWaitHolder(holder);
            delete holder;
            delete ethernetInterface;
        }
        else
        {
            // obtain interface state
            result = ethernetInterface->GetMediaType(&requested, &current, &eventCounter);
            NN_ABORT_UNLESS(result.IsSuccess());

            if (!EthLinkUp(current))
            {
                NN_SDK_LOG("[%s] Lost link\n", ethernetInterface->GetInterfaceName());
                ConfigureInterface(ethernetInterface->GetInterfaceName(), false);
            } else {
                NN_SDK_LOG("[%s] Link: %s, Speed: %d, Fullduplex: %s, Flow: %s, Type: %s\n",
                    ethernetInterface->GetInterfaceName(),
                    EthLinkUp(current)     ? "Up"   : "Down",
                    EthSpeed(current),
                    EthFullDuplex(current) ? "On"   : "Off",
                    EthFlow(current)       ? "On"   : "Off",
                    EthAuto(current)       ? "Auto" : "Manual");

                ConfigureInterface(ethernetInterface->GetInterfaceName(), true);
            }
        }
    }

    void ClientThread()
    NN_NOEXCEPT
    {
        nn::Result result;

        m_CurrentInterfaceList.adapterCount = 0;

        m_EthernetInterfaceGroup = new nn::eth::client::InterfaceGroupHandler;
        NN_ABORT_UNLESS(m_EthernetInterfaceGroup != nullptr);

        // open session to the server
        result = m_EthernetInterfaceGroup->Initialize(nn::os::EventClearMode_ManualClear);
        NN_ABORT_UNLESS(result.IsSuccess());

        // add group interface handler to multi wait
        nn::os::InitializeMultiWait(&m_MultiWait);
        SetupWaitHolder<nn::eth::client::InterfaceGroupHandler>(&m_InterfaceGroupHolder, m_EthernetInterfaceGroup);

        while (m_Running)
        {
            // wait for group change or link state change
            nn::os::MultiWaitHolderType* pendingEventHolder = nn::os::WaitAny(&m_MultiWait);

            if (pendingEventHolder == &m_InterfaceGroupHolder)
            {
                // interface was added or removed
                HandleInterfaceGroupEvent();
            }
            else
            {
                // link state has changed
                HandleInterfaceMediaEvent(pendingEventHolder);
            }
        }
    }

    static void ClientThreadEntry(void* arg)
    NN_NOEXCEPT
    {
        Client* p = (Client*)arg;
        p->ClientThread();
    }

public:

    void Initialize()
    NN_NOEXCEPT
    {
        m_Running = true;
        nn::os::CreateThread(&m_ClientThread, ClientThreadEntry, this, m_Stack, sizeof(m_Stack), nn::os::DefaultThreadPriority);
        nn::os::StartThread(&m_ClientThread);
    }

    void Wait()
    NN_NOEXCEPT
    {
        nn::os::WaitThread(&m_ClientThread);
    }
};

static void EthClientInit()
{
#if !DISABLE_IF
    static Client s_Client;
    s_Client.Initialize();
#endif
}

}}}

