﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>

#include <nn/os.h>
#include <nn/tsc/tsc_Types.h>

// Debug flag -------------------------------------------------------------------------------------
//#define NN_DETAIL_TSC_UTIL_ENABLE_DBG_PRINT
// ------------------------------------------------------------------------------------------------

#ifdef NN_DETAIL_TSC_UTIL_ENABLE_DBG_PRINT
#define NN_DETAIL_TSC_PRINT_DBG(format, ...)  NN_SDK_LOG("****** [TSC]" format, ##__VA_ARGS__)
#else
#define NN_DETAIL_TSC_PRINT_DBG(format, ...)
#endif
#define NN_DETAIL_TSC_PRINT_WARN(format, ...) NN_SDK_LOG("[TSC]" format, ##__VA_ARGS__)

#define NN_DETAIL_TSC_RETURN_ON_ERROR(result)                                                     \
    if(result.IsFailure())                                                                        \
    {                                                                                             \
        NN_DETAIL_TSC_PRINT_DBG("%s returning upon error.\n", __FUNCTION__);                      \
        return result;                                                                            \
    }

#define NN_DETAIL_TSC_RETURN_ON_INITIALIZED_STATE()                                               \
    if(g_pObjStackConfig != nullptr)                                                              \
    {                                                                                             \
        return nn::ResultSuccess();                                                               \
    }

#define NN_DETAIL_TSC_RETURN_ON_UNINITIALIZED_STATE()                                             \
    if(g_pObjStackConfig == nullptr)                                                              \
    {                                                                                             \
        NN_DETAIL_TSC_PRINT_DBG("TSC library is not initialzed yet.\n");                          \
        return tsc::ResultLibraryNotInitialized();                                                \
    }

#define NN_DETAIL_TSC_GET_CONFIG_OR_RETURN_ON_UNINITIALIZED_STATE(pConfigOut)                     \
    do                                                                                            \
    {                                                                                             \
        pConfigOut = g_pObjStackConfig;                                                           \
        if(pConfigOut == nullptr)                                                                 \
        {                                                                                         \
            NN_DETAIL_TSC_PRINT_DBG("TSC library is not initialzed yet.\n");                      \
            return tsc::ResultLibraryNotInitialized();                                            \
        }                                                                                         \
    } while( NN_STATIC_CONDITION(false) )

#define NN_DETAIL_TSC_GET_MIN(param1, param2)      ((param1 > param2)?(param2):(param1))
#define NN_DETAIL_TSC_GET_THREAD_STACK_SIZE()      ((nn::os::StackRegionAlignment < 4096)?        \
                                                       (4096):(nn::os::StackRegionAlignment))

namespace nn { namespace tsc {
namespace detail {
class TcpIpStackConfigBase;
}
extern class detail::TcpIpStackConfigBase* g_pObjStackConfig;
extern nn::os::Mutex g_initMutex;

namespace detail {

// ------------------------------------------------------------------------------------------------
// Internal state class for asynchronous operation management
// ------------------------------------------------------------------------------------------------
class TscInternalState
{
public:
    enum AsyncState
    {
        AsyncState_Invalid,    // Not initialized
        AsyncState_Completed,  // Previous operation was completed
        AsyncState_Init,       // Initialized
        AsyncState_InProgress, // Operation in progress
        AsyncState_Done,       // Operation is already done
        AsyncState_Aborted,    // Operation aborted
        AsyncState_Timeout     // Operation timeout
    };

private:
    AsyncState m_AsyncState;

public:
    TscInternalState()
    {
        m_AsyncState = AsyncState_Invalid;
    }

    virtual ~TscInternalState()
    {
    }

    void       SetAsyncState(AsyncState state) {m_AsyncState = state;}
    AsyncState GetAsyncState(){return m_AsyncState;}
};

// ------------------------------------------------------------------------------------------------
// Worker thread class
// ------------------------------------------------------------------------------------------------
class TscWorker
{
    typedef nn::Result (*OperationFunction)();
    typedef void       (*DoneCallbackFunction)();

    static const int s_WorkerThreadStackSize = NN_DETAIL_TSC_GET_THREAD_STACK_SIZE();

    NN_ALIGNAS(nn::os::StackRegionAlignment) char m_WorkerThreadStack[s_WorkerThreadStackSize];
    nn::os::ThreadType                            m_WorkerThread;
    OperationFunction                             m_pOperation;
    DoneCallbackFunction                          m_pDoneCallback;
    bool                                          m_IsOperationInprogress;
    bool                                          m_IsThreadInit;
    bool                                          m_IsInitialized;
    nn::Result                                    m_Result;

    static void ExecuteWorker(void* arg);
    nn::Result ExecuteCallback(void);

public:
    TscWorker(); // Constructor

    nn::Result Initialize(OperationFunction pInOpFunction, DoneCallbackFunction pInDoneCallback);
    nn::Result Finalize();
    nn::Result Start();
    nn::Result GetResult(nn::Result* pOutResult);
};

class ScopedMutexLock
{
    NN_DISALLOW_COPY(ScopedMutexLock);

public:
    explicit ScopedMutexLock(nn::os::Mutex& unlockedMutex) NN_NOEXCEPT
        : m_mutex(unlockedMutex)
    {
        m_mutex.Lock();
    }

    ~ScopedMutexLock() NN_NOEXCEPT
    {
        m_mutex.Unlock();
    }

private:
    nn::os::Mutex& m_mutex;
};

// ------------------------------------------------------------------------------------------------
// APIs to verify configuration
// ------------------------------------------------------------------------------------------------
inline Result VerifyIpConfigMethod(Ipv4ConfigMethod method)
{
    if ((method < Ipv4ConfigMethod_Static) || (method > Ipv4ConfigMethod_Dhcp))
    {
        return tsc::ResultIpv4ConfigMethodInvalid();
    }
    return nn::ResultSuccess();
}

inline Result VerifyInterfaceAddress(Ipv4AddrStorage address)
{
    // Verify address
    // TODO: More accurate validation may be required
    if(address.storage == 0)
    {
        return nn::tsc::ResultIpv4AddressInvalid();
    }
    return nn::ResultSuccess();
}

inline Result VerifySubnetMask(Ipv4AddrStorage address)
{
    // Verify address
    // TODO: More accurate validation may be required
    if(address.storage == 0)
    {
        return nn::tsc::ResultIpv4SubnetMaskInvalid();
    }
    return nn::ResultSuccess();
}

inline Result VerifyDefaultGateway(Ipv4AddrStorage address)
{
    // Verify address
    // TODO: More accurate validation may be required
    if(address.storage == 0)
    {
        return nn::tsc::ResultIpv4DefaultGatewayInvalid();
    }
    return nn::ResultSuccess();
}

inline Result VerifyPreferredDnsAddress(Ipv4AddrStorage address)
{
    // Verify address
    // TODO: More accurate validation may be required
    if(address.storage == 0xffffff)
    {
        return nn::tsc::ResultIpv4PreferredDnsInvalid();
    }
    return nn::ResultSuccess();
}

inline Result VerifyAlternativeDnsAddress(Ipv4AddrStorage address)
{
    // Verify address
    // TODO: More accurate validation may be required
    if(address.storage == 0xffffff)
    {
        return nn::tsc::ResultIpv4AlternativeDnsInvalid();
    }
    return nn::ResultSuccess();
}

inline Result VerifyMtu(uint32_t mtuValue)
{
    if((mtuValue > g_MaxMtuSize) || (mtuValue < g_MinMtuSize))
    {
        return nn::tsc::ResultMtuInvalid();
    }
    return nn::ResultSuccess();
}

inline Result VerifyInterfaceName(const char* pInInterfaceName)
{
    NN_SDK_REQUIRES_NOT_NULL(pInInterfaceName);

    if(pInInterfaceName[0] == '\0')
    {
        return nn::tsc::ResultInterfaceNameInvalid();
    }

    uint32_t nameLength = strnlen(pInInterfaceName, g_MaxInterfaceNameLength);
    if(nameLength > g_MaxInterfaceNameLength - 1)
    {
        return nn::tsc::ResultInterfaceNameInvalid();
    }

    return nn::ResultSuccess();
}

}}}
