﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>

#include <nn/socket/socket_Types.h>

#include <nn/os/os_SystemEvent.h>
#include <nn/bsdsocket/cfg/cfg_Result.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>

#include <nn/tsc/tsc_Result.h>
#include <nn/tsc/tsc_Types.h>
#include <nn/tsc/tsc_ConfigContexts.h>
#include <nn/tsc/tsc_ActiveConfigContext.h>

#include "tsc_TcpipStackConfigBase.h"
#include "tsc_TcpipStackConfigBsd.h"
#include "tsc_Util.h"

namespace nn { namespace tsc { namespace detail {

nn::Result ConvertErrorFromBsdcfgToTsc(nn::Result bsdCfgResult)
{
    if (bsdCfgResult.IsSuccess())
    {
        return ResultSuccess();
    }
    else if (bsdsocket::ResultIfInvalid::Includes(bsdCfgResult))
    {
        return tsc::ResultInterfaceNameInvalid();
    }
    else if (bsdsocket::ResultMtuSizeError::Includes(bsdCfgResult))
    {
        return tsc::ResultMtuInvalid();
    }
    else if (bsdsocket::ResultIfRemoved::Includes(bsdCfgResult))
    {
        // Specified interface doesn't exist anymore
        // Should be same as invalid interface name
        return tsc::ResultInterfaceNameInvalid();
    }
    else if (bsdsocket::ResultRouteError::Includes(bsdCfgResult))
    {
        // TODO: Should it return new error?
        return tsc::ResultIpv4SubnetMaskInvalid();
    }
    else if (bsdsocket::ResultIfIsUp::Includes(bsdCfgResult))
    {
        return tsc::ResultProcessAlreadyDone();
    }
    else if (bsdsocket::ResultInvalidIfAddressMode::Includes(bsdCfgResult))
    {
        return tsc::ResultIpv4ConfigMethodInvalid();
    }
    else if (bsdsocket::ResultInvalidIfAddress::Includes(bsdCfgResult))
    {
        return tsc::ResultIpv4AddressInvalid();
    }
    else if (bsdsocket::ResultMaximumExceeded::Includes(bsdCfgResult))
    {
        return tsc::ResultMaxInterfaces();
    }
    else if (bsdsocket::ResultBusy::Includes(bsdCfgResult))
    {
        return tsc::ResultResourceBusy();
    }
    else if (bsdsocket::ResultCancelled::Includes(bsdCfgResult))
    {
        return tsc::ResultProcessAborted();
    }
    else if (bsdsocket::ResultIfSendError::Includes(bsdCfgResult))
    {
        return tsc::ResultInterfaceNameInvalid();
    }
    else if (bsdsocket::ResultDuplicateIp::Includes(bsdCfgResult))
    {
        return tsc::ResultDuplicateIp();
    }
    else if (bsdsocket::ResultInvalidReissuedLease::Includes(bsdCfgResult))
    {
        return tsc::ResultInvalidReissuedLease();
    }
    else if (bsdsocket::ResultArpEntryNotFound::Includes(bsdCfgResult))
    {
        return tsc::ResultArpEntryNotFound();
    }
    else if (bsdsocket::ResultIpv4AddressSizeInvalid::Includes(bsdCfgResult))
    {
        return tsc::ResultIpv4AddressSizeInvalid();
    }
    else if (bsdsocket::ResultMacAddressSizeInvalid::Includes(bsdCfgResult))
    {
        return tsc::ResultMacAddressSizeInvalid();
    }
    else if (bsdsocket::ResultInvalidPacket::Includes(bsdCfgResult))
    {
        return tsc::ResultInvalidResponse();
    }
    else
    {
        NN_DETAIL_TSC_PRINT_DBG("[%s]Got unexpected result (Module:%d Desc:%d)\n",
            __FUNCTION__,
            bsdCfgResult.GetModule(), bsdCfgResult.GetDescription());
        return tsc::ResultInvalidInternalLogic();
    }
}

namespace
{
    const int32_t       g_HardCodedMetric = 0;

    // --------------------------------------------------------------------------------------------
    // Local class declarations
    // --------------------------------------------------------------------------------------------
    class BsdIpv4ActiveConfig
    {
    private:
        nn::bsdsocket::cfg::IfState m_Config;
    public:
        ~BsdIpv4ActiveConfig()                          {}
        BsdIpv4ActiveConfig()                           {memset(&m_Config, 0x00, sizeof(m_Config));}
        nn::bsdsocket::cfg::IfState* GetConfigPointer() {return &m_Config;}
        nn::socket::InAddrT GetInterfaceAddress()                 {return m_Config.addr.S_addr;}
        nn::socket::InAddrT GetSubnetMask()                       {return m_Config.subnetMask.S_addr;}
        nn::socket::InAddrT GetDefaultGateway()                   {return m_Config.gatewayAddr.S_addr;}
        nn::socket::InAddrT GetPreferredDns()                     {return m_Config.dnsAddrs[0].S_addr;}
        nn::socket::InAddrT GetAlternativeDns()                   {return m_Config.dnsAddrs[1].S_addr;}
        uint32_t  GetMtu()                              {return m_Config.mtu;}

        void GetInterfaceStatistics(ActiveConfigContext::InterfaceStatistics* pOutStats)
        {
            pOutStats->dhcp.renewedLeaseCount = m_Config.u.modeDhcp.renewedLeaseCount;
            pOutStats->dhcp.reboundLeaseCount = m_Config.u.modeDhcp.reboundLeaseCount;
            pOutStats->interfaceStatus        = ConvertErrorFromBsdcfgToTsc(m_Config.status);
        }
        Result UpdateConfig(const char* pInInterfaceName)
        {
            // TODO: Caching for less IPC?
            return nn::bsdsocket::cfg::GetIfState(pInInterfaceName, &m_Config);
        }
    };

    class BsdIpv4Config
    {
    private:
        nn::bsdsocket::cfg::IfSettings m_Config;
    public:
        ~BsdIpv4Config() {}
        BsdIpv4Config()  {m_Config = bsdsocket::cfg::DefaultIfSettings;}
        nn::bsdsocket::cfg::IfSettings* GetConfigPointer() {return &m_Config;}

        nn::socket::InAddr GetInterfaceAddress()                  {return m_Config.u.modeStatic.addr;}
        nn::socket::InAddr GetSubnetMask()                        {return m_Config.u.modeStatic.subnetMask;}
        nn::socket::InAddr GetDefaultGateway()                    {return m_Config.u.modeStatic.gatewayAddr;}
        nn::socket::InAddr GetPreferredDns()                      {return m_Config.dnsAddrs[0];}
        nn::socket::InAddr GetAlternativeDns()                    {return m_Config.dnsAddrs[1];}
        uint32_t  GetDuplicateIpWaitTime()             {return m_Config.duplicateIpWaitTime; }
        uint32_t  GetMtu()                             {return m_Config.mtu;}
        void SetInterfaceAddress(Ipv4AddrStorage addr) {m_Config.u.modeStatic.addr.S_addr = addr.storage;}
        void SetSubnetMask(Ipv4AddrStorage addr)       {m_Config.u.modeStatic.subnetMask.S_addr = addr.storage;}
        void SetDefaultGateway(Ipv4AddrStorage addr)   {m_Config.u.modeStatic.gatewayAddr.S_addr = addr.storage;}
        void SetPreferredDns(Ipv4AddrStorage addr)     {m_Config.dnsAddrs[0].S_addr = addr.storage;}
        void SetAlternativeDns(Ipv4AddrStorage addr)   {m_Config.dnsAddrs[1].S_addr = addr.storage;}
        void SetDuplicateIpWaitTime(uint32_t dupeIpWaitTime) {m_Config.duplicateIpWaitTime = dupeIpWaitTime; }
        void SetMtu(uint32_t mtu)                      {m_Config.mtu = mtu;}
        void SetConfigMethod(Ipv4ConfigMethod method)
        {
            if(method == Ipv4ConfigMethod_Static)
            {
                m_Config.isManualDns = true;
                m_Config.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
            }
            else if (method == Ipv4ConfigMethod_DhcpHybrid)
            {
                m_Config.isManualDns = true;
                m_Config.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
            }
            else
            {
                m_Config.isManualDns = false;
                m_Config.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
            }
        }
        void SetIpInfo(Ipv4Config* pInIpInfo)
        {
            NN_SDK_REQUIRES_NOT_NULL(pInIpInfo);
            SetInterfaceAddress(pInIpInfo->interfaceAddress);
            SetSubnetMask(pInIpInfo->subnetMask);
            SetDefaultGateway(pInIpInfo->defaultGateway);
            SetPreferredDns(pInIpInfo->preferredDns);
            SetAlternativeDns(pInIpInfo->alternativeDns);
            SetConfigMethod(pInIpInfo->method);

            m_Config.metric = g_HardCodedMetric; // TODO: Support setting metric?
        }
        const uint32_t DuplicateIpWaitTimeMs = 1000;
    };

    // --------------------------------------------------------------------------------------------
    // Local objects
    // --------------------------------------------------------------------------------------------
    BsdIpv4Config       g_BsdConfig;
    BsdIpv4ActiveConfig g_BsdActiveConfig;
    TscWorker           g_ApplyIpConfigWorker;
    Ipv4ConfigContext*  g_pIpv4ConfigContext;
    bool                g_IsConfigurationInProgress;
    nn::os::Event       g_applyConfigDone(nn::os::EventClearMode_ManualClear);

    // --------------------------------------------------------------------------------------------
    // Local functions
    // --------------------------------------------------------------------------------------------

    // Function for worker thread
    Result ExecuteIfUp()
    {
        char interfaceName[g_MaxInterfaceNameLength] = {0};
        nn::Result result;

        {
            detail::ScopedMutexLock mutexLock(g_initMutex);
            if( nullptr == g_pIpv4ConfigContext )
            {
                NN_DETAIL_TSC_PRINT_DBG("ExecuteIfUp: g_pIpv4ConfigContext == nullptr\n");
                return nn::tsc::ResultContextInvalid();
            }

            result = g_pIpv4ConfigContext->GetInterfaceName(interfaceName, g_MaxInterfaceNameLength);
        }

        NN_TSC_ASSERT_AND_RETURN_ERROR(result.IsSuccess(), result);

        NN_DETAIL_TSC_PRINT_DBG("Call SetIfUp for if:%s\n", interfaceName);

        if (g_pIpv4ConfigContext->IsUseSystemEvent())
        {
            NN_DETAIL_TSC_PRINT_DBG(" Call SetIfUp with system event\n");
            result = nn::bsdsocket::cfg::SetIfUp(
                interfaceName,
                g_BsdConfig.GetConfigPointer(),
                g_pIpv4ConfigContext->GetSystemEventPointer(),
                *g_pIpv4ConfigContext->GetEventClearModePointer());
        }
        else
        {
            result = nn::bsdsocket::cfg::SetIfUp(interfaceName, g_BsdConfig.GetConfigPointer());
        }
        NN_DETAIL_TSC_PRINT_DBG("[%s]SetIfUp result (Module:%d Desc:%d)\n",
                                __FUNCTION__, result.GetModule(), result.GetDescription());

        result = ConvertErrorFromBsdcfgToTsc(result);

        {
            detail::ScopedMutexLock mutexLock(g_initMutex);
            if( nullptr == g_pIpv4ConfigContext )
            {
                NN_DETAIL_TSC_PRINT_DBG("ExecuteIfUp: g_pIpv4ConfigContext == nullptr\n");
                return nn::tsc::ResultContextInvalid();
            }

            g_pIpv4ConfigContext->SetApplyResult(result);
        }

        return result;
    }

    // Called after ExecuteIfUp returns
    void SetIfUpCallback()
    {


        { // Scope for mutex lock
            detail::ScopedMutexLock mutexLock(g_initMutex);
            g_IsConfigurationInProgress = false;
            g_applyConfigDone.Signal();

            NN_SDK_ASSERT(g_pIpv4ConfigContext != nullptr);
            if( g_pIpv4ConfigContext )
            {
                if (g_pIpv4ConfigContext->GetMsecTimeout() > 0)
                {
                    g_pIpv4ConfigContext->GetTimerEventPointer()->Signal();
                }
                else
                {
                    g_pIpv4ConfigContext->GetEventPointer()->Signal();
                }
            }
        }
    }
} // namespace

Result TcpIpStackConfigBsd::Initialize()
{
    //
    // No initialization is needed by BsdsocketCfg library
    //
    g_pIpv4ConfigContext = nullptr;
    g_IsConfigurationInProgress = false;

    nn::Result result = g_ApplyIpConfigWorker.Initialize(ExecuteIfUp, SetIfUpCallback);
    NN_SDK_ASSERT(result.IsSuccess());

    return result;
}

Result TcpIpStackConfigBsd::Finalize()
{
    //
    // No finalization is needed by BsdsocketCfg library
    //

    // Return if there's still ongoing process
    if (g_IsConfigurationInProgress)
    {
        NN_DETAIL_TSC_PRINT_WARN("TSC library is still in the process of ApplyIpConfig.\n");
        return tsc::ResultConfigurationInProgress();
    }

    nn::Result result = TcpIpStackConfigBase::Finalize();
    NN_DETAIL_TSC_RETURN_ON_ERROR(result);

    result = g_ApplyIpConfigWorker.Finalize();
    NN_TSC_ASSERT_AND_RETURN_ERROR(result.IsSuccess(), result);

    g_pIpv4ConfigContext = nullptr;
    g_IsConfigurationInProgress = false;
    return result;
}

Result TcpIpStackConfigBsd::CleanupConfigContext(const Ipv4ConfigContext* pInContext)
{
    if(g_pIpv4ConfigContext == nullptr)
    {
        NN_DETAIL_TSC_PRINT_DBG("No context is found in the library.\n");
        return nn::tsc::ResultNoContext();
    }

    // We check before the mutex lock so that we don't block if a different
    //  context is being used.
    if(g_pIpv4ConfigContext != pInContext)
    {
        NN_DETAIL_TSC_PRINT_DBG("Invalid context was passed (inernal:%p passed:%p).\n",
                      g_pIpv4ConfigContext, pInContext);
        return nn::tsc::ResultContextInvalid();
    }

    { // Scope for mutex lock
        detail::ScopedMutexLock mutexLock(g_initMutex);

        // We must check again after we lock just in case
        //  a different context was set.
        if(g_pIpv4ConfigContext != pInContext)
        {
            NN_DETAIL_TSC_PRINT_DBG("Invalid context was passed (inernal:%p passed:%p).\n",
                      g_pIpv4ConfigContext, pInContext);
            return nn::tsc::ResultContextInvalid();
        }

        if( g_IsConfigurationInProgress )
        {
            char interfaceName[g_MaxInterfaceNameLength] = {0};
            nn::Result result = g_pIpv4ConfigContext->GetInterfaceName(interfaceName, g_MaxInterfaceNameLength);
            if( result.IsFailure() )
            {
                NN_DETAIL_TSC_PRINT_DBG("Failed to get interface name.\n");
                return nn::tsc::ResultInternalModuleFailed();
            }

            NN_DETAIL_TSC_PRINT_DBG("Configuration in progress. Canceling configuration...\n");
            nn::bsdsocket::cfg::CancelIf(interfaceName);
        }

        g_pIpv4ConfigContext = nullptr;
    } // Mutex lock scope

    NN_DETAIL_TSC_PRINT_DBG("Cleared context.\n");
    return nn::ResultSuccess();
}

Result TcpIpStackConfigBsd::NotifyInterfaceDown(const char* pInInterfaceName)
{
    NN_SDK_REQUIRES_NOT_NULL(pInInterfaceName);

    NN_DETAIL_TSC_PRINT_DBG("[%s]Call SetIfDown for if:%s\n", __FUNCTION__, pInInterfaceName);
    nn::Result result = nn::bsdsocket::cfg::SetIfDown(pInInterfaceName);

    // ConvertErrorFromBsdcfgToTsc is not used because retrned result from SetIfDown
    // has different meanings

    if(bsdsocket::ResultBusy::Includes(result))
    {
        NN_DETAIL_TSC_PRINT_DBG("[%s]SetIfDown for if:%s failed due to ongoing process\n",
                                __FUNCTION__, pInInterfaceName);
        result = tsc::ResultConfigurationInProgress();
    }
    else if(bsdsocket::ResultIfInvalid::Includes(result))
    {
        NN_DETAIL_TSC_PRINT_DBG("[%s]SetIfDown for if:%s failed due to invalid name\n",
                                __FUNCTION__, pInInterfaceName);
        result = tsc::ResultInterfaceNotReady();
    }
    else
    {
        // Let ConvertErrorFromBsdcfgToTsc convert the error because program comes here
        // when unexpected error is returned
        result = ConvertErrorFromBsdcfgToTsc(result);
    }

    return result;
}

// ------------------------------------------------------------------------------------------------
// Functions for apply/cancel configuration process
// ------------------------------------------------------------------------------------------------
Result TcpIpStackConfigBsd::ApplyIpConfig(Ipv4ConfigContext* pInContext, int modeMask)
{
    //
    // Ipv4ConfigContext information is already verified in Ipv4ConfigContext::ApplyConfig
    //
    NN_SDK_REQUIRES_NOT_NULL(pInContext);

    { // Scope for mutex lock
        detail::ScopedMutexLock mutexLock(g_initMutex);

        if( nullptr == g_pObjStackConfig )
        {
            NN_DETAIL_TSC_PRINT_DBG("TSC library is not initialzed yet.\n");
            return tsc::ResultLibraryNotInitialized();
        }

        if( g_IsConfigurationInProgress )
        {
            NN_DETAIL_TSC_PRINT_DBG("TSC configuration already in progress.\n");
            return tsc::ResultConfigurationInProgress();
        }

        g_IsConfigurationInProgress = true;
        g_pIpv4ConfigContext = pInContext;
    } // Mutex lock scope

    // Setup BSD config object
    Ipv4Config tmpConfig;
    nn::Result result = pInContext->GetConfig(&tmpConfig);
    NN_TSC_ASSERT_AND_RETURN_ERROR(result.IsSuccess(), result);

    g_BsdConfig.SetIpInfo(&tmpConfig);

    uint32_t mtuValue;
    result = pInContext->GetMtu(&mtuValue);
    NN_TSC_ASSERT_AND_RETURN_ERROR(result.IsSuccess(), result);

    g_BsdConfig.SetMtu(mtuValue);
    if ((modeMask & IpApplyModeMask_DuplicateIpCheck))
    {
        g_BsdConfig.SetDuplicateIpWaitTime(g_BsdConfig.DuplicateIpWaitTimeMs);
    }
    else
    {
        //Setting to zero disables the Duplicate Ip Check.
        uint32_t dupeIpWaitTimeMs = 0;
        g_BsdConfig.SetDuplicateIpWaitTime(dupeIpWaitTimeMs);
    }

    if(pInContext->GetMsecTimeout() > 0)
    {
        // Start timer if it is required
        nn::TimeSpan maxWaitTime = nn::TimeSpan::FromMilliSeconds(pInContext->GetMsecTimeout());
        pInContext->GetTimerEventPointer()->StartOneShot(maxWaitTime);
    }

    // Start worker thread
    result = g_ApplyIpConfigWorker.Start();
    if(result.IsFailure())
    {
        g_IsConfigurationInProgress = false;
        return ResultInternalModuleFailed();
    }

    pInContext->SetApplyResult(tsc::ResultConfigurationInProgress());

    return ResultSuccess();
}

Result TcpIpStackConfigBsd::CancelApplyIpConfig()
{
    nn::Result result = nn::ResultSuccess();
    char       interfaceName[g_MaxInterfaceNameLength] = {0};

    {
        detail::ScopedMutexLock mutexLock(g_initMutex);
        if( nullptr == g_pIpv4ConfigContext || !g_IsConfigurationInProgress )
        {
            return tsc::ResultNoOngoingProcess();
        }

        result = g_pIpv4ConfigContext->GetInterfaceName(interfaceName, g_MaxInterfaceNameLength);
        NN_TSC_ASSERT_AND_RETURN_ERROR(result.IsSuccess(), result);

        NN_DETAIL_TSC_PRINT_DBG("Aborting SetIfUp for if:%s\n", interfaceName);
        result = nn::bsdsocket::cfg::CancelIf(interfaceName);
        if(result.IsFailure())
        {
            NN_DETAIL_TSC_PRINT_DBG("CancelIf() failed (module:%d desc:%d).\n",
                                    result.GetModule(), result.GetDescription());
            if(bsdsocket::ResultIfInvalid::Includes(result))
            {
                // This means there's no interface in the process of configuration
                // (logically same as no process to cancel)
                result = tsc::ResultNoOngoingProcess();
            }
            else if(bsdsocket::ResultIfIsUp::Includes(result))
            {
                result = tsc::ResultProcessAlreadyDone();
            }
            else
            {
                result = tsc::ResultInternalModuleFailed();
            }
        }

        g_applyConfigDone.Clear();
    }

    // Wait for Configuration to be fully canceled.
    g_applyConfigDone.Wait();

    return result;
}

Result TcpIpStackConfigBsd::CheckDuplicateIp(int msecTimeout)
{
    // Nothing to do with CTR socket
    NN_DETAIL_TSC_PRINT_WARN("Duplicate IP check is not implemented yet.\n");

    return nn::ResultSuccess();
}

// ------------------------------------------------------------------------------------------------
// Functions for active IP information
// ------------------------------------------------------------------------------------------------
Result TcpIpStackConfigBsd::GetActiveInterfaceAddress(
    const char* pInInterfaceName,
    Ipv4AddrStorage* pOutAddress)
{
    NN_UNUSED(pInInterfaceName);
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    VerifyInterfaceName(pInInterfaceName);

    nn::Result result = g_BsdActiveConfig.UpdateConfig(pInInterfaceName);
    if(result.IsSuccess())
    {
        pOutAddress->storage = g_BsdActiveConfig.GetInterfaceAddress();
        return nn::ResultSuccess();
    }

    return result;
}

Result TcpIpStackConfigBsd::GetActiveSubnetMask(
    const char* pInInterfaceName,
    Ipv4AddrStorage* pOutAddress)
{
    NN_UNUSED(pInInterfaceName);
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    VerifyInterfaceName(pInInterfaceName);

    nn::Result result = g_BsdActiveConfig.UpdateConfig(pInInterfaceName);
    if(result.IsSuccess())
    {
        pOutAddress->storage = g_BsdActiveConfig.GetSubnetMask();
        return nn::ResultSuccess();
    }

    return result;
}

Result TcpIpStackConfigBsd::GetActiveDefaultGateway(
    const char* pInInterfaceName,
    Ipv4AddrStorage* pOutAddress)
{
    NN_UNUSED(pInInterfaceName);
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    VerifyInterfaceName(pInInterfaceName);

    nn::Result result = g_BsdActiveConfig.UpdateConfig(pInInterfaceName);
    if(result.IsSuccess())
    {
        pOutAddress->storage = g_BsdActiveConfig.GetDefaultGateway();;
        return nn::ResultSuccess();
    }

    return result;
}

Result TcpIpStackConfigBsd::GetActivePreferredDns(
    const char* pInInterfaceName,
    Ipv4AddrStorage* pOutAddress)
{
    NN_UNUSED(pInInterfaceName);
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    VerifyInterfaceName(pInInterfaceName);

    nn::Result result = g_BsdActiveConfig.UpdateConfig(pInInterfaceName);
    if(result.IsSuccess())
    {
        pOutAddress->storage = g_BsdActiveConfig.GetPreferredDns();;
        return nn::ResultSuccess();
    }

    return result;
}

Result TcpIpStackConfigBsd::GetActiveAlternativeDns(
    const char* pInInterfaceName,
    Ipv4AddrStorage* pOutAddress)
{
    NN_UNUSED(pInInterfaceName);
    NN_SDK_REQUIRES_NOT_NULL(pOutAddress);
    VerifyInterfaceName(pInInterfaceName);

    nn::Result result = g_BsdActiveConfig.UpdateConfig(pInInterfaceName);
    if(result.IsSuccess())
    {
        pOutAddress->storage = g_BsdActiveConfig.GetAlternativeDns();;
        return nn::ResultSuccess();
    }

    return result;
}

Result TcpIpStackConfigBsd::GetActiveMtu(
    const char* pInInterfaceName,
    uint32_t* pOutMtuValue)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMtuValue);
    VerifyInterfaceName(pInInterfaceName);

    nn::Result result = g_BsdActiveConfig.UpdateConfig(pInInterfaceName);
    if(result.IsSuccess())
    {
        *pOutMtuValue = g_BsdActiveConfig.GetMtu();;
        return nn::ResultSuccess();
    }

    return nn::ResultSuccess();
}

Result TcpIpStackConfigBsd::GetInterfaceStatistics(
    const char* pInInterfaceName,
    ActiveConfigContext::InterfaceStatistics* pOutStats)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStats);
    VerifyInterfaceName(pInInterfaceName);

    nn::Result result = g_BsdActiveConfig.UpdateConfig(pInInterfaceName);
    if(result.IsSuccess())
    {
        g_BsdActiveConfig.GetInterfaceStatistics(pOutStats);
    }

    return result;
}

}}}
